"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role, props) {
    var _a, _b;
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        _role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    _role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = ((_a = props.endpointConfigProps) === null || _a === void 0 ? void 0 : _a.productionVariants)[0].acceleratorType;
        if (acceleratorType !== undefined) {
            _role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    _role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    _role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefaultPolicy = (_b = _role.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    utils_1.addCfnSuppressRules(roleDefaultPolicy, [
        {
            id: 'W12',
            reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
        },
        {
            id: 'W76',
            reason: 'Complex role becuase Sagemaker needs permissions to access several services',
        }
    ]);
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined &&
                ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            /** return [endpoint, endpointConfig, model] */
            return deploySagemakerEndpoint(scope, props);
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return [props.existingSagemakerEndpointObj];
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependsOn(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependsOn(endpointConfig);
        return [endpoint, endpointConfig, model];
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        utils_1.addCfnSuppressRules(cfnSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig);
        // Override default model properties
        finalModelProps = utils_1.overrideProps(finalModelProps, modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Get default EndpointConfig props
    finalEndpointConfigProps = sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId);
    // Overwrite default EndpointConfig properties
    if (endpointConfigProps) {
        finalEndpointConfigProps = utils_1.overrideProps(finalEndpointConfigProps, endpointConfigProps);
    }
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Get default Endpoint props
    finalEndpointProps = sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName);
    // Overwrite default Endpoint properties
    if (endpointProps) {
        finalEndpointProps = utils_1.overrideProps(finalEndpointProps, endpointProps);
    }
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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