"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    var _a, _b;
    let logGroup;
    let _smProps;
    // If they sent a logGroup in stateMachineProps
    if ((_a = stateMachineProps.logs) === null || _a === void 0 ? void 0 : _a.destination) {
        logGroup = (_b = stateMachineProps.logs) === null || _b === void 0 ? void 0 : _b.destination;
        _smProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!(consolidatedLogGroupProps === null || consolidatedLogGroupProps === void 0 ? void 0 : consolidatedLogGroupProps.logGroupName)) {
            const logGroupPrefix = '/aws/vendedlogs/states/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    utils_1.addCfnSuppressRules(cfnDefaultPolicy, [
        {
            id: 'W12',
            reason: `The 'LogDelivery' actions do not support resource-level authorizations`
        }
    ]);
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RlcC1mdW5jdGlvbi1oZWxwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGVwLWZ1bmN0aW9uLWhlbHBlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUlILHFDQUFxQztBQUNyQyx1REFBdUQ7QUFDdkQsa0RBQWtEO0FBQ2xELG1DQUFtRjtBQUNuRix3Q0FBd0M7QUFDeEMsc0RBQXNEO0FBQ3RELCtFQUE4RDtBQUU5RDs7OztHQUlHO0FBQ0gsU0FBZ0IsaUJBQWlCLENBQUMsS0FBb0IsRUFBRSxpQkFBd0MsRUFDOUYsYUFBa0M7O0lBRWxDLElBQUksUUFBd0IsQ0FBQztJQUM3QixJQUFJLFFBQVEsQ0FBQztJQUViLCtDQUErQztJQUMvQyxVQUFJLGlCQUFpQixDQUFDLElBQUksMENBQUUsV0FBVyxFQUFFO1FBQ3ZDLFFBQVEsU0FBRyxpQkFBaUIsQ0FBQyxJQUFJLDBDQUFFLFdBQVcsQ0FBQztRQUMvQyxRQUFRLEdBQUcsaUJBQWlCLENBQUM7S0FDOUI7U0FBTTtRQUNMLHNCQUFzQjtRQUN0Qiw4RUFBOEU7UUFDOUUsdUZBQXVGO1FBQ3ZGLDZFQUE2RTtRQUM3RSxJQUFJLHlCQUF5QixHQUFHLGFBQWEsQ0FBQztRQUU5QyxJQUFJLENBQUMseUJBQXlCLEVBQUU7WUFDOUIseUJBQXlCLEdBQUcsRUFBRSxDQUFDO1NBQ2hDO1FBQ0QsSUFBSSxFQUFDLHlCQUF5QixhQUF6Qix5QkFBeUIsdUJBQXpCLHlCQUF5QixDQUFFLFlBQVksQ0FBQSxFQUFFO1lBQzVDLE1BQU0sY0FBYyxHQUFHLHlCQUF5QixDQUFDO1lBQ2pELE1BQU0scUJBQXFCLEdBQUcsR0FBRyxHQUFHLGNBQWMsQ0FBQyxNQUFNLENBQUM7WUFDMUQsTUFBTSxTQUFTLEdBQWE7Z0JBQzFCLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVM7Z0JBQzdCLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDYixpQkFBaUIsQ0FBYyw0Q0FBNEM7YUFDNUUsQ0FBQztZQUVGLE1BQU0sWUFBWSxHQUFHLGNBQWMsR0FBRyw0QkFBb0IsQ0FBQyxTQUFTLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUM3Rix5QkFBeUIsR0FBRyxxQkFBYSxDQUFDLHlCQUF5QixFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztTQUN4RjtRQUVELGtFQUFrRTtRQUNsRSxRQUFRLEdBQUcsMkNBQWEsQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUUseUJBQXlCLENBQUMsQ0FBQztRQUVuRixxREFBcUQ7UUFDckQsUUFBUSxHQUFHLHFCQUFhLENBQUMsVUFBVSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7S0FDNUY7SUFFRCxtRUFBbUU7SUFDbkUsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDbEUsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFhLENBQUM7SUFDcEQsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBNkIsQ0FBQztJQUVqRyw2REFBNkQ7SUFDN0QsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsbUNBQW1DLEVBQ3RFO1FBQ0Usd0JBQXdCO1FBQ3hCLHFCQUFxQjtRQUNyQix3QkFBd0I7UUFDeEIsd0JBQXdCO1FBQ3hCLHdCQUF3QjtLQUN6QixDQUFDLENBQUM7SUFFTCx1RUFBdUU7SUFDdkUsMkJBQW1CLENBQUMsZ0JBQWdCLEVBQUU7UUFDcEM7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSx3RUFBd0U7U0FDakY7S0FDRixDQUFDLENBQUM7SUFFSCwrRUFBK0U7SUFDL0UsR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDMUMsT0FBTyxFQUFFO1lBQ1Asd0JBQXdCO1lBQ3hCLCtCQUErQjtZQUMvQix3QkFBd0I7U0FDekI7UUFDRCxTQUFTLEVBQUUsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxTQUFTLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUM7S0FDdkYsQ0FBQyxDQUFDLENBQUM7SUFFSixPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0FBQ3pCLENBQUM7QUExRUQsOENBMEVDO0FBRUQsU0FBZ0IseUJBQXlCLENBQUMsS0FBb0IsRUFBRSxFQUFvQjtJQUNsRixvQ0FBb0M7SUFDcEMsTUFBTSxNQUFNLEdBQXVCLElBQUksS0FBSyxFQUFFLENBQUM7SUFFL0Msb0ZBQW9GO0lBQ3BGLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxzQkFBc0IsRUFBRTtRQUM5RCxNQUFNLEVBQUUsRUFBRSxDQUFDLFlBQVksRUFBRTtRQUN6QixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLEtBQUs7UUFDaEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDhFQUE4RTtLQUNqRyxDQUFDLENBQUMsQ0FBQztJQUVKLDJGQUEyRjtJQUMzRixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUU7UUFDakUsTUFBTSxFQUFFLEVBQUUsQ0FBQyxlQUFlLEVBQUU7UUFDNUIsU0FBUyxFQUFFLENBQUM7UUFDWixpQkFBaUIsRUFBRSxDQUFDO1FBQ3BCLFNBQVMsRUFBRSxLQUFLO1FBQ2hCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7UUFDakMsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLGtDQUFrQztRQUNwRixnQkFBZ0IsRUFBRSxpRkFBaUY7S0FDcEcsQ0FBQyxDQUFDLENBQUM7SUFFSixxRkFBcUY7SUFDckYsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLHVCQUF1QixFQUFFO1FBQy9ELE1BQU0sRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFO1FBQzFCLFNBQVMsRUFBRSxDQUFDO1FBQ1osaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixTQUFTLEVBQUUsU0FBUztRQUNwQixNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1FBQ2pDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxrQ0FBa0M7UUFDcEYsZ0JBQWdCLEVBQUUsK0VBQStFO0tBQ2xHLENBQUMsQ0FBQyxDQUFDO0lBRUosT0FBTyxNQUFNLENBQUM7QUFDaEIsQ0FBQztBQXRDRCw4REFzQ0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc21EZWZhdWx0cyBmcm9tICcuL3N0ZXAtZnVuY3Rpb24tZGVmYXVsdHMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMsIGdlbmVyYXRlUmVzb3VyY2VOYW1lLCBhZGRDZm5TdXBwcmVzc1J1bGVzIH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IGJ1aWxkTG9nR3JvdXAgfSBmcm9tICcuL2Nsb3Vkd2F0Y2gtbG9nLWdyb3VwLWhlbHBlcic7XG5cbi8qKlxuICogQnVpbGRzIGFuZCByZXR1cm5zIGEgU3RhdGVNYWNoaW5lLlxuICogQHBhcmFtIHNjb3BlIC0gdGhlIGNvbnN0cnVjdCB0byB3aGljaCB0aGUgU3RhdGVNYWNoaW5lIHNob3VsZCBiZSBhdHRhY2hlZCB0by5cbiAqIEBwYXJhbSBzdGF0ZU1hY2hpbmVQcm9wcyAtIHVzZXItc3BlY2lmaWVkIHByb3BlcnRpZXMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcGVydGllcy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkU3RhdGVNYWNoaW5lKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBzdGF0ZU1hY2hpbmVQcm9wczogc2ZuLlN0YXRlTWFjaGluZVByb3BzLFxuICBsb2dHcm91cFByb3BzPzogbG9ncy5Mb2dHcm91cFByb3BzKTogW3Nmbi5TdGF0ZU1hY2hpbmUsIGxvZ3MuSUxvZ0dyb3VwXSB7XG5cbiAgbGV0IGxvZ0dyb3VwOiBsb2dzLklMb2dHcm91cDtcbiAgbGV0IF9zbVByb3BzO1xuXG4gIC8vIElmIHRoZXkgc2VudCBhIGxvZ0dyb3VwIGluIHN0YXRlTWFjaGluZVByb3BzXG4gIGlmIChzdGF0ZU1hY2hpbmVQcm9wcy5sb2dzPy5kZXN0aW5hdGlvbikge1xuICAgIGxvZ0dyb3VwID0gc3RhdGVNYWNoaW5lUHJvcHMubG9ncz8uZGVzdGluYXRpb247XG4gICAgX3NtUHJvcHMgPSBzdGF0ZU1hY2hpbmVQcm9wcztcbiAgfSBlbHNlIHtcbiAgICAvLyBUaHJlZSBwb3NzaWJpbGl0aWVzXG4gICAgLy8gMSkgbG9nR3JvdXBQcm9wcyBub3QgcHJvdmlkZWQgLSBjcmVhdGUgbG9nR3JvdXBQcm9wcyB3aXRoIGp1c3QgbG9nR3JvdXBOYW1lXG4gICAgLy8gMikgbG9nR3JvdXBQcm9wcyBwcm92aWRlZCB3aXRoIG5vIGxvZ0dyb3VwTmFtZSAtIG92ZXJyaWRlIGxvZ0dyb3VwUHJvcHMubG9nR3JvdXBOYW1lXG4gICAgLy8gMykgbG9nR3JvdXBQcm9wcyBwcm92aWRlZCB3aXRoIGxvZ0dyb3VwTmFtZSAtIHBhc3MgdW5hbHRlcmVkIGxvZ0dyb3VwUHJvcHNcbiAgICBsZXQgY29uc29saWRhdGVkTG9nR3JvdXBQcm9wcyA9IGxvZ0dyb3VwUHJvcHM7XG5cbiAgICBpZiAoIWNvbnNvbGlkYXRlZExvZ0dyb3VwUHJvcHMpIHtcbiAgICAgIGNvbnNvbGlkYXRlZExvZ0dyb3VwUHJvcHMgPSB7fTtcbiAgICB9XG4gICAgaWYgKCFjb25zb2xpZGF0ZWRMb2dHcm91cFByb3BzPy5sb2dHcm91cE5hbWUpIHtcbiAgICAgIGNvbnN0IGxvZ0dyb3VwUHJlZml4ID0gJy9hd3MvdmVuZGVkbG9ncy9zdGF0ZXMvJztcbiAgICAgIGNvbnN0IG1heFJlc291cmNlTmFtZUxlbmd0aCA9IDI1NSAtIGxvZ0dyb3VwUHJlZml4Lmxlbmd0aDtcbiAgICAgIGNvbnN0IG5hbWVQYXJ0czogc3RyaW5nW10gPSBbXG4gICAgICAgIGNkay5TdGFjay5vZihzY29wZSkuc3RhY2tOYW1lLCAvLyBOYW1lIG9mIHRoZSBzdGFja1xuICAgICAgICBzY29wZS5ub2RlLmlkLCAgICAgICAgICAgICAgICAgLy8gQ29uc3RydWN0IElEXG4gICAgICAgICdTdGF0ZU1hY2hpbmVMb2cnICAgICAgICAgICAgICAvLyBMaXRlcmFsIHN0cmluZyBmb3IgbG9nIGdyb3VwIG5hbWUgcG9ydGlvblxuICAgICAgXTtcblxuICAgICAgY29uc3QgbG9nR3JvdXBOYW1lID0gbG9nR3JvdXBQcmVmaXggKyBnZW5lcmF0ZVJlc291cmNlTmFtZShuYW1lUGFydHMsIG1heFJlc291cmNlTmFtZUxlbmd0aCk7XG4gICAgICBjb25zb2xpZGF0ZWRMb2dHcm91cFByb3BzID0gb3ZlcnJpZGVQcm9wcyhjb25zb2xpZGF0ZWRMb2dHcm91cFByb3BzLCB7IGxvZ0dyb3VwTmFtZSB9KTtcbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgbmV3IENsb3Vkd2F0Y2ggbG9nIGdyb3VwIGZvciBTdGVwIGZ1bmN0aW9uIFN0YXRlIE1hY2hpbmVcbiAgICBsb2dHcm91cCA9IGJ1aWxkTG9nR3JvdXAoc2NvcGUsICdTdGF0ZU1hY2hpbmVMb2dHcm91cCcsIGNvbnNvbGlkYXRlZExvZ0dyb3VwUHJvcHMpO1xuXG4gICAgLy8gT3ZlcnJpZGUgdGhlIGRlZmF1bHRzIHdpdGggdGhlIHVzZXIgcHJvdmlkZWQgcHJvcHNcbiAgICBfc21Qcm9wcyA9IG92ZXJyaWRlUHJvcHMoc21EZWZhdWx0cy5EZWZhdWx0U3RhdGVNYWNoaW5lUHJvcHMobG9nR3JvdXApLCBzdGF0ZU1hY2hpbmVQcm9wcyk7XG4gIH1cblxuICAvLyBPdmVycmlkZSB0aGUgQ2xvdWR3YXRjaCBwZXJtaXNzaW9ucyB0byBtYWtlIGl0IG1vcmUgZmluZSBncmFpbmVkXG4gIGNvbnN0IF9zbSA9IG5ldyBzZm4uU3RhdGVNYWNoaW5lKHNjb3BlLCAnU3RhdGVNYWNoaW5lJywgX3NtUHJvcHMpO1xuICBjb25zdCByb2xlID0gX3NtLm5vZGUuZmluZENoaWxkKCdSb2xlJykgYXMgaWFtLlJvbGU7XG4gIGNvbnN0IGNmbkRlZmF1bHRQb2xpY3kgPSByb2xlLm5vZGUuZmluZENoaWxkKCdEZWZhdWx0UG9saWN5Jykubm9kZS5kZWZhdWx0Q2hpbGQgYXMgaWFtLkNmblBvbGljeTtcblxuICAvLyBSZWR1Y2UgdGhlIHNjb3BlIG9mIGFjdGlvbnMgZm9yIHRoZSBleGlzdGluZyBEZWZhdWx0UG9saWN5XG4gIGNmbkRlZmF1bHRQb2xpY3kuYWRkUHJvcGVydHlPdmVycmlkZSgnUG9saWN5RG9jdW1lbnQuU3RhdGVtZW50LjAuQWN0aW9uJyxcbiAgICBbXG4gICAgICBcImxvZ3M6Q3JlYXRlTG9nRGVsaXZlcnlcIixcbiAgICAgICdsb2dzOkdldExvZ0RlbGl2ZXJ5JyxcbiAgICAgICdsb2dzOlVwZGF0ZUxvZ0RlbGl2ZXJ5JyxcbiAgICAgICdsb2dzOkRlbGV0ZUxvZ0RlbGl2ZXJ5JyxcbiAgICAgICdsb2dzOkxpc3RMb2dEZWxpdmVyaWVzJ1xuICAgIF0pO1xuXG4gIC8vIE92ZXJyaWRlIENmbiBOYWcgd2FybmluZyBXMTI6IElBTSBwb2xpY3kgc2hvdWxkIG5vdCBhbGxvdyAqIHJlc291cmNlXG4gIGFkZENmblN1cHByZXNzUnVsZXMoY2ZuRGVmYXVsdFBvbGljeSwgW1xuICAgIHtcbiAgICAgIGlkOiAnVzEyJyxcbiAgICAgIHJlYXNvbjogYFRoZSAnTG9nRGVsaXZlcnknIGFjdGlvbnMgZG8gbm90IHN1cHBvcnQgcmVzb3VyY2UtbGV2ZWwgYXV0aG9yaXphdGlvbnNgXG4gICAgfVxuICBdKTtcblxuICAvLyBBZGQgYSBuZXcgcG9saWN5IHdpdGggbG9nZ2luZyBwZXJtaXNzaW9ucyBmb3IgdGhlIGdpdmVuIGNsb3Vkd2F0Y2ggbG9nIGdyb3VwXG4gIF9zbS5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgIGFjdGlvbnM6IFtcbiAgICAgICdsb2dzOlB1dFJlc291cmNlUG9saWN5JyxcbiAgICAgICdsb2dzOkRlc2NyaWJlUmVzb3VyY2VQb2xpY2llcycsXG4gICAgICAnbG9nczpEZXNjcmliZUxvZ0dyb3VwcydcbiAgICBdLFxuICAgIHJlc291cmNlczogW2Bhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06bG9nczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06KmBdXG4gIH0pKTtcblxuICByZXR1cm4gW19zbSwgbG9nR3JvdXBdO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRTdGVwRnVuY3Rpb25DV0FsYXJtcyhzY29wZTogY2RrLkNvbnN0cnVjdCwgc206IHNmbi5TdGF0ZU1hY2hpbmUpOiBjbG91ZHdhdGNoLkFsYXJtW10ge1xuICAvLyBTZXR1cCBDVyBBbGFybXMgZm9yIFN0YXRlIE1hY2hpbmVcbiAgY29uc3QgYWxhcm1zOiBjbG91ZHdhdGNoLkFsYXJtW10gPSBuZXcgQXJyYXkoKTtcblxuICAvLyBTdW0gb2YgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgaXMgPj0gMSBmb3IgNSBtaW51dGVzLCAxIGNvbnNlY3V0aXZlIHRpbWVcbiAgYWxhcm1zLnB1c2gobmV3IGNsb3Vkd2F0Y2guQWxhcm0oc2NvcGUsICdFeGVjdXRpb25GYWlsZWRBbGFybScsIHtcbiAgICBtZXRyaWM6IHNtLm1ldHJpY0ZhaWxlZCgpLFxuICAgIHRocmVzaG9sZDogMSxcbiAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMzAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0FsYXJtIGZvciB0aGUgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgZXhjZWVkZWQgdGhlIHRocmVzaG9sZCBvZiAxLiAnXG4gIH0pKTtcblxuICAvLyBTdW0gb2YgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgbWF4aW11bSBpcyA+PSAxIGZvciA1IG1pbnV0ZSwgMSBjb25zZWN1dGl2ZSB0aW1lXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnRXhlY3V0aW9uVGhyb3R0bGVkQWxhcm0nLCB7XG4gICAgbWV0cmljOiBzbS5tZXRyaWNUaHJvdHRsZWQoKSxcbiAgICB0aHJlc2hvbGQ6IDEsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwMCksXG4gICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgIGFsYXJtRGVzY3JpcHRpb246ICdBbGFybSBmb3IgdGhlIG51bWJlciBvZiBleGVjdXRpb25zIHRoYXQgdGhyb3R0bGVkIGV4Y2VlZGVkIHRoZSB0aHJlc2hvbGQgb2YgMS4gJ1xuICB9KSk7XG5cbiAgLy8gTnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBhYm9ydGVkIG1heGltdW0gaXMgPj0gMSBmb3IgNSBtaW51dGUsIDEgY29uc2VjdXRpdmUgdGltZVxuICBhbGFybXMucHVzaChuZXcgY2xvdWR3YXRjaC5BbGFybShzY29wZSwgJ0V4ZWN1dGlvbkFib3J0ZWRBbGFybScsIHtcbiAgICBtZXRyaWM6IHNtLm1ldHJpY0Fib3J0ZWQoKSxcbiAgICB0aHJlc2hvbGQ6IDEsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnTWF4aW11bScsXG4gICAgcGVyaW9kOiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQWxhcm0gZm9yIHRoZSBudW1iZXIgb2YgZXhlY3V0aW9ucyB0aGF0IGFib3J0ZWQgZXhjZWVkZWQgdGhlIHRocmVzaG9sZCBvZiAxLiAnXG4gIH0pKTtcblxuICByZXR1cm4gYWxhcm1zO1xufSJdfQ==