"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    var _a;
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    if (props === null || props === void 0 ? void 0 : props.userVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps);
    }
    if (props === null || props === void 0 ? void 0 : props.constructVpcProps) {
        cumulativeProps = utils_1.overrideProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    }
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        utils_1.addCfnSuppressRules(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
            }
        ]);
    });
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = (_a = flowLog.logGroup) === null || _a === void 0 ? void 0 : _a.node.defaultChild;
    utils_1.addCfnSuppressRules(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
        }
    ]);
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEPFUNCTIONS"] = "STEPFUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (!vpc.node.children.some((child) => child.node.id === interfaceTag)) {
        const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
        if (!service) {
            throw new Error("Unsupported Service sent to AddServiceEndpoint");
        }
        if (service.endpointType === EndpointTypes.GATEWAY) {
            vpc.addGatewayEndpoint(interfaceTag, {
                service: service.endpointGatewayService,
            });
        }
        if (service.endpointType === EndpointTypes.INTERFACE) {
            const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, `${scope.node.id}-${service.endpointName}`, {
                vpc,
                allowAllOutbound: true,
            }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
            vpc.addInterfaceEndpoint(interfaceTag, {
                service: service.endpointInterfaceService,
                securityGroups: [endpointDefaultSecurityGroup],
            });
        }
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
//# sourceMappingURL=data:application/json;base64,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