"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const dynamodb = require("@aws-cdk/aws-dynamodb");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const mediastore = require("@aws-cdk/aws-mediastore");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const mediastore_defaults_1 = require("../lib/mediastore-defaults");
const sagemaker_helper_1 = require("../lib/sagemaker-helper");
const test_helper_1 = require("./test-helper");
test('Test with valid props', () => {
    const props = {};
    defaults.CheckProps(props);
});
test('Test fail DynamoDB table check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingTableObj: new dynamodb.Table(stack, 'placeholder', defaults.DefaultTableProps),
        dynamoTableProps: defaults.DefaultTableProps,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
test("Test fail Lambda function check", () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
test("Test fail SQS Queue check", () => {
    const stack = new core_1.Stack();
    const props = {
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'placeholder', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps or existingQueueObj, but not both.\n');
});
test('Test fail Dead Letter Queue check', () => {
    const props = {
        deployDeadLetterQueue: false,
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n');
});
test('Test fail Dead Letter Queue check with no deployDeadLetterQueue flag', () => {
    const props = {
        deadLetterQueueProps: {},
    };
    //  Should not flag an error
    defaults.CheckProps(props);
});
test("Test fail MediaStore container check", () => {
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, "placeholder", mediastore_defaults_1.MediaStoreContainerProps());
    const props = {
        mediaStoreContainerProps: mediastore_defaults_1.MediaStoreContainerProps(),
        existingMediaStoreContainerObj: mediaStoreContainer,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n");
});
test('Test fail Kinesis stream check', () => {
    const stack = new core_1.Stack();
    const stream = new kinesis.Stream(stack, 'placeholder', {});
    const props = {
        existingStreamObj: stream,
        kinesisStreamProps: {}
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
test('Test fail S3 check', () => {
    const stack = new core_1.Stack();
    const props = {
        existingBucketObj: test_helper_1.CreateScrapBucket(stack, {}),
        bucketProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
test('Test fail SNS topic check', () => {
    const stack = new core_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check with bad topic attribute name', () => {
    const stack = new core_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail Glue job check', () => {
    const stack = new core_1.Stack();
    const _jobRole = new iam.Role(stack, 'CustomETLJobRole', {
        assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
    });
    const jobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: new s3.Bucket(stack, 'ScriptBucket').bucketArn,
        },
        role: new iam.Role(stack, 'JobRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
        }).roleArn
    }, 'testETLJob', {});
    const job = new glue.CfnJob(stack, 'placeholder', jobProps);
    const props = {
        glueJobProps: jobProps,
        existingGlueJob: job
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide glueJobProps or existingGlueJob, but not both.\n');
});
test('Test fail SageMaker endpoint check', () => {
    const stack = new core_1.Stack();
    // Build Sagemaker Inference Endpoint
    const modelProps = {
        primaryContainer: {
            image: "<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest",
            modelDataUrl: "s3://<bucket-name>/<prefix>/model.tar.gz",
        },
    };
    const [endpoint] = sagemaker_helper_1.BuildSagemakerEndpoint(stack, { modelProps });
    const props = {
        existingSagemakerEndpointObj: endpoint,
        endpointProps: {
            endpointConfigName: 'placeholder'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
test('Test fail Secret check', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail encryption key check', () => {
    const stack = new core_1.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        enableKeyRotation: false
    });
    const props = {
        encryptionKey: key,
        encryptionKeyProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Vpc check with deployVpc', () => {
    const stack = new core_1.Stack();
    const props = {
        deployVpc: true,
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail Vpc check with vpcProps', () => {
    const stack = new core_1.Stack();
    const props = {
        vpcProps: defaults.DefaultPublicPrivateVpcProps(),
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail multiple failures message', () => {
    const stack = new core_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n' +
        'Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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