"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const ec2 = require("@aws-cdk/aws-ec2");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment with VPC
// --------------------------------------------------------------
test('Test deployment with VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    let sagemaker;
    let vpc;
    let sg;
    // Build Sagemaker Notebook Instance
    [sagemaker, vpc, sg] = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.privateSubnets.length).toEqual(2);
    expect(vpc === null || vpc === void 0 ? void 0 : vpc.publicSubnets.length).toEqual(2);
    expect(sagemaker.instanceType).toEqual('ml.t2.medium');
    expect(sg).toBeInstanceOf(ec2.SecurityGroup);
});
// --------------------------------------------------------------
// Test deployment witout VPC
// --------------------------------------------------------------
test('Test deployment w/o VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: false,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment in existing VPC
// --------------------------------------------------------------
test('Test deployment w/ existing VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef'],
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        SecurityGroupIds: ['sg-deadbeef'],
        SubnetId: 'subnet-deadbeef',
    });
});
// --------------------------------------------------------------
// Test deployment with override
// --------------------------------------------------------------
test('Test deployment w/ override', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    const key = new kms.Key(stack, 'MyEncryptionKey');
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        sagemakerNotebookProps: {
            instanceType: 'ml.c4.2xlarge',
            kmsKeyId: key.keyArn,
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        InstanceType: 'ml.c4.2xlarge',
        KmsKeyId: {
            'Fn::GetAtt': ['MyEncryptionKeyD795679F', 'Arn'],
        },
    });
});
// ----------------------------------------------------------
// Test deployment with existing Sagemaker Notebook instance
// ----------------------------------------------------------
test('Test deployment with existing Sagemaker Notebook instance', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    const [noteBookInstance] = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        existingNotebookObj: noteBookInstance,
        role: sagemakerRole,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test exception
// --------------------------------------------------------------
test('Test exception', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    expect(() => {
        // Build Sagemaker Notebook Instance
        defaults.buildSagemakerNotebook(stack, {
            role: sagemakerRole,
            deployInsideVpc: true,
            sagemakerNotebookProps: {
                subnetId: 'subnet-deadbeef',
            },
        });
    }).toThrowError();
});
// --------------------------------------------------------------------------------------
// Test minimal deployment of Sagemaker Inference Endpoint no VPC using internal IAM role
// --------------------------------------------------------------------------------------
test('Test minimal deployment with no properties using internal IAM role', () => {
    // Stack
    const stack = new core_1.Stack();
    // Build Sagemaker Inference Endpoint
    defaults.BuildSagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// ----------------------------------------------------------------
// Test minimal deployment of Sagemaker Inference Endpoint with VPC
// ----------------------------------------------------------------
test('Test minimal deployment of Sagemaker Inference Endpoint with VPC', () => {
    // Stack
    const stack = new core_1.Stack();
    // create a VPC with required VPC S3 gateway and SAGEMAKER_RUNTIME Interface
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // Build Sagemaker Inference Endpoint
    defaults.BuildSagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        vpc,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// -------------------------------------------------------------------------
// Test deployment of Sagemaker Inference Endpoint with properties overwrite
// -------------------------------------------------------------------------
test('Test deployment of Sagemaker Inference Endpoint with properties overwrite', () => {
    // Stack
    const stack = new core_1.Stack();
    // create a VPC with required VPC S3 gateway and SAGEMAKER_RUNTIME Interface
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    // Add S3 VPC Gateway Endpint, required by Sagemaker to access Models artifacts via AWS private network
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.S3);
    // Add SAGEMAKER_RUNTIME VPC Interface Endpint, required by the lambda function to invoke the SageMaker endpoint
    defaults.AddAwsServiceEndpoint(stack, vpc, defaults.ServiceEndpointTypes.SAGEMAKER_RUNTIME);
    // create encryption key
    const encryptionkey = new kms.Key(stack, 'MyEndpointConfigEncryptionKey');
    // Build Sagemaker Inference Endpoint
    defaults.BuildSagemakerEndpoint(stack, {
        modelProps: {
            modelName: 'linear-learner-model',
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
        endpointConfigProps: {
            endpointConfigName: 'linear-learner-endpoint-config',
            productionVariants: [
                {
                    modelName: 'linear-learner-model',
                    initialInstanceCount: 1,
                    initialVariantWeight: 1.0,
                    instanceType: 'ml.m4.large',
                    variantName: 'AllTraffic',
                    acceleratorType: 'ml.eia2.medium',
                },
            ],
            kmsKeyId: encryptionkey.keyArn,
        },
        endpointProps: {
            endpointConfigName: 'linear-learner-endpoint-config',
            endpointName: 'linear-learner-endpoint',
        },
        vpc,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment of existing Sagemaker Endpoint
// --------------------------------------------------------------
test('Test deployment of existing Sagemaker Endpoint', () => {
    // Stack
    const stack = new core_1.Stack();
    const [sagemakerEndpoint] = defaults.deploySagemakerEndpoint(stack, {
        modelProps: {
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    // Build Sagemaker Inference Endpoint
    defaults.BuildSagemakerEndpoint(stack, {
        existingSagemakerEndpointObj: sagemakerEndpoint,
    });
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// ------------------------------------------------------------------------
// Test deployment of sagemaker endpoint with a customer provided role
// ------------------------------------------------------------------------
test('Test deployment of sagemaker endpoint with a customer provided role', () => {
    // Stack
    const stack = new core_1.Stack();
    // Create IAM Role to be assumed by Sagemaker
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    sagemakerRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'));
    // Build Sagemaker Inference Endpoint
    defaults.BuildSagemakerEndpoint(stack, {
        modelProps: {
            executionRoleArn: sagemakerRole.roleArn,
            primaryContainer: {
                image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
            },
        },
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// ---------------------------------------------------------------
// Test exception for not providing primaryContainer in modelProps
// ---------------------------------------------------------------
test('Test exception for not providing primaryContainer in modelProps', () => {
    // Stack
    const stack = new core_1.Stack();
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, {
            modelProps: {},
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps', () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -------------------------------------------------------------------------
// Test exception for not providing modelProps or existingSagemkaerObj
// -------------------------------------------------------------------------
test('Test exception for not providing modelProps or existingSagemkaerObj', () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
// -----------------------------------------------------------------------------------------
// Test exception for not providing private or isolated subnets in an existing vpc
// -----------------------------------------------------------------------------------------
test('Test exception for not providing private or isolated subnets in an existing vpc', () => {
    // Stack
    const stack = new core_1.Stack();
    // create a VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        userVpcProps: {
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'public',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
            ],
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, {
            modelProps: {
                primaryContainer: {
                    image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                    modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
                },
            },
            vpc,
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLWhlbHBlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2FnZW1ha2VyLWhlbHBlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCx3Q0FBc0M7QUFDdEMsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsZ0NBQWdDO0FBQ2hDLDRDQUE2QztBQUM3QyxnQ0FBOEI7QUFFOUIsaUVBQWlFO0FBQ2pFLDZDQUE2QztBQUM3QyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLDRDQUE0QyxFQUFFLEdBQUcsRUFBRTtJQUN0RCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUN6RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7S0FDL0QsQ0FBQyxDQUFDO0lBQ0gsb0NBQW9DO0lBQ3BDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDckMsSUFBSSxFQUFFLGFBQWE7S0FDcEIsQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFFSCxpRUFBaUU7QUFDakUsMkJBQTJCO0FBQzNCLGlFQUFpRTtBQUNqRSxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO0lBQ3BDLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQ3pELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyx5QkFBeUIsQ0FBQztLQUMvRCxDQUFDLENBQUM7SUFDSCxJQUFJLFNBQVMsQ0FBQztJQUNkLElBQUksR0FBRyxDQUFDO0lBQ1IsSUFBSSxFQUFFLENBQUM7SUFFUCxvQ0FBb0M7SUFDcEMsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDNUQsSUFBSSxFQUFFLGFBQWE7S0FDcEIsQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7SUFDN0QsTUFBTSxDQUFDLEdBQUcsYUFBSCxHQUFHLHVCQUFILEdBQUcsQ0FBRSxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzlDLE1BQU0sQ0FBQyxHQUFHLGFBQUgsR0FBRyx1QkFBSCxHQUFHLENBQUUsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM3QyxNQUFNLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN2RCxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztBQUMvQyxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSw2QkFBNkI7QUFDN0IsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDbkMsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUNILG9DQUFvQztJQUNwQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ3JDLElBQUksRUFBRSxhQUFhO1FBQ25CLGVBQWUsRUFBRSxLQUFLO0tBQ3ZCLENBQUMsQ0FBQztJQUNILFlBQVk7SUFDWixNQUFNLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQy9ELENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGtDQUFrQztBQUNsQyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMzQyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUN6RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7S0FDL0QsQ0FBQyxDQUFDO0lBQ0gsb0NBQW9DO0lBQ3BDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDckMsSUFBSSxFQUFFLGFBQWE7UUFDbkIsZUFBZSxFQUFFLElBQUk7UUFDckIsc0JBQXNCLEVBQUU7WUFDdEIsUUFBUSxFQUFFLGlCQUFpQjtZQUMzQixnQkFBZ0IsRUFBRSxDQUFDLGFBQWEsQ0FBQztTQUNsQztLQUNGLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUU7UUFDL0Qsb0JBQW9CLEVBQUUsVUFBVTtRQUNoQyxnQkFBZ0IsRUFBRSxDQUFDLGFBQWEsQ0FBQztRQUNqQyxRQUFRLEVBQUUsaUJBQWlCO0tBQzVCLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGdDQUFnQztBQUNoQyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUN2QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtRQUN6RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7S0FDL0QsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBQ2xELG9DQUFvQztJQUNwQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ3JDLElBQUksRUFBRSxhQUFhO1FBQ25CLHNCQUFzQixFQUFFO1lBQ3RCLFlBQVksRUFBRSxlQUFlO1lBQzdCLFFBQVEsRUFBRSxHQUFHLENBQUMsTUFBTTtTQUNyQjtLQUNGLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUU7UUFDL0Qsb0JBQW9CLEVBQUUsVUFBVTtRQUNoQyxZQUFZLEVBQUUsZUFBZTtRQUM3QixRQUFRLEVBQUU7WUFDUixZQUFZLEVBQUUsQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLENBQUM7U0FDakQ7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILDZEQUE2RDtBQUM3RCw0REFBNEQ7QUFDNUQsNkRBQTZEO0FBQzdELElBQUksQ0FBQywyREFBMkQsRUFBRSxHQUFHLEVBQUU7SUFDckUsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUNILG9DQUFvQztJQUNwQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ2hFLElBQUksRUFBRSxhQUFhO0tBQ3BCLENBQUMsQ0FBQztJQUVILG9DQUFvQztJQUNwQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ3JDLG1CQUFtQixFQUFFLGdCQUFnQjtRQUNyQyxJQUFJLEVBQUUsYUFBYTtLQUNwQixDQUFDLENBQUM7SUFDSCxZQUFZO0lBQ1osTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSxpQkFBaUI7QUFDakIsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7SUFDMUIsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixvQ0FBb0M7UUFDcEMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLEtBQUssRUFBRTtZQUNyQyxJQUFJLEVBQUUsYUFBYTtZQUNuQixlQUFlLEVBQUUsSUFBSTtZQUNyQixzQkFBc0IsRUFBRTtnQkFDdEIsUUFBUSxFQUFFLGlCQUFpQjthQUM1QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDO0FBQ3BCLENBQUMsQ0FBQyxDQUFDO0FBRUgseUZBQXlGO0FBQ3pGLHlGQUF5RjtBQUN6Rix5RkFBeUY7QUFDekYsSUFBSSxDQUFDLG9FQUFvRSxFQUFFLEdBQUcsRUFBRTtJQUM5RSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixxQ0FBcUM7SUFDckMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLEtBQUssRUFBRTtRQUNyQyxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsWUFBWSxFQUFFLDBDQUEwQzthQUN6RDtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFFSCxtRUFBbUU7QUFDbkUsbUVBQW1FO0FBQ25FLG1FQUFtRTtBQUNuRSxJQUFJLENBQUMsa0VBQWtFLEVBQUUsR0FBRyxFQUFFO0lBQzVFLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLDRFQUE0RTtJQUM1RSxNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRTtRQUNuQyxlQUFlLEVBQUUsUUFBUSxDQUFDLHVCQUF1QixFQUFFO1FBQ25ELGlCQUFpQixFQUFFO1lBQ2pCLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QjtLQUNGLENBQUMsQ0FBQztJQUVILHVHQUF1RztJQUN2RyxRQUFRLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxRQUFRLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDN0UsZ0hBQWdIO0lBQ2hILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBRTVGLHFDQUFxQztJQUNyQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1FBQ3JDLFVBQVUsRUFBRTtZQUNWLGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsa0VBQWtFO2dCQUN6RSxZQUFZLEVBQUUsMENBQTBDO2FBQ3pEO1NBQ0Y7UUFDRCxHQUFHO0tBQ0osQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFFSCw0RUFBNEU7QUFDNUUsNEVBQTRFO0FBQzVFLDRFQUE0RTtBQUM1RSxJQUFJLENBQUMsMkVBQTJFLEVBQUUsR0FBRyxFQUFFO0lBQ3JGLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLDRFQUE0RTtJQUM1RSxNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRTtRQUNuQyxlQUFlLEVBQUUsUUFBUSxDQUFDLHVCQUF1QixFQUFFO1FBQ25ELGlCQUFpQixFQUFFO1lBQ2pCLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QjtLQUNGLENBQUMsQ0FBQztJQUVILHVHQUF1RztJQUN2RyxRQUFRLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxRQUFRLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDN0UsZ0hBQWdIO0lBQ2hILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBRTVGLHdCQUF3QjtJQUN4QixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLCtCQUErQixDQUFDLENBQUM7SUFDMUUscUNBQXFDO0lBQ3JDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDckMsVUFBVSxFQUFFO1lBQ1YsU0FBUyxFQUFFLHNCQUFzQjtZQUNqQyxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsWUFBWSxFQUFFLDBDQUEwQzthQUN6RDtTQUNGO1FBQ0QsbUJBQW1CLEVBQUU7WUFDbkIsa0JBQWtCLEVBQUUsZ0NBQWdDO1lBQ3BELGtCQUFrQixFQUFFO2dCQUNsQjtvQkFDRSxTQUFTLEVBQUUsc0JBQXNCO29CQUNqQyxvQkFBb0IsRUFBRSxDQUFDO29CQUN2QixvQkFBb0IsRUFBRSxHQUFHO29CQUN6QixZQUFZLEVBQUUsYUFBYTtvQkFDM0IsV0FBVyxFQUFFLFlBQVk7b0JBQ3pCLGVBQWUsRUFBRSxnQkFBZ0I7aUJBQ2xDO2FBQ0Y7WUFDRCxRQUFRLEVBQUUsYUFBYSxDQUFDLE1BQU07U0FDL0I7UUFDRCxhQUFhLEVBQUU7WUFDYixrQkFBa0IsRUFBRSxnQ0FBZ0M7WUFDcEQsWUFBWSxFQUFFLHlCQUF5QjtTQUN4QztRQUNELEdBQUc7S0FDSixDQUFDLENBQUM7SUFDSCxZQUFZO0lBQ1osTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSxpREFBaUQ7QUFDakQsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFDMUQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsUUFBUSxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRTtRQUNsRSxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsWUFBWSxFQUFFLDBDQUEwQzthQUN6RDtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgscUNBQXFDO0lBQ3JDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLEVBQUU7UUFDckMsNEJBQTRCLEVBQUUsaUJBQWlCO0tBQ2hELENBQUMsQ0FBQztJQUNILFlBQVk7SUFDWixNQUFNLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQy9ELENBQUMsQ0FBQyxDQUFDO0FBRUgsMkVBQTJFO0FBQzNFLHNFQUFzRTtBQUN0RSwyRUFBMkU7QUFDM0UsSUFBSSxDQUFDLHFFQUFxRSxFQUFFLEdBQUcsRUFBRTtJQUMvRSxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQiw2Q0FBNkM7SUFDN0MsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDekQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO0tBQy9ELENBQUMsQ0FBQztJQUNILGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDJCQUEyQixDQUFDLENBQUMsQ0FBQztJQUV4RyxxQ0FBcUM7SUFDckMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLEtBQUssRUFBRTtRQUNyQyxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRSxhQUFhLENBQUMsT0FBTztZQUN2QyxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLGtFQUFrRTtnQkFDekUsWUFBWSxFQUFFLDBDQUEwQzthQUN6RDtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsY0FBYztJQUNkLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFFSCxrRUFBa0U7QUFDbEUsa0VBQWtFO0FBQ2xFLGtFQUFrRTtBQUNsRSxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO0lBQzNFLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLHFDQUFxQztRQUNyQyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFO1lBQ3JDLFVBQVUsRUFBRSxFQUFFO1NBQ2YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILDRFQUE0RTtBQUM1RSw4Q0FBOEM7QUFDOUMsNEVBQTRFO0FBQzVFLElBQUksQ0FBQyw2Q0FBNkMsRUFBRSxHQUFHLEVBQUU7SUFDdkQsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7UUFDbkMsZUFBZSxFQUFFLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRTtRQUNuRCxpQkFBaUIsRUFBRTtZQUNqQixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLGdCQUFnQixFQUFFLElBQUk7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixxQ0FBcUM7UUFDckMsUUFBUSxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFDbkQsQ0FBQyxDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILDRFQUE0RTtBQUM1RSxzRUFBc0U7QUFDdEUsNEVBQTRFO0FBQzVFLElBQUksQ0FBQyxxRUFBcUUsRUFBRSxHQUFHLEVBQUU7SUFDL0UsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7UUFDbkMsZUFBZSxFQUFFLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRTtRQUNuRCxpQkFBaUIsRUFBRTtZQUNqQixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLGdCQUFnQixFQUFFLElBQUk7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixxQ0FBcUM7UUFDckMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFDbEQsQ0FBQyxDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztBQUM3QixDQUFDLENBQUMsQ0FBQztBQUVILDRGQUE0RjtBQUM1RixrRkFBa0Y7QUFDbEYsNEZBQTRGO0FBQzVGLElBQUksQ0FBQyxpRkFBaUYsRUFBRSxHQUFHLEVBQUU7SUFDM0YsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFFMUIsZUFBZTtJQUNmLE1BQU0sR0FBRyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFO1FBQ25DLGVBQWUsRUFBRSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7UUFDbkQsWUFBWSxFQUFFO1lBQ1osV0FBVyxFQUFFLENBQUM7WUFDZCxtQkFBbUIsRUFBRTtnQkFDbkI7b0JBQ0UsUUFBUSxFQUFFLEVBQUU7b0JBQ1osSUFBSSxFQUFFLFFBQVE7b0JBQ2QsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTTtpQkFDbEM7YUFDRjtTQUNGO1FBQ0QsaUJBQWlCLEVBQUU7WUFDakIsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixnQkFBZ0IsRUFBRSxJQUFJO1NBQ3ZCO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YscUNBQXFDO1FBQ3JDLFFBQVEsQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUU7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLGdCQUFnQixFQUFFO29CQUNoQixLQUFLLEVBQUUsa0VBQWtFO29CQUN6RSxZQUFZLEVBQUUsMENBQTBDO2lCQUN6RDthQUNGO1lBQ0QsR0FBRztTQUNKLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztJQUNGLGNBQWM7SUFDZCxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDN0IsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICcuLi8nO1xuaW1wb3J0IHsgU3ludGhVdGlscyB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggbm8gcHJvcGVydGllc1xuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggbm8gcHJvcGVydGllcycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3Qgc2FnZW1ha2VyUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1NhZ2VtYWtlclJvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3NhZ2VtYWtlci5hbWF6b25hd3MuY29tJyksXG4gIH0pO1xuICAvLyBCdWlsZCBTYWdlbWFrZXIgTm90ZWJvb2sgSW5zdGFuY2VcbiAgZGVmYXVsdHMuYnVpbGRTYWdlbWFrZXJOb3RlYm9vayhzdGFjaywge1xuICAgIHJvbGU6IHNhZ2VtYWtlclJvbGUsXG4gIH0pO1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgd2l0aCBWUENcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgd2l0aCBWUEMnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcbiAgbGV0IHNhZ2VtYWtlcjtcbiAgbGV0IHZwYztcbiAgbGV0IHNnO1xuXG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICBbc2FnZW1ha2VyLCB2cGMsIHNnXSA9IGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICByb2xlOiBzYWdlbWFrZXJSb2xlLFxuICB9KTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spKS50b01hdGNoU25hcHNob3QoKTtcbiAgZXhwZWN0KHZwYz8ucHJpdmF0ZVN1Ym5ldHMubGVuZ3RoKS50b0VxdWFsKDIpO1xuICBleHBlY3QodnBjPy5wdWJsaWNTdWJuZXRzLmxlbmd0aCkudG9FcXVhbCgyKTtcbiAgZXhwZWN0KHNhZ2VtYWtlci5pbnN0YW5jZVR5cGUpLnRvRXF1YWwoJ21sLnQyLm1lZGl1bScpO1xuICBleHBlY3Qoc2cpLnRvQmVJbnN0YW5jZU9mKGVjMi5TZWN1cml0eUdyb3VwKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHdpdG91dCBWUENcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy9vIFZQQycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3Qgc2FnZW1ha2VyUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1NhZ2VtYWtlclJvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3NhZ2VtYWtlci5hbWF6b25hd3MuY29tJyksXG4gIH0pO1xuICAvLyBCdWlsZCBTYWdlbWFrZXIgTm90ZWJvb2sgSW5zdGFuY2VcbiAgZGVmYXVsdHMuYnVpbGRTYWdlbWFrZXJOb3RlYm9vayhzdGFjaywge1xuICAgIHJvbGU6IHNhZ2VtYWtlclJvbGUsXG4gICAgZGVwbG95SW5zaWRlVnBjOiBmYWxzZSxcbiAgfSk7XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgZGVwbG95bWVudCBpbiBleGlzdGluZyBWUENcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gZXhpc3RpbmcgVlBDJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBjb25zdCBzYWdlbWFrZXJSb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnU2FnZW1ha2VyUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgfSk7XG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICBkZWZhdWx0cy5idWlsZFNhZ2VtYWtlck5vdGVib29rKHN0YWNrLCB7XG4gICAgcm9sZTogc2FnZW1ha2VyUm9sZSxcbiAgICBkZXBsb3lJbnNpZGVWcGM6IHRydWUsXG4gICAgc2FnZW1ha2VyTm90ZWJvb2tQcm9wczoge1xuICAgICAgc3VibmV0SWQ6ICdzdWJuZXQtZGVhZGJlZWYnLFxuICAgICAgc2VjdXJpdHlHcm91cElkczogWydzZy1kZWFkYmVlZiddLFxuICAgIH0sXG4gIH0pO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlNhZ2VNYWtlcjo6Tm90ZWJvb2tJbnN0YW5jZScsIHtcbiAgICBEaXJlY3RJbnRlcm5ldEFjY2VzczogJ0Rpc2FibGVkJyxcbiAgICBTZWN1cml0eUdyb3VwSWRzOiBbJ3NnLWRlYWRiZWVmJ10sXG4gICAgU3VibmV0SWQ6ICdzdWJuZXQtZGVhZGJlZWYnLFxuICB9KTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHdpdGggb3ZlcnJpZGVcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gb3ZlcnJpZGUnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcbiAgY29uc3Qga2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdNeUVuY3J5cHRpb25LZXknKTtcbiAgLy8gQnVpbGQgU2FnZW1ha2VyIE5vdGVib29rIEluc3RhbmNlXG4gIGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICByb2xlOiBzYWdlbWFrZXJSb2xlLFxuICAgIHNhZ2VtYWtlck5vdGVib29rUHJvcHM6IHtcbiAgICAgIGluc3RhbmNlVHlwZTogJ21sLmM0LjJ4bGFyZ2UnLFxuICAgICAga21zS2V5SWQ6IGtleS5rZXlBcm4sXG4gICAgfSxcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6U2FnZU1ha2VyOjpOb3RlYm9va0luc3RhbmNlJywge1xuICAgIERpcmVjdEludGVybmV0QWNjZXNzOiAnRGlzYWJsZWQnLFxuICAgIEluc3RhbmNlVHlwZTogJ21sLmM0LjJ4bGFyZ2UnLFxuICAgIEttc0tleUlkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFsnTXlFbmNyeXB0aW9uS2V5RDc5NTY3OUYnLCAnQXJuJ10sXG4gICAgfSxcbiAgfSk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHdpdGggZXhpc3RpbmcgU2FnZW1ha2VyIE5vdGVib29rIGluc3RhbmNlXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgd2l0aCBleGlzdGluZyBTYWdlbWFrZXIgTm90ZWJvb2sgaW5zdGFuY2UnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcbiAgLy8gQnVpbGQgU2FnZW1ha2VyIE5vdGVib29rIEluc3RhbmNlXG4gIGNvbnN0IFtub3RlQm9va0luc3RhbmNlXSA9IGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICByb2xlOiBzYWdlbWFrZXJSb2xlLFxuICB9KTtcblxuICAvLyBCdWlsZCBTYWdlbWFrZXIgTm90ZWJvb2sgSW5zdGFuY2VcbiAgZGVmYXVsdHMuYnVpbGRTYWdlbWFrZXJOb3RlYm9vayhzdGFjaywge1xuICAgIGV4aXN0aW5nTm90ZWJvb2tPYmo6IG5vdGVCb29rSW5zdGFuY2UsXG4gICAgcm9sZTogc2FnZW1ha2VyUm9sZSxcbiAgfSk7XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgZXhjZXB0aW9uXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBleGNlcHRpb24nLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IHNhZ2VtYWtlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTYWdlbWFrZXJSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICB9KTtcblxuICBleHBlY3QoKCkgPT4ge1xuICAgIC8vIEJ1aWxkIFNhZ2VtYWtlciBOb3RlYm9vayBJbnN0YW5jZVxuICAgIGRlZmF1bHRzLmJ1aWxkU2FnZW1ha2VyTm90ZWJvb2soc3RhY2ssIHtcbiAgICAgIHJvbGU6IHNhZ2VtYWtlclJvbGUsXG4gICAgICBkZXBsb3lJbnNpZGVWcGM6IHRydWUsXG4gICAgICBzYWdlbWFrZXJOb3RlYm9va1Byb3BzOiB7XG4gICAgICAgIHN1Ym5ldElkOiAnc3VibmV0LWRlYWRiZWVmJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pLnRvVGhyb3dFcnJvcigpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IG1pbmltYWwgZGVwbG95bWVudCBvZiBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50IG5vIFZQQyB1c2luZyBpbnRlcm5hbCBJQU0gcm9sZVxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggbm8gcHJvcGVydGllcyB1c2luZyBpbnRlcm5hbCBJQU0gcm9sZScsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgLy8gQnVpbGQgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludFxuICBkZWZhdWx0cy5CdWlsZFNhZ2VtYWtlckVuZHBvaW50KHN0YWNrLCB7XG4gICAgbW9kZWxQcm9wczoge1xuICAgICAgcHJpbWFyeUNvbnRhaW5lcjoge1xuICAgICAgICBpbWFnZTogJzxBY2NvdW50SWQ+LmRrci5lY3IuPHJlZ2lvbj4uYW1hem9uYXdzLmNvbS9saW5lYXItbGVhcm5lcjpsYXRlc3QnLFxuICAgICAgICBtb2RlbERhdGFVcmw6ICdzMzovLzxidWNrZXQtbmFtZT4vPHByZWZpeD4vbW9kZWwudGFyLmd6JyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBtaW5pbWFsIGRlcGxveW1lbnQgb2YgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludCB3aXRoIFZQQ1xuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBtaW5pbWFsIGRlcGxveW1lbnQgb2YgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludCB3aXRoIFZQQycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAvLyBjcmVhdGUgYSBWUEMgd2l0aCByZXF1aXJlZCBWUEMgUzMgZ2F0ZXdheSBhbmQgU0FHRU1BS0VSX1JVTlRJTUUgSW50ZXJmYWNlXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIEFkZCBTMyBWUEMgR2F0ZXdheSBFbmRwaW50LCByZXF1aXJlZCBieSBTYWdlbWFrZXIgdG8gYWNjZXNzIE1vZGVscyBhcnRpZmFjdHMgdmlhIEFXUyBwcml2YXRlIG5ldHdvcmtcbiAgZGVmYXVsdHMuQWRkQXdzU2VydmljZUVuZHBvaW50KHN0YWNrLCB2cGMsIGRlZmF1bHRzLlNlcnZpY2VFbmRwb2ludFR5cGVzLlMzKTtcbiAgLy8gQWRkIFNBR0VNQUtFUl9SVU5USU1FIFZQQyBJbnRlcmZhY2UgRW5kcGludCwgcmVxdWlyZWQgYnkgdGhlIGxhbWJkYSBmdW5jdGlvbiB0byBpbnZva2UgdGhlIFNhZ2VNYWtlciBlbmRwb2ludFxuICBkZWZhdWx0cy5BZGRBd3NTZXJ2aWNlRW5kcG9pbnQoc3RhY2ssIHZwYywgZGVmYXVsdHMuU2VydmljZUVuZHBvaW50VHlwZXMuU0FHRU1BS0VSX1JVTlRJTUUpO1xuXG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBJbmZlcmVuY2UgRW5kcG9pbnRcbiAgZGVmYXVsdHMuQnVpbGRTYWdlbWFrZXJFbmRwb2ludChzdGFjaywge1xuICAgIG1vZGVsUHJvcHM6IHtcbiAgICAgIHByaW1hcnlDb250YWluZXI6IHtcbiAgICAgICAgaW1hZ2U6ICc8QWNjb3VudElkPi5ka3IuZWNyLjxyZWdpb24+LmFtYXpvbmF3cy5jb20vbGluZWFyLWxlYXJuZXI6bGF0ZXN0JyxcbiAgICAgICAgbW9kZWxEYXRhVXJsOiAnczM6Ly88YnVja2V0LW5hbWU+LzxwcmVmaXg+L21vZGVsLnRhci5neicsXG4gICAgICB9LFxuICAgIH0sXG4gICAgdnBjLFxuICB9KTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spKS50b01hdGNoU25hcHNob3QoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgb2YgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludCB3aXRoIHByb3BlcnRpZXMgb3ZlcndyaXRlXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgb2YgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludCB3aXRoIHByb3BlcnRpZXMgb3ZlcndyaXRlJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIC8vIGNyZWF0ZSBhIFZQQyB3aXRoIHJlcXVpcmVkIFZQQyBTMyBnYXRld2F5IGFuZCBTQUdFTUFLRVJfUlVOVElNRSBJbnRlcmZhY2VcbiAgY29uc3QgdnBjID0gZGVmYXVsdHMuYnVpbGRWcGMoc3RhY2ssIHtcbiAgICBkZWZhdWx0VnBjUHJvcHM6IGRlZmF1bHRzLkRlZmF1bHRJc29sYXRlZFZwY1Byb3BzKCksXG4gICAgY29uc3RydWN0VnBjUHJvcHM6IHtcbiAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IHRydWUsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gQWRkIFMzIFZQQyBHYXRld2F5IEVuZHBpbnQsIHJlcXVpcmVkIGJ5IFNhZ2VtYWtlciB0byBhY2Nlc3MgTW9kZWxzIGFydGlmYWN0cyB2aWEgQVdTIHByaXZhdGUgbmV0d29ya1xuICBkZWZhdWx0cy5BZGRBd3NTZXJ2aWNlRW5kcG9pbnQoc3RhY2ssIHZwYywgZGVmYXVsdHMuU2VydmljZUVuZHBvaW50VHlwZXMuUzMpO1xuICAvLyBBZGQgU0FHRU1BS0VSX1JVTlRJTUUgVlBDIEludGVyZmFjZSBFbmRwaW50LCByZXF1aXJlZCBieSB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRvIGludm9rZSB0aGUgU2FnZU1ha2VyIGVuZHBvaW50XG4gIGRlZmF1bHRzLkFkZEF3c1NlcnZpY2VFbmRwb2ludChzdGFjaywgdnBjLCBkZWZhdWx0cy5TZXJ2aWNlRW5kcG9pbnRUeXBlcy5TQUdFTUFLRVJfUlVOVElNRSk7XG5cbiAgLy8gY3JlYXRlIGVuY3J5cHRpb24ga2V5XG4gIGNvbnN0IGVuY3J5cHRpb25rZXkgPSBuZXcga21zLktleShzdGFjaywgJ015RW5kcG9pbnRDb25maWdFbmNyeXB0aW9uS2V5Jyk7XG4gIC8vIEJ1aWxkIFNhZ2VtYWtlciBJbmZlcmVuY2UgRW5kcG9pbnRcbiAgZGVmYXVsdHMuQnVpbGRTYWdlbWFrZXJFbmRwb2ludChzdGFjaywge1xuICAgIG1vZGVsUHJvcHM6IHtcbiAgICAgIG1vZGVsTmFtZTogJ2xpbmVhci1sZWFybmVyLW1vZGVsJyxcbiAgICAgIHByaW1hcnlDb250YWluZXI6IHtcbiAgICAgICAgaW1hZ2U6ICc8QWNjb3VudElkPi5ka3IuZWNyLjxyZWdpb24+LmFtYXpvbmF3cy5jb20vbGluZWFyLWxlYXJuZXI6bGF0ZXN0JyxcbiAgICAgICAgbW9kZWxEYXRhVXJsOiAnczM6Ly88YnVja2V0LW5hbWU+LzxwcmVmaXg+L21vZGVsLnRhci5neicsXG4gICAgICB9LFxuICAgIH0sXG4gICAgZW5kcG9pbnRDb25maWdQcm9wczoge1xuICAgICAgZW5kcG9pbnRDb25maWdOYW1lOiAnbGluZWFyLWxlYXJuZXItZW5kcG9pbnQtY29uZmlnJyxcbiAgICAgIHByb2R1Y3Rpb25WYXJpYW50czogW1xuICAgICAgICB7XG4gICAgICAgICAgbW9kZWxOYW1lOiAnbGluZWFyLWxlYXJuZXItbW9kZWwnLFxuICAgICAgICAgIGluaXRpYWxJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICAgIGluaXRpYWxWYXJpYW50V2VpZ2h0OiAxLjAsXG4gICAgICAgICAgaW5zdGFuY2VUeXBlOiAnbWwubTQubGFyZ2UnLFxuICAgICAgICAgIHZhcmlhbnROYW1lOiAnQWxsVHJhZmZpYycsXG4gICAgICAgICAgYWNjZWxlcmF0b3JUeXBlOiAnbWwuZWlhMi5tZWRpdW0nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIGttc0tleUlkOiBlbmNyeXB0aW9ua2V5LmtleUFybixcbiAgICB9LFxuICAgIGVuZHBvaW50UHJvcHM6IHtcbiAgICAgIGVuZHBvaW50Q29uZmlnTmFtZTogJ2xpbmVhci1sZWFybmVyLWVuZHBvaW50LWNvbmZpZycsXG4gICAgICBlbmRwb2ludE5hbWU6ICdsaW5lYXItbGVhcm5lci1lbmRwb2ludCcsXG4gICAgfSxcbiAgICB2cGMsXG4gIH0pO1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgb2YgZXhpc3RpbmcgU2FnZW1ha2VyIEVuZHBvaW50XG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBkZXBsb3ltZW50IG9mIGV4aXN0aW5nIFNhZ2VtYWtlciBFbmRwb2ludCcsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBbc2FnZW1ha2VyRW5kcG9pbnRdID0gZGVmYXVsdHMuZGVwbG95U2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHtcbiAgICBtb2RlbFByb3BzOiB7XG4gICAgICBwcmltYXJ5Q29udGFpbmVyOiB7XG4gICAgICAgIGltYWdlOiAnPEFjY291bnRJZD4uZGtyLmVjci48cmVnaW9uPi5hbWF6b25hd3MuY29tL2xpbmVhci1sZWFybmVyOmxhdGVzdCcsXG4gICAgICAgIG1vZGVsRGF0YVVybDogJ3MzOi8vPGJ1Y2tldC1uYW1lPi88cHJlZml4Pi9tb2RlbC50YXIuZ3onLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBCdWlsZCBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50XG4gIGRlZmF1bHRzLkJ1aWxkU2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHtcbiAgICBleGlzdGluZ1NhZ2VtYWtlckVuZHBvaW50T2JqOiBzYWdlbWFrZXJFbmRwb2ludCxcbiAgfSk7XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgb2Ygc2FnZW1ha2VyIGVuZHBvaW50IHdpdGggYSBjdXN0b21lciBwcm92aWRlZCByb2xlXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgZGVwbG95bWVudCBvZiBzYWdlbWFrZXIgZW5kcG9pbnQgd2l0aCBhIGN1c3RvbWVyIHByb3ZpZGVkIHJvbGUnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIENyZWF0ZSBJQU0gUm9sZSB0byBiZSBhc3N1bWVkIGJ5IFNhZ2VtYWtlclxuICBjb25zdCBzYWdlbWFrZXJSb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnU2FnZW1ha2VyUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgfSk7XG4gIHNhZ2VtYWtlclJvbGUuYWRkTWFuYWdlZFBvbGljeShpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNhZ2VNYWtlckZ1bGxBY2Nlc3MnKSk7XG5cbiAgLy8gQnVpbGQgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludFxuICBkZWZhdWx0cy5CdWlsZFNhZ2VtYWtlckVuZHBvaW50KHN0YWNrLCB7XG4gICAgbW9kZWxQcm9wczoge1xuICAgICAgZXhlY3V0aW9uUm9sZUFybjogc2FnZW1ha2VyUm9sZS5yb2xlQXJuLFxuICAgICAgcHJpbWFyeUNvbnRhaW5lcjoge1xuICAgICAgICBpbWFnZTogJzxBY2NvdW50SWQ+LmRrci5lY3IuPHJlZ2lvbj4uYW1hem9uYXdzLmNvbS9saW5lYXItbGVhcm5lcjpsYXRlc3QnLFxuICAgICAgICBtb2RlbERhdGFVcmw6ICdzMzovLzxidWNrZXQtbmFtZT4vPHByZWZpeD4vbW9kZWwudGFyLmd6JyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gQXNzZXJ0aW9uIDFcbiAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBleGNlcHRpb24gZm9yIG5vdCBwcm92aWRpbmcgcHJpbWFyeUNvbnRhaW5lciBpbiBtb2RlbFByb3BzXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgZXhjZXB0aW9uIGZvciBub3QgcHJvdmlkaW5nIHByaW1hcnlDb250YWluZXIgaW4gbW9kZWxQcm9wcycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgLy8gQnVpbGQgU2FnZW1ha2VyIEluZmVyZW5jZSBFbmRwb2ludFxuICAgIGRlZmF1bHRzLkJ1aWxkU2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHtcbiAgICAgIG1vZGVsUHJvcHM6IHt9LFxuICAgIH0pO1xuICB9O1xuICAvLyBBc3NlcnRpb24gMVxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBCdWlsZCBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50XG4gICAgZGVmYXVsdHMuZGVwbG95U2FnZW1ha2VyRW5kcG9pbnQoc3RhY2ssIHsgdnBjIH0pO1xuICB9O1xuICAvLyBBc3NlcnRpb24gMVxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzIG9yIGV4aXN0aW5nU2FnZW1rYWVyT2JqXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBtb2RlbFByb3BzIG9yIGV4aXN0aW5nU2FnZW1rYWVyT2JqJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IHZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBkZWZhdWx0cy5EZWZhdWx0SXNvbGF0ZWRWcGNQcm9wcygpLFxuICAgIGNvbnN0cnVjdFZwY1Byb3BzOiB7XG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgIH0sXG4gIH0pO1xuXG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICAvLyBCdWlsZCBTYWdlbWFrZXIgSW5mZXJlbmNlIEVuZHBvaW50XG4gICAgZGVmYXVsdHMuQnVpbGRTYWdlbWFrZXJFbmRwb2ludChzdGFjaywgeyB2cGMgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvbiAxXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGV4Y2VwdGlvbiBmb3Igbm90IHByb3ZpZGluZyBwcml2YXRlIG9yIGlzb2xhdGVkIHN1Ym5ldHMgaW4gYW4gZXhpc3RpbmcgdnBjXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBleGNlcHRpb24gZm9yIG5vdCBwcm92aWRpbmcgcHJpdmF0ZSBvciBpc29sYXRlZCBzdWJuZXRzIGluIGFuIGV4aXN0aW5nIHZwYycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAvLyBjcmVhdGUgYSBWUENcbiAgY29uc3QgdnBjID0gZGVmYXVsdHMuYnVpbGRWcGMoc3RhY2ssIHtcbiAgICBkZWZhdWx0VnBjUHJvcHM6IGRlZmF1bHRzLkRlZmF1bHRJc29sYXRlZFZwY1Byb3BzKCksXG4gICAgdXNlclZwY1Byb3BzOiB7XG4gICAgICBuYXRHYXRld2F5czogMCxcbiAgICAgIHN1Ym5ldENvbmZpZ3VyYXRpb246IFtcbiAgICAgICAge1xuICAgICAgICAgIGNpZHJNYXNrOiAxOCxcbiAgICAgICAgICBuYW1lOiAncHVibGljJyxcbiAgICAgICAgICBzdWJuZXRUeXBlOiBlYzIuU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgY29uc3RydWN0VnBjUHJvcHM6IHtcbiAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IHRydWUsXG4gICAgfSxcbiAgfSk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIC8vIEJ1aWxkIFNhZ2VtYWtlciBJbmZlcmVuY2UgRW5kcG9pbnRcbiAgICBkZWZhdWx0cy5kZXBsb3lTYWdlbWFrZXJFbmRwb2ludChzdGFjaywge1xuICAgICAgbW9kZWxQcm9wczoge1xuICAgICAgICBwcmltYXJ5Q29udGFpbmVyOiB7XG4gICAgICAgICAgaW1hZ2U6ICc8QWNjb3VudElkPi5ka3IuZWNyLjxyZWdpb24+LmFtYXpvbmF3cy5jb20vbGluZWFyLWxlYXJuZXI6bGF0ZXN0JyxcbiAgICAgICAgICBtb2RlbERhdGFVcmw6ICdzMzovLzxidWNrZXQtbmFtZT4vPHByZWZpeD4vbW9kZWwudGFyLmd6JyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICB2cGMsXG4gICAgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvbiAxXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigpO1xufSk7XG4iXX0=