import asyncio
from typing import Optional, Protocol

from .abstract import AbstractService
from .container import ServiceContainerMixin
from .exceptions import UnhealthyException
from .tasks import TasksMixin
from .bus import ServiceBusMixin, ServiceBus


class WithServiceBus(Protocol):
    def set_service_bus(self, bus: ServiceBus): ...


class Service(ServiceBusMixin, ServiceContainerMixin, TasksMixin, AbstractService):
    """Base service class.

    Your services should be inherited from this class.

    Service class provides a basic asynchronous service lifecycle methods.
    You should extend it for your needs.
    """
    _monitoring_task: Optional[asyncio.Task] = None
    #: interval in seconds to sleep between healthcheck runs
    _monitoring_interval: float = .1

    def __init__(self, *, loop=None, bus: ServiceBus = None, monitoring_interval: float = .1):
        self._loop = loop
        self._monitoring_interval = monitoring_interval
        super().__init__()
        self.set_service_bus(bus)

    async def start(self):
        """Start service.

        Set `running` flag to `True`, start service tasks, nested services
        and create monitoring task.

        You can override this method in your service implementation to apply custom
        start logic. But don't forget to invoke super implementation.
        """
        self.log.debug("Starting")
        self.running = True
        self.log.debug("Starting service tasks...")
        await self._start_service_tasks()
        try:
            self.log.debug("Starting nested services...")
            await self._start_nested_services()
        except Exception:
            self.log.exception("Failed to start nested service")
            self.running = False
            self.should_stop = True
            await self._stop_service_tasks()
            raise
        self._monitoring_task = self.loop.create_task(self.monitoring_task(),
                                                      name=f"{self.name}.monitoring_task")
        await self.start_bus_reader()
        self.log.debug("Service was started")

    async def stop(self):
        """Stop service.

        Set `should_stop` flag to `True`, `running` to `False` and start shutdown sequence.

        Nested services will be stopped first, service tasks will be cancelled than.

        You can override this method in your service implementation to apply custom
        start logic. But don't forget to invoke super implementation.
        """
        self.should_stop = True
        self.running = False
        await self.stop_bus_reader()
        if self._monitoring_task:
            self._monitoring_task.cancel()
        self.log.debug("Stopping nested services...")
        await self._stop_nested_services()
        self.log.debug("Stopping service tasks...")
        await self._stop_service_tasks()
        self.log.debug("Service was stopped")

    async def monitoring_task(self):
        """Monitoring task.

        Started with a service. Run healthcheck periodically and force service
        to stop if it failed.
        """
        while not self.should_stop:
            try:
                await self.healthcheck()
            except UnhealthyException:
                self.log.exception("Healthcheck failed with exception")
                break
            except Exception:  # noqa
                self.log.exception("Service healthcheck failed with unexpected exception")
                break
            await asyncio.sleep(self._monitoring_interval)
        # terminate service on exit
        if not self.should_stop:
            await self.stop()

    def nested_service_pre_start(self, service: WithServiceBus):
        if not self.bus:  # pragma: no cover
            raise RuntimeError("Service bus is not initialized")
        service.set_service_bus(self.bus)
        return service
