"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("@aws-cdk/core");
const cfn_parse = require("@aws-cdk/core/lib/helpers-internal");
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_1 = require("@aws-cdk/core");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_cloudformation_include_CfnIncludeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CfnInclude);
            }
            throw error;
        }
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = props.preserveLogicalIds ?? true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core_1.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core_1.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core_1.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core_1.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core_1.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     * @param nestedStackProps the properties of the included child Stack
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     */
    loadNestedStack(logicalId, nestedStackProps) {
        try {
            jsiiDeprecationWarnings._aws_cdk_cloudformation_include_CfnIncludeProps(nestedStackProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.loadNestedStack);
            }
            throw error;
        }
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = cfnParser.parseValue(hookAttributes.Properties) ?? {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    return self.resources[elementName] ?? self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId, cycleChain = []) {
        cycleChain = cycleChain.concat([logicalId]);
        if (cycleChain.length !== new Set(cycleChain).size) {
            throw new Error(`Found a cycle between resources in the template: ${cycleChain.join(' depends on ')}`);
        }
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId, cycleChain);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: cfn_parse.FromCloudFormation.getStringArray(nestedStackProps.NotificationARNs).value,
            timeout: this.timeoutForNestedStack(nestedStackProps.TimeoutInMinutes),
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = this.nestedStacksToInclude[nestedStackId].parameters ?? {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = cfn_parse.FromCloudFormation.getString(parameters[paramName]).value;
            }
        }
        return ret;
    }
    timeoutForNestedStack(value) {
        if (value == null) {
            return undefined;
        }
        return core.Duration.minutes(cfn_parse.FromCloudFormation.getNumber(value).value);
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
_a = JSII_RTTI_SYMBOL_1;
CfnInclude[_a] = { fqn: "@aws-cdk/cloudformation-include.CfnInclude", version: "1.181.0" };
//# sourceMappingURL=data:application/json;base64,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