# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['numba_passthru', 'numba_passthru.test']

package_data = \
{'': ['*']}

install_requires = \
['numba>=0.51.2']

setup_kwargs = {
    'name': 'numba-passthru',
    'version': '0.3.0',
    'description': 'Stand-alone package for Numba passthrough types until PR 3640 has landed',
    'long_description': 'Pass-through type for Numba\n===========================\n\nTools to ferry arbitrary Python objects through `nopython` mode. This is a stand-alone version of the Numba internal\nimplementation [PR 3640](https://github.com/numba/numba/pull/3640). \n\nThis has two typical use-cases:\n  1. ferrying data structures not currently supported by Numba into `objmode` blocks via `PassThruContainer`\n  2. creating extension types that are simplified representations of the Python class and keep a link to the\n     Python object\n     \nIt\'s not unlikely both can be avoided completely when starting from scratch but potentially require extensive\nrefactoring when moving Numba into an existing code base.\n\nFerrying objects into `objmode` using `PassThruContainer`\n---------------------------------------------------------\n`PassThruContainer` can be used to make Python objects not supported by `nopython` available inside\n`objmode` blocks. Outside `objmode` blocks the only supported operation on `PassThruContainer` \nis `==`. Two instances are equal if the wrapped objects are identical, ie `a.obj is b.obj`.\n\nIn the following example an object unsupported in `nopython` is ferried into an `objmode` block and\nmutated there. \n```python\nfrom __future__ import annotations\nfrom numba import jit, objmode\nfrom numba_passthru import PassThruContainer\n\nclass Testee:\n    def __init__(self, value: int, invert=False):\n        self.value = value\n        self.numba_will_not_like_this = compile(\'pass\', \'N/A\', \'single\')\n        \n    def __gt__(self, other):    # will be used in next example\n        return self.value > other.value\n\ntestee = Testee(1)  \ncontainer = PassThruContainer(testee)\n\n@jit(nopython=True)\ndef do_something(container):\n    with objmode(value=\'int64\'):\n        setattr(container.obj, \'value_2\', 2)\n        value = container.obj.value\n        \n    return container, value\n    \nresult, value = do_something(container)\n\nassert container is result\nassert value == 1\nassert result.obj.value_2 == 2\n```\n\nThere will be no speed-up for the code inside the `objmode` block and `container` is (un)boxed twice adding further\noverhead. Hence, this only makes sense in rarely visited code-path and if refactoring into a more Numba friendly\n form is not an option.\n \nNote that the example above already contains the most common pattern that is pretty unpleasant to refactor into a \nNumba friendly form in requiring object identity being preserved through `nopython` (`assert container is result`).\nObviously, this is a highly artificial requirement in this toy example but might get more real if the pass-through\nobject is part of conditional branches. \n\nCreating custom passthrough types\n---------------------------------\n`PassThroughContainer` does not allow attribute access on the wrapped object in `nopython` and there is no way\nto dispatch on the type of the wrapped object. To get both you can create a Numba extension type using `pass_thru_type`.\n`pass_thru_type` holds a pointer to the `PyObject` and manages references. `pass_thru_type` can be used like any \nmem-managed member on an [extension type](http://numba.pydata.org/numba-doc/latest/extending/index.html). (Some\nfamiliarity with Numba extension types is expected for the following.)  \n \nContinuing the example above let\'s try to get the following code working in `nopython` (another toy\nexample, no speed-up expected):\n```python\ndef find_max(testees: List[Testee]) -> Testee:\n    result = testees[0]  # testees must not be empty\n    for testee in testees[1:]:\n        if testee > result:\n            result = testee\n\n    return result     \n``` \n`PassThroughContainer` will not help here as there would be no way to dispatch `>`  to `Testee.__gt__` and there would\nbe no way to access `.value` from `nopython` inside `Testee.__gt__`. Still, since `Testee.value` is the only attribute \nbeing accessed from `nopython` there is a realistic chance to get this working. Indeed, assuming we already had the \n(un)boxer this is a straight forward Numba extension type:\n```python\nfrom numba import jit, types\nfrom numba.extending import overload\nfrom numba.typing.typeof import typeof_impl\nimport operator\n\nclass TesteeType(PassThruType):\n    def __init__(self):\n        super(TesteeType, self).__init__(name=\'Testee\')\n\ntestee_type = TesteeType()\n\n@typeof_impl.register(Testee)\ndef type_testee(val, context):\n    return testee_type\n\n@overload(operator.gt)\ndef testee_gt(self, other):\n    if self is testee_type and other is testee_type:\n        return Testee.__gt__\n\nfind_max = jit(nopython=True)(find_max)\n```\n\nTrying to implement the (un)boxer to somehow pass the `.numba_will_not_like_this` attribute around `nopython` (sharing\na dict between boxer/unboxer etc) is not straight forward to get working for `find_max` alone and it is impossible\nto get the reference counting right in the general case. The clean approach is to have the Numba runtime manage the\nreferences by putting a NRT managed reference to the original Python object onto the extension type\'s data model.  \n\n`pass_thru_type` helps with the boiler-plate of boxing/unboxing the required `MemInfoPointer`. The `PyObject` \npassed into the unboxer can be unboxed directly into a `pass_thru_type`. On the way out the original `PyObject` is \nrecovered  by boxing the `pass_thru_type`. \n\n```python\nfrom numba import cgutils\nfrom numba.datamodel import models\nfrom numba.extending import make_attribute_wrapper, overload, overload_method, register_model\nfrom numba.pythonapi import NativeValue, unbox, box\nfrom numba.targets.hashing import _Py_hash_t\n\nfrom numba_passthru import pass_thru_type\n\n@register_model(TesteeType)\nclass TesteeModel(models.StructModel):\n    def __init__(self, dmm, fe_typ):\n        members = [\n            (\'parent\', pass_thru_type),\n            (\'value\', types.intp),\n        ]\n        super(TesteeModel, self).__init__(dmm, fe_typ, members)\n        \nmake_attribute_wrapper(TesteeType, \'value\', \'value\')\n\n@unbox(TesteeType)\ndef unbox_testee(typ, obj, context):\n    testee = cgutils.create_struct_proxy(typ)(context.context, context.builder)\n    \n    testee.parent = context.unbox(pass_thru_type, obj).value\n    \n    value = context.pyapi.object_getattr_string(obj, "value")\n    native_value = context.unbox(types.intp, value)\n    context.pyapi.decref(value)\n\n    testee.value = native_value.value\n\n    is_error = cgutils.is_not_null(context.builder, context.pyapi.err_occurred())\n    return NativeValue(testee._getvalue(), is_error=is_error)\n    \n@box(TesteeType)\ndef box_testee(typ, val, context):\n    val = cgutils.create_struct_proxy(typ)(context.context, context.builder, value=val)\n    obj = context.box(pass_thru_type, val.parent)\n\n    return obj\n```\nGiven the implementation above `TesteeType` is immutable from `nopython` (`make_attribute_wrapper` creates read-only\nattributes). If you made a pass-through type mutable from `nopython` you had to make sure to reflect changes back\nto the Python object in the boxer. However, given the [experience with reflected lists and sets](http://numba.pydata.org/numba-doc/latest/reference/deprecation.html#deprecation-of-reflection-for-list-and-set-types)\nthere are good reasons to be careful about this. \n\nUpward compatibility notice\n---------------------------\nThis is a stand-alone version of Numba [PR 3640](https://github.com/numba/numba/pull/3640). Import of\n`PassThruType`, `pass_thru_type`, `PassThruContainer`, `pass_thru_container_type` from `numba` is attempted first \nhence you will get the Numba internal implementations once the PR has landed.\n\nThis package contains an overload of `int(Opaque)` (essentially `ptrtoint`) that might break future Numba versions \nif Numba created diverging implementations.\n\nThis was considered too unlikely to put a version constraint on the Numba dependency (which would require a new release\nof `numba-passthru` every time a new Numba versions is released)\n',
    'author': 'Andreas Sodeur',
    'author_email': 'andreas.sodeur@rwe.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/asodeur/numba-passthru',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6',
}


setup(**setup_kwargs)
