"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerService = exports.Database = exports.KeyCloak = exports.KeycloakVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const certmgr = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
// regional availibility for aurora serverless
// see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/Concepts.AuroraFeaturesRegionsDBEngines.grids.html
const AURORA_SERVERLESS_SUPPORTED_REGIONS = [
    'us-east-1',
    'us-east-2',
    'us-west-1',
    'us-west-2',
    'ap-south-1',
    'ap-northeast-1',
    'ap-northeast-2',
    'ap-southeast-1',
    'ap-southeast-2',
    'ca-central-1',
    'eu-central-1',
    'eu-west-1',
    'eu-west-2',
    'eu-west-3',
    'cn-northwest-1',
];
/**
 * Keycloak  version.
 *
 * @stability stable
 */
class KeycloakVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version.
     *
     * @param version custom version number.
     * @stability stable
     */
    static of(version) { return new KeycloakVersion(version); }
}
exports.KeycloakVersion = KeycloakVersion;
_a = JSII_RTTI_SYMBOL_1;
KeycloakVersion[_a] = { fqn: "cdk-keycloak.KeycloakVersion", version: "0.2.42" };
/**
 * Keycloak version 12.0.4.
 *
 * @stability stable
 */
KeycloakVersion.V12_0_4 = KeycloakVersion.of('12.0.4');
/**
 * Keycloak version 15.0.0.
 *
 * @stability stable
 */
KeycloakVersion.V15_0_0 = KeycloakVersion.of('15.0.0');
/**
 * Keycloak version 15.0.1.
 *
 * @stability stable
 */
KeycloakVersion.V15_0_1 = KeycloakVersion.of('15.0.1');
/**
 * Keycloak version 15.0.2.
 *
 * @stability stable
 */
KeycloakVersion.V15_0_2 = KeycloakVersion.of('15.0.2');
const KEYCLOAK_DOCKER_IMAGE_URI_MAP = {
    'aws': 'jboss/keycloak:',
    'aws-cn': '048912060910.dkr.ecr.cn-northwest-1.amazonaws.com.cn/dockerhub/jboss/keycloak:',
};
/**
 * @stability stable
 */
class KeyCloak extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e;
        super(scope, id);
        const region = cdk.Stack.of(this).region;
        const regionIsResolved = !cdk.Token.isUnresolved(region);
        if (props.auroraServerless && regionIsResolved && !AURORA_SERVERLESS_SUPPORTED_REGIONS.includes(region)) {
            throw new Error(`Aurora serverless is not supported in ${region}`);
        }
        this.vpc = (_e = props.vpc) !== null && _e !== void 0 ? _e : getOrCreateVpc(this);
        this.db = this.addDatabase({
            vpc: this.vpc,
            databaseSubnets: props.databaseSubnets,
            instanceType: props.databaseInstanceType,
            instanceEngine: props.instanceEngine,
            clusterEngine: props.clusterEngine,
            auroraServerless: props.auroraServerless,
            singleDbInstance: props.singleDbInstance,
            backupRetention: props.backupRetention,
        });
        this.addKeyCloakContainerService({
            database: this.db,
            vpc: this.vpc,
            keycloakVersion: props.keycloakVersion,
            publicSubnets: props.publicSubnets,
            privateSubnets: props.privateSubnets,
            keycloakSecret: this._generateKeycloakSecret(),
            certificate: certmgr.Certificate.fromCertificateArn(this, 'ACMCert', props.certificateArn),
            bastion: props.bastion,
            nodeCount: props.nodeCount,
            stickinessCookieDuration: props.stickinessCookieDuration,
            autoScaleTask: props.autoScaleTask,
            env: props.env,
        });
        if (!cdk.Stack.of(this).templateOptions.description) {
            cdk.Stack.of(this).templateOptions.description = '(SO8021) - Deploy keycloak on AWS with cdk-keycloak construct library';
        }
    }
    /**
     * @stability stable
     */
    addDatabase(props) {
        return new Database(this, 'Database', props);
    }
    /**
     * @stability stable
     */
    addKeyCloakContainerService(props) {
        return new ContainerService(this, 'KeyCloakContainerSerivce', props);
    }
    _generateKeycloakSecret() {
        return new secretsmanager.Secret(this, 'KCSecret', {
            generateSecretString: {
                generateStringKey: 'password',
                excludePunctuation: true,
                passwordLength: 12,
                secretStringTemplate: JSON.stringify({ username: 'keycloak' }),
            },
        });
    }
}
exports.KeyCloak = KeyCloak;
_b = JSII_RTTI_SYMBOL_1;
KeyCloak[_b] = { fqn: "cdk-keycloak.KeyCloak", version: "0.2.42" };
/**
 * Represents the database instance or database cluster.
 *
 * @stability stable
 */
class Database extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._mysqlListenerPort = 3306;
        this.vpc = props.vpc;
        const config = props.auroraServerless ? this._createServerlessCluster(props)
            : props.singleDbInstance ? this._createRdsInstance(props) : this._createRdsCluster(props);
        this.secret = config.secret;
        // allow internally from the same security group
        config.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the whole vpc cidr
        config.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.clusterEndpointHostname = config.endpoint;
        this.clusterIdentifier = config.identifier;
        this.connections = config.connections;
        printOutput(this, 'DBSecretArn', config.secret.secretArn);
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
    }
    _createRdsInstance(props) {
        var _e, _f, _g;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            databaseName: 'keycloak',
            vpcSubnets: props.databaseSubnets,
            engine: (_e = props.instanceEngine) !== null && _e !== void 0 ? _e : rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_21,
            }),
            storageEncrypted: true,
            backupRetention: (_f = props.backupRetention) !== null && _f !== void 0 ? _f : cdk.Duration.days(7),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_g = props.instanceType) !== null && _g !== void 0 ? _g : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            deletionProtection: true,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        return {
            connections: dbInstance.connections,
            endpoint: dbInstance.dbInstanceEndpointAddress,
            identifier: dbInstance.instanceIdentifier,
            secret: dbInstance.secret,
        };
    }
    // create a RDS for MySQL DB cluster
    _createRdsCluster(props) {
        var _e, _f, _g;
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: (_e = props.clusterEngine) !== null && _e !== void 0 ? _e : rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_09_1,
            }),
            defaultDatabaseName: 'keycloak',
            deletionProtection: true,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceProps: {
                vpc: props.vpc,
                vpcSubnets: props.databaseSubnets,
                instanceType: (_f = props.instanceType) !== null && _f !== void 0 ? _f : new ec2.InstanceType('r5.large'),
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            backup: {
                retention: (_g = props.backupRetention) !== null && _g !== void 0 ? _g : cdk.Duration.days(7),
            },
            storageEncrypted: true,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
    _createServerlessCluster(props) {
        var _e;
        const dbCluster = new rds.ServerlessCluster(this, 'AuroraServerlessCluster', {
            engine: rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc: props.vpc,
            defaultDatabaseName: 'keycloak',
            vpcSubnets: props.databaseSubnets,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            backupRetention: (_e = props.backupRetention) !== null && _e !== void 0 ? _e : cdk.Duration.days(7),
            deletionProtection: true,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
}
exports.Database = Database;
_c = JSII_RTTI_SYMBOL_1;
Database[_c] = { fqn: "cdk-keycloak.Database", version: "0.2.42" };
/**
 * @stability stable
 */
class ContainerService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        const vpc = props.vpc;
        const cluster = new ecs.Cluster(this, 'Cluster', { vpc });
        cluster.node.addDependency(props.database);
        const taskRole = new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com')),
        });
        const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDef', {
            cpu: 4096,
            memoryLimitMiB: 8192,
            executionRole: taskRole,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        const kc = taskDefinition.addContainer('keycloak', {
            image: ecs.ContainerImage.fromRegistry(this.getKeyCloakDockerImageUri(props.keycloakVersion.version)),
            environment: Object.assign({
                DB_ADDR: props.database.clusterEndpointHostname,
                DB_DATABASE: 'keycloak',
                DB_PORT: '3306',
                DB_USER: 'admin',
                DB_VENDOR: 'mysql',
                // KEYCLOAK_LOGLEVEL: 'DEBUG',
                PROXY_ADDRESS_FORWARDING: 'true',
                JDBC_PARAMS: 'useSSL=false',
                JGROUPS_DISCOVERY_PROTOCOL: 'JDBC_PING',
            }, props.env),
            secrets: {
                DB_PASSWORD: ecs.Secret.fromSecretsManager(props.database.secret, 'password'),
                KEYCLOAK_USER: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'username'),
                KEYCLOAK_PASSWORD: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'password'),
            },
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'keycloak',
                logGroup,
            }),
        });
        kc.addPortMappings({ containerPort: 8443 }, // HTTPS web port
        { containerPort: 7600 }, // jgroups-tcp
        { containerPort: 57600 }, // jgroups-tcp-fd
        { containerPort: 55200, protocol: ecs.Protocol.UDP }, // jgroups-udp
        { containerPort: 54200, protocol: ecs.Protocol.UDP });
        // we need extra privileges to fetch keycloak docker images from China mirror site
        (_e = taskDefinition.executionRole) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.service = new ecs.FargateService(this, 'Service', {
            cluster,
            taskDefinition,
            circuitBreaker: props.circuitBreaker ? { rollback: true } : undefined,
            desiredCount: (_f = props.nodeCount) !== null && _f !== void 0 ? _f : 2,
            healthCheckGracePeriod: cdk.Duration.seconds(120),
        });
        // we need to allow traffic from the same secret group for keycloak cluster with jdbc_ping
        this.service.connections.allowFrom(this.service.connections, ec2.Port.tcp(7600), 'kc jgroups-tcp');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.tcp(57600), 'kc jgroups-tcp-fd');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.udp(55200), 'kc jgroups-udp');
        this.service.connections.allowFrom(this.service.connections, ec2.Port.udp(54200), 'kc jgroups-udp-fd');
        if (props.autoScaleTask) {
            const minCapacity = (_h = (_g = props.autoScaleTask.min) !== null && _g !== void 0 ? _g : props.nodeCount) !== null && _h !== void 0 ? _h : 2;
            const scaling = this.service.autoScaleTaskCount({
                minCapacity,
                maxCapacity: (_j = props.autoScaleTask.max) !== null && _j !== void 0 ? _j : minCapacity + 5,
            });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_k = props.autoScaleTask.targetCpuUtilization) !== null && _k !== void 0 ? _k : 75,
            });
        }
        ;
        const alb = new elbv2.ApplicationLoadBalancer(this, 'ALB', {
            vpc,
            vpcSubnets: props.publicSubnets,
            internetFacing: true,
        });
        printOutput(this, 'EndpointURL', `https://${alb.loadBalancerDnsName}`);
        const listener = alb.addListener('HttpsListener', {
            protocol: elbv2.ApplicationProtocol.HTTPS,
            certificates: [{ certificateArn: props.certificate.certificateArn }],
        });
        listener.addTargets('ECSTarget', {
            targets: [this.service],
            healthCheck: {
                healthyThresholdCount: 3,
            },
            // set slow_start.duration_seconds to 60
            // see https://docs.aws.amazon.com/cli/latest/reference/elbv2/modify-target-group-attributes.html
            slowStart: cdk.Duration.seconds(60),
            stickinessCookieDuration: (_l = props.stickinessCookieDuration) !== null && _l !== void 0 ? _l : cdk.Duration.days(1),
            port: 8443,
            protocol: elbv2.ApplicationProtocol.HTTPS,
        });
        // allow task execution role to read the secrets
        props.database.secret.grantRead(taskDefinition.executionRole);
        props.keycloakSecret.grantRead(taskDefinition.executionRole);
        // allow ecs task connect to database
        props.database.connections.allowDefaultPortFrom(this.service);
        // create a bastion host
        if (props.bastion === true) {
            const bast = new ec2.BastionHostLinux(this, 'Bast', {
                vpc,
                instanceType: new ec2.InstanceType('m5.large'),
            });
            props.database.connections.allowDefaultPortFrom(bast);
        }
    }
    getImageUriFromMap(map, version, id) {
        const stack = cdk.Stack.of(this);
        if (cdk.Token.isUnresolved(stack.region)) {
            const mapping = {};
            for (let [partition, uri] of Object.entries(map)) {
                uri += version;
                mapping[partition] = { uri };
            }
            const imageMap = new cdk.CfnMapping(this, id, { mapping });
            return imageMap.findInMap(cdk.Aws.PARTITION, 'uri');
        }
        else {
            if (stack.region.startsWith('cn-')) {
                return map['aws-cn'] += version;
            }
            else {
                return map.aws += version;
            }
        }
    }
    getKeyCloakDockerImageUri(version) {
        return this.getImageUriFromMap(KEYCLOAK_DOCKER_IMAGE_URI_MAP, version, 'KeycloakImageMap');
    }
}
exports.ContainerService = ContainerService;
_d = JSII_RTTI_SYMBOL_1;
ContainerService[_d] = { fqn: "cdk-keycloak.ContainerService", version: "0.2.42" };
/**
 * Create or import VPC
 * @param scope the cdk scope
 */
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1' ?
        ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2V5Y2xvYWsuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMva2V5Y2xvYWsudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwyREFBMkQ7QUFDM0Qsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyw2REFBNkQ7QUFDN0Qsd0NBQXdDO0FBQ3hDLDBDQUEwQztBQUMxQyx3Q0FBd0M7QUFDeEMsOERBQThEO0FBQzlELHFDQUFxQztBQUdyQyw4Q0FBOEM7QUFDOUMsc0hBQXNIO0FBQ3RILE1BQU0sbUNBQW1DLEdBQUc7SUFDMUMsV0FBVztJQUNYLFdBQVc7SUFDWCxXQUFXO0lBQ1gsV0FBVztJQUNYLFlBQVk7SUFDWixnQkFBZ0I7SUFDaEIsZ0JBQWdCO0lBQ2hCLGdCQUFnQjtJQUNoQixnQkFBZ0I7SUFDaEIsY0FBYztJQUNkLGNBQWM7SUFDZCxXQUFXO0lBQ1gsV0FBVztJQUNYLFdBQVc7SUFDWCxnQkFBZ0I7Q0FDakIsQ0FBQzs7Ozs7O0FBR0YsTUFBYSxlQUFlO0lBZTFCOzs7T0FHRztJQUNILFlBQW9DLE9BQWU7UUFBZixZQUFPLEdBQVAsT0FBTyxDQUFRO0lBQUksQ0FBQzs7Ozs7OztJQUxqRCxNQUFNLENBQUMsRUFBRSxDQUFDLE9BQWUsSUFBSSxPQUFPLElBQUksZUFBZSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQzs7QUFkNUUsMENBb0JDOzs7Ozs7OztBQWxCd0IsdUJBQU8sR0FBRyxlQUFlLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDOzs7Ozs7QUFHdkMsdUJBQU8sR0FBRyxlQUFlLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDOzs7Ozs7QUFHdkMsdUJBQU8sR0FBRyxlQUFlLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDOzs7Ozs7QUFHdkMsdUJBQU8sR0FBRyxlQUFlLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBa0JoRSxNQUFNLDZCQUE2QixHQUFtQjtJQUNwRCxLQUFLLEVBQUUsaUJBQWlCO0lBQ3hCLFFBQVEsRUFBRSxnRkFBZ0Y7Q0FDM0YsQ0FBQzs7OztBQWlERixNQUFhLFFBQVMsU0FBUSxHQUFHLENBQUMsU0FBUzs7OztJQUd6QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQW9COztRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sTUFBTSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUN6QyxNQUFNLGdCQUFnQixHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFekQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksZ0JBQWdCLElBQUksQ0FBQyxtQ0FBbUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDdkcsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsTUFBTSxFQUFFLENBQUMsQ0FBQztTQUNwRTtRQUVELElBQUksQ0FBQyxHQUFHLFNBQUcsS0FBSyxDQUFDLEdBQUcsbUNBQUksY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQztZQUN6QixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWU7WUFDdEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7WUFDeEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1NBQ3ZDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQywyQkFBMkIsQ0FBQztZQUMvQixRQUFRLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDakIsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2IsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsY0FBYyxFQUFFLElBQUksQ0FBQyx1QkFBdUIsRUFBRTtZQUM5QyxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDMUYsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQix3QkFBd0IsRUFBRSxLQUFLLENBQUMsd0JBQXdCO1lBQ3hELGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRTtZQUNuRCxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxlQUFlLENBQUMsV0FBVyxHQUFHLHVFQUF1RSxDQUFDO1NBQzFIO0lBQ0gsQ0FBQzs7OztJQUNNLFdBQVcsQ0FBQyxLQUFvQjtRQUNyQyxPQUFPLElBQUksUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDL0MsQ0FBQzs7OztJQUNNLDJCQUEyQixDQUFDLEtBQTRCO1FBQzdELE9BQU8sSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDdkUsQ0FBQztJQUNPLHVCQUF1QjtRQUM3QixPQUFPLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2pELG9CQUFvQixFQUFFO2dCQUNwQixpQkFBaUIsRUFBRSxVQUFVO2dCQUM3QixrQkFBa0IsRUFBRSxJQUFJO2dCQUN4QixjQUFjLEVBQUUsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsQ0FBQzthQUMvRDtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBekRILDRCQTBEQzs7Ozs7Ozs7QUFtQ0QsTUFBYSxRQUFTLFNBQVEsR0FBRyxDQUFDLFNBQVM7Ozs7SUFRekMsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSEYsdUJBQWtCLEdBQVcsSUFBSSxDQUFDO1FBSWpELElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztRQUNyQixNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxLQUFLLENBQUM7WUFDMUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDNUYsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDO1FBQzVCLGdEQUFnRDtRQUNoRCxNQUFNLENBQUMsV0FBVyxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBQzFFLGdDQUFnQztRQUNoQyxNQUFNLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFDM0csSUFBSSxDQUFDLHVCQUF1QixHQUFHLE1BQU0sQ0FBQyxRQUFRLENBQUM7UUFDL0MsSUFBSSxDQUFDLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUM7UUFDM0MsSUFBSSxDQUFDLFdBQVcsR0FBRyxNQUFNLENBQUMsV0FBVyxDQUFDO1FBQ3RDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDMUQsV0FBVyxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRSxJQUFJLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUMzRSxXQUFXLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFDTyxrQkFBa0IsQ0FBQyxLQUFvQjs7UUFDN0MsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUM5RCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxZQUFZLEVBQUUsVUFBVTtZQUN4QixVQUFVLEVBQUUsS0FBSyxDQUFDLGVBQWU7WUFDakMsTUFBTSxRQUFFLEtBQUssQ0FBQyxjQUFjLG1DQUFJLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLENBQUM7Z0JBQy9ELE9BQU8sRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsVUFBVTthQUMzQyxDQUFDO1lBQ0YsZ0JBQWdCLEVBQUUsSUFBSTtZQUN0QixlQUFlLFFBQUUsS0FBSyxDQUFDLGVBQWUsbUNBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQzlELFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQztZQUN6RCxZQUFZLFFBQUUsS0FBSyxDQUFDLFlBQVksbUNBQUksSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQztZQUNwRSxjQUFjLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsa0JBQWtCLENBQUM7WUFDckcsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1NBQ3hDLENBQUMsQ0FBQztRQUNILE9BQU87WUFDTCxXQUFXLEVBQUUsVUFBVSxDQUFDLFdBQVc7WUFDbkMsUUFBUSxFQUFFLFVBQVUsQ0FBQyx5QkFBeUI7WUFDOUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxrQkFBa0I7WUFDekMsTUFBTSxFQUFFLFVBQVUsQ0FBQyxNQUFPO1NBQzNCLENBQUM7SUFDSixDQUFDO0lBQ0Qsb0NBQW9DO0lBQzVCLGlCQUFpQixDQUFDLEtBQW9COztRQUM1QyxNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUMzRCxNQUFNLFFBQUUsS0FBSyxDQUFDLGFBQWEsbUNBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLFdBQVcsQ0FBQztnQkFDbkUsT0FBTyxFQUFFLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxVQUFVO2FBQ2pELENBQUM7WUFDRixtQkFBbUIsRUFBRSxVQUFVO1lBQy9CLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDO1lBQ3pELGFBQWEsRUFBRTtnQkFDYixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLEtBQUssQ0FBQyxlQUFlO2dCQUNqQyxZQUFZLFFBQUUsS0FBSyxDQUFDLFlBQVksbUNBQUksSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQzthQUNyRTtZQUNELGNBQWMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSx5QkFBeUIsQ0FBQztZQUM1RyxNQUFNLEVBQUU7Z0JBQ04sU0FBUyxRQUFFLEtBQUssQ0FBQyxlQUFlLG1DQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzthQUN6RDtZQUNELGdCQUFnQixFQUFFLElBQUk7WUFDdEIsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTTtTQUN4QyxDQUFDLENBQUM7UUFDSCxPQUFPO1lBQ0wsV0FBVyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1lBQ2xDLFFBQVEsRUFBRSxTQUFTLENBQUMsZUFBZSxDQUFDLFFBQVE7WUFDNUMsVUFBVSxFQUFFLFNBQVMsQ0FBQyxpQkFBaUI7WUFDdkMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxNQUFPO1NBQzFCLENBQUM7SUFDSixDQUFDO0lBQ08sd0JBQXdCLENBQUMsS0FBb0I7O1FBQ25ELE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUMzRSxNQUFNLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLFlBQVk7WUFDOUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1lBQ2QsbUJBQW1CLEVBQUUsVUFBVTtZQUMvQixVQUFVLEVBQUUsS0FBSyxDQUFDLGVBQWU7WUFDakMsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDO1lBQ3pELGVBQWUsUUFBRSxLQUFLLENBQUMsZUFBZSxtQ0FBSSxHQUFHLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDOUQsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ3ZDLGNBQWMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSx5QkFBeUIsQ0FBQztTQUM3RyxDQUFDLENBQUM7UUFDSCxPQUFPO1lBQ0wsV0FBVyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1lBQ2xDLFFBQVEsRUFBRSxTQUFTLENBQUMsZUFBZSxDQUFDLFFBQVE7WUFDNUMsVUFBVSxFQUFFLFNBQVMsQ0FBQyxpQkFBaUI7WUFDdkMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxNQUFPO1NBQzFCLENBQUM7SUFDSixDQUFDOztBQTlGSCw0QkErRkM7Ozs7OztBQWdDRCxNQUFhLGdCQUFpQixTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7O0lBRWpELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7O1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztRQUN0QixNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDMUQsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FDbkMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsRUFDN0MsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUMsQ0FDcEQ7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3BFLEdBQUcsRUFBRSxJQUFJO1lBQ1QsY0FBYyxFQUFFLElBQUk7WUFDcEIsYUFBYSxFQUFFLFFBQVE7U0FDeEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxRQUFRLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbkQsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztZQUN2QyxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1NBQ3hDLENBQUMsQ0FBQztRQUVILE1BQU0sRUFBRSxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsVUFBVSxFQUFFO1lBQ2pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNyRyxXQUFXLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQztnQkFDekIsT0FBTyxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsdUJBQXVCO2dCQUMvQyxXQUFXLEVBQUUsVUFBVTtnQkFDdkIsT0FBTyxFQUFFLE1BQU07Z0JBQ2YsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLFNBQVMsRUFBRSxPQUFPO2dCQUNsQiw4QkFBOEI7Z0JBQzlCLHdCQUF3QixFQUFFLE1BQU07Z0JBQ2hDLFdBQVcsRUFBRSxjQUFjO2dCQUMzQiwwQkFBMEIsRUFBRSxXQUFXO2FBS3hDLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQztZQUNiLE9BQU8sRUFBRTtnQkFDUCxXQUFXLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxVQUFVLENBQUM7Z0JBQzdFLGFBQWEsRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUUsVUFBVSxDQUFDO2dCQUM5RSxpQkFBaUIsRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUUsVUFBVSxDQUFDO2FBQ25GO1lBQ0QsT0FBTyxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDO2dCQUM5QixZQUFZLEVBQUUsVUFBVTtnQkFDeEIsUUFBUTthQUNULENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsZUFBZSxDQUNoQixFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsRUFBRSxpQkFBaUI7UUFDMUMsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLEVBQUUsY0FBYztRQUN2QyxFQUFFLGFBQWEsRUFBRSxLQUFLLEVBQUUsRUFBRSxpQkFBaUI7UUFDM0MsRUFBRSxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxFQUFFLGNBQWM7UUFDcEUsRUFBRSxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxDQUNyRCxDQUFDO1FBRUYsa0ZBQWtGO1FBQ2xGLE1BQUEsY0FBYyxDQUFDLGFBQWEsMENBQUUsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxvQ0FBb0MsQ0FBQyxFQUFFO1FBRWpJLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDckQsT0FBTztZQUNQLGNBQWM7WUFDZCxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDckUsWUFBWSxRQUFFLEtBQUssQ0FBQyxTQUFTLG1DQUFJLENBQUM7WUFDbEMsc0JBQXNCLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUNILDBGQUEwRjtRQUMxRixJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUNuRyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUN2RyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUNwRyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUV2RyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsTUFBTSxXQUFXLGVBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxHQUFHLG1DQUFJLEtBQUssQ0FBQyxTQUFTLG1DQUFJLENBQUMsQ0FBQztZQUNwRSxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGtCQUFrQixDQUFDO2dCQUM5QyxXQUFXO2dCQUNYLFdBQVcsUUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLEdBQUcsbUNBQUksV0FBVyxHQUFHLENBQUM7YUFDeEQsQ0FBQyxDQUFDO1lBQ0gsT0FBTyxDQUFDLHFCQUFxQixDQUFDLFlBQVksRUFBRTtnQkFDMUMsd0JBQXdCLFFBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxvQkFBb0IsbUNBQUksRUFBRTthQUN6RSxDQUFDLENBQUM7U0FDSjtRQUFBLENBQUM7UUFFRixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO1lBQ3pELEdBQUc7WUFDSCxVQUFVLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDL0IsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsV0FBVyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUUsV0FBVyxHQUFHLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBRXZFLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsZUFBZSxFQUFFO1lBQ2hELFFBQVEsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSztZQUN6QyxZQUFZLEVBQUUsQ0FBQyxFQUFFLGNBQWMsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRSxDQUFDO1NBQ3JFLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFO1lBQy9CLE9BQU8sRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUM7WUFDdkIsV0FBVyxFQUFFO2dCQUNYLHFCQUFxQixFQUFFLENBQUM7YUFDekI7WUFDRCx3Q0FBd0M7WUFDeEMsaUdBQWlHO1lBQ2pHLFNBQVMsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDbkMsd0JBQXdCLFFBQUUsS0FBSyxDQUFDLHdCQUF3QixtQ0FBSSxHQUFHLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDaEYsSUFBSSxFQUFFLElBQUk7WUFDVixRQUFRLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLEtBQUs7U0FDMUMsQ0FBQyxDQUFDO1FBRUgsZ0RBQWdEO1FBQ2hELEtBQUssQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsYUFBYyxDQUFDLENBQUM7UUFDL0QsS0FBSyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLGFBQWMsQ0FBQyxDQUFDO1FBRTlELHFDQUFxQztRQUNyQyxLQUFLLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFHOUQsd0JBQXdCO1FBQ3hCLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxJQUFJLEVBQUU7WUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsR0FBRztnQkFDSCxZQUFZLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQzthQUMvQyxDQUFDLENBQUM7WUFDSCxLQUFLLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN2RDtJQUNILENBQUM7SUFDTyxrQkFBa0IsQ0FBQyxHQUFtQixFQUFFLE9BQWUsRUFBRSxFQUFVO1FBQ3pFLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sT0FBTyxHQUE0QyxFQUFFLENBQUM7WUFDNUQsS0FBSyxJQUFJLENBQUMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ2hELEdBQUcsSUFBSSxPQUFPLENBQUM7Z0JBQ2YsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUM7YUFDOUI7WUFDRCxNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDM0QsT0FBTyxRQUFRLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1NBQ3JEO2FBQU07WUFDTCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUNsQyxPQUFPLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxPQUFPLENBQUM7YUFDakM7aUJBQU07Z0JBQ0wsT0FBTyxHQUFHLENBQUMsR0FBRyxJQUFJLE9BQU8sQ0FBQzthQUMzQjtTQUNGO0lBQ0gsQ0FBQztJQUNPLHlCQUF5QixDQUFDLE9BQWU7UUFDL0MsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsNkJBQTZCLEVBQUUsT0FBTyxFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFDN0YsQ0FBQzs7QUFySkgsNENBc0pDOzs7QUFFRDs7O0dBR0c7QUFDSCxTQUFTLGNBQWMsQ0FBQyxLQUFvQjtJQUMxQywwQ0FBMEM7SUFDMUMsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxDQUFDO1FBQzFELEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7WUFDdEMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNyRixJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsV0FBVyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDL0QsQ0FBQztBQUVELFNBQVMsV0FBVyxDQUFDLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEdBQW9CO0lBQ3pFLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDdkQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNlcnRtZ3IgZnJvbSAnQGF3cy1jZGsvYXdzLWNlcnRpZmljYXRlbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBlY3MgZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgKiBhcyBlbGJ2MiBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCAqIGFzIHJkcyBmcm9tICdAYXdzLWNkay9hd3MtcmRzJztcbmltcG9ydCAqIGFzIHNlY3JldHNtYW5hZ2VyIGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cblxuLy8gcmVnaW9uYWwgYXZhaWxpYmlsaXR5IGZvciBhdXJvcmEgc2VydmVybGVzc1xuLy8gc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25SRFMvbGF0ZXN0L0F1cm9yYVVzZXJHdWlkZS9Db25jZXB0cy5BdXJvcmFGZWF0dXJlc1JlZ2lvbnNEQkVuZ2luZXMuZ3JpZHMuaHRtbFxuY29uc3QgQVVST1JBX1NFUlZFUkxFU1NfU1VQUE9SVEVEX1JFR0lPTlMgPSBbXG4gICd1cy1lYXN0LTEnLFxuICAndXMtZWFzdC0yJyxcbiAgJ3VzLXdlc3QtMScsXG4gICd1cy13ZXN0LTInLFxuICAnYXAtc291dGgtMScsXG4gICdhcC1ub3J0aGVhc3QtMScsXG4gICdhcC1ub3J0aGVhc3QtMicsXG4gICdhcC1zb3V0aGVhc3QtMScsXG4gICdhcC1zb3V0aGVhc3QtMicsXG4gICdjYS1jZW50cmFsLTEnLFxuICAnZXUtY2VudHJhbC0xJyxcbiAgJ2V1LXdlc3QtMScsXG4gICdldS13ZXN0LTInLFxuICAnZXUtd2VzdC0zJyxcbiAgJ2NuLW5vcnRod2VzdC0xJyxcbl07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBLZXljbG9ha1ZlcnNpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxMl8wXzQgPSBLZXljbG9ha1ZlcnNpb24ub2YoJzEyLjAuNCcpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjE1XzBfMCA9IEtleWNsb2FrVmVyc2lvbi5vZignMTUuMC4wJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWMTVfMF8xID0gS2V5Y2xvYWtWZXJzaW9uLm9mKCcxNS4wLjEnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxNV8wXzIgPSBLZXljbG9ha1ZlcnNpb24ub2YoJzE1LjAuMicpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBvZih2ZXJzaW9uOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBLZXljbG9ha1ZlcnNpb24odmVyc2lvbik7IH1cbiAgLyoqXG4gICAqXG4gICAqIEBwYXJhbSB2ZXJzaW9uIGNsdXN0ZXIgdmVyc2lvbiBudW1iZXJcbiAgICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHZlcnNpb246IHN0cmluZykgeyB9XG59XG5cbi8vIGNvbnN0IEtFWUNMT0FLX1ZFUlNJT04gPSAnMTIuMC40JztcblxuaW50ZXJmYWNlIGRvY2tlckltYWdlTWFwIHtcbiAgJ2F3cyc6IHN0cmluZztcbiAgJ2F3cy1jbic6IHN0cmluZztcbn1cblxuY29uc3QgS0VZQ0xPQUtfRE9DS0VSX0lNQUdFX1VSSV9NQVA6IGRvY2tlckltYWdlTWFwID0ge1xuICAnYXdzJzogJ2pib3NzL2tleWNsb2FrOicsXG4gICdhd3MtY24nOiAnMDQ4OTEyMDYwOTEwLmRrci5lY3IuY24tbm9ydGh3ZXN0LTEuYW1hem9uYXdzLmNvbS5jbi9kb2NrZXJodWIvamJvc3Mva2V5Y2xvYWs6Jyxcbn07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQXV0b1NjYWxlVGFzayB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWluPzogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXg/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGFyZ2V0Q3B1VXRpbGl6YXRpb24/OiBudW1iZXI7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgS2V5Q2xvYWtQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkga2V5Y2xvYWtWZXJzaW9uOiBLZXljbG9ha1ZlcnNpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVudj86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjZXJ0aWZpY2F0ZUFybjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBiYXN0aW9uPzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbm9kZUNvdW50PzogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHVibGljU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcml2YXRlU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkYXRhYmFzZVN1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkYXRhYmFzZUluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlRW5naW5lPzogcmRzLklJbnN0YW5jZUVuZ2luZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2x1c3RlckVuZ2luZT86IHJkcy5JQ2x1c3RlckVuZ2luZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXVyb3JhU2VydmVybGVzcz86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNpbmdsZURiSW5zdGFuY2U/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBiYWNrdXBSZXRlbnRpb24/OiBjZGsuRHVyYXRpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN0aWNraW5lc3NDb29raWVEdXJhdGlvbj86IGNkay5EdXJhdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdXRvU2NhbGVUYXNrPzogQXV0b1NjYWxlVGFzaztcbn1cblxuZXhwb3J0IGNsYXNzIEtleUNsb2FrIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG4gIHJlYWRvbmx5IGRiPzogRGF0YWJhc2U7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogS2V5Q2xvYWtQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCByZWdpb24gPSBjZGsuU3RhY2sub2YodGhpcykucmVnaW9uO1xuICAgIGNvbnN0IHJlZ2lvbklzUmVzb2x2ZWQgPSAhY2RrLlRva2VuLmlzVW5yZXNvbHZlZChyZWdpb24pO1xuXG4gICAgaWYgKHByb3BzLmF1cm9yYVNlcnZlcmxlc3MgJiYgcmVnaW9uSXNSZXNvbHZlZCAmJiAhQVVST1JBX1NFUlZFUkxFU1NfU1VQUE9SVEVEX1JFR0lPTlMuaW5jbHVkZXMocmVnaW9uKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBdXJvcmEgc2VydmVybGVzcyBpcyBub3Qgc3VwcG9ydGVkIGluICR7cmVnaW9ufWApO1xuICAgIH1cblxuICAgIHRoaXMudnBjID0gcHJvcHMudnBjID8/IGdldE9yQ3JlYXRlVnBjKHRoaXMpO1xuICAgIHRoaXMuZGIgPSB0aGlzLmFkZERhdGFiYXNlKHtcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBkYXRhYmFzZVN1Ym5ldHM6IHByb3BzLmRhdGFiYXNlU3VibmV0cyxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuZGF0YWJhc2VJbnN0YW5jZVR5cGUsXG4gICAgICBpbnN0YW5jZUVuZ2luZTogcHJvcHMuaW5zdGFuY2VFbmdpbmUsXG4gICAgICBjbHVzdGVyRW5naW5lOiBwcm9wcy5jbHVzdGVyRW5naW5lLFxuICAgICAgYXVyb3JhU2VydmVybGVzczogcHJvcHMuYXVyb3JhU2VydmVybGVzcyxcbiAgICAgIHNpbmdsZURiSW5zdGFuY2U6IHByb3BzLnNpbmdsZURiSW5zdGFuY2UsXG4gICAgICBiYWNrdXBSZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbixcbiAgICB9KTtcbiAgICB0aGlzLmFkZEtleUNsb2FrQ29udGFpbmVyU2VydmljZSh7XG4gICAgICBkYXRhYmFzZTogdGhpcy5kYixcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBrZXljbG9ha1ZlcnNpb246IHByb3BzLmtleWNsb2FrVmVyc2lvbixcbiAgICAgIHB1YmxpY1N1Ym5ldHM6IHByb3BzLnB1YmxpY1N1Ym5ldHMsXG4gICAgICBwcml2YXRlU3VibmV0czogcHJvcHMucHJpdmF0ZVN1Ym5ldHMsXG4gICAgICBrZXljbG9ha1NlY3JldDogdGhpcy5fZ2VuZXJhdGVLZXljbG9ha1NlY3JldCgpLFxuICAgICAgY2VydGlmaWNhdGU6IGNlcnRtZ3IuQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHRoaXMsICdBQ01DZXJ0JywgcHJvcHMuY2VydGlmaWNhdGVBcm4pLFxuICAgICAgYmFzdGlvbjogcHJvcHMuYmFzdGlvbixcbiAgICAgIG5vZGVDb3VudDogcHJvcHMubm9kZUNvdW50LFxuICAgICAgc3RpY2tpbmVzc0Nvb2tpZUR1cmF0aW9uOiBwcm9wcy5zdGlja2luZXNzQ29va2llRHVyYXRpb24sXG4gICAgICBhdXRvU2NhbGVUYXNrOiBwcm9wcy5hdXRvU2NhbGVUYXNrLFxuICAgICAgZW52OiBwcm9wcy5lbnYsXG4gICAgfSk7XG4gICAgaWYgKCFjZGsuU3RhY2sub2YodGhpcykudGVtcGxhdGVPcHRpb25zLmRlc2NyaXB0aW9uKSB7XG4gICAgICBjZGsuU3RhY2sub2YodGhpcykudGVtcGxhdGVPcHRpb25zLmRlc2NyaXB0aW9uID0gJyhTTzgwMjEpIC0gRGVwbG95IGtleWNsb2FrIG9uIEFXUyB3aXRoIGNkay1rZXljbG9hayBjb25zdHJ1Y3QgbGlicmFyeSc7XG4gICAgfVxuICB9XG4gIHB1YmxpYyBhZGREYXRhYmFzZShwcm9wczogRGF0YWJhc2VQcm9wcyk6IERhdGFiYXNlIHtcbiAgICByZXR1cm4gbmV3IERhdGFiYXNlKHRoaXMsICdEYXRhYmFzZScsIHByb3BzKTtcbiAgfVxuICBwdWJsaWMgYWRkS2V5Q2xvYWtDb250YWluZXJTZXJ2aWNlKHByb3BzOiBDb250YWluZXJTZXJ2aWNlUHJvcHMpIHtcbiAgICByZXR1cm4gbmV3IENvbnRhaW5lclNlcnZpY2UodGhpcywgJ0tleUNsb2FrQ29udGFpbmVyU2VyaXZjZScsIHByb3BzKTtcbiAgfVxuICBwcml2YXRlIF9nZW5lcmF0ZUtleWNsb2FrU2VjcmV0KCk6IHNlY3JldHNtYW5hZ2VyLklTZWNyZXQge1xuICAgIHJldHVybiBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHRoaXMsICdLQ1NlY3JldCcsIHtcbiAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgIGdlbmVyYXRlU3RyaW5nS2V5OiAncGFzc3dvcmQnLFxuICAgICAgICBleGNsdWRlUHVuY3R1YXRpb246IHRydWUsXG4gICAgICAgIHBhc3N3b3JkTGVuZ3RoOiAxMixcbiAgICAgICAgc2VjcmV0U3RyaW5nVGVtcGxhdGU6IEpTT04uc3RyaW5naWZ5KHsgdXNlcm5hbWU6ICdrZXljbG9haycgfSksXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YWJhc2VTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlRW5naW5lPzogcmRzLklJbnN0YW5jZUVuZ2luZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2x1c3RlckVuZ2luZT86IHJkcy5JQ2x1c3RlckVuZ2luZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdXJvcmFTZXJ2ZXJsZXNzPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzaW5nbGVEYkluc3RhbmNlPzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYmFja3VwUmV0ZW50aW9uPzogY2RrLkR1cmF0aW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VDb2ZpZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3JldDogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5kcG9pbnQ6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpZGVudGlmaWVyOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIERhdGFiYXNlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG4gIHJlYWRvbmx5IGNsdXN0ZXJFbmRwb2ludEhvc3RuYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNsdXN0ZXJJZGVudGlmaWVyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNlY3JldDogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcbiAgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcbiAgcHJpdmF0ZSByZWFkb25seSBfbXlzcWxMaXN0ZW5lclBvcnQ6IG51bWJlciA9IDMzMDY7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEYXRhYmFzZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLnZwYyA9IHByb3BzLnZwYztcbiAgICBjb25zdCBjb25maWcgPSBwcm9wcy5hdXJvcmFTZXJ2ZXJsZXNzID8gdGhpcy5fY3JlYXRlU2VydmVybGVzc0NsdXN0ZXIocHJvcHMpXG4gICAgICA6IHByb3BzLnNpbmdsZURiSW5zdGFuY2UgPyB0aGlzLl9jcmVhdGVSZHNJbnN0YW5jZShwcm9wcykgOiB0aGlzLl9jcmVhdGVSZHNDbHVzdGVyKHByb3BzKTtcbiAgICB0aGlzLnNlY3JldCA9IGNvbmZpZy5zZWNyZXQ7XG4gICAgLy8gYWxsb3cgaW50ZXJuYWxseSBmcm9tIHRoZSBzYW1lIHNlY3VyaXR5IGdyb3VwXG4gICAgY29uZmlnLmNvbm5lY3Rpb25zLmFsbG93SW50ZXJuYWxseShlYzIuUG9ydC50Y3AodGhpcy5fbXlzcWxMaXN0ZW5lclBvcnQpKTtcbiAgICAvLyBhbGxvdyBmcm9tIHRoZSB3aG9sZSB2cGMgY2lkclxuICAgIGNvbmZpZy5jb25uZWN0aW9ucy5hbGxvd0Zyb20oZWMyLlBlZXIuaXB2NChwcm9wcy52cGMudnBjQ2lkckJsb2NrKSwgZWMyLlBvcnQudGNwKHRoaXMuX215c3FsTGlzdGVuZXJQb3J0KSk7XG4gICAgdGhpcy5jbHVzdGVyRW5kcG9pbnRIb3N0bmFtZSA9IGNvbmZpZy5lbmRwb2ludDtcbiAgICB0aGlzLmNsdXN0ZXJJZGVudGlmaWVyID0gY29uZmlnLmlkZW50aWZpZXI7XG4gICAgdGhpcy5jb25uZWN0aW9ucyA9IGNvbmZpZy5jb25uZWN0aW9ucztcbiAgICBwcmludE91dHB1dCh0aGlzLCAnREJTZWNyZXRBcm4nLCBjb25maWcuc2VjcmV0LnNlY3JldEFybik7XG4gICAgcHJpbnRPdXRwdXQodGhpcywgJ2NsdXN0ZXJFbmRwb2ludEhvc3RuYW1lJywgdGhpcy5jbHVzdGVyRW5kcG9pbnRIb3N0bmFtZSk7XG4gICAgcHJpbnRPdXRwdXQodGhpcywgJ2NsdXN0ZXJJZGVudGlmaWVyJywgdGhpcy5jbHVzdGVySWRlbnRpZmllcik7XG4gIH1cbiAgcHJpdmF0ZSBfY3JlYXRlUmRzSW5zdGFuY2UocHJvcHM6IERhdGFiYXNlUHJvcHMpOiBEYXRhYmFzZUNvZmlnIHtcbiAgICBjb25zdCBkYkluc3RhbmNlID0gbmV3IHJkcy5EYXRhYmFzZUluc3RhbmNlKHRoaXMsICdEQkluc3RhbmNlJywge1xuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICBkYXRhYmFzZU5hbWU6ICdrZXljbG9haycsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5kYXRhYmFzZVN1Ym5ldHMsXG4gICAgICBlbmdpbmU6IHByb3BzLmluc3RhbmNlRW5naW5lID8/IHJkcy5EYXRhYmFzZUluc3RhbmNlRW5naW5lLm15c3FsKHtcbiAgICAgICAgdmVyc2lvbjogcmRzLk15c3FsRW5naW5lVmVyc2lvbi5WRVJfOF8wXzIxLFxuICAgICAgfSksXG4gICAgICBzdG9yYWdlRW5jcnlwdGVkOiB0cnVlLFxuICAgICAgYmFja3VwUmV0ZW50aW9uOiBwcm9wcy5iYWNrdXBSZXRlbnRpb24gPz8gY2RrLkR1cmF0aW9uLmRheXMoNyksXG4gICAgICBjcmVkZW50aWFsczogcmRzLkNyZWRlbnRpYWxzLmZyb21HZW5lcmF0ZWRTZWNyZXQoJ2FkbWluJyksXG4gICAgICBpbnN0YW5jZVR5cGU6IHByb3BzLmluc3RhbmNlVHlwZSA/PyBuZXcgZWMyLkluc3RhbmNlVHlwZSgncjUubGFyZ2UnKSxcbiAgICAgIHBhcmFtZXRlckdyb3VwOiByZHMuUGFyYW1ldGVyR3JvdXAuZnJvbVBhcmFtZXRlckdyb3VwTmFtZSh0aGlzLCAnUGFyYW1ldGVyR3JvdXAnLCAnZGVmYXVsdC5teXNxbDguMCcpLFxuICAgICAgZGVsZXRpb25Qcm90ZWN0aW9uOiB0cnVlLFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgIH0pO1xuICAgIHJldHVybiB7XG4gICAgICBjb25uZWN0aW9uczogZGJJbnN0YW5jZS5jb25uZWN0aW9ucyxcbiAgICAgIGVuZHBvaW50OiBkYkluc3RhbmNlLmRiSW5zdGFuY2VFbmRwb2ludEFkZHJlc3MsXG4gICAgICBpZGVudGlmaWVyOiBkYkluc3RhbmNlLmluc3RhbmNlSWRlbnRpZmllcixcbiAgICAgIHNlY3JldDogZGJJbnN0YW5jZS5zZWNyZXQhLFxuICAgIH07XG4gIH1cbiAgLy8gY3JlYXRlIGEgUkRTIGZvciBNeVNRTCBEQiBjbHVzdGVyXG4gIHByaXZhdGUgX2NyZWF0ZVJkc0NsdXN0ZXIocHJvcHM6IERhdGFiYXNlUHJvcHMpOiBEYXRhYmFzZUNvZmlnIHtcbiAgICBjb25zdCBkYkNsdXN0ZXIgPSBuZXcgcmRzLkRhdGFiYXNlQ2x1c3Rlcih0aGlzLCAnREJDbHVzdGVyJywge1xuICAgICAgZW5naW5lOiBwcm9wcy5jbHVzdGVyRW5naW5lID8/IHJkcy5EYXRhYmFzZUNsdXN0ZXJFbmdpbmUuYXVyb3JhTXlzcWwoe1xuICAgICAgICB2ZXJzaW9uOiByZHMuQXVyb3JhTXlzcWxFbmdpbmVWZXJzaW9uLlZFUl8yXzA5XzEsXG4gICAgICB9KSxcbiAgICAgIGRlZmF1bHREYXRhYmFzZU5hbWU6ICdrZXljbG9haycsXG4gICAgICBkZWxldGlvblByb3RlY3Rpb246IHRydWUsXG4gICAgICBjcmVkZW50aWFsczogcmRzLkNyZWRlbnRpYWxzLmZyb21HZW5lcmF0ZWRTZWNyZXQoJ2FkbWluJyksXG4gICAgICBpbnN0YW5jZVByb3BzOiB7XG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgICB2cGNTdWJuZXRzOiBwcm9wcy5kYXRhYmFzZVN1Ym5ldHMsXG4gICAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlID8/IG5ldyBlYzIuSW5zdGFuY2VUeXBlKCdyNS5sYXJnZScpLFxuICAgICAgfSxcbiAgICAgIHBhcmFtZXRlckdyb3VwOiByZHMuUGFyYW1ldGVyR3JvdXAuZnJvbVBhcmFtZXRlckdyb3VwTmFtZSh0aGlzLCAnUGFyYW1ldGVyR3JvdXAnLCAnZGVmYXVsdC5hdXJvcmEtbXlzcWw1LjcnKSxcbiAgICAgIGJhY2t1cDoge1xuICAgICAgICByZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbiA/PyBjZGsuRHVyYXRpb24uZGF5cyg3KSxcbiAgICAgIH0sXG4gICAgICBzdG9yYWdlRW5jcnlwdGVkOiB0cnVlLFxuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgIH0pO1xuICAgIHJldHVybiB7XG4gICAgICBjb25uZWN0aW9uczogZGJDbHVzdGVyLmNvbm5lY3Rpb25zLFxuICAgICAgZW5kcG9pbnQ6IGRiQ2x1c3Rlci5jbHVzdGVyRW5kcG9pbnQuaG9zdG5hbWUsXG4gICAgICBpZGVudGlmaWVyOiBkYkNsdXN0ZXIuY2x1c3RlcklkZW50aWZpZXIsXG4gICAgICBzZWNyZXQ6IGRiQ2x1c3Rlci5zZWNyZXQhLFxuICAgIH07XG4gIH1cbiAgcHJpdmF0ZSBfY3JlYXRlU2VydmVybGVzc0NsdXN0ZXIocHJvcHM6IERhdGFiYXNlUHJvcHMpOiBEYXRhYmFzZUNvZmlnIHtcbiAgICBjb25zdCBkYkNsdXN0ZXIgPSBuZXcgcmRzLlNlcnZlcmxlc3NDbHVzdGVyKHRoaXMsICdBdXJvcmFTZXJ2ZXJsZXNzQ2x1c3RlcicsIHtcbiAgICAgIGVuZ2luZTogcmRzLkRhdGFiYXNlQ2x1c3RlckVuZ2luZS5BVVJPUkFfTVlTUUwsXG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgIGRlZmF1bHREYXRhYmFzZU5hbWU6ICdrZXljbG9haycsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5kYXRhYmFzZVN1Ym5ldHMsXG4gICAgICBjcmVkZW50aWFsczogcmRzLkNyZWRlbnRpYWxzLmZyb21HZW5lcmF0ZWRTZWNyZXQoJ2FkbWluJyksXG4gICAgICBiYWNrdXBSZXRlbnRpb246IHByb3BzLmJhY2t1cFJldGVudGlvbiA/PyBjZGsuRHVyYXRpb24uZGF5cyg3KSxcbiAgICAgIGRlbGV0aW9uUHJvdGVjdGlvbjogdHJ1ZSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LlJFVEFJTixcbiAgICAgIHBhcmFtZXRlckdyb3VwOiByZHMuUGFyYW1ldGVyR3JvdXAuZnJvbVBhcmFtZXRlckdyb3VwTmFtZSh0aGlzLCAnUGFyYW1ldGVyR3JvdXAnLCAnZGVmYXVsdC5hdXJvcmEtbXlzcWw1LjcnKSxcbiAgICB9KTtcbiAgICByZXR1cm4ge1xuICAgICAgY29ubmVjdGlvbnM6IGRiQ2x1c3Rlci5jb25uZWN0aW9ucyxcbiAgICAgIGVuZHBvaW50OiBkYkNsdXN0ZXIuY2x1c3RlckVuZHBvaW50Lmhvc3RuYW1lLFxuICAgICAgaWRlbnRpZmllcjogZGJDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyLFxuICAgICAgc2VjcmV0OiBkYkNsdXN0ZXIuc2VjcmV0ISxcbiAgICB9O1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyU2VydmljZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW52PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrZXljbG9ha1ZlcnNpb246IEtleWNsb2FrVmVyc2lvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwdWJsaWNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YWJhc2U6IERhdGFiYXNlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGtleWNsb2FrU2VjcmV0OiBzZWNyZXRzbWFuYWdlci5JU2VjcmV0O1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlOiBjZXJ0bWdyLklDZXJ0aWZpY2F0ZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJhc3Rpb24/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjaXJjdWl0QnJlYWtlcj86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG5vZGVDb3VudD86IG51bWJlcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3RpY2tpbmVzc0Nvb2tpZUR1cmF0aW9uPzogY2RrLkR1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXV0b1NjYWxlVGFzaz86IEF1dG9TY2FsZVRhc2s7XG59XG5cbmV4cG9ydCBjbGFzcyBDb250YWluZXJTZXJ2aWNlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IHNlcnZpY2U6IGVjcy5GYXJnYXRlU2VydmljZTtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDb250YWluZXJTZXJ2aWNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgdnBjID0gcHJvcHMudnBjO1xuICAgIGNvbnN0IGNsdXN0ZXIgPSBuZXcgZWNzLkNsdXN0ZXIodGhpcywgJ0NsdXN0ZXInLCB7IHZwYyB9KTtcbiAgICBjbHVzdGVyLm5vZGUuYWRkRGVwZW5kZW5jeShwcm9wcy5kYXRhYmFzZSk7XG4gICAgY29uc3QgdGFza1JvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1Rhc2tSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLkNvbXBvc2l0ZVByaW5jaXBhbChcbiAgICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlY3MuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Vjcy10YXNrcy5hbWF6b25hd3MuY29tJyksXG4gICAgICApLFxuICAgIH0pO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24odGhpcywgJ1Rhc2tEZWYnLCB7XG4gICAgICBjcHU6IDQwOTYsXG4gICAgICBtZW1vcnlMaW1pdE1pQjogODE5MixcbiAgICAgIGV4ZWN1dGlvblJvbGU6IHRhc2tSb2xlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbG9nR3JvdXAgPSBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnTG9nR3JvdXAnLCB7XG4gICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEgsXG4gICAgICByZW1vdmFsUG9saWN5OiBjZGsuUmVtb3ZhbFBvbGljeS5SRVRBSU4sXG4gICAgfSk7XG5cbiAgICBjb25zdCBrYyA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcigna2V5Y2xvYWsnLCB7XG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSh0aGlzLmdldEtleUNsb2FrRG9ja2VySW1hZ2VVcmkocHJvcHMua2V5Y2xvYWtWZXJzaW9uLnZlcnNpb24pKSxcbiAgICAgIGVudmlyb25tZW50OiBPYmplY3QuYXNzaWduKHtcbiAgICAgICAgREJfQUREUjogcHJvcHMuZGF0YWJhc2UuY2x1c3RlckVuZHBvaW50SG9zdG5hbWUsXG4gICAgICAgIERCX0RBVEFCQVNFOiAna2V5Y2xvYWsnLFxuICAgICAgICBEQl9QT1JUOiAnMzMwNicsXG4gICAgICAgIERCX1VTRVI6ICdhZG1pbicsXG4gICAgICAgIERCX1ZFTkRPUjogJ215c3FsJyxcbiAgICAgICAgLy8gS0VZQ0xPQUtfTE9HTEVWRUw6ICdERUJVRycsXG4gICAgICAgIFBST1hZX0FERFJFU1NfRk9SV0FSRElORzogJ3RydWUnLFxuICAgICAgICBKREJDX1BBUkFNUzogJ3VzZVNTTD1mYWxzZScsXG4gICAgICAgIEpHUk9VUFNfRElTQ09WRVJZX1BST1RPQ09MOiAnSkRCQ19QSU5HJyxcbiAgICAgICAgLy8gV2UgZG9uJ3QgbmVlZCB0byBzcGVjaWZ5IGBpbml0aWFsaXplX3NxbGAgc3RyaW5nIGludG8gYEpHUk9VUFNfRElTQ09WRVJZX1BST1BFUlRJRVNgIHByb3BlcnR5LFxuICAgICAgICAvLyBiZWNhdXNlIHRoZSBkZWZhdWx0IGBpbml0aWFsaXplX3NxbGAgaXMgY29tcGF0aWJsZSB3aXRoIE15U1FMLiAoU2VlOiBodHRwczovL2dpdGh1Yi5jb20vYmVsYWJhbi9KR3JvdXBzL2Jsb2IvbWFzdGVyL3NyYy9vcmcvamdyb3Vwcy9wcm90b2NvbHMvSkRCQ19QSU5HLmphdmEjTDU1LUw2MClcbiAgICAgICAgLy8gQnV0IHlvdSBuZWVkIHRvIHNwZWNpZnkgYGluaXRpYWxpemVfc3FsYCBmb3IgUG9zdGdyZVNRTCwgYmVjYXVzZSBgdmFyYmluYXJ5YCBzY2hlbWEgaXMgbm90IHN1cHBvcnRlZC4gKFNlZTogaHR0cHM6Ly9naXRodWIuY29tL2tleWNsb2FrL2tleWNsb2FrLWNvbnRhaW5lcnMvYmxvYi9kNGNlNDQ2ZGRlMzAyNmY4OWY2NmZhODZiNThjMmQwZDYxMzJjZTRkL2RvY2tlci1jb21wb3NlLWV4YW1wbGVzL2tleWNsb2FrLXBvc3RncmVzLWpkYmMtcGluZy55bWwjTDQ5KVxuICAgICAgICAvLyBKR1JPVVBTX0RJU0NPVkVSWV9QUk9QRVJUSUVTOiAnJyxcbiAgICAgIH0sIHByb3BzLmVudiksXG4gICAgICBzZWNyZXRzOiB7XG4gICAgICAgIERCX1BBU1NXT1JEOiBlY3MuU2VjcmV0LmZyb21TZWNyZXRzTWFuYWdlcihwcm9wcy5kYXRhYmFzZS5zZWNyZXQsICdwYXNzd29yZCcpLFxuICAgICAgICBLRVlDTE9BS19VU0VSOiBlY3MuU2VjcmV0LmZyb21TZWNyZXRzTWFuYWdlcihwcm9wcy5rZXljbG9ha1NlY3JldCwgJ3VzZXJuYW1lJyksXG4gICAgICAgIEtFWUNMT0FLX1BBU1NXT1JEOiBlY3MuU2VjcmV0LmZyb21TZWNyZXRzTWFuYWdlcihwcm9wcy5rZXljbG9ha1NlY3JldCwgJ3Bhc3N3b3JkJyksXG4gICAgICB9LFxuICAgICAgbG9nZ2luZzogZWNzLkxvZ0RyaXZlcnMuYXdzTG9ncyh7XG4gICAgICAgIHN0cmVhbVByZWZpeDogJ2tleWNsb2FrJyxcbiAgICAgICAgbG9nR3JvdXAsXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICBrYy5hZGRQb3J0TWFwcGluZ3MoXG4gICAgICB7IGNvbnRhaW5lclBvcnQ6IDg0NDMgfSwgLy8gSFRUUFMgd2ViIHBvcnRcbiAgICAgIHsgY29udGFpbmVyUG9ydDogNzYwMCB9LCAvLyBqZ3JvdXBzLXRjcFxuICAgICAgeyBjb250YWluZXJQb3J0OiA1NzYwMCB9LCAvLyBqZ3JvdXBzLXRjcC1mZFxuICAgICAgeyBjb250YWluZXJQb3J0OiA1NTIwMCwgcHJvdG9jb2w6IGVjcy5Qcm90b2NvbC5VRFAgfSwgLy8gamdyb3Vwcy11ZHBcbiAgICAgIHsgY29udGFpbmVyUG9ydDogNTQyMDAsIHByb3RvY29sOiBlY3MuUHJvdG9jb2wuVURQIH0sIC8vIGpncm91cHMtdWRwLWZkXG4gICAgKTtcblxuICAgIC8vIHdlIG5lZWQgZXh0cmEgcHJpdmlsZWdlcyB0byBmZXRjaCBrZXljbG9hayBkb2NrZXIgaW1hZ2VzIGZyb20gQ2hpbmEgbWlycm9yIHNpdGVcbiAgICB0YXNrRGVmaW5pdGlvbi5leGVjdXRpb25Sb2xlPy5hZGRNYW5hZ2VkUG9saWN5KGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQW1hem9uRUMyQ29udGFpbmVyUmVnaXN0cnlSZWFkT25seScpKTtcblxuICAgIHRoaXMuc2VydmljZSA9IG5ldyBlY3MuRmFyZ2F0ZVNlcnZpY2UodGhpcywgJ1NlcnZpY2UnLCB7XG4gICAgICBjbHVzdGVyLFxuICAgICAgdGFza0RlZmluaXRpb24sXG4gICAgICBjaXJjdWl0QnJlYWtlcjogcHJvcHMuY2lyY3VpdEJyZWFrZXIgPyB7IHJvbGxiYWNrOiB0cnVlIH0gOiB1bmRlZmluZWQsXG4gICAgICBkZXNpcmVkQ291bnQ6IHByb3BzLm5vZGVDb3VudCA/PyAyLFxuICAgICAgaGVhbHRoQ2hlY2tHcmFjZVBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMTIwKSxcbiAgICB9KTtcbiAgICAvLyB3ZSBuZWVkIHRvIGFsbG93IHRyYWZmaWMgZnJvbSB0aGUgc2FtZSBzZWNyZXQgZ3JvdXAgZm9yIGtleWNsb2FrIGNsdXN0ZXIgd2l0aCBqZGJjX3BpbmdcbiAgICB0aGlzLnNlcnZpY2UuY29ubmVjdGlvbnMuYWxsb3dGcm9tKHRoaXMuc2VydmljZS5jb25uZWN0aW9ucywgZWMyLlBvcnQudGNwKDc2MDApLCAna2Mgamdyb3Vwcy10Y3AnKTtcbiAgICB0aGlzLnNlcnZpY2UuY29ubmVjdGlvbnMuYWxsb3dGcm9tKHRoaXMuc2VydmljZS5jb25uZWN0aW9ucywgZWMyLlBvcnQudGNwKDU3NjAwKSwgJ2tjIGpncm91cHMtdGNwLWZkJyk7XG4gICAgdGhpcy5zZXJ2aWNlLmNvbm5lY3Rpb25zLmFsbG93RnJvbSh0aGlzLnNlcnZpY2UuY29ubmVjdGlvbnMsIGVjMi5Qb3J0LnVkcCg1NTIwMCksICdrYyBqZ3JvdXBzLXVkcCcpO1xuICAgIHRoaXMuc2VydmljZS5jb25uZWN0aW9ucy5hbGxvd0Zyb20odGhpcy5zZXJ2aWNlLmNvbm5lY3Rpb25zLCBlYzIuUG9ydC51ZHAoNTQyMDApLCAna2Mgamdyb3Vwcy11ZHAtZmQnKTtcblxuICAgIGlmIChwcm9wcy5hdXRvU2NhbGVUYXNrKSB7XG4gICAgICBjb25zdCBtaW5DYXBhY2l0eSA9IHByb3BzLmF1dG9TY2FsZVRhc2subWluID8/IHByb3BzLm5vZGVDb3VudCA/PyAyO1xuICAgICAgY29uc3Qgc2NhbGluZyA9IHRoaXMuc2VydmljZS5hdXRvU2NhbGVUYXNrQ291bnQoe1xuICAgICAgICBtaW5DYXBhY2l0eSxcbiAgICAgICAgbWF4Q2FwYWNpdHk6IHByb3BzLmF1dG9TY2FsZVRhc2subWF4ID8/IG1pbkNhcGFjaXR5ICsgNSxcbiAgICAgIH0pO1xuICAgICAgc2NhbGluZy5zY2FsZU9uQ3B1VXRpbGl6YXRpb24oJ0NwdVNjYWxpbmcnLCB7XG4gICAgICAgIHRhcmdldFV0aWxpemF0aW9uUGVyY2VudDogcHJvcHMuYXV0b1NjYWxlVGFzay50YXJnZXRDcHVVdGlsaXphdGlvbiA/PyA3NSxcbiAgICAgIH0pO1xuICAgIH07XG5cbiAgICBjb25zdCBhbGIgPSBuZXcgZWxidjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgJ0FMQicsIHtcbiAgICAgIHZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnB1YmxpY1N1Ym5ldHMsXG4gICAgICBpbnRlcm5ldEZhY2luZzogdHJ1ZSxcbiAgICB9KTtcbiAgICBwcmludE91dHB1dCh0aGlzLCAnRW5kcG9pbnRVUkwnLCBgaHR0cHM6Ly8ke2FsYi5sb2FkQmFsYW5jZXJEbnNOYW1lfWApO1xuXG4gICAgY29uc3QgbGlzdGVuZXIgPSBhbGIuYWRkTGlzdGVuZXIoJ0h0dHBzTGlzdGVuZXInLCB7XG4gICAgICBwcm90b2NvbDogZWxidjIuQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUyxcbiAgICAgIGNlcnRpZmljYXRlczogW3sgY2VydGlmaWNhdGVBcm46IHByb3BzLmNlcnRpZmljYXRlLmNlcnRpZmljYXRlQXJuIH1dLFxuICAgIH0pO1xuXG4gICAgbGlzdGVuZXIuYWRkVGFyZ2V0cygnRUNTVGFyZ2V0Jywge1xuICAgICAgdGFyZ2V0czogW3RoaXMuc2VydmljZV0sXG4gICAgICBoZWFsdGhDaGVjazoge1xuICAgICAgICBoZWFsdGh5VGhyZXNob2xkQ291bnQ6IDMsXG4gICAgICB9LFxuICAgICAgLy8gc2V0IHNsb3dfc3RhcnQuZHVyYXRpb25fc2Vjb25kcyB0byA2MFxuICAgICAgLy8gc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbGkvbGF0ZXN0L3JlZmVyZW5jZS9lbGJ2Mi9tb2RpZnktdGFyZ2V0LWdyb3VwLWF0dHJpYnV0ZXMuaHRtbFxuICAgICAgc2xvd1N0YXJ0OiBjZGsuRHVyYXRpb24uc2Vjb25kcyg2MCksXG4gICAgICBzdGlja2luZXNzQ29va2llRHVyYXRpb246IHByb3BzLnN0aWNraW5lc3NDb29raWVEdXJhdGlvbiA/PyBjZGsuRHVyYXRpb24uZGF5cygxKSxcbiAgICAgIHBvcnQ6IDg0NDMsXG4gICAgICBwcm90b2NvbDogZWxidjIuQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUyxcbiAgICB9KTtcblxuICAgIC8vIGFsbG93IHRhc2sgZXhlY3V0aW9uIHJvbGUgdG8gcmVhZCB0aGUgc2VjcmV0c1xuICAgIHByb3BzLmRhdGFiYXNlLnNlY3JldC5ncmFudFJlYWQodGFza0RlZmluaXRpb24uZXhlY3V0aW9uUm9sZSEpO1xuICAgIHByb3BzLmtleWNsb2FrU2VjcmV0LmdyYW50UmVhZCh0YXNrRGVmaW5pdGlvbi5leGVjdXRpb25Sb2xlISk7XG5cbiAgICAvLyBhbGxvdyBlY3MgdGFzayBjb25uZWN0IHRvIGRhdGFiYXNlXG4gICAgcHJvcHMuZGF0YWJhc2UuY29ubmVjdGlvbnMuYWxsb3dEZWZhdWx0UG9ydEZyb20odGhpcy5zZXJ2aWNlKTtcblxuXG4gICAgLy8gY3JlYXRlIGEgYmFzdGlvbiBob3N0XG4gICAgaWYgKHByb3BzLmJhc3Rpb24gPT09IHRydWUpIHtcbiAgICAgIGNvbnN0IGJhc3QgPSBuZXcgZWMyLkJhc3Rpb25Ib3N0TGludXgodGhpcywgJ0Jhc3QnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgZWMyLkluc3RhbmNlVHlwZSgnbTUubGFyZ2UnKSxcbiAgICAgIH0pO1xuICAgICAgcHJvcHMuZGF0YWJhc2UuY29ubmVjdGlvbnMuYWxsb3dEZWZhdWx0UG9ydEZyb20oYmFzdCk7XG4gICAgfVxuICB9XG4gIHByaXZhdGUgZ2V0SW1hZ2VVcmlGcm9tTWFwKG1hcDogZG9ja2VySW1hZ2VNYXAsIHZlcnNpb246IHN0cmluZywgaWQ6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2YodGhpcyk7XG4gICAgaWYgKGNkay5Ub2tlbi5pc1VucmVzb2x2ZWQoc3RhY2sucmVnaW9uKSkge1xuICAgICAgY29uc3QgbWFwcGluZzogeyBbazE6IHN0cmluZ106IHsgW2syOiBzdHJpbmddOiBhbnkgfSB9ID0ge307XG4gICAgICBmb3IgKGxldCBbcGFydGl0aW9uLCB1cmldIG9mIE9iamVjdC5lbnRyaWVzKG1hcCkpIHtcbiAgICAgICAgdXJpICs9IHZlcnNpb247XG4gICAgICAgIG1hcHBpbmdbcGFydGl0aW9uXSA9IHsgdXJpIH07XG4gICAgICB9XG4gICAgICBjb25zdCBpbWFnZU1hcCA9IG5ldyBjZGsuQ2ZuTWFwcGluZyh0aGlzLCBpZCwgeyBtYXBwaW5nIH0pO1xuICAgICAgcmV0dXJuIGltYWdlTWFwLmZpbmRJbk1hcChjZGsuQXdzLlBBUlRJVElPTiwgJ3VyaScpO1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAoc3RhY2sucmVnaW9uLnN0YXJ0c1dpdGgoJ2NuLScpKSB7XG4gICAgICAgIHJldHVybiBtYXBbJ2F3cy1jbiddICs9IHZlcnNpb247XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gbWFwLmF3cyArPSB2ZXJzaW9uO1xuICAgICAgfVxuICAgIH1cbiAgfVxuICBwcml2YXRlIGdldEtleUNsb2FrRG9ja2VySW1hZ2VVcmkodmVyc2lvbjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5nZXRJbWFnZVVyaUZyb21NYXAoS0VZQ0xPQUtfRE9DS0VSX0lNQUdFX1VSSV9NQVAsIHZlcnNpb24sICdLZXljbG9ha0ltYWdlTWFwJyk7XG4gIH1cbn1cblxuLyoqXG4gKiBDcmVhdGUgb3IgaW1wb3J0IFZQQ1xuICogQHBhcmFtIHNjb3BlIHRoZSBjZGsgc2NvcGVcbiAqL1xuZnVuY3Rpb24gZ2V0T3JDcmVhdGVWcGMoc2NvcGU6IGNkay5Db25zdHJ1Y3QpOiBlYzIuSVZwYyB7XG4gIC8vIHVzZSBhbiBleGlzdGluZyB2cGMgb3IgY3JlYXRlIGEgbmV3IG9uZVxuICByZXR1cm4gc2NvcGUubm9kZS50cnlHZXRDb250ZXh0KCd1c2VfZGVmYXVsdF92cGMnKSA9PT0gJzEnID9cbiAgICBlYzIuVnBjLmZyb21Mb29rdXAoc2NvcGUsICdWcGMnLCB7IGlzRGVmYXVsdDogdHJ1ZSB9KSA6XG4gICAgc2NvcGUubm9kZS50cnlHZXRDb250ZXh0KCd1c2VfdnBjX2lkJykgP1xuICAgICAgZWMyLlZwYy5mcm9tTG9va3VwKHNjb3BlLCAnVnBjJywgeyB2cGNJZDogc2NvcGUubm9kZS50cnlHZXRDb250ZXh0KCd1c2VfdnBjX2lkJykgfSkgOlxuICAgICAgbmV3IGVjMi5WcGMoc2NvcGUsICdWcGMnLCB7IG1heEF6czogMywgbmF0R2F0ZXdheXM6IDEgfSk7XG59XG5cbmZ1bmN0aW9uIHByaW50T3V0cHV0KHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBrZXk6IHN0cmluZyB8IG51bWJlcikge1xuICBuZXcgY2RrLkNmbk91dHB1dChzY29wZSwgaWQsIHsgdmFsdWU6IFN0cmluZyhrZXkpIH0pO1xufVxuIl19