"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk8s_1 = require("cdk8s");
/**
 * A CDK8s app which allows implementing Kubernetes operators using CDK8s constructs.
 *
 * @stability stable
 */
class Operator extends cdk8s_1.App {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        var _a, _b;
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk8s'));
        super({ outdir });
        this.providers = [];
        this.inputFile = (_b = (_a = props.inputFile) !== null && _a !== void 0 ? _a : process.argv[2]) !== null && _b !== void 0 ? _b : '/dev/stdin';
        this.outputFile = props.outputFile;
    }
    /**
     * Adds a custom resource provider to this operator.
     *
     * @param provider The provider to add.
     * @stability stable
     */
    addProvider(provider) {
        this.providers.push(provider);
    }
    /**
     * Reads a Kubernetes manifest in JSON format from STDIN or the file specified as the first positional command-line argument.
     *
     * This manifest is expected to
     * include a single Kubernetes resource. Then, we match `apiVersion` and
     * `kind` to one of the registered providers and if we do, we invoke
     * `apply()`, passing it the `spec` of the input manifest and a chart as a
     * scope. The chart is then synthesized and the output manifest is written to
     * STDOUT.
     *
     * @stability stable
     */
    synth() {
        var _a, _b, _c;
        const input = JSON.parse(fs.readFileSync(this.inputFile, 'utf-8'));
        let write;
        if (this.outputFile) {
            const outfile = this.outputFile;
            write = (data) => fs.writeFileSync(outfile, data);
        }
        else {
            write = (data) => process.stdout.write(data);
        }
        if (typeof (input) !== 'object') {
            throw new Error('input must be a single kubernetes resource');
        }
        const provider = this.findProvider(input);
        const name = (_a = input.metadata) === null || _a === void 0 ? void 0 : _a.name;
        if (!name) {
            throw new Error('"metadata.name" must be defined');
        }
        const namespace = (_b = input.metadata) === null || _b === void 0 ? void 0 : _b.namespace;
        // TODO: namespace
        const spec = (_c = input.spec) !== null && _c !== void 0 ? _c : {};
        const chart = new cdk8s_1.Chart(this, name, { namespace });
        console.error(`Synthesizing ${input.kind}.${input.apiVersion}`);
        provider.handler.apply(chart, name, spec);
        super.synth();
        for (const file of fs.readdirSync(this.outdir)) {
            const filepath = path.join(this.outdir, file);
            const manifest = fs.readFileSync(filepath);
            write(manifest);
        }
    }
    findProvider(input) {
        const { apiVersion, kind } = input;
        if (!apiVersion) {
            throw new Error('"apiVersion" is required');
        }
        if (!kind) {
            throw new Error('"kind" is required');
        }
        for (const p of this.providers) {
            if (p.apiVersion === apiVersion && p.kind === kind) {
                return p;
            }
        }
        throw new Error(`No custom resource provider found for ${kind}.${apiVersion}`);
    }
}
exports.Operator = Operator;
//# sourceMappingURL=data:application/json;base64,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