"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
/**
 * Adds a header to the received email
 */
class ReceiptRuleAddHeaderAction {
    constructor(props) {
        if (!/^[a-zA-Z0-9-]{1,50}$/.test(props.name)) {
            // tslint:disable:max-line-length
            throw new Error('Header `name` must be between 1 and 50 characters, inclusive, and consist of alphanumeric (a-z, A-Z, 0-9) characters and dashes only.');
            // tslint:enable:max-line-length
        }
        if (!/^[^\n\r]{0,2047}$/.test(props.value)) {
            throw new Error('Header `value` must be less than 2048 characters, and must not contain newline characters ("\r" or "\n").');
        }
        this.name = props.name;
        this.value = props.value;
    }
    render() {
        return {
            addHeaderAction: {
                headerName: this.name,
                headerValue: this.value
            }
        };
    }
}
exports.ReceiptRuleAddHeaderAction = ReceiptRuleAddHeaderAction;
/**
 * A bounce action template.
 */
class ReceiptRuleBounceActionTemplate {
    constructor(props) {
        this.message = props.message;
        this.smtpReplyCode = props.smtpReplyCode;
        this.statusCode = props.statusCode;
    }
}
ReceiptRuleBounceActionTemplate.MailboxDoesNotExist = new ReceiptRuleBounceActionTemplate({
    message: 'Mailbox does not exist',
    smtpReplyCode: '550',
    statusCode: '5.1.1'
});
ReceiptRuleBounceActionTemplate.MessageTooLarge = new ReceiptRuleBounceActionTemplate({
    message: 'Message too large',
    smtpReplyCode: '552',
    statusCode: '5.3.4'
});
ReceiptRuleBounceActionTemplate.MailboxFull = new ReceiptRuleBounceActionTemplate({
    message: 'Mailbox full',
    smtpReplyCode: '552',
    statusCode: '5.2.2'
});
ReceiptRuleBounceActionTemplate.MessageContentRejected = new ReceiptRuleBounceActionTemplate({
    message: 'Message content rejected',
    smtpReplyCode: '500',
    statusCode: '5.6.1'
});
ReceiptRuleBounceActionTemplate.TemporaryFailure = new ReceiptRuleBounceActionTemplate({
    message: 'Temporary failure',
    smtpReplyCode: '450',
    statusCode: '4.0.0'
});
exports.ReceiptRuleBounceActionTemplate = ReceiptRuleBounceActionTemplate;
/**
 * Rejects the received email by returning a bounce response to the sender and,
 * optionally, publishes a notification to Amazon SNS.
 */
class ReceiptRuleBounceAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            bounceAction: {
                sender: this.props.sender,
                smtpReplyCode: this.props.template.smtpReplyCode,
                message: this.props.template.message,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
                statusCode: this.props.template.statusCode
            }
        };
    }
}
exports.ReceiptRuleBounceAction = ReceiptRuleBounceAction;
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["Event"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["RequestResponse"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class ReceiptRuleLambdaAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.node.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.accountId
            });
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleLambdaAction = ReceiptRuleLambdaAction;
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class ReceiptRuleS3Action {
    constructor(props) {
        this.props = props;
    }
    render() {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement()
            .addAction('s3:PutObject')
            .addServicePrincipal('ses.amazonaws.com')
            .addResource(this.props.bucket.arnForObjects(`${keyPattern}*`))
            .addCondition('StringEquals', {
            'aws:Referer': cdk.Aws.accountId
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement()
                .addActions('km:Encrypt', 'kms:GenerateDataKey')
                .addServicePrincipal('ses.amazonaws.com')
                .addAllResources()
                .addConditions({
                Null: {
                    'kms:EncryptionContext:aws:ses:rule-name': 'false',
                    'kms:EncryptionContext:aws:ses:message-id': 'false'
                },
                StringEquals: {
                    'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.accountId
                }
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleS3Action = ReceiptRuleS3Action;
/**
 * The type of email encoding to use for a SNS action.
 */
var EmailEncoding;
(function (EmailEncoding) {
    /**
     * Base 64
     */
    EmailEncoding["Base64"] = "Base64";
    /**
     * UTF-8
     */
    EmailEncoding["UTF8"] = "UTF-8";
})(EmailEncoding = exports.EmailEncoding || (exports.EmailEncoding = {}));
/**
 * Publishes the email content within a notification to Amazon SNS.
 */
class ReceiptRuleSnsAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            snsAction: {
                encoding: this.props.encoding,
                topicArn: this.props.topic.topicArn
            }
        };
    }
}
exports.ReceiptRuleSnsAction = ReceiptRuleSnsAction;
/**
 * Terminates the evaluation of the receipt rule set and optionally publishes a
 * notification to Amazon SNS.
 */
class ReceiptRuleStopAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            stopAction: {
                scope: 'RuleSet',
                topicArn: this.props && this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleStopAction = ReceiptRuleStopAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVjZWlwdC1ydWxlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlY2VpcHQtcnVsZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBeUM7QUFLekMsb0NBQXFDO0FBMkVyQzs7R0FFRztBQUNILE1BQWEsMEJBQTBCO0lBSXJDLFlBQVksS0FBc0M7UUFDaEQsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDNUMsaUNBQWlDO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUlBQXVJLENBQUMsQ0FBQztZQUN6SixnQ0FBZ0M7U0FDakM7UUFFRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLDJHQUEyRyxDQUFDLENBQUM7U0FDOUg7UUFFRCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdkIsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO0lBQzNCLENBQUM7SUFFTSxNQUFNO1FBQ1gsT0FBTztZQUNMLGVBQWUsRUFBRTtnQkFDZixVQUFVLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ3JCLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSzthQUN4QjtTQUNGLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUEzQkQsZ0VBMkJDO0FBMEJEOztHQUVHO0FBQ0gsTUFBYSwrQkFBK0I7SUFtQzFDLFlBQVksS0FBMkM7UUFDckQsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUN6QyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7SUFDckMsQ0FBQzs7QUF0Q3NCLG1EQUFtQixHQUFHLElBQUksK0JBQStCLENBQUM7SUFDL0UsT0FBTyxFQUFFLHdCQUF3QjtJQUNqQyxhQUFhLEVBQUUsS0FBSztJQUNwQixVQUFVLEVBQUUsT0FBTztDQUNwQixDQUFDLENBQUM7QUFFb0IsK0NBQWUsR0FBRyxJQUFJLCtCQUErQixDQUFDO0lBQzNFLE9BQU8sRUFBRSxtQkFBbUI7SUFDNUIsYUFBYSxFQUFFLEtBQUs7SUFDcEIsVUFBVSxFQUFFLE9BQU87Q0FDcEIsQ0FBQyxDQUFDO0FBRW9CLDJDQUFXLEdBQUcsSUFBSSwrQkFBK0IsQ0FBQztJQUN2RSxPQUFPLEVBQUUsY0FBYztJQUN2QixhQUFhLEVBQUUsS0FBSztJQUNwQixVQUFVLEVBQUUsT0FBTztDQUNwQixDQUFDLENBQUM7QUFFb0Isc0RBQXNCLEdBQUcsSUFBSSwrQkFBK0IsQ0FBQztJQUNsRixPQUFPLEVBQUUsMEJBQTBCO0lBQ25DLGFBQWEsRUFBRSxLQUFLO0lBQ3BCLFVBQVUsRUFBRSxPQUFPO0NBQ3BCLENBQUMsQ0FBQztBQUVvQixnREFBZ0IsR0FBRyxJQUFJLCtCQUErQixDQUFDO0lBQzVFLE9BQU8sRUFBRSxtQkFBbUI7SUFDNUIsYUFBYSxFQUFFLEtBQUs7SUFDcEIsVUFBVSxFQUFFLE9BQU87Q0FDcEIsQ0FBQyxDQUFDO0FBN0JMLDBFQXdDQztBQXlCRDs7O0dBR0c7QUFDSCxNQUFhLHVCQUF1QjtJQUNsQyxZQUE2QixLQUFtQztRQUFuQyxVQUFLLEdBQUwsS0FBSyxDQUE4QjtJQUNoRSxDQUFDO0lBRU0sTUFBTTtRQUNYLE9BQU87WUFDTCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTTtnQkFDekIsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLGFBQWE7Z0JBQ2hELE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPO2dCQUNwQyxRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDbEUsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQVU7YUFDM0M7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBZkQsMERBZUM7QUFFRDs7R0FFRztBQUNILElBQVksb0JBWVg7QUFaRCxXQUFZLG9CQUFvQjtJQUM5Qjs7T0FFRztJQUNILHVDQUFlLENBQUE7SUFFZjs7OztPQUlHO0lBQ0gsMkRBQW1DLENBQUE7QUFDckMsQ0FBQyxFQVpXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBWS9CO0FBMEJEOzs7R0FHRztBQUNILE1BQWEsdUJBQXVCO0lBQ2xDLFlBQTZCLEtBQW1DO1FBQW5DLFVBQUssR0FBTCxLQUFLLENBQThCO0lBQ2hFLENBQUM7SUFFTSxNQUFNO1FBQ1gsc0NBQXNDO1FBQ3RDLGdJQUFnSTtRQUNoSSxNQUFNLFlBQVksR0FBRyxVQUFVLENBQUM7UUFDaEMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDeEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRTtnQkFDOUMsTUFBTSxFQUFFLHVCQUF1QjtnQkFDL0IsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO2dCQUN4RCxhQUFhLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTO2FBQ2pDLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTztZQUNMLFlBQVksRUFBRTtnQkFDWixXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsV0FBVztnQkFDNUMsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYztnQkFDekMsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDbkU7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBeEJELDBEQXdCQztBQWtDRDs7O0dBR0c7QUFDSCxNQUFhLG1CQUFtQjtJQUM5QixZQUE2QixLQUErQjtRQUEvQixVQUFLLEdBQUwsS0FBSyxDQUEwQjtJQUM1RCxDQUFDO0lBRU0sTUFBTTtRQUNYLGtDQUFrQztRQUNsQyw0SEFBNEg7UUFDNUgsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLElBQUksRUFBRSxDQUFDO1FBRXBELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsRUFBRTthQUMxQyxTQUFTLENBQUMsY0FBYyxDQUFDO2FBQ3pCLG1CQUFtQixDQUFDLG1CQUFtQixDQUFDO2FBQ3hDLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxVQUFVLEdBQUcsQ0FBQyxDQUFDO2FBQzlELFlBQVksQ0FBQyxjQUFjLEVBQUU7WUFDNUIsYUFBYSxFQUFFLEdBQUcsQ0FBQyxHQUFHLENBQUMsU0FBUztTQUNqQyxDQUFDLENBQUM7UUFFTCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUVuRCxrQ0FBa0M7UUFDbEMsNkhBQTZIO1FBQzdILElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDM0UsTUFBTSxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxFQUFFO2lCQUMzQyxVQUFVLENBQUMsWUFBWSxFQUFFLHFCQUFxQixDQUFDO2lCQUMvQyxtQkFBbUIsQ0FBQyxtQkFBbUIsQ0FBQztpQkFDeEMsZUFBZSxFQUFFO2lCQUNqQixhQUFhLENBQUM7Z0JBQ2IsSUFBSSxFQUFFO29CQUNKLHlDQUF5QyxFQUFFLE9BQU87b0JBQ2xELDBDQUEwQyxFQUFFLE9BQU87aUJBQ3BEO2dCQUNELFlBQVksRUFBRTtvQkFDWiw4Q0FBOEMsRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLFNBQVM7aUJBQ2xFO2FBQ0YsQ0FBQyxDQUFDO1lBRUwsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDckQ7UUFFRCxPQUFPO1lBQ0wsUUFBUSxFQUFFO2dCQUNSLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUN4QyxTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDbkUsZUFBZSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZTtnQkFDM0MsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDbkU7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBaERELGtEQWdEQztBQUVEOztHQUVHO0FBQ0gsSUFBWSxhQVVYO0FBVkQsV0FBWSxhQUFhO0lBQ3ZCOztPQUVHO0lBQ0gsa0NBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCwrQkFBYyxDQUFBO0FBQ2hCLENBQUMsRUFWVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQVV4QjtBQW1CRDs7R0FFRztBQUNILE1BQWEsb0JBQW9CO0lBQy9CLFlBQTZCLEtBQWdDO1FBQWhDLFVBQUssR0FBTCxLQUFLLENBQTJCO0lBQzdELENBQUM7SUFFTSxNQUFNO1FBQ1gsT0FBTztZQUNMLFNBQVMsRUFBRTtnQkFDVCxRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRO2dCQUM3QixRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsUUFBUTthQUNwQztTQUNGLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFaRCxvREFZQztBQVlEOzs7R0FHRztBQUNILE1BQWEscUJBQXFCO0lBQ2hDLFlBQTZCLEtBQWtDO1FBQWxDLFVBQUssR0FBTCxLQUFLLENBQTZCO0lBQy9ELENBQUM7SUFFTSxNQUFNO1FBQ1gsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixLQUFLLEVBQUUsU0FBUztnQkFDaEIsUUFBUSxFQUFFLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUNqRjtTQUNGLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFaRCxzREFZQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQga21zID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWttcycpO1xuaW1wb3J0IGxhbWJkYSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbmltcG9ydCBzMyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1zMycpO1xuaW1wb3J0IHNucyA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1zbnMnKTtcbmltcG9ydCBjZGsgPSByZXF1aXJlKCdAYXdzLWNkay9jZGsnKTtcbmltcG9ydCB7IENmblJlY2VpcHRSdWxlIH0gZnJvbSAnLi9zZXMuZ2VuZXJhdGVkJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIHJlY2VpcHQgcnVsZSBhY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjZWlwdFJ1bGVBY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBBZGRzIGEgaGVhZGVyIHRvIHRoZSByZWNlaXZlZCBlbWFpbC5cbiAgICovXG4gIHJlYWRvbmx5IGFkZEhlYWRlckFjdGlvbj86IENmblJlY2VpcHRSdWxlLkFkZEhlYWRlckFjdGlvblByb3BlcnR5XG5cbiAgLyoqXG4gICAqIFJlamVjdHMgdGhlIHJlY2VpdmVkIGVtYWlsIGJ5IHJldHVybmluZyBhIGJvdW5jZSByZXNwb25zZSB0byB0aGUgc2VuZGVyIGFuZCxcbiAgICogb3B0aW9uYWxseSwgcHVibGlzaGVzIGEgbm90aWZpY2F0aW9uIHRvIEFtYXpvbiBTTlMuXG4gICAqL1xuICByZWFkb25seSBib3VuY2VBY3Rpb24/OiBDZm5SZWNlaXB0UnVsZS5Cb3VuY2VBY3Rpb25Qcm9wZXJ0eTtcblxuICAvKipcbiAgICogQ2FsbHMgYW4gQVdTIExhbWJkYSBmdW5jdGlvbiwgYW5kIG9wdGlvbmFsbHksIHB1Ymxpc2hlcyBhIG5vdGlmaWNhdGlvbiB0b1xuICAgKiBBbWF6b24gU05TLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFtYmRhQWN0aW9uPzogQ2ZuUmVjZWlwdFJ1bGUuTGFtYmRhQWN0aW9uUHJvcGVydHk7XG5cbiAgLyoqXG4gICAqIFNhdmVzIHRoZSByZWNlaXZlZCBtZXNzYWdlIHRvIGFuIEFtYXpvbiBTMyBidWNrZXQgYW5kLCBvcHRpb25hbGx5LCBwdWJsaXNoZXNcbiAgICogYSBub3RpZmljYXRpb24gdG8gQW1hem9uIFNOUy5cbiAgICovXG4gIHJlYWRvbmx5IHMzQWN0aW9uPzogQ2ZuUmVjZWlwdFJ1bGUuUzNBY3Rpb25Qcm9wZXJ0eTtcblxuICAvKipcbiAgICogUHVibGlzaGVzIHRoZSBlbWFpbCBjb250ZW50IHdpdGhpbiBhIG5vdGlmaWNhdGlvbiB0byBBbWF6b24gU05TLlxuICAgKi9cbiAgcmVhZG9ubHkgc25zQWN0aW9uPzogQ2ZuUmVjZWlwdFJ1bGUuU05TQWN0aW9uUHJvcGVydHk7XG5cbiAgLyoqXG4gICAqIFRlcm1pbmF0ZXMgdGhlIGV2YWx1YXRpb24gb2YgdGhlIHJlY2VpcHQgcnVsZSBzZXQgYW5kIG9wdGlvbmFsbHkgcHVibGlzaGVzIGFcbiAgICogbm90aWZpY2F0aW9uIHRvIEFtYXpvbiBTTlMuXG4gICAqL1xuICByZWFkb25seSBzdG9wQWN0aW9uPzogQ2ZuUmVjZWlwdFJ1bGUuU3RvcEFjdGlvblByb3BlcnR5O1xuXG4gIC8qKlxuICAgKiBDYWxscyBBbWF6b24gV29ya01haWwgYW5kLCBvcHRpb25hbGx5LCBwdWJsaXNoZXMgYSBub3RpZmljYXRpb24gdG8gQW1hem9uIFNOUy5cbiAgICovXG4gIHJlYWRvbmx5IHdvcmttYWlsQWN0aW9uPzogQ2ZuUmVjZWlwdFJ1bGUuV29ya21haWxBY3Rpb25Qcm9wZXJ0eTtcbn1cblxuLyoqXG4gKiBBbiBhYnN0cmFjdCBhY3Rpb24gZm9yIGEgcmVjZWlwdCBydWxlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSZWNlaXB0UnVsZUFjdGlvbiB7XG4gIC8qKlxuICAgKiBSZW5kZXJzIHRoZSBhY3Rpb24gc3BlY2lmaWNhdGlvblxuICAgKi9cbiAgcmVuZGVyKCk6IFJlY2VpcHRSdWxlQWN0aW9uUHJvcHM7XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjZWlwdFJ1bGVBZGRIZWFkZXJBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjZWlwdFJ1bGVBZGRIZWFkZXJBY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgaGVhZGVyIHRvIGFkZC4gTXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDUwIGNoYXJhY3RlcnMsXG4gICAqIGluY2x1c2l2ZSwgYW5kIGNvbnNpc3Qgb2YgYWxwaGFudW1lcmljIChhLXosIEEtWiwgMC05KSBjaGFyYWN0ZXJzXG4gICAqIGFuZCBkYXNoZXMgb25seS5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHZhbHVlIG9mIHRoZSBoZWFkZXIgdG8gYWRkLiBNdXN0IGJlIGxlc3MgdGhhbiAyMDQ4IGNoYXJhY3RlcnMsXG4gICAqIGFuZCBtdXN0IG5vdCBjb250YWluIG5ld2xpbmUgY2hhcmFjdGVycyAoXCJcXHJcIiBvciBcIlxcblwiKS5cbiAgICovXG4gIHJlYWRvbmx5IHZhbHVlOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQWRkcyBhIGhlYWRlciB0byB0aGUgcmVjZWl2ZWQgZW1haWxcbiAqL1xuZXhwb3J0IGNsYXNzIFJlY2VpcHRSdWxlQWRkSGVhZGVyQWN0aW9uIGltcGxlbWVudHMgSVJlY2VpcHRSdWxlQWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogUmVjZWlwdFJ1bGVBZGRIZWFkZXJBY3Rpb25Qcm9wcykge1xuICAgIGlmICghL15bYS16QS1aMC05LV17MSw1MH0kLy50ZXN0KHByb3BzLm5hbWUpKSB7XG4gICAgICAvLyB0c2xpbnQ6ZGlzYWJsZTptYXgtbGluZS1sZW5ndGhcbiAgICAgIHRocm93IG5ldyBFcnJvcignSGVhZGVyIGBuYW1lYCBtdXN0IGJlIGJldHdlZW4gMSBhbmQgNTAgY2hhcmFjdGVycywgaW5jbHVzaXZlLCBhbmQgY29uc2lzdCBvZiBhbHBoYW51bWVyaWMgKGEteiwgQS1aLCAwLTkpIGNoYXJhY3RlcnMgYW5kIGRhc2hlcyBvbmx5LicpO1xuICAgICAgLy8gdHNsaW50OmVuYWJsZTptYXgtbGluZS1sZW5ndGhcbiAgICB9XG5cbiAgICBpZiAoIS9eW15cXG5cXHJdezAsMjA0N30kLy50ZXN0KHByb3BzLnZhbHVlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdIZWFkZXIgYHZhbHVlYCBtdXN0IGJlIGxlc3MgdGhhbiAyMDQ4IGNoYXJhY3RlcnMsIGFuZCBtdXN0IG5vdCBjb250YWluIG5ld2xpbmUgY2hhcmFjdGVycyAoXCJcXHJcIiBvciBcIlxcblwiKS4nKTtcbiAgICB9XG5cbiAgICB0aGlzLm5hbWUgPSBwcm9wcy5uYW1lO1xuICAgIHRoaXMudmFsdWUgPSBwcm9wcy52YWx1ZTtcbiAgfVxuXG4gIHB1YmxpYyByZW5kZXIoKTogUmVjZWlwdFJ1bGVBY3Rpb25Qcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGFkZEhlYWRlckFjdGlvbjoge1xuICAgICAgICBoZWFkZXJOYW1lOiB0aGlzLm5hbWUsXG4gICAgICAgIGhlYWRlclZhbHVlOiB0aGlzLnZhbHVlXG4gICAgICB9XG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciBhIFJlY2VpcHRSdWxlQm91bmNlQWN0aW9uVGVtcGxhdGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjZWlwdFJ1bGVCb3VuY2VBY3Rpb25UZW1wbGF0ZVByb3BzIHtcbiAgLyoqXG4gICAqIEh1bWFuLXJlYWRhYmxlIHRleHQgdG8gaW5jbHVkZSBpbiB0aGUgYm91bmNlIG1lc3NhZ2UuXG4gICAqL1xuICByZWFkb25seSBtZXNzYWdlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBTTVRQIHJlcGx5IGNvZGUsIGFzIGRlZmluZWQgYnkgUkZDIDUzMjEuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly90b29scy5pZXRmLm9yZy9odG1sL3JmYzUzMjFcbiAgICovXG4gIHJlYWRvbmx5IHNtdHBSZXBseUNvZGU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFNNVFAgZW5oYW5jZWQgc3RhdHVzIGNvZGUsIGFzIGRlZmluZWQgYnkgUkZDIDM0NjMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly90b29scy5pZXRmLm9yZy9odG1sL3JmYzM0NjNcbiAgICovXG4gIHJlYWRvbmx5IHN0YXR1c0NvZGU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSBib3VuY2UgYWN0aW9uIHRlbXBsYXRlLlxuICovXG5leHBvcnQgY2xhc3MgUmVjZWlwdFJ1bGVCb3VuY2VBY3Rpb25UZW1wbGF0ZSB7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTWFpbGJveERvZXNOb3RFeGlzdCA9IG5ldyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvblRlbXBsYXRlKHtcbiAgICBtZXNzYWdlOiAnTWFpbGJveCBkb2VzIG5vdCBleGlzdCcsXG4gICAgc210cFJlcGx5Q29kZTogJzU1MCcsXG4gICAgc3RhdHVzQ29kZTogJzUuMS4xJ1xuICB9KTtcblxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1lc3NhZ2VUb29MYXJnZSA9IG5ldyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvblRlbXBsYXRlKHtcbiAgICBtZXNzYWdlOiAnTWVzc2FnZSB0b28gbGFyZ2UnLFxuICAgIHNtdHBSZXBseUNvZGU6ICc1NTInLFxuICAgIHN0YXR1c0NvZGU6ICc1LjMuNCdcbiAgfSk7XG5cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBNYWlsYm94RnVsbCA9IG5ldyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvblRlbXBsYXRlKHtcbiAgICBtZXNzYWdlOiAnTWFpbGJveCBmdWxsJyxcbiAgICBzbXRwUmVwbHlDb2RlOiAnNTUyJyxcbiAgICBzdGF0dXNDb2RlOiAnNS4yLjInXG4gIH0pO1xuXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTWVzc2FnZUNvbnRlbnRSZWplY3RlZCA9IG5ldyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvblRlbXBsYXRlKHtcbiAgICBtZXNzYWdlOiAnTWVzc2FnZSBjb250ZW50IHJlamVjdGVkJyxcbiAgICBzbXRwUmVwbHlDb2RlOiAnNTAwJyxcbiAgICBzdGF0dXNDb2RlOiAnNS42LjEnXG4gIH0pO1xuXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVGVtcG9yYXJ5RmFpbHVyZSA9IG5ldyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvblRlbXBsYXRlKHtcbiAgICBtZXNzYWdlOiAnVGVtcG9yYXJ5IGZhaWx1cmUnLFxuICAgIHNtdHBSZXBseUNvZGU6ICc0NTAnLFxuICAgIHN0YXR1c0NvZGU6ICc0LjAuMCdcbiAgfSk7XG5cbiAgcHVibGljIHJlYWRvbmx5IG1lc3NhZ2U6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHNtdHBSZXBseUNvZGU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHN0YXR1c0NvZGU/OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IFJlY2VpcHRSdWxlQm91bmNlQWN0aW9uVGVtcGxhdGVQcm9wcykge1xuICAgIHRoaXMubWVzc2FnZSA9IHByb3BzLm1lc3NhZ2U7XG4gICAgdGhpcy5zbXRwUmVwbHlDb2RlID0gcHJvcHMuc210cFJlcGx5Q29kZTtcbiAgICB0aGlzLnN0YXR1c0NvZGUgPSBwcm9wcy5zdGF0dXNDb2RlO1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjZWlwdFJ1bGVCb3VuY2VBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjZWlwdFJ1bGVCb3VuY2VBY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgdGVtcGxhdGUgY29udGFpbmluZyB0aGUgbWVzc2FnZSwgcmVwbHkgY29kZSBhbmQgc3RhdHVzIGNvZGUuXG4gICAqL1xuICByZWFkb25seSB0ZW1wbGF0ZTogUmVjZWlwdFJ1bGVCb3VuY2VBY3Rpb25UZW1wbGF0ZTtcblxuICAvKipcbiAgICogVGhlIGVtYWlsIGFkZHJlc3Mgb2YgdGhlIHNlbmRlciBvZiB0aGUgYm91bmNlZCBlbWFpbC4gVGhpcyBpcyB0aGUgYWRkcmVzc1xuICAgKiBmcm9tIHdoaWNoIHRoZSBib3VuY2UgbWVzc2FnZSB3aWxsIGJlIHNlbnQuXG4gICAqL1xuICByZWFkb25seSBzZW5kZXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBub3RpZnkgd2hlbiB0aGUgYm91bmNlIGFjdGlvbiBpcyB0YWtlbi5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gbm90aWZpY2F0aW9uXG4gICAqL1xuICByZWFkb25seSB0b3BpYz86IHNucy5JVG9waWM7XG59XG5cbi8qKlxuICogUmVqZWN0cyB0aGUgcmVjZWl2ZWQgZW1haWwgYnkgcmV0dXJuaW5nIGEgYm91bmNlIHJlc3BvbnNlIHRvIHRoZSBzZW5kZXIgYW5kLFxuICogb3B0aW9uYWxseSwgcHVibGlzaGVzIGEgbm90aWZpY2F0aW9uIHRvIEFtYXpvbiBTTlMuXG4gKi9cbmV4cG9ydCBjbGFzcyBSZWNlaXB0UnVsZUJvdW5jZUFjdGlvbiBpbXBsZW1lbnRzIElSZWNlaXB0UnVsZUFjdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFJlY2VpcHRSdWxlQm91bmNlQWN0aW9uUHJvcHMpIHtcbiAgfVxuXG4gIHB1YmxpYyByZW5kZXIoKTogUmVjZWlwdFJ1bGVBY3Rpb25Qcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGJvdW5jZUFjdGlvbjoge1xuICAgICAgICBzZW5kZXI6IHRoaXMucHJvcHMuc2VuZGVyLFxuICAgICAgICBzbXRwUmVwbHlDb2RlOiB0aGlzLnByb3BzLnRlbXBsYXRlLnNtdHBSZXBseUNvZGUsXG4gICAgICAgIG1lc3NhZ2U6IHRoaXMucHJvcHMudGVtcGxhdGUubWVzc2FnZSxcbiAgICAgICAgdG9waWNBcm46IHRoaXMucHJvcHMudG9waWMgPyB0aGlzLnByb3BzLnRvcGljLnRvcGljQXJuIDogdW5kZWZpbmVkLFxuICAgICAgICBzdGF0dXNDb2RlOiB0aGlzLnByb3BzLnRlbXBsYXRlLnN0YXR1c0NvZGVcbiAgICAgIH1cbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogVGhlIHR5cGUgb2YgaW52b2NhdGlvbiB0byB1c2UgZm9yIGEgTGFtYmRhIEFjdGlvbi5cbiAqL1xuZXhwb3J0IGVudW0gTGFtYmRhSW52b2NhdGlvblR5cGUge1xuICAvKipcbiAgICogVGhlIGZ1bmN0aW9uIHdpbGwgYmUgaW52b2tlZCBhc3luY2hyb25vdXNseS5cbiAgICovXG4gIEV2ZW50ID0gJ0V2ZW50JyxcblxuICAvKipcbiAgICogVGhlIGZ1bmN0aW9uIHdpbGwgYmUgaW52b2tlZCBzeWNocm9ub3VzbHkuIFVzZSBSZXF1ZXN0UmVzcG9uc2Ugb25seSB3aGVuXG4gICAqIHlvdSB3YW50IHRvIG1ha2UgYSBtYWlsIGZsb3cgZGVjaXNpb24sIHN1Y2ggYXMgd2hldGhlciB0byBzdG9wIHRoZSByZWNlaXB0XG4gICAqIHJ1bGUgb3IgdGhlIHJlY2VpcHQgcnVsZSBzZXQuXG4gICAqL1xuICBSZXF1ZXN0UmVzcG9uc2UgPSAnUmVxdWVzdFJlc3BvbnNlJyxcbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3IgYSBSZWNlaXB0UnVsZUxhbWJkYUFjdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZWNlaXB0UnVsZUxhbWJkYUFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBMYW1iZGEgZnVuY3Rpb24gdG8gaW52b2tlLlxuICAgKi9cbiAgcmVhZG9ubHkgZnVuY3Rpb246IGxhbWJkYS5JRnVuY3Rpb25cblxuICAvKipcbiAgICogVGhlIGludm9jYXRpb24gdHlwZSBvZiB0aGUgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCBFdmVudFxuICAgKi9cbiAgcmVhZG9ubHkgaW52b2NhdGlvblR5cGU/OiBMYW1iZGFJbnZvY2F0aW9uVHlwZTtcblxuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBub3RpZnkgd2hlbiB0aGUgTGFtYmRhIGFjdGlvbiBpcyB0YWtlbi5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gbm90aWZpY2F0aW9uXG4gICAqL1xuICByZWFkb25seSB0b3BpYz86IHNucy5JVG9waWM7XG59XG5cbi8qKlxuICogQ2FsbHMgYW4gQVdTIExhbWJkYSBmdW5jdGlvbiwgYW5kIG9wdGlvbmFsbHksIHB1Ymxpc2hlcyBhIG5vdGlmaWNhdGlvbiB0b1xuICogQW1hem9uIFNOUy5cbiAqL1xuZXhwb3J0IGNsYXNzIFJlY2VpcHRSdWxlTGFtYmRhQWN0aW9uIGltcGxlbWVudHMgSVJlY2VpcHRSdWxlQWN0aW9uIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogUmVjZWlwdFJ1bGVMYW1iZGFBY3Rpb25Qcm9wcykge1xuICB9XG5cbiAgcHVibGljIHJlbmRlcigpOiBSZWNlaXB0UnVsZUFjdGlvblByb3BzIHtcbiAgICAvLyBBbGxvdyBTRVMgdG8gaW52b2tlIExhbWJkYSBmdW5jdGlvblxuICAgIC8vIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9EZXZlbG9wZXJHdWlkZS9yZWNlaXZpbmctZW1haWwtcGVybWlzc2lvbnMuaHRtbCNyZWNlaXZpbmctZW1haWwtcGVybWlzc2lvbnMtbGFtYmRhXG4gICAgY29uc3QgcGVybWlzc2lvbklkID0gJ0FsbG93U2VzJztcbiAgICBpZiAoIXRoaXMucHJvcHMuZnVuY3Rpb24ubm9kZS50cnlGaW5kQ2hpbGQocGVybWlzc2lvbklkKSkge1xuICAgICAgdGhpcy5wcm9wcy5mdW5jdGlvbi5hZGRQZXJtaXNzaW9uKHBlcm1pc3Npb25JZCwge1xuICAgICAgICBhY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICBwcmluY2lwYWw6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2VzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgc291cmNlQWNjb3VudDogY2RrLkF3cy5hY2NvdW50SWRcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBsYW1iZGFBY3Rpb246IHtcbiAgICAgICAgZnVuY3Rpb25Bcm46IHRoaXMucHJvcHMuZnVuY3Rpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgIGludm9jYXRpb25UeXBlOiB0aGlzLnByb3BzLmludm9jYXRpb25UeXBlLFxuICAgICAgICB0b3BpY0FybjogdGhpcy5wcm9wcy50b3BpYyA/IHRoaXMucHJvcHMudG9waWMudG9waWNBcm4gOiB1bmRlZmluZWRcbiAgICAgIH1cbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjZWlwdFJ1bGVTM0FjdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZWNlaXB0UnVsZVMzQWN0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIFMzIGJ1Y2tldCB0aGF0IGluY29taW5nIGVtYWlsIHdpbGwgYmUgc2F2ZWQgdG8uXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXN0ZXIga2V5IHRoYXQgU0VTIHNob3VsZCB1c2UgdG8gZW5jcnlwdCB5b3VyIGVtYWlscyBiZWZvcmUgc2F2aW5nXG4gICAqIHRoZW0gdG8gdGhlIFMzIGJ1Y2tldC5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gZW5jcnlwdGlvblxuICAgKi9cbiAgcmVhZG9ubHkga21zS2V5Pzoga21zLklFbmNyeXB0aW9uS2V5O1xuXG4gIC8qKlxuICAgKiBUaGUga2V5IHByZWZpeCBvZiB0aGUgUzMgYnVja2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBubyBwcmVmaXhcbiAgICovXG4gIHJlYWRvbmx5IG9iamVjdEtleVByZWZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBub3RpZnkgd2hlbiB0aGUgUzMgYWN0aW9uIGlzIHRha2VuLlxuICAgKlxuICAgKiBAZGVmYXVsdCBubyBub3RpZmljYXRpb25cbiAgICovXG4gIHJlYWRvbmx5IHRvcGljPzogc25zLklUb3BpYztcbn1cblxuLyoqXG4gKiBTYXZlcyB0aGUgcmVjZWl2ZWQgbWVzc2FnZSB0byBhbiBBbWF6b24gUzMgYnVja2V0IGFuZCwgb3B0aW9uYWxseSwgcHVibGlzaGVzXG4gKiBhIG5vdGlmaWNhdGlvbiB0byBBbWF6b24gU05TLlxuICovXG5leHBvcnQgY2xhc3MgUmVjZWlwdFJ1bGVTM0FjdGlvbiBpbXBsZW1lbnRzIElSZWNlaXB0UnVsZUFjdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFJlY2VpcHRSdWxlUzNBY3Rpb25Qcm9wcykge1xuICB9XG5cbiAgcHVibGljIHJlbmRlcigpOiBSZWNlaXB0UnVsZUFjdGlvblByb3BzIHtcbiAgICAvLyBBbGxvdyBTRVMgdG8gd3JpdGUgdG8gUzMgYnVja2V0XG4gICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL3JlY2VpdmluZy1lbWFpbC1wZXJtaXNzaW9ucy5odG1sI3JlY2VpdmluZy1lbWFpbC1wZXJtaXNzaW9ucy1zM1xuICAgIGNvbnN0IGtleVBhdHRlcm4gPSB0aGlzLnByb3BzLm9iamVjdEtleVByZWZpeCB8fCAnJztcblxuICAgIGNvbnN0IHMzU3RhdGVtZW50ID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoKVxuICAgICAgLmFkZEFjdGlvbignczM6UHV0T2JqZWN0JylcbiAgICAgIC5hZGRTZXJ2aWNlUHJpbmNpcGFsKCdzZXMuYW1hem9uYXdzLmNvbScpXG4gICAgICAuYWRkUmVzb3VyY2UodGhpcy5wcm9wcy5idWNrZXQuYXJuRm9yT2JqZWN0cyhgJHtrZXlQYXR0ZXJufSpgKSlcbiAgICAgIC5hZGRDb25kaXRpb24oJ1N0cmluZ0VxdWFscycsIHtcbiAgICAgICAgJ2F3czpSZWZlcmVyJzogY2RrLkF3cy5hY2NvdW50SWRcbiAgICAgIH0pO1xuXG4gICAgdGhpcy5wcm9wcy5idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShzM1N0YXRlbWVudCk7XG5cbiAgICAvLyBBbGxvdyBTRVMgdG8gdXNlIEtNUyBtYXN0ZXIga2V5XG4gICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0RldmVsb3Blckd1aWRlL3JlY2VpdmluZy1lbWFpbC1wZXJtaXNzaW9ucy5odG1sI3JlY2VpdmluZy1lbWFpbC1wZXJtaXNzaW9ucy1rbXNcbiAgICBpZiAodGhpcy5wcm9wcy5rbXNLZXkgJiYgIS9hbGlhc1xcL2F3c1xcL3NlcyQvLnRlc3QodGhpcy5wcm9wcy5rbXNLZXkua2V5QXJuKSkge1xuICAgICAgY29uc3Qga21zU3RhdGVtZW50ID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoKVxuICAgICAgICAuYWRkQWN0aW9ucygna206RW5jcnlwdCcsICdrbXM6R2VuZXJhdGVEYXRhS2V5JylcbiAgICAgICAgLmFkZFNlcnZpY2VQcmluY2lwYWwoJ3Nlcy5hbWF6b25hd3MuY29tJylcbiAgICAgICAgLmFkZEFsbFJlc291cmNlcygpXG4gICAgICAgIC5hZGRDb25kaXRpb25zKHtcbiAgICAgICAgICBOdWxsOiB7XG4gICAgICAgICAgICAna21zOkVuY3J5cHRpb25Db250ZXh0OmF3czpzZXM6cnVsZS1uYW1lJzogJ2ZhbHNlJyxcbiAgICAgICAgICAgICdrbXM6RW5jcnlwdGlvbkNvbnRleHQ6YXdzOnNlczptZXNzYWdlLWlkJzogJ2ZhbHNlJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAna21zOkVuY3J5cHRpb25Db250ZXh0OmF3czpzZXM6c291cmNlLWFjY291bnQnOiBjZGsuQXdzLmFjY291bnRJZFxuICAgICAgICAgIH1cbiAgICAgICAgfSk7XG5cbiAgICAgIHRoaXMucHJvcHMua21zS2V5LmFkZFRvUmVzb3VyY2VQb2xpY3koa21zU3RhdGVtZW50KTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgczNBY3Rpb246IHtcbiAgICAgICAgYnVja2V0TmFtZTogdGhpcy5wcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAga21zS2V5QXJuOiB0aGlzLnByb3BzLmttc0tleSA/IHRoaXMucHJvcHMua21zS2V5LmtleUFybiA6IHVuZGVmaW5lZCxcbiAgICAgICAgb2JqZWN0S2V5UHJlZml4OiB0aGlzLnByb3BzLm9iamVjdEtleVByZWZpeCxcbiAgICAgICAgdG9waWNBcm46IHRoaXMucHJvcHMudG9waWMgPyB0aGlzLnByb3BzLnRvcGljLnRvcGljQXJuIDogdW5kZWZpbmVkXG4gICAgICB9XG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIFRoZSB0eXBlIG9mIGVtYWlsIGVuY29kaW5nIHRvIHVzZSBmb3IgYSBTTlMgYWN0aW9uLlxuICovXG5leHBvcnQgZW51bSBFbWFpbEVuY29kaW5nIHtcbiAgLyoqXG4gICAqIEJhc2UgNjRcbiAgICovXG4gIEJhc2U2NCA9ICdCYXNlNjQnLFxuXG4gIC8qKlxuICAgKiBVVEYtOFxuICAgKi9cbiAgVVRGOCA9ICdVVEYtOCcsXG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjZWlwdFJ1bGVTbnNBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVjZWlwdFJ1bGVTbnNBY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgZW5jb2RpbmcgdG8gdXNlIGZvciB0aGUgZW1haWwgd2l0aGluIHRoZSBBbWF6b24gU05TIG5vdGlmaWNhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgVVRGLThcbiAgICovXG4gIHJlYWRvbmx5IGVuY29kaW5nPzogRW1haWxFbmNvZGluZztcblxuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBub3RpZnkuXG4gICAqL1xuICByZWFkb25seSB0b3BpYzogc25zLklUb3BpYztcbn1cblxuLyoqXG4gKiBQdWJsaXNoZXMgdGhlIGVtYWlsIGNvbnRlbnQgd2l0aGluIGEgbm90aWZpY2F0aW9uIHRvIEFtYXpvbiBTTlMuXG4gKi9cbmV4cG9ydCBjbGFzcyBSZWNlaXB0UnVsZVNuc0FjdGlvbiBpbXBsZW1lbnRzIElSZWNlaXB0UnVsZUFjdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFJlY2VpcHRSdWxlU25zQWN0aW9uUHJvcHMpIHtcbiAgfVxuXG4gIHB1YmxpYyByZW5kZXIoKTogUmVjZWlwdFJ1bGVBY3Rpb25Qcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNuc0FjdGlvbjoge1xuICAgICAgICBlbmNvZGluZzogdGhpcy5wcm9wcy5lbmNvZGluZyxcbiAgICAgICAgdG9waWNBcm46IHRoaXMucHJvcHMudG9waWMudG9waWNBcm5cbiAgICAgIH1cbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgUmVjZWlwdFJ1bGVTdG9wQWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlY2VpcHRSdWxlU3RvcEFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBTTlMgdG9waWMgdG8gbm90aWZ5IHdoZW4gdGhlIHN0b3AgYWN0aW9uIGlzIHRha2VuLlxuICAgKi9cbiAgcmVhZG9ubHkgdG9waWM/OiBzbnMuSVRvcGljO1xufVxuXG4vKipcbiAqIFRlcm1pbmF0ZXMgdGhlIGV2YWx1YXRpb24gb2YgdGhlIHJlY2VpcHQgcnVsZSBzZXQgYW5kIG9wdGlvbmFsbHkgcHVibGlzaGVzIGFcbiAqIG5vdGlmaWNhdGlvbiB0byBBbWF6b24gU05TLlxuICovXG5leHBvcnQgY2xhc3MgUmVjZWlwdFJ1bGVTdG9wQWN0aW9uIGltcGxlbWVudHMgSVJlY2VpcHRSdWxlQWN0aW9uIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wcz86IFJlY2VpcHRSdWxlU3RvcEFjdGlvblByb3BzKSB7XG4gIH1cblxuICBwdWJsaWMgcmVuZGVyKCk6IFJlY2VpcHRSdWxlQWN0aW9uUHJvcHMge1xuICAgIHJldHVybiB7XG4gICAgICBzdG9wQWN0aW9uOiB7XG4gICAgICAgIHNjb3BlOiAnUnVsZVNldCcsXG4gICAgICAgIHRvcGljQXJuOiB0aGlzLnByb3BzICYmIHRoaXMucHJvcHMudG9waWMgPyB0aGlzLnByb3BzLnRvcGljLnRvcGljQXJuIDogdW5kZWZpbmVkXG4gICAgICB9XG4gICAgfTtcbiAgfVxufVxuIl19