import { Construct, IResource, Resource } from '@aws-cdk/cdk';
import { ReceiptRule, ReceiptRuleOptions } from './receipt-rule';
/**
 * A receipt rule set.
 */
export interface IReceiptRuleSet extends IResource {
    /**
     * The receipt rule set name.
     */
    readonly name: string;
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id: string, options?: ReceiptRuleOptions): ReceiptRule;
    /**
     * Exports this receipt rule set from the stack.
     */
    export(): ReceiptRuleSetImportProps;
}
/**
 * Construction properties for a ReceiptRuleSet.
 */
export interface ReceiptRuleSetProps {
    /**
     * The name for the receipt rule set.
     *
     * @default a CloudFormation generated name
     */
    readonly name?: string;
    /**
     * The list of rules to add to this rule set. Rules are added in the same
     * order as they appear in the list.
     */
    readonly rules?: ReceiptRuleOptions[];
    /**
     * Whether to add a first rule to stop processing messages
     * that have at least one spam indicator.
     *
     * @default false
     */
    readonly dropSpam?: boolean;
}
/**
 * A new or imported receipt rule set.
 */
export declare abstract class ReceiptRuleSetBase extends Resource implements IReceiptRuleSet {
    abstract readonly name: string;
    private lastAddedRule?;
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id: string, options?: ReceiptRuleOptions): ReceiptRule;
    abstract export(): ReceiptRuleSetImportProps;
    /**
     * Adds a drop spam rule
     */
    protected addDropSpamRule(): void;
}
/**
 * A new receipt rule set.
 */
export declare class ReceiptRuleSet extends ReceiptRuleSetBase implements IReceiptRuleSet {
    /**
     * Import an exported receipt rule set.
     */
    static import(scope: Construct, id: string, props: ReceiptRuleSetImportProps): IReceiptRuleSet;
    readonly name: string;
    constructor(scope: Construct, id: string, props?: ReceiptRuleSetProps);
    /**
     * Exports this receipt rule set from the stack.
     */
    export(): ReceiptRuleSetImportProps;
}
/**
 * Construction properties for an ImportedReceiptRuleSet.
 */
export interface ReceiptRuleSetImportProps {
    /**
     * The receipt rule set name.
     */
    readonly name: string;
}
