"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends cdk_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, {
            after: this.lastAddedRule ? this.lastAddedRule : undefined,
            ruleSet: this,
            ...options
        });
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
exports.ReceiptRuleSetBase = ReceiptRuleSetBase;
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    /**
     * Import an exported receipt rule set.
     */
    static import(scope, id, props) {
        return new ImportedReceiptRuleSet(scope, id, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: props ? props.name : undefined
        });
        this.name = resource.receiptRuleSetName;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
    /**
     * Exports this receipt rule set from the stack.
     */
    export() {
        return {
            name: new cdk_1.CfnOutput(this, 'ReceiptRuleSetName', { value: this.name }).makeImportValue().toString()
        };
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
/**
 * An imported receipt rule set.
 */
class ImportedReceiptRuleSet extends ReceiptRuleSetBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.name = props.name;
    }
    /**
     * Exports this receipt rule set from the stack.
     */
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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