import { Construct, IResource, Resource } from '@aws-cdk/cdk';
import { IReceiptRuleAction } from './receipt-rule-action';
import { IReceiptRuleSet } from './receipt-rule-set';
/**
 * A receipt rule.
 */
export interface IReceiptRule extends IResource {
    /**
     * The name of the receipt rule.
     */
    readonly name: string;
    /**
     * Exports this receipt rule from the stack.
     */
    export(): ReceiptRuleImportProps;
}
/**
 * The type of TLS policy for a receipt rule.
 */
export declare enum TlsPolicy {
    /**
     * Do not check for TLS.
     */
    Optional = "Optional",
    /**
     * Bounce emails that are not received over TLS.
     */
    Require = "Require"
}
/**
 * Options to add a receipt rule to a receipt rule set.
 */
export interface ReceiptRuleOptions {
    /**
     * An ordered list of actions to perform on messages that match at least
     * one of the recipient email addresses or domains specified in the
     * receipt rule.
     */
    readonly actions?: IReceiptRuleAction[];
    /**
     * An existing rule after which the new rule will be placed.
     *
     * @default the new rule is inserted at the beginning of the rule list
     */
    readonly after?: IReceiptRule;
    /**
     * Whether the rule is active.
     *
     * @default true
     */
    readonly enabled?: boolean;
    /**
     * The name for the rule
     *
     * @default a CloudFormation generated name
     */
    readonly name?: string;
    /**
     * The recipient domains and email addresses that the receipt rule applies to.
     *
     * @default match all recipients under all verified domains.
     */
    readonly recipients?: string[];
    /**
     * Wheter to scan for spam and viruses.
     *
     * @default false
     */
    readonly scanEnabled?: boolean;
    /**
     * The TLS policy
     *
     * @default Optional
     */
    readonly tlsPolicy?: TlsPolicy;
}
/**
 * Construction properties for a ReceiptRule.
 */
export interface ReceiptRuleProps extends ReceiptRuleOptions {
    /**
     * The name of the rule set that the receipt rule will be added to.
     */
    readonly ruleSet: IReceiptRuleSet;
}
/**
 * A new receipt rule.
 */
export declare class ReceiptRule extends Resource implements IReceiptRule {
    /**
     * Import an exported receipt rule.
     */
    static import(scope: Construct, id: string, props: ReceiptRuleImportProps): IReceiptRule;
    readonly name: string;
    private readonly renderedActions;
    constructor(scope: Construct, id: string, props: ReceiptRuleProps);
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action: IReceiptRuleAction): void;
    /**
     * Exports this receipt rule from the stack.
     */
    export(): ReceiptRuleImportProps;
    private getRenderedActions;
}
export interface ReceiptRuleImportProps {
    /**
     * The name of the receipt rule.
     */
    readonly name: string;
}
export interface DropSpamReceiptRuleProps extends ReceiptRuleProps {
}
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
export declare class DropSpamReceiptRule extends Construct {
    readonly rule: ReceiptRule;
    constructor(scope: Construct, id: string, props: DropSpamReceiptRuleProps);
}
