"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const cdk_1 = require("@aws-cdk/cdk");
const receipt_rule_action_1 = require("./receipt-rule-action");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["Optional"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["Require"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends cdk_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.renderedActions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.name : undefined,
            rule: {
                actions: new cdk_1.Token(() => this.getRenderedActions()),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: props.name,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy
            },
            ruleSetName: props.ruleSet.name
        });
        this.name = resource.receiptRuleName;
        if (props.actions) {
            props.actions.forEach(action => this.addAction(action));
        }
    }
    /**
     * Import an exported receipt rule.
     */
    static import(scope, id, props) {
        return new ImportedReceiptRule(scope, id, props);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        const renderedAction = action.render();
        this.renderedActions.push(renderedAction);
    }
    /**
     * Exports this receipt rule from the stack.
     */
    export() {
        return {
            name: new cdk_1.CfnOutput(this, 'ReceiptRuleName', { value: this.name }).makeImportValue().toString()
        };
    }
    getRenderedActions() {
        if (this.renderedActions.length === 0) {
            return undefined;
        }
        return this.renderedActions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * An imported receipt rule.
 */
class ImportedReceiptRule extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.name = props.name;
    }
    /**
     * Exports this receipt rule from the stack.
     */
    export() {
        return this.props;
    }
}
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NodeJS810,
            handler: 'index.handler',
            code: lambda.Code.inline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16'
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                new receipt_rule_action_1.ReceiptRuleLambdaAction({
                    function: fn,
                    invocationType: receipt_rule_action_1.LambdaInvocationType.RequestResponse
                })
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log("SES Notification:\n", JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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