import asyncio
import logging
import httpx
import json
from typing import Any, Dict, Optional
from uuid import uuid4
from ..model.agent_card import AgentCard as AliasAgentCard
from a2a.client import Client, ClientConfig
from a2a.types import AgentCard, MessageSendParams, SendMessageRequest, Message, AgentCapabilities
from a2a.utils import new_agent_text_message  # Use to create user message
from a2a.client import ClientFactory  # Import ClientFactory

class A2AAgentCaller:
    def __init__(self, alias_agent_card: AliasAgentCard):
        skills = []
        agent_card = AgentCard(
            name=alias_agent_card.name,
            url=alias_agent_card.url,
            description=alias_agent_card.description or "",
            skills=skills,
            capabilities=AgentCapabilities(
                streaming=True,
                push_notifications=True,
                state_transition_history=True,
            ),
            version="1.0",  # Default version, can be customized
            default_input_modes=["text"],  # Default input modes, can be customized
            default_output_modes=["text"],  # Default output modes, can be customized
        )
        self.agent_card = agent_card
    
    async def call(self, tool_name: str, arguments: Optional[dict[str, Any]] = None) -> Any:
        logging.info(f"Calling A2A agent {self.agent_card.name} with tool {tool_name} and arguments: {arguments} at {self.agent_card.url}")
        
        """Call a specific tool on the agent with given arguments."""
        async with httpx.AsyncClient() as http_client:
            try:
                # Initialize Client
                factory = ClientFactory(ClientConfig())
                client = factory.create(card=self.agent_card)
                
                # skill_ids = [skill.id for skill in self.agent_card.skills]
                # if tool_name not in skill_ids:
                #    raise ValueError(f"Tool '{tool_name}' not found. Available tools: {skill_ids}")
                
                # Construct the message text for tool call
                message_text = f"Use tool {tool_name} with arguments {json.dumps(arguments or {})}"
                
                # Create the user message
                message = new_agent_text_message(message_text)  # Use utility to create user message
                params = MessageSendParams(message=message)
                # Create the request
                send_message_payload = {
                    'message': message,
                }
                request = SendMessageRequest(id=100,  params=MessageSendParams(**send_message_payload))
                
                # Try Client.send_message
                try:
                    async for response in client.send_message(request=message):
                        logging.info(f"Response: {response}")
                        if isinstance(response, Message):
                            if response.parts and hasattr(response.parts[0], "root") and hasattr(response.parts[0].root, "text"):
                                result_text = response.parts[0].root.text
                                print("Agent response text:", result_text)
                            else:
                                print("No text part found in response.")
                        else:
                            logging.info(f"Received non-message response: {response}")
                        result = response
                        break  # Break after first valid response
                    return result
                except AttributeError as e:
                    raise e
            except httpx.HTTPError as e:
                raise Exception(f"HTTP error calling tool '{tool_name}': {e}")
            except Exception as e:
                logging.error(e)
                raise Exception(f"Error calling tool '{tool_name}': {e}. Check Client API with: help(a2a.client.Client)")