import os
import json
from typing import List, Dict, Any
from any_agent import AnyAgent, AgentConfig
from a2a.types import AgentCard, AgentSkill
from a2a.client import AgentClient  # or your actual A2A client lib

def create_selector_agent(framework: str, model: str):
    framework = framework or os.getenv("AGENT_SELECTOR_FRAMEWORK", "openai")
    model = model or os.getenv("AGENT_SELECTOR_MODEL", "gpt-4o-mini")
    print(f"[Router] Using selector framework={framework}, model={model}")

    return AnyAgent.create(
        framework,
        AgentConfig(
            model_id=model,
            instructions=(
                "You are an AI router. "
                "Given the user's request and a list of agents (with names and skills), "
                "choose exactly ONE agent and ONE method in JSON: "
                '{"agent": "...", "method": "..."}'
            ),
        )
    )

def select_agent_and_method_llm(
    input_params: Dict[str, Any],
    agents: List[AgentCard],
    selector_agent
) -> (AgentCard, AgentSkill):
    agents_metadata = [
        {
            "name": agent.name,
            "description": getattr(agent, "description", ""),
            "skills": [
                {"name": skill.name, "description": getattr(skill, "description", "")}
                for skill in agent.skills
            ]
        }
        for agent in agents
    ]

    prompt = {"task": input_params, "agents": agents_metadata}
    response_text = selector_agent.run(json.dumps(prompt))
    
    try:
        choice = json.loads(response_text)
        agent_name = choice["agent"]
        method_name = choice["method"]
    except Exception as e:
        raise ValueError(f"Invalid selector output: {response_text}") from e

    for agent in agents:
        if agent.name == agent_name:
            for skill in agent.skills:
                if skill.name == method_name:
                    return agent, skill

    raise ValueError(f"Agent or method not found: {choice}")

def route_and_execute(
    input_params: Dict[str, Any],
    agents: List[AgentCard],
    registry_url: str,
    framework: str,
    model: str
) -> Any:
    selector_agent = create_selector_agent(framework, model)
    agent, skill = select_agent_and_method_llm(input_params, agents, selector_agent)

    client = AgentClient(registry_url, agent_id=agent.id)
    print(f"[Router] Routed to {agent.name}.{skill.name} …")
    return client.call(skill.name, **input_params)

if __name__ == "__main__":
    agents = [
        AgentCard(id="weather-001", name="WeatherBot", skills=[AgentSkill(name="get_weather", description="Weather forecast")]),
        AgentCard(id="math-001", name="MathBot", skills=[AgentSkill(name="calculate", description="Math operations")]),
    ]
    params = {"task": "What’s the weather in Paris tomorrow?", "city": "Paris"}
    result = route_and_execute(params, agents, registry_url="https://a2a-registry.example.com")
    print("Result:", result)
