# src/aans_sdk/agent.py
from typing import Any, Dict, Optional
import logging
import requests
from .model.agent_card import AgentCard
from .model.message import Message
from .model.typings import AuthCallback
from .model.typings import VerbType
from .a2a.a2a_agent_caller import A2AAgentCaller
from .mcp.mcp_agent_caller import MCPAgentCaller
from .llm.llm_agent_caller import LLMAgentCaller
from .agent_resolver import AgentResolver

# from pydantic_ai import Agent as PydanticAgent


class Agent:
    def __init__(self, card: AgentCard):
        # self.agent = PydanticAgent(**card.model_dump())
        self.card = card

    async def invoke(
        self,
        arguments: Dict[str, Any] | list[Message],
        verb: Optional[str] = None,
        verb_type: Optional[VerbType] = None,
        auth_callback: Optional[AuthCallback] = None,
    ) -> Dict[str, Any] | list[Message] | Message:
        logging.info(
            f"Performing verb '{verb}' on agent {self.card.name} with arguments: {arguments}"
        )
        # Assuming the agent's URL can handle different verbs as part of the path or query
        # This is a placeholder; you might need to adjust the URL or request body
        # based on how your agent API is designed to handle different verbs.

        if self.card.type == AgentCard.TYPE_A2A:
            a2a_agent_caller = A2AAgentCaller(self.card)
            return await a2a_agent_caller.call(verb, arguments)
        elif self.card.type == AgentCard.TYPE_MCP:
            mcp_agent_caller = MCPAgentCaller(self.card)
            if verb_type == VerbType.PROMPTS:
                return await mcp_agent_caller.get_prompt(verb, arguments)
            elif verb_type == VerbType.RESOURCES:
                return await mcp_agent_caller.read_resource(verb)
            # default to tools
            return await mcp_agent_caller.call_tool(verb, arguments)
        elif self.card.type == AgentCard.TYPE_LLM:
            llm_agent_caller = LLMAgentCaller(self.card)
            return await llm_agent_caller.call(verb, arguments)
        response = requests.post(f"{self.card.url}/{verb}", json=arguments)
        response.raise_for_status()
        return response.json()

    def describe(self) -> Dict[str, Any]:
        return self.card.model_dump()
