import os
import json
import anthropic
import logging
from typing import List, Dict, Any, Optional
from any_agent import AnyAgent, AgentConfig, AgentFramework

from .agent import Agent
from .model.agent_card import AgentCard, Skill
from .model.typings import AuthCallback
from .llm.llm_agent import ClaudeAgent

class AgentRouter:
    instructions=(
        "You are an AI router that returns only JSON. "
        "Given the user's request and a list of agents, select one agent, one method and the parameter if applicable. "
        "Your response must be ONLY this JSON format with no other text: "
        '{"agent": "agent_name", "method": "method_name", "parameter": "parameter_value"}' 
        "return '{}' if no suitable agent is found."
    )

    def select_agent_and_method_llm(
        self,
        input_params: Dict[str, Any],
        agents: List[AgentCard],
        selector_agent
    ) -> tuple[AgentCard, Skill, Any]:
        agents_metadata = [
            {
                "name": agent.name,
                "description": getattr(agent, "description", ""),
                "skills": [
                    {"name": skill.get("name") if isinstance(skill, dict) else getattr(skill, "name", None), 
                    "description": skill.get("description") if isinstance(skill, dict) else getattr(skill, "description", None), 
                    "input_schema": skill.get("input_schema") if isinstance(skill, dict) else getattr(skill, "input_schema", None)}
                    for skill in getattr(agent, "skills", None)
                            or getattr(agent, "tools", None)
                            or []
                ]
            }
            for agent in agents
        ]

        prompt = {"task": input_params, "agents": agents_metadata}
        raw_response = selector_agent.run(json.dumps(prompt))
        
        def _extract_text_from_response(r: Any) -> str:
            # If it's already a string, return it.
            if isinstance(r, str):
                return r
            # Message-like objects often have a .content list of text blocks.
            content = None
            if hasattr(r, "content"):
                content = getattr(r, "content")
            elif isinstance(r, dict) and "content" in r:
                content = r.get("content")
            if content:
                parts: List[str] = []
                for block in content:
                    if isinstance(block, str):
                        parts.append(block)
                    elif hasattr(block, "text"):
                        parts.append(getattr(block, "text") or "")
                    elif isinstance(block, dict) and "text" in block:
                        parts.append(block.get("text") or "")
                return "".join(parts)
            # Fallback: use .text if present, else str()
            if hasattr(r, "text"):
                return getattr(r, "text") or ""
            return str(r)

        response_text = _extract_text_from_response(raw_response)
        try:
            logging.info(f"Selector response: {raw_response}")
            choice = json.loads(response_text)
            if choice == {}:
                raise ValueError("No suitable agent found")
            agent_name = choice["agent"]
            method_name = choice["method"]
            params = choice.get("parameter", {})
        except Exception as e:
            logging.error(f"Selector error: {e}; raw_response={raw_response}; extracted_text={response_text}")
            raise ValueError(f"Invalid selector output: {raw_response}") from e

        for agent in agents:
            if agent.name == agent_name:
                for skill in (agent.skills or agent.tools or []): # type: ignore agent.skills:
                    if isinstance(skill, dict):
                        skill_name = skill.get("name")
                    else:
                        skill_name = getattr(skill, "name", None)
                    if skill_name == method_name:
                        return agent, skill, params or input_params

        raise ValueError(f"Agent or method not found: {choice}")

    async def route_and_execute(
        self,
        input_params: Dict[str, Any],
        agent_cards: List[AgentCard],
        verb: Optional[str] = None,
        framework: Optional[str] = None,
        model: Optional[str] = None,
        auth_callback: Optional[AuthCallback] = None,
    ) -> Any:
        try:
            selector_agent = self._create_selector_agent(framework, model)
            agent_card, skill, params = self.select_agent_and_method_llm(input_params, agent_cards, selector_agent)

            agent = Agent(agent_card)
            if isinstance(skill, dict):
                skill_name = skill.get("name")
            else:
                skill_name = getattr(skill, "name", None)
            print(f"[Router] Routed to {agent_card.name}.{skill_name} …")
            return await agent.invoke(verb=skill_name, arguments=params)
        except Exception as e:
            print(f"[Router] Error occurred: {e}")
            return None

    def _create_selector_agent(self, framework: Optional[str], model: Optional[str]):
        framework = framework or os.getenv("AGENT_SELECTOR_FRAMEWORK", "anthropic")
        model = model or os.getenv("AGENT_SELECTOR_MODEL", "claude-3-5-haiku-latest")
        print(f"[Router] Using selector framework={framework}, model={model}")

        if framework == "anthropic":
            api_key = os.getenv("ANTHROPIC_API_KEY")
            return ClaudeAgent(api_key, model, AgentRouter.instructions)

        return AnyAgent.create(
            framework,
            AgentConfig(
                model_id=model,
                instructions=AgentRouter.instructions,
            )
        )