import requests
import os
import logging
from dotenv import load_dotenv
from typing import Optional
from .model.agent_card import AgentCard, Authentication
from .model.typings import AuthCallback
from .agent import Agent
from .agent_router import AgentRouter
from typing import Any, Dict, List, Callable

class AgentNameServiceClient:
    DEFAULT_SERVICE_URL = "https://app.clearentitlement.com"
    PATH_GET_AGENTS = "/ce/admin/Agents"
    PATH_QUERY_ENTITLEMENTS = "/ce/entitlement/queryfunctional"
    
    agent_cards: dict[str, AgentCard]
    entitlements: dict[str, dict]

    def __init__(self, service_url: Optional[str] = None):
        self.service_url = (service_url or self.DEFAULT_SERVICE_URL).rstrip("/")
        self.agent_cards = dict()  # Initialize agent_cards dict
        self.entitlements = dict()  # Initialize entitlements dict

    """
    Invokes the agent with the given question and returns the response.
        verb: if the developer know what verb or action should be taken
        agent_cards: a list of agent card

    """

    async def invoke(
        self,
        uid: str,
        question: Dict[str, Any],
        verb: Optional[str] = None,
        agent_cards: Optional[List[AgentCard]] = None,
        auth_callback: Optional[AuthCallback] = None,
    ) -> Optional[Dict[str, Any]]:
        if question is None or not question:
            return None
        agent_router = AgentRouter()
        if agent_cards is None or not agent_cards:
            agent_cards = list(self._get_agents(uid).values())
        if len(agent_cards) == 0:
            return None
        elif len(agent_cards) == 1:
            pass
            # agent = Agent(agent_cards[0])
            # return await agent.invoke(question, verb, auth_callback = auth_callback)
        return await agent_router.route_and_execute(
            input_params=question, verb=verb, agent_cards=agent_cards, auth_callback = auth_callback
        )

    def lookup(
        self, uid: str, name: Optional[str] = None
    ) -> dict[str, AgentCard] | AgentCard | None:
        agents = self._get_agents(uid)
        if name is None:
            return agents
        return agents.get(name)

    def _get_agents(self, uid: str) -> dict[str, AgentCard]:
        if self.agent_cards:
            return self.agent_cards
        """
        Fetch agents from the API and return a dict mapping agent name to Agent instance.
        """
        print("Fetching agents from API...")
        url = f"{self.service_url}{self.PATH_GET_AGENTS}"
        api_key = os.getenv("AGNS_API_KEY")
        if not api_key:
            raise ValueError("AGNS_API_KEY environment variable is not set")
        # print("API Key:", api_key)  # Debugging line to check API key

        headers = {"ApiKey": api_key}
        response = requests.get(url, headers=headers)
        response.raise_for_status()
        agent_cards_json = response.json()
        print("API response:", agent_cards_json)

        agent_cards_dict = dict[str, AgentCard]()
        for agent_dict in agent_cards_json:
            try:
                card = self._convert_agent_api_response(agent_dict)
                agent_cards_dict[card.name] = card
            except Exception as e:
                logging.error(
                    f"Skipping agent {agent_dict.get('name')} due to error: {e}, data: {agent_dict}"
                )

        logging.info(
            f"Total agents fetched: {len(agent_cards_dict)}, agents: {list(agent_cards_dict.keys())}"
        )
        _query_entitlements = self._query_entitlements(uid)
        print("Entitlements fetched:", _query_entitlements)
        # Filter agents based on entitlements if needed
        functionalAccess = _query_entitlements.get("functionalAccess", [])
        if functionalAccess:
            allowed = set()
            for entry in functionalAccess:
                if not isinstance(entry, dict):
                    continue
                resource = (entry.get("resource") or "").strip()
                action = entry.get("action")
                if action != "access":
                    continue
                if not resource.startswith("agent/"):
                    continue
                parts = resource.split("/", 2)
                if len(parts) >= 2 and parts[1]:
                    allowed.add(parts[1])
            if allowed:
                agent_cards_dict = {
                    name: card
                    for name, card in agent_cards_dict.items()
                    if name in allowed
                }
            else:
                agent_cards_dict = {}
            logging.info(
                f"Agents after entitlement filtering: {list(agent_cards_dict.keys())}"
            )
        else:
            agent_cards_dict = {}
        self.agent_cards = agent_cards_dict
        return self.agent_cards

    def _query_entitlements(self, uid: str) -> dict:
        if uid in self.entitlements:
            return self.entitlements[uid]
        
        """
        Fetch entitlements from the API.
        """
        print("Fetching entitlements from API...")
        url = f"{self.service_url}{self.PATH_QUERY_ENTITLEMENTS}"
        api_key = os.getenv("AGNS_API_KEY")
        if not api_key:
            raise ValueError("AGNS_API_KEY environment variable is not set")
        # print("API Key:", api_key)  # Debugging line to check API key

        headers = {"ApiKey": api_key}
        data = {"app":"cdd85dc6-a31e-446c-ae01-d54050c28c65","resource":"agent","context":{},"action":"access","user": uid}
        response = requests.post(url, headers=headers, json=data)

        response.raise_for_status()
        entitlements_json = response.json()
        print("API response:", entitlements_json)

        self.entitlements[uid] = entitlements_json
        return entitlements_json
    

    def _convert_agent_api_response(self, agent_dict: dict[str, Any]) -> AgentCard:
        """
        Convert API agent dict to AgentCard-compatible object.
        Adjust the mapping as needed based on your API response.
        """
        metadata = agent_dict.get("metadata", {})
        authentication = agent_dict.get("authentication", {})
        if authentication:
            authentication["auth_method"] = authentication.get("authMethod")
            del authentication["authMethod"]
        metadata["id"] = agent_dict.get("id")
        metadata["org_id"] = agent_dict.get("orgId")
        metadata["alias"] = agent_dict.get("name")
        metadata["authentication"] = Authentication(**authentication) if authentication else None
        return AgentCard(**metadata)

class LazyClient:
    def __init__(self):
        self._client: Optional[AgentNameServiceClient] = None

    def _get_default_client(self) -> AgentNameServiceClient:
        load_dotenv()  # Load .env once, only when needed
        return AgentNameServiceClient()

    def __getattr__(self, item: str) -> Any:
        if self._client is None:
            self._client = self._get_default_client()
        return getattr(self._client, item)
