from datetime import datetime, timezone
import logging
from typing import Iterable
import anthropic
from anthropic.types import MessageParam
from anthropic.types.message import Message as AnthropicMessage
from openai import OpenAI

import ollama
from ollama._types import Message as OllamaMessage, ChatResponse as OllamaChatResponse

from ..model.message import Message

class ClaudeAgent:
    def __init__(
        self,
        api_key: str,
        model: str = "claude-3-haiku-20240307",
        instructions: str = ""
    ):
        self.client = anthropic.Client(api_key=api_key)
        self.model = model
        self.instructions = instructions

    def run(self, messages: str | dict | list[Message]) -> Message:
        user_input_text = messages
        # Messages API expects a list of message dicts with roles
        anthropic_messages = self._to_anthropic_messages(messages)
        logging.info(f"Claude API request: {anthropic_messages}")
        prompt = f"{self.instructions}\n\nHuman: {user_input_text}\n\nAssistant:"

        response = self.client.messages.create(
            model=self.model,
            system=self.instructions,
            messages=anthropic_messages,
            max_tokens=1000,
        )
        logging.info(f"Claude API response type: {type(response)}")
        logging.info(f"Claude API response: {response}")
        return self._to_message(response)
    
    def _to_anthropic_messages(self, messages) -> Iterable[MessageParam]:
        if (isinstance(messages, list)):
            return [self._to_anthropic_message(msg) for msg in messages]
        return [self._to_anthropic_message(messages)]

    def _to_anthropic_message(self, message) -> MessageParam:
        if isinstance(message, str):
            return {"role": "user", "content": message}
        role = message.get("role", "user")
        message = message.get("message", str(message))
        result = MessageParam(role=role, content=message)
        return result
    
    def _to_message(self, anthropic_message: AnthropicMessage) -> Message:
        return Message(
            sender=anthropic_message.role,
            receiver="user",
            role=anthropic_message.role,
            content=anthropic_message.content,
            model=anthropic_message.model,
            created_at=datetime.now(timezone.utc).isoformat()
        )

    
class OpenaiAgent:
    def __init__(self, api_key: str, model: str, instructions: str = ""):
        self.client = OpenAI(api_key=api_key)
        self.model = model
        self.instructions = instructions

    async def run(self, messages: str | dict | list[Message]) -> str:
        """
        Call OpenAI LLM with optional system instructions.
        """
        messages = []
        if self.instructions:
            messages.append({"role": "system", "content": self.instructions})
        messages.append({"role": "user", "content": input_text})

        # Using the Chat Completions API
        response = self.client.chat.completions.create(
            model=self.model,
            messages=messages,
        )
        return response.choices[0].message.content or ""
    
class LLAMA3Agent:
    def __init__(self, api_key: str = '', model: str = "llama3.1:8b-q4", url: str = "http://localhost:11434", instructions: str = ""):
        self.api_key = api_key
        self.model = model
        self.url = url
        self.instructions = instructions

    def run(self, messages: str | dict | list[Message]) -> Message:
        client = ollama.Client(self.url)
        response = client.chat(self.model, messages=self._to_llama_messages(messages))
        logging.info(f"LLAMA3 API response type: {type(response)}")
        logging.info(f"LLAMA3 API response: {response}")
        return self._to_message(response)
    
    def _to_llama_messages(self, messages) -> list[OllamaMessage]:
        if (isinstance(messages, list)):
            return [self._to_llama_message(msg) for msg in messages]
        return [self._to_llama_message(messages)]

    def _to_llama_message(self, message) -> OllamaMessage:
        if isinstance(message, str):
            return OllamaMessage(role="user", content=message)
        role = message.get("role", "user")
        message = message.get("message", str(message))
        result = OllamaMessage(role=role, content=message)
        return result
    
    def _to_message(self, ollama_response: OllamaChatResponse) -> Message:
        return Message(
            sender=ollama_response.message.role,
            receiver="user",
            role=ollama_response.message.role,
            content=ollama_response.message.content,
            model=self.model,
            created_at=getattr(ollama_response, "created_at", None)
        )
