import logging
import json
import os
from .llm_agent import ClaudeAgent, LLAMA3Agent
from typing import Any, Dict, Optional
from uuid import uuid4
from ..model.agent_card import AgentCard
from ..model.message import Message
from any_agent import AnyAgent, AgentConfig, AgentFramework, AgentTrace

class LLMAgentCaller:
    
    INSTRUCTIONS = 'You are a smart LLM Agent'
    def __init__(self, agent_card: AgentCard):
        self.agent_card = agent_card
    
    async def call(self, tool_name: str, arguments: str | dict[str, Any] | list[Message]) -> Message:
        logging.info(f"Calling LLM agent {self.agent_card.name} with tool {tool_name} and arguments: {arguments} at {self.agent_card.url}")
        llm_agent = self.__create_llm_agent()
        if isinstance(llm_agent, AnyAgent):
            return self._to_message(llm_agent.run(json.dumps(arguments)))
        return self._generate_response_message(llm_agent.run(arguments))

    def _generate_response_message(self, original_message: Message) -> Message:
        result = original_message
        result.agent_id = self.agent_card.id
        result.agent_name = self.agent_card.name
        return result
    
    def _to_message(self, original_message: AgentTrace) -> Message:
        return Message(
            sender=self.agent_card.name,
            receiver="user",
            role="assistant",
            content=original_message.final_output,
            model=self.agent_card.url,
            agent_id=self.agent_card.id,
            agent_name=self.agent_card.name
        )
    
    def __create_llm_agent(self):
        [framework, model] = self.agent_card.url.split('://')
        print(f"[Router] Using selector framework={framework}, model={model}")

        if framework == "anthropic":
            authentication = getattr(self.agent_card, 'authentication', None)
            if authentication is not None and hasattr(authentication, 'secret'):
                api_key = authentication.secret
            else:
                api_key = os.getenv("ANTHROPIC_API_KEY")
            return ClaudeAgent(api_key, model, LLMAgentCaller.INSTRUCTIONS)
        elif framework == "local_llama":
            return LLAMA3Agent(model=model) 

        return AnyAgent.create(
            framework,
            AgentConfig(
                model_id=model,
                instructions=LLMAgentCaller.INSTRUCTIONS,
            )
        )