import logging
from typing import Any, Dict, Optional
from mcp import ClientSession
from mcp.client.streamable_http import streamablehttp_client
from ..model.agent_card import AgentCard
from pydantic import AnyUrl

class MCPAgentCaller:
    def __init__(self, agent_card: AgentCard):
        self.agent_card = agent_card

    async def call_tool(self, tool_name: str, arguments: Optional[Dict[str, Any]] = None) -> Any:

        logging.info(f"Calling MCP agent {self.agent_card.name} with tool {tool_name} and arguments: {arguments} at {self.agent_card.url}")
        
        if arguments is None:
            arguments = {}
        async with streamablehttp_client(self.agent_card.url) as (read, write, _):
            async with ClientSession(read, write) as session:
                await session.initialize()    
                params = {"args": arguments, "kwargs": {}}  # Example parameters, adjust as needed
                result = await session.call_tool(tool_name, params)
                logging.info(f"{tool_name, arguments}, result: {result}")
                if result.isError:
                    logging.error(f"Error calling tool {tool_name}: {result}")
                    return {"isError": True, "error": result}
                else:
                    sc = getattr(result, "structuredContent", None)
                    return sc if sc is not None else result.content

    async def get_prompt(self, prompt_name: str, arguments: Optional[Dict[str, Any]] = None) -> Any:

        logging.info(f"Calling MCP agent {self.agent_card.name} with prompt {prompt_name} and arguments: {arguments} at {self.agent_card.url}")

        if arguments is None:
            arguments = {}
        async with streamablehttp_client(self.agent_card.url) as (read, write, _):
            async with ClientSession(read, write) as session:
                await session.initialize()    
                params = {"args": arguments, "kwargs": {}}  # Example parameters, adjust as needed
                result = await session.get_prompt(prompt_name, params)
                print(f"{prompt_name, arguments}, result: {result.structuredContent if hasattr(result, 'structuredContent') else result}")
                return result

    async def read_resource(self, resource_name: str) -> Any:

        logging.info(f"Calling MCP agent {self.agent_card.name} with resource {resource_name} at {self.agent_card.url}")
        uri = self.agent_card.resources[resource_name]
        async with streamablehttp_client(self.agent_card.url) as (read, write, _):
            async with ClientSession(read, write) as session:
                await session.initialize()
                result = await session.read_resource(uri)
                print(f"{uri}, result: {result.structuredContent if hasattr(result, 'structuredContent') else result}")
                return result