import asyncio
from typing import Dict, Any
from pydantic import create_model
from mcp.client.session import ClientSession
from mcp.client.transport.http import HttpTransport
import openai
import json

openai.api_key = "YOUR_OPENAI_KEY"

async def call_best_tool_with_llm(server_url: str, prompt: str = None, args: Dict[str, Any] = None):
    async with ClientSession(HttpTransport(server_url)) as client:
        # 1. List available tools
        tools = await client.list_tools()
        print(f"Discovered tools: {[t.name for t in tools]}")

        chosen_tool = None
        final_args = args

        if args:
            # Pick tool whose required fields match args
            for t in tools:
                required = set(t.inputSchema.get("required", []))
                if required.issubset(args.keys()):
                    chosen_tool = t
                    break
        elif prompt:
            # Use LLM to select tool and generate arguments
            tool_names = [t.name for t in tools]
            # Prompt engineering: ask LLM to pick tool and fill arguments
            llm_prompt = f"""
You have the following tools available: {tool_names}.
Given the user prompt: "{prompt}",
1) Select the best tool name.
2) Provide the arguments in JSON format matching the tool's schema.
Only output a JSON object with keys: "tool": tool_name, "args": {{...}}
"""
            resp = openai.ChatCompletion.create(
                model="gpt-4",
                messages=[{"role": "user", "content": llm_prompt}],
                temperature=0,
            )
            llm_text = resp.choices[0].message.content.strip()
            try:
                llm_output = json.loads(llm_text)
                tool_name = llm_output["tool"]
                final_args = llm_output["args"]
                # Find chosen tool object
                chosen_tool = next(t for t in tools if t.name == tool_name)
            except Exception as e:
                raise ValueError(f"Failed to parse LLM output: {llm_text}") from e

        if not chosen_tool:
            raise ValueError("No suitable tool found for given prompt/args")

        print(f"Chosen tool: {chosen_tool.name}")
        print(f"Arguments: {final_args}")

        # 2. Build parameter model from schema
        props = chosen_tool.inputSchema.get("properties", {})
        required_fields = chosen_tool.inputSchema.get("required", [])

        model_fields = {}
        for field_name, field_spec in props.items():
            py_type = str
            t = field_spec.get("type")
            if t == "integer": py_type = int
            elif t == "number": py_type = float
            elif t == "boolean": py_type = bool
            elif t == "array": py_type = list
            elif t == "object": py_type = dict
            model_fields[field_name] = (py_type, ... if field_name in required_fields else None)

        ArgsModel = create_model(f"{chosen_tool.name}Args", **model_fields)

        # Validate arguments
        args_obj = ArgsModel(**final_args) if final_args else ArgsModel()

        # 3. Call the tool
        result = await client.call_tool(chosen_tool.name, args_obj.dict())
        return result

# Example usage
async def main():
    server_url = "http://localhost:8000"
    prompt = "Add 100 and 200 using the math agent"
    result = await call_best_tool_with_llm(server_url, prompt=prompt)
    print("Result:", result)

if __name__ == "__main__":
    asyncio.run(main())
