'Amino Acid sequence processing utilities.'
from collections import OrderedDict

IUPAC_CODES = OrderedDict(
    [
        ('Ala', 'A'),
        ('Asx', 'B'),  # Aspartate or Asparagine
        ('Cys', 'C'),
        ('Asp', 'D'),
        ('Glu', 'E'),
        ('Phe', 'F'),
        ('Gly', 'G'),
        ('His', 'H'),
        ('Ile', 'I'),
        ('Lys', 'K'),
        ('Leu', 'L'),
        ('Met', 'M'),
        ('Asn', 'N'),
        ('Pyl', 'O'),  # Pyrrolysin
        ('Pro', 'P'),
        ('Gln', 'Q'),
        ('Arg', 'R'),
        ('Ser', 'S'),
        ('Thr', 'T'),
        ('Sec', 'U'),  # Selenocysteine
        ('Val', 'V'),
        ('Trp', 'W'),
        ('Xaa', 'X'),  # Any AA
        ('Tyr', 'Y'),
        ('Glx', 'Z'),  # Glutamate or Glutamine
    ]
)

IUPAC_VOCAB = OrderedDict(
    [
        ('<PAD>', 0),
        ('<MASK>', 1),
        ('<CLS>', 2),
        ('<SEP>', 3),
        ('<UNK>', 4),
        ('A', 5),
        ('B', 6),
        ('C', 7),
        ('D', 8),
        ('E', 9),
        ('F', 10),
        ('G', 11),
        ('H', 12),
        ('I', 13),
        ('K', 14),
        ('L', 15),
        ('M', 16),
        ('N', 17),
        ('O', 18),
        ('P', 19),
        ('Q', 20),
        ('R', 21),
        ('S', 22),
        ('T', 23),
        ('U', 24),
        ('V', 25),
        ('W', 26),
        ('X', 27),
        ('Y', 28),
        ('Z', 29),
        ('<START>', 30),
        ('<STOP>', 31),
        ('-', 32),
    ]
)

UNIREP_VOCAB = OrderedDict(
    [
        ('<PAD>', 0),
        ('M', 1),
        ('R', 2),
        ('H', 3),
        ('K', 4),
        ('D', 5),
        ('E', 6),
        ('S', 7),
        ('T', 8),
        ('N', 9),
        ('Q', 10),
        ('C', 11),
        ('U', 12),
        ('G', 13),
        ('P', 14),
        ('A', 15),
        ('V', 16),
        ('I', 17),
        ('F', 18),
        ('Y', 19),
        ('W', 20),
        ('L', 21),
        ('O', 22),
        ('X', 23),
        ('Z', 23),
        ('B', 23),
        ('J', 23),
        ('<CLS>', 24),
        ('<SEP>', 25),
        ('<START>', 26),
        ('<STOP>', 27),
        ('<UNK>', 28),
        ('<MASK>', 29),
    ]
)

HUMAN_KINASE_ALIGNMENT_VOCAB = OrderedDict(
    [
        ('<PAD>', 0),
        ('<MASK>', 1),
        ('<CLS>', 2),
        ('<SEP>', 3),
        ('<UNK>', 4),
        ('-', 5),
        ('A', 6),
        ('C', 7),
        ('D', 8),
        ('E', 9),
        ('F', 10),
        ('G', 11),
        ('H', 12),
        ('I', 13),
        ('K', 14),
        ('L', 15),
        ('M', 16),
        ('N', 17),
        ('P', 18),
        ('Q', 19),
        ('R', 20),
        ('S', 21),
        ('T', 22),
        ('V', 23),
        ('W', 24),
        ('Y', 25),
        ('<START>', 26),
        ('<STOP>', 27),
    ]
)

REAL_AAS = list(filter(lambda x: x.isalpha(), HUMAN_KINASE_ALIGNMENT_VOCAB.keys()))

"""
Polarity, Charge, Hydophobicity, Aromaticity, Ionizability, StartStop
Nonpolar = -1, Polar = 1
Negative = -1, Neutral = 0, Positive = 1
Hydrophobic = -1, Hydrophilic = 1
NonAromatic = -1, Aromatic = 1
NonIonizable = -1, Ionizable = 1
Stop = -1, Start = 1, Neither = 0
"""
AA_PROPERTIES_NUM = OrderedDict(
    [
        ('A', (-1, 0, -1, -1, -1, 0)),
        ('B', (1, -0.5, 1, -1, 0, 0)),  # mean of D, N
        ('C', (1, 0, 1, -1, 1, 0)),
        ('D', (1, -1, 1, -1, 1, 0)),
        ('E', (1, -1, 1, -1, 1, 0)),
        ('F', (-1, 0, -1, 1, -1, 0)),
        ('G', (-1, 0, -1, -1, -1, 0)),
        ('H', (1, 1, 1, -1, 1, 0)),
        ('I', (-1, 0, -1, -1, -1, 0)),
        ('K', (1, 1, 1, -1, 1, 0)),
        ('L', (-1, 0, -1, -1, -1, 0)),
        ('M', (-1, 0, -1, -1, -1, 0)),
        ('N', (1, 0, 1, -1, -1, 0)),
        ('O', (1, 0, 1, 1, 1, 0)),  # Pyrrolysine
        ('P', (-1, 0, -1, -1, -1, 0)),
        ('Q', (1, 0, 1, -1, -1, 0)),
        ('R', (1, 1, 1, -1, 1, 0)),
        ('S', (1, 0, 1, -1, -1, 0)),
        ('T', (1, 0, 1, -1, -1, 0)),
        ('U', (-1, 0, -1, -1, 1, 0)),  # Selenocyteine
        ('V', (-1, 0, -1, -1, -1, 0)),
        ('W', (-1, 0, -1, 1, -1, 0)),
        ('X', (0.2, 0, 0.1, -0.7, -0.2, 0)),  # mean AA (Unknown)
        ('Y', (1, 0, -1, 1, 1, 0)),
        ('Z', (1, -0.5, 1, -1, 0, 0)),  # mean of E, Q
        ('<PAD>', (0, 0, 0, 0, 0, 0)),
        ('<UNK>', (0, 0, 0, 0, 0, 0)),  # PAD and UNK are identical
        ('<START>', (0, 0, 0, 0, 0, 1)),
        ('<STOP>', (0, 0, 0, 0, 0, -1)),
        ('-', (0, 0, 0, 0, 0, 0.5)),
    ]
)
"""
Molecular Weight, Residue Weight, pKa, pKb, pKx, pI, Hydrophobicity at pH2
Taken from: https://www.sigmaaldrich.com/life-science/metabolomics/learning-center/amino-acid-reference-chart.html
"""
AA_FEAT = OrderedDict(
    [
        ('A', (89.1, 71.08, 2.34, 9.69, 0, 6, 47, 0)),
        ('B', (132.615, 114.6, 1.95, 9.2, 1.825, 4.09, -29.5, 0)),  # D/N mean
        ('C', (121.16, 103.15, 1.96, 10.28, 8.18, 5.07, 52, 0)),
        ('D', (133.11, 115.09, 1.88, 9.6, 3.65, 2.77, -18, 0)),
        ('E', (147.13, 129.12, 2.19, 9.67, 4.25, 3.22, 8, 0)),
        ('F', (165.19, 147.18, 1.83, 9.13, 0, 5.48, 92, 0)),
        ('G', (75.07, 57.05, 2.34, 9.6, 0, 5.97, 0, 0)),
        ('H', (155.16, 137.14, 1.82, 9.17, 6, 7.59, -42, 0)),
        ('I', (131.18, 113.16, 2.36, 9.6, 0, 6.02, 100, 0)),
        ('K', (146.19, 128.18, 2.18, 8.95, 10.53, 9.74, -37, 0)),
        ('L', (131.18, 113.16, 2.36, 9.6, 0, 5.98, 100, 0)),
        ('M', (149.21, 131.2, 2.28, 9.21, 0, 5.74, 74, 0)),
        ('N', (132.12, 114.11, 2.02, 8.8, 0, 5.41, -41, 0)),
        ('O', (131.13, 113.11, 1.82, 9.65, 0, 0, 0, 0)),  # Pyrrolysine
        ('P', (115.13, 97.12, 1.99, 10.6, 0, 6.3, -46, 0)),
        ('Q', (146.15, 128.13, 2.17, 9.13, 0, 5.65, -18, 0)),
        ('R', (174.2, 156.19, 2.17, 9.04, 12.48, 10.76, -26, 0)),
        ('S', (105.09, 87.08, 2.21, 9.15, 0, 5.68, -7, 0)),
        ('T', (119.12, 101.11, 2.09, 9.1, 0, 5.6, 13, 0)),
        ('U', (168.07, 150.05, 5.47, 10.28, 0, 3.9, 52, 0)),  # Selenocyteine
        ('V', (117.15, 99.13, 2.32, 9.62, 0, 5.96, 97, 0)),
        ('W', (204.23, 186.22, 2.83, 9.39, 0, 5.89, 84, 0)),
        ('X', (136.74, 118.73, 2.06, 9.00, 2.51, 5.74, 21.86, 0)),  # mean AA (Unknown)
        ('Y', (181.19, 163.18, 2.2, 9.11, 10.07, 5.66, 49, 0)),
        ('Z', (146.64, 128.625, 2.18, 9.4, 2.125, 4.435, -5, 0)),  # mean of E, Q
        ('<PAD>', (0, 0, 0, 0, 0, 0, 0, 0)),
        ('<START>', (0, 0, 0, 0, 0, 0, 0, 1)),
        ('<STOP>', (0, 0, 0, 0, 0, 0, 0, -1)),
        ('<UNK>', (0, 0, 0, 0, 0, 0, 0, 0)),  # PAD and UNK are identical
        ('-', (0, 0, 0, 0, 0, 0, 0, 0.5)),
    ]
)
"""
Taken from: https://www.ncbi.nlm.nih.gov/Class/FieldGuide/BLOSUM62.txt
"""
# yapf: disable
BLOSUM62 = OrderedDict(
    [

        (
            'A', (
                4, -1, -2, -2, 0, -1, -1, 0, -2, -1, -1, -1, -1, -2, -1, 1, 0,
                -3, -2, 0, -2, -1, 0, -4, -4, -4
            )
        ),
        (
            'B', (
                -2, -1, 3, 4, -3, 0, 1, -1, 0, -3, -4, 0, -3, -3, -2, 0, -1,
                -4, -3, -3, 4, 1, -1, -4, -4, -4
            )
        ),
        (
            'C', (
                0, -3, -3, -3, 9, -3, -4, -3, -3, -1, -1, -3, -1, -2, -3, -1,
                -1, -2, -2, -1, -3, -3, -2, -4, -4, -4
            )
        ),
        (
            'D', (
                -2, -2, 1, 6, -3, 0, 2, -1, -1, -3, -4, -1, -3, -3, -1, 0, -1,
                -4, -3, -3, 4, 1, -1, -4, -4, -4
            )
        ),
        (
            'E', (
                -1, 0, 0, 2, -4, 2, 5, -2, 0, -3, -3, 1, -2, -3, -1, 0, -1, -3,
                -2, -2, 1, 4, -1, -4, -4, -4
            )
        ),
        (
            'F', (
                -2, -3, -3, -3, -2, -3, -3, -3, -1, 0, 0, -3, 0, 6, -4, -2, -2,
                1, 3, -1, -3, -3, -1, -4, -4, -4
            )
        ),
        (
            'G', (
                0, -2, 0, -1, -3, -2, -2, 6, -2, -4, -4, -2, -3, -3, -2, 0, -2,
                -2, -3, -3, -1, -2, -1, -4, -4, -4
            )
        ),
        (
            'H', (
                -2, 0, 1, -1, -3, 0, 0, -2, 8, -3, -3, -1, -2, -1, -2, -1, -2,
                -2, 2, -3, 0, 0, -1, -4, -4, -4
            )
        ),
        (
            'I', (
                -1, -3, -3, -3, -1, -3, -3, -4, -3, 4, 2, -3, 1, 0, -3, -2, -1,
                -3, -1, 3, -3, -3, -1, -4, -4, -4
            )
        ),
        (
            'K', (
                -1, 2, 0, -1, -3, 1, 1, -2, -1, -3, -2, 5, -1, -3, -1, 0, -1,
                -3, -2, -2, 0, 1, -1, -4, -4, -4
            )
        ),
        (
            'L', (
                -1, -2, -3, -4, -1, -2, -3, -4, -3, 2, 4, -2, 2, 0, -3, -2, -1,
                -2, -1, 1, -4, -3, -1, -4, -4, -4
            )
        ),
        (
            'M', (
                -1, -1, -2, -3, -1, 0, -2, -3, -2, 1, 2, -1, 5, 0, -2, -1, -1,
                -1, -1, 1, -3, -1, -1, -4, -4, -4
            )
        ),
        (
            'N', (
                -2, 0, 6, 1, -3, 0, 0, 0, 1, -3, -3, 0, -2, -3, -2, 1, 0, -4,
                -2, -3, 3, 0, -1, -4, -4, -4
            )
        ),
        (
            'O', (
                0, -1, -1, -1, -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, 0,
                0, -2, -1, -1, -1, -1, -1, -4, -4, -4
            )
        ),  # Pyrrolysine encoded as unknown
        (
            'P', (
                -1, -2, -2, -1, -3, -1, -1, -2, -2, -3, -3, -1, -2, -4, 7, -1,
                -1, -4, -3, -2, -2, -1, -2, -4, -4, -4
            )
        ),
        (
            'Q', (
                -1, 1, 0, 0, -3, 5, 2, -2, 0, -3, -2, 1, 0, -3, -1, 0, -1, -2,
                -1, -2, 0, 3, -1, -4, -4, -4
            )
        ),
        (
            'R', (
                -1, 5, 0, -2, -3, 1, 0, -2, 0, -3, -2, 2, -1, -3, -2, -1, -1,
                -3, -2, -3, -1, 0, -1, -4, -4, -4
            )
        ),
        (
            'S', (
                1, -1, 1, 0, -1, 0, 0, 0, -1, -2, -2, 0, -1, -2, -1, 4, 1, -3,
                -2, -2, 0, 0, 0, -4, -4, -4
            )
        ),
        (
            'T', (
                0, -1, 0, -1, -1, -1, -1, -2, -2, -1, -1, -1, -1, -2, -1, 1, 5,
                -2, -2, 0, -1, -1, 0, -4, -4, -4
            )
        ),
        (
            'U', (
                0, -1, -1, -1, -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, 0,
                0, -2, -1, -1, -1, -1, -1, -4, -4, -4
            )
        ),  # Selenocysteine encoded as unknown
        (
            'V', (
                0, -3, -3, -3, -1, -2, -2, -3, -3, 3, 1, -2, 1, -1, -2, -2, 0,
                -3, -1, 4, -3, -2, -1, -4, -4, -4
            )
        ),
        (
            'W', (
                -3, -3, -4, -4, -2, -2, -3, -2, -2, -3, -2, -3, -1, 1, -4, -3,
                -2, 11, 2, -3, -4, -3, -2, -4, -4, -4
            )
        ),
        (
            'X', (
                0, -1, -1, -1, -2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -2, 0,
                0, -2, -1, -1, -1, -1, -1, -4, -4, -4
            )
        ),
        (
            'Y', (
                -2, -2, -2, -3, -2, -1, -2, -3, 2, -1, -1, -2, -1, 3, -3, -2,
                -2, 2, 7, -1, -3, -2, -1, -4, -4, -4
            )
        ),
        (
            'Z', (
                -1, 0, 0, 1, -3, 3, 4, -2, 0, -3, -3, 1, -1, -3, -1, 0, -1, -3,
                -2, -2, 1, 4, -1, -4, -4, -4
            )
        ),
        (
            '<PAD>', (
                -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
                -4, -4, -4, -4, -4, -4, -4, 10, -4, -4
            )
        ),
        (
            '<START>', (
                -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
                -4, -4, -4, -4, -4, -4, -4, -4, 10, -4
            )
        ),
        (
            '<STOP>', (
                -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
                -4, -4, -4, -4, -4, -4, -4, -4, -4, 10
            )
        ),
        (
            '<UNK>', (
                -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
                -4, -4, -4, -4, -4, -4, -4, 10, -4, -4
            )
        ),  # UNK is identical to PAD
        (
            '-', (
                -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4, -4,
                -4, -4, -4, -4, -4, -4, -4, 4, -4, -4
            )
        ),  # dash is half-way to a <START>
    ]
)

BLOSUM62_NORM = OrderedDict(
    [
        (
            'A', (
                0.2901, 0.0310, 0.0256, 0.0297, 0.0216, 0.0256, 0.0405, 0.0783,
                0.0148, 0.0432, 0.0594, 0.0445, 0.0175, 0.0216, 0.0297, 0.0850,
                0.0499, 0.0054, 0.0175, 0.0688, 0, 0
            )
        ),
        (
            'B', (
                0.0419, 0.0374, 0.1930, 0.2403, 0.0083, 0.0318, 0.0704, 0.0559,
                0.0251, 0.0225, 0.0298, 0.0494, 0.0103, 0.0165, 0.0213, 0.0610,
                0.0424, 0.0041, 0.0135, 0.0257, 0, 0
            )
        ),
        (
            'C', (
                0.0650, 0.0163, 0.0163, 0.0163, 0.4837, 0.0122, 0.0163, 0.0325,
                0.0081, 0.0447, 0.0650, 0.0203, 0.0163, 0.0203, 0.0163, 0.0407,
                0.0366, 0.0041, 0.0122, 0.0569, 0, 0
            )
        ),
        (
            'D', (
                0.0410, 0.0299, 0.0690, 0.3974, 0.0075, 0.0299, 0.0914, 0.0466,
                0.0187, 0.0224, 0.0280, 0.0448, 0.0093, 0.0149, 0.0224, 0.0522,
                0.0354, 0.0037, 0.0112, 0.0243, 0, 0
            )
        ),
        (
            'E', (
                0.0552, 0.0497, 0.0405, 0.0902, 0.0074, 0.0645, 0.2965, 0.0350,
                0.0258, 0.0221, 0.0368, 0.0755, 0.0129, 0.0166, 0.0258, 0.0552,
                0.0368, 0.0055, 0.0166, 0.0313, 0, 0
            )
        ),
        (
            'F', (
                0.0338, 0.0190, 0.0169, 0.0169, 0.0106, 0.0106, 0.0190, 0.0254,
                0.0169, 0.0634, 0.1142, 0.0190, 0.0254, 0.3869, 0.0106, 0.0254,
                0.0254, 0.0169, 0.0888, 0.0550, 0, 0
            )
        ),
        (
            'G', (
                0.0783, 0.0229, 0.0391, 0.0337, 0.0108, 0.0189, 0.0256, 0.5101,
                0.0135, 0.0189, 0.0283, 0.0337, 0.0094, 0.0162, 0.0189, 0.0513,
                0.0297, 0.0054, 0.0108, 0.0243, 0, 0
            )
        ),
        (
            'H', (
                0.0420, 0.0458, 0.0534, 0.0382, 0.0076, 0.0382, 0.0534, 0.0382,
                0.3550, 0.0229, 0.0382, 0.0458, 0.0153, 0.0305, 0.0191, 0.0420,
                0.0267, 0.0076, 0.0573, 0.0229, 0, 0
            )
        ),
        (
            'I', (
                0.0471, 0.0177, 0.0147, 0.0177, 0.0162, 0.0133, 0.0177, 0.0206,
                0.0088, 0.2710, 0.1679, 0.0236, 0.0368, 0.0442, 0.0147, 0.0250,
                0.0398, 0.0059, 0.0206, 0.1767, 0, 0
            )
        ),
        (
            'K', (
                0.0570, 0.1071, 0.0415, 0.0415, 0.0086, 0.0535, 0.0708, 0.0432,
                0.0207, 0.0276, 0.0432, 0.2781, 0.0155, 0.0155, 0.0276, 0.0535,
                0.0397, 0.0052, 0.0173, 0.0328, 0, 0
            )
        ),
        (
            'L', (
                0.0445, 0.0243, 0.0142, 0.0152, 0.0162, 0.0162, 0.0202, 0.0213,
                0.0101, 0.1154, 0.3755, 0.0253, 0.0496, 0.0547, 0.0142, 0.0243,
                0.0334, 0.0071, 0.0223, 0.0962, 0, 0
            )
        ),
        (
            'M', (
                0.0522, 0.0321, 0.0201, 0.0201, 0.0161, 0.0281, 0.0281, 0.0281,
                0.0161, 0.1004, 0.1968, 0.0361, 0.1606, 0.0482, 0.0161, 0.0361,
                0.0402, 0.0080, 0.0241, 0.0924, 0, 0
            )
        ),
        (
            'N', (
                0.0427, 0.0449, 0.3169, 0.0831, 0.0090, 0.0337, 0.0494, 0.0652,
                0.0315, 0.0225, 0.0315, 0.0539, 0.0112, 0.0180, 0.0202, 0.0697,
                0.0494, 0.0045, 0.0157, 0.0270, 0, 0
            )
        ),
        (
            'O', (
                0.0649, 0.0515, 0.0528, 0.0616, 0.0332, 0.0416, 0.0616, 0.0608,
                0.0346, 0.0554, 0.0793, 0.0575, 0.0252, 0.0471, 0.0416, 0.0555,
                0.0484, 0.0291, 0.0381, 0.0602, 0, 0
            )
        ),  # Pyrrolysin encoded as unknown
        (
            'P', (
                0.0568, 0.0258, 0.0233, 0.0310, 0.0103, 0.0207, 0.0362, 0.0362,
                0.0129, 0.0258, 0.0362, 0.0413, 0.0103, 0.0129, 0.4935, 0.0439,
                0.0362, 0.0026, 0.0129, 0.0310, 0, 0
            )
        ),
        (
            'Q', (
                0.0559, 0.0735, 0.0441, 0.0471, 0.0088, 0.2147, 0.1029, 0.0412,
                0.0294, 0.0265, 0.0471, 0.0912, 0.0206, 0.0147, 0.0235, 0.0559,
                0.0412, 0.0059, 0.0206, 0.0353, 0, 0
            )
        ),
        (
            'R', (
                0.0446, 0.3450, 0.0388, 0.0310, 0.0078, 0.0484, 0.0523, 0.0329,
                0.0233, 0.0233, 0.0465, 0.1202, 0.0155, 0.0174, 0.0194, 0.0446,
                0.0349, 0.0058, 0.0174, 0.0310, 0, 0
            )
        ),
        (
            'S', (
                0.1099, 0.0401, 0.0541, 0.0489, 0.0175, 0.0332, 0.0524, 0.0663,
                0.0192, 0.0297, 0.0419, 0.0541, 0.0157, 0.0209, 0.0297, 0.2199,
                0.0820, 0.0052, 0.0175, 0.0419, 0, 0
            )
        ),
        (
            'T', (
                0.0730, 0.0355, 0.0434, 0.0375, 0.0178, 0.0276, 0.0394, 0.0434,
                0.0138, 0.0533, 0.0651, 0.0454, 0.0197, 0.0237, 0.0276, 0.0927,
                0.2465, 0.0059, 0.0178, 0.0710, 0, 0
            )
        ),
        (
            'U', (
                0.0649, 0.0515, 0.0528, 0.0616, 0.0332, 0.0416, 0.0616, 0.0608,
                0.0346, 0.0554, 0.0793, 0.0575, 0.0252, 0.0471, 0.0416, 0.0555,
                0.0484, 0.0291, 0.0381, 0.0602, 0, 0
            )
        ),  # Selenocysteine encoded as unknown
        (
            'V', (
                0.0700, 0.0219, 0.0165, 0.0178, 0.0192, 0.0165, 0.0233, 0.0247,
                0.0082, 0.1646, 0.1303, 0.0261, 0.0316, 0.0357, 0.0165, 0.0329,
                0.0494, 0.0055, 0.0206, 0.2689, 0, 0
            )
        ),
        (
            'W', (
                0.0303, 0.0227, 0.0152, 0.0152, 0.0076, 0.0152, 0.0227, 0.0303,
                0.0152, 0.0303, 0.0530, 0.0227, 0.0152, 0.0606, 0.0076, 0.0227,
                0.0227, 0.4924, 0.0682, 0.0303, 0, 0
            )
        ),
        (
            'X', (
                0.0649, 0.0515, 0.0528, 0.0616, 0.0332, 0.0416, 0.0616, 0.0608,
                0.0346, 0.0554, 0.0793, 0.0575, 0.0252, 0.0471, 0.0416, 0.0555,
                0.0484, 0.0291, 0.0381, 0.0602, 0, 0
            )
        ),
        (
            'Y', (
                0.0405, 0.0280, 0.0218, 0.0187, 0.0093, 0.0218, 0.0280, 0.0249,
                0.0467, 0.0436, 0.0685, 0.0312, 0.0187, 0.1308, 0.0156, 0.0312,
                0.0280, 0.0280, 0.3178, 0.0467, 0, 0
            )
        ),
        (
            'Z', (
                0.0556, 0.0616, 0.0423, 0.0687, 0.0081, 0.1396, 0.1997, 0.0381,
                0.0276, 0.0243, 0.0420, 0.0834, 0.0168, 0.0157, 0.0247, 0.0556,
                0.0390, 0.0057, 0.0186, 0.0333, 0, 0
            )
        ),
        (
            '<PAD>', (
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0
            )
        ),
        (
            '<START>', (
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,
                0
            )
        ),
        (
            '<STOP>', (
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                1
            )
        ),
        (
            '<UNK>', (
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                0
            )
        ),  # UNK is identical to PAD
        (
            '-', (
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.5,
                0
            )
        ),
    ]
)
