"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_substrings_1 = require("@udondan/common-substrings");
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * (experimental) Adds "action" functionality to the Policy Statement.
 *
 * @experimental
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.accessLevelList = {};
        this.useNotActions = false;
        /**
         * @experimental
         */
        this.actions = [];
        this.cdkActionsApplied = false;
        this.isCompact = false;
    }
    /**
     * (experimental) Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     *
     * @experimental
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            if (this.isCompact) {
                this.compactActions();
            }
            const actions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            statement[mode] = actions.length > 1 ? actions : actions[0];
        }
        return statement;
    }
    /**
     * @experimental
     */
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            if (this.isCompact) {
                this.compactActions();
            }
            const uniqueActions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * (experimental) Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * @experimental
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * (experimental) Checks weather actions have been applied to the policy.
     *
     * @experimental
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * (experimental) Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * If the action does not contain a colon, the action will be prefixed with the service prefix of the class, e.g. `ec2:`
     *
     * @param action Actions that will be added to the statement.
     * @experimental
     */
    to(action) {
        if (this.servicePrefix.length && action.indexOf(':') < 0) {
            action = this.servicePrefix + ':' + action;
        }
        this.actions.push(action);
        return this;
    }
    /**
     * (experimental) Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`.
     *
     * @experimental
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.to(`${this.servicePrefix}:*`);
        }
        else {
            this.to('*');
        }
        return this;
    }
    /**
     * (experimental) Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions.
     * @experimental
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [_, actions] of Object.entries(this.accessLevelList)) {
                actions.forEach((action) => {
                    if (action.match(RegexParser(expression))) {
                        this.to(`${this.servicePrefix}:${action}`);
                    }
                });
            }
        });
        return this;
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement.
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     *
     * @experimental
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement.
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     *
     * @experimental
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement.
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     *
     * @experimental
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement.
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     *
     * @experimental
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement.
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     *
     * @experimental
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        if (accessLevel in this.accessLevelList) {
            this.accessLevelList[accessLevel].forEach((action) => {
                this.to(`${this.servicePrefix}:${action}`);
            });
        }
        return this;
    }
    /**
     * (experimental) Condense action list down to a list of patterns.
     *
     * Using this method can help to reduce the policy size.
     *
     * For example, all actions with access level `list` could be reduced to a small pattern `List*`.
     *
     * @experimental
     */
    compact() {
        this.isCompact = true;
        return this;
    }
    compactActions() {
        // actions that will be included, service prefix is removed
        const includeActions = this.actions.map((elem) => {
            return elem.substr(elem.indexOf(':') + 1);
        });
        // actions that will not be included, the opposite of includeActions
        const excludeActions = [];
        for (const [_, actions] of Object.entries(this.accessLevelList)) {
            actions.forEach((action) => {
                if (!includeActions.includes(action)) {
                    excludeActions.push(`^${action}$`);
                }
            });
        }
        // will contain the index of elements that are covered by substrings
        let covered = [];
        const subs = common_substrings_1.default(includeActions.map((action) => {
            return `^${action}$`;
        }), {
            minLength: 3,
            minOccurrence: 2,
        })
            .filter((sub) => {
            // remove all substrings, that match an action we have not selected
            for (let action of excludeActions) {
                if (action.includes(sub.name)) {
                    return false;
                }
            }
            return true;
        })
            .sort((a, b) => {
            // sort list by the number of matches
            if (a.source.length < b.source.length)
                return 1;
            if (a.source.length > b.source.length)
                return -1;
            return 0;
        })
            .filter((sub) => {
            // removes substrings, that have already been covered by other substrings
            const sources = sub.source.filter((source) => {
                return !covered.includes(source);
            });
            if (sources.length > 1) {
                //add list of sources to the global list of covered actions
                covered = covered.concat(sources);
                return true;
            }
            return false;
        });
        // stores the list of patterns
        const compactActionList = [];
        subs.forEach((sub) => {
            compactActionList.push(`${this.servicePrefix}:*${sub.name}*`
                .replace('*^', '')
                .replace('$*', ''));
            sub.source.forEach((source) => {
                includeActions[source] = ''; // invalidate, will be filtered later
            });
        });
        includeActions
            .filter((action) => {
            // remove elements that have been covered by patterns, we invalidated them above
            return action.length > 0;
        })
            .forEach((action) => {
            // add actions that have not been covered by patterns to the new action list
            compactActionList.push(`${this.servicePrefix}:${action}`);
        });
        // we're done. override action list
        this.actions = compactActionList;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithActions[_a] = { fqn: "iam-floyd.PolicyStatementWithActions", version: "0.301.0" };
//# sourceMappingURL=data:application/json;base64,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