"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonFileSeeds = exports.InlineSeeds = exports.S3Seeds = exports.Seeds = void 0;
const fs = require("fs");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
class Seeds {
    /**
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3Seeds` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Seeds(bucket, key, objectVersion);
    }
    /**
     * @param seeds The actual json code (limited to 4KiB).
     * @returns `InlineSeeds` with inline seeds.
     */
    static fromInline(seeds) {
        return new InlineSeeds(JSON.stringify(seeds));
    }
    /**
     * Loads the seeds from a local disk path and uploads it to s3.
     *
     * @param path Path to json seeds file.
     * @returns `JsonFileSeeds` associated with the specified S3 object.
     */
    static fromJsonFile(path, options) {
        return new JsonFileSeeds(path, options);
    }
}
exports.Seeds = Seeds;
/**
 * Seeds from an S3 archive.
 */
class S3Seeds extends Seeds {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('BucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * Called when the seeder is initialized to allow this object to bind to the stack.
     */
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Seeds = S3Seeds;
/**
 * Seeds from an inline json object (limited to 4KiB).
 */
class InlineSeeds extends Seeds {
    constructor(seeds) {
        super();
        this.seeds = seeds;
        if (seeds.length === 0) {
            throw new Error('Inline seeds cannot be empty');
        }
        if (seeds.length > 4096) {
            throw new Error(`Inline seeds are too large, must be <= 4096 but is ${seeds.length}`);
        }
    }
    /**
     * Called when the seeder is initialized to allow this object to bind to the stack.
     */
    bind(_scope) {
        return {
            inlineSeeds: this.seeds,
        };
    }
}
exports.InlineSeeds = InlineSeeds;
/**
 * Seeds from a local json file.
 */
class JsonFileSeeds extends Seeds {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    /**
     * Called when the seeder is initialized to allow this object to bind to the stack.
     */
    bind(scope) {
        // If the same JsonFileSeeds is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new aws_s3_assets_1.Asset(scope, 'JsonFileSeeds', {
                path: this.path,
                ...this.options,
            });
        }
        try {
            JSON.parse(fs.readFileSync(this.path, 'utf-8'));
        }
        catch (e) {
            throw new Error('Could not convert file to JSON');
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.JsonFileSeeds = JsonFileSeeds;
//# sourceMappingURL=data:application/json;base64,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