"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SagemakerTrainTask {
    constructor(props) {
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSizeInGB: 10
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1)
        };
        // check that either algorithm name or image is defined
        if ((!props.algorithmSpecification.algorithmName) && (!props.algorithmSpecification.trainingImage)) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = (props.algorithmSpecification.trainingInputMode) ?
            (props.algorithmSpecification) :
            ({ ...props.algorithmSpecification, trainingInputMode: sagemaker_task_base_types_1.InputMode.FILE });
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map(config => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return Object.assign({}, config, { dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX } } });
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = (props.vpcConfig.subnets) ?
                (this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds) : this.vpc.selectSubnets().subnetIds;
        }
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    bind(task) {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role = this.props.role || new iam.Role(task, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateTrainingJob: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'cloudwatch:PutMetricData',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                                'logs:CreateLogGroup',
                                'logs:DescribeLogStreams',
                                'ecr:GetAuthorizationToken',
                                ...this.props.vpcConfig
                                    ? [
                                        'ec2:CreateNetworkInterface',
                                        'ec2:CreateNetworkInterfacePermission',
                                        'ec2:DeleteNetworkInterface',
                                        'ec2:DeleteNetworkInterfacePermission',
                                        'ec2:DescribeNetworkInterfaces',
                                        'ec2:DescribeVpcs',
                                        'ec2:DescribeDhcpOptions',
                                        'ec2:DescribeSubnets',
                                        'ec2:DescribeSecurityGroups',
                                    ]
                                    : [],
                            ],
                            resources: ['*'],
                        })
                    ]
                }),
            }
        });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(task, 'TrainJobSecurityGroup', {
                vpc: this.vpc
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...(this.renderAlgorithmSpecification(this.algorithmSpecification)),
            ...(this.renderInputDataConfig(this.inputDataConfig)),
            ...(this.renderOutputDataConfig(this.props.outputDataConfig)),
            ...(this.renderResourceConfig(this.resourceConfig)),
            ...(this.renderStoppingCondition(this.stoppingCondition)),
            ...(this.renderHyperparameters(this.props.hyperparameters)),
            ...(this.renderTags(this.props.tags)),
            ...(this.renderVpcConfig(this.props.vpcConfig)),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage) ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {},
                ...(spec.algorithmName) ? { AlgorithmName: spec.algorithmName } : {},
                ...(spec.metricDefinitions) ?
                    { MetricDefinitions: spec.metricDefinitions
                            .map(metric => ({ Name: metric.name, Regex: metric.regex })) } : {}
            }
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map(channel => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType) ?
                            { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType } : {},
                        ...(channel.dataSource.s3DataSource.attributeNames) ?
                            { AtttributeNames: channel.dataSource.s3DataSource.attributeNames } : {},
                    }
                },
                ...(channel.compressionType) ? { CompressionType: channel.compressionType } : {},
                ...(channel.contentType) ? { ContentType: channel.contentType } : {},
                ...(channel.inputMode) ? { InputMode: channel.inputMode } : {},
                ...(channel.recordWrapperType) ? { RecordWrapperType: channel.recordWrapperType } : {},
            }))
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey) ? { KmsKeyId: config.encryptionKey.keyArn } : {},
            }
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSizeInGB,
                ...(config.volumeEncryptionKey) ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {},
            }
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds()
            }
        };
    }
    renderHyperparameters(params) {
        return (params) ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return (config) ? { VpcConfig: {
                SecurityGroupIds: core_1.Lazy.listValue({ produce: () => (this.securityGroups.map(sg => (sg.securityGroupId))) }),
                Subnets: this.subnets,
            } } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.Data.isJsonPathString(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`
                    })
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*']
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' }
                }
            })
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule'
                    })]
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTrainTask = SagemakerTrainTask;
//# sourceMappingURL=data:application/json;base64,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