"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A StepFunctions Task to send messages to SQS queue.
 *
 * A Function can be used directly as a Resource, but this class mirrors
 * integration with other AWS services via a specific class instance.
 */
class SendToQueue {
    constructor(queue, props) {
        this.queue = queue;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SQS.`);
        }
        if (props.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.messageBody)) {
                throw new Error('Task Token is missing in messageBody (pass Context.taskToken somewhere in messageBody)');
            }
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sqs', 'sendMessage', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    actions: ['sqs:SendMessage'],
                    resources: [this.queue.queueArn]
                })],
            parameters: {
                QueueUrl: this.queue.queueUrl,
                MessageBody: this.props.messageBody.value,
                DelaySeconds: this.props.delay && this.props.delay.toSeconds(),
                MessageDeduplicationId: this.props.messageDeduplicationId,
                MessageGroupId: this.props.messageGroupId,
            }
        };
    }
}
exports.SendToQueue = SendToQueue;
//# sourceMappingURL=data:application/json;base64,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