"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let clusterRole;
let serviceRole;
let autoScalingRole;
beforeEach(() => {
    var _a;
    // GIVEN
    stack = new cdk.Stack();
    clusterRole = new iam.Role(stack, 'ClusterRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
    });
    serviceRole = new iam.Role(stack, 'ServiceRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com')
    });
    autoScalingRole = new iam.Role(stack, 'AutoScalingRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com')
    });
    (_a = autoScalingRole.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        principals: [
            new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        ],
        actions: [
            'sts:AssumeRole'
        ]
    }));
});
test('Create Cluster with FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with SYNC integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with clusterConfiguration Name from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: sfn.TaskInput.fromDataAt('$.ClusterName').value,
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Name.$': '$.ClusterName',
            'Instances': {
                KeepJobFlowAliveWhenNoSteps: true
            },
            'VisibleToAllUsers': true,
            'JobFlowRole': {
                Ref: 'ClusterRoleD9CA7471',
            },
            'ServiceRole': {
                Ref: 'ServiceRole4288B192'
            },
            'AutoScalingRole': {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with Tags', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            tags: [{
                    key: 'Key',
                    value: 'Value'
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Tags: [{
                    Key: 'Key',
                    Value: 'Value'
                }]
        },
    });
});
test('Create Cluster with Applications', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            applications: [
                { name: 'Hive', version: '0.0' },
                { name: 'Spark', version: '0.0' }
            ],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Applications: [
                { Name: 'Hive', Version: '0.0' },
                { Name: 'Spark', Version: '0.0' }
            ]
        },
    });
});
test('Create Cluster with Bootstrap Actions', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            bootstrapActions: [{
                    name: 'Bootstrap',
                    scriptBootstrapAction: {
                        path: 's3://null',
                        args: ['Arg']
                    }
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            BootstrapActions: [{
                    Name: 'Bootstrap',
                    ScriptBootstrapAction: {
                        Path: 's3://null',
                        Args: ['Arg']
                    }
                }]
        },
    });
});
test('Create Cluster with Configurations', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            configurations: [{
                    classification: 'classification',
                    properties: {
                        Key: 'Value'
                    }
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Configurations: [{
                    Classification: 'classification',
                    Properties: {
                        Key: 'Value'
                    }
                }]
        },
    });
});
test('Create Cluster with KerberosAttributes', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            kerberosAttributes: {
                realm: 'realm',
                adDomainJoinPassword: 'password1',
                adDomainJoinUser: 'user',
                crossRealmTrustPrincipalPassword: 'password2',
                kdcAdminPassword: 'password3'
            },
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            KerberosAttributes: {
                Realm: 'realm',
                ADDomainJoinPassword: 'password1',
                ADDomainJoinUser: 'user',
                CrossRealmTrustPrincipalPassword: 'password2',
                KdcAdminPassword: 'password3'
            }
        },
    });
});
test('Create Cluster without Roles', () => {
    // WHEN
    const createClusterTask = new tasks.EmrCreateCluster({
        instances: {},
        name: 'Cluster',
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC
    });
    const task = new sfn.Task(stack, 'Task', { task: createClusterTask });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'TaskInstanceRoleB72072BF'
            },
            ServiceRole: {
                Ref: 'TaskServiceRoleBF55F61E'
            },
            AutoScalingRole: {
                Ref: 'TaskAutoScalingRoleD06F8423'
            }
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
    // The stack renders the ec2.amazonaws.com Service principal id with a
    // Join to the URLSuffix
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: {
                            'Fn::Join': [
                                '',
                                [
                                    'ec2.',
                                    {
                                        Ref: 'AWS::URLSuffix'
                                    }
                                ]
                            ]
                        }
                    },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                },
                {
                    Principal: { Service: 'application-autoscaling.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
});
test('Create Cluster with Instances configuration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                additionalMasterSecurityGroups: ['MasterGroup'],
                additionalSlaveSecurityGroups: ['SlaveGroup'],
                ec2KeyName: 'Ec2KeyName',
                ec2SubnetId: 'Ec2SubnetId',
                ec2SubnetIds: ['Ec2SubnetId'],
                emrManagedMasterSecurityGroup: 'MasterGroup',
                emrManagedSlaveSecurityGroup: 'SlaveGroup',
                hadoopVersion: 'HadoopVersion',
                instanceCount: 1,
                masterInstanceType: 'MasterInstanceType',
                placement: {
                    availabilityZone: 'AvailabilityZone',
                    availabilityZones: ['AvailabilityZone']
                },
                serviceAccessSecurityGroup: 'ServiceAccessGroup',
                slaveInstanceType: 'SlaveInstanceType',
                terminationProtected: true
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                AdditionalMasterSecurityGroups: ['MasterGroup'],
                AdditionalSlaveSecurityGroups: ['SlaveGroup'],
                Ec2KeyName: 'Ec2KeyName',
                Ec2SubnetId: 'Ec2SubnetId',
                Ec2SubnetIds: ['Ec2SubnetId'],
                EmrManagedMasterSecurityGroup: 'MasterGroup',
                EmrManagedSlaveSecurityGroup: 'SlaveGroup',
                HadoopVersion: 'HadoopVersion',
                InstanceCount: 1,
                KeepJobFlowAliveWhenNoSteps: true,
                MasterInstanceType: 'MasterInstanceType',
                Placement: {
                    AvailabilityZone: 'AvailabilityZone',
                    AvailabilityZones: ['AvailabilityZone']
                },
                ServiceAccessSecurityGroup: 'ServiceAccessGroup',
                SlaveInstanceType: 'SlaveInstanceType',
                TerminationProtected: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with InstanceFleet', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                instanceFleets: [{
                        instanceFleetType: tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                        instanceTypeConfigs: [{
                                bidPrice: '1',
                                bidPriceAsPercentageOfOnDemandPrice: 1,
                                configurations: [{
                                        classification: 'Classification',
                                        properties: {
                                            Key: 'Value'
                                        }
                                    }],
                                ebsConfiguration: {
                                    ebsBlockDeviceConfigs: [{
                                            volumeSpecification: {
                                                iops: 1,
                                                sizeInGB: 1,
                                                volumeType: tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                            },
                                            volumesPerInstance: 1
                                        }],
                                    ebsOptimized: true
                                },
                                instanceType: 'm5.xlarge',
                                weightedCapacity: 1
                            }],
                        launchSpecifications: {
                            spotSpecification: {
                                blockDurationMinutes: 1,
                                timeoutAction: tasks.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                                timeoutDurationMinutes: 1
                            }
                        },
                        name: 'Master',
                        targetOnDemandCapacity: 1,
                        targetSpotCapacity: 1
                    }]
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceFleets: [{
                        InstanceFleetType: 'MASTER',
                        InstanceTypeConfigs: [{
                                BidPrice: '1',
                                BidPriceAsPercentageOfOnDemandPrice: 1,
                                Configurations: [{
                                        Classification: 'Classification',
                                        Properties: {
                                            Key: 'Value'
                                        }
                                    }],
                                EbsConfiguration: {
                                    EbsBlockDeviceConfigs: [{
                                            VolumeSpecification: {
                                                Iops: 1,
                                                SizeInGB: 1,
                                                VolumeType: 'standard'
                                            },
                                            VolumesPerInstance: 1
                                        }],
                                    EbsOptimized: true
                                },
                                InstanceType: 'm5.xlarge',
                                WeightedCapacity: 1
                            }],
                        LaunchSpecifications: {
                            SpotSpecification: {
                                BlockDurationMinutes: 1,
                                TimeoutAction: 'TERMINATE_CLUSTER',
                                TimeoutDurationMinutes: 1
                            }
                        },
                        Name: 'Master',
                        TargetOnDemandCapacity: 1,
                        TargetSpotCapacity: 1
                    }]
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            }
        },
    });
});
test('Create Cluster with InstanceGroup', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                instanceGroups: [{
                        autoScalingPolicy: {
                            constraints: {
                                maxCapacity: 2,
                                minCapacity: 1,
                            },
                            rules: [{
                                    action: {
                                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                        simpleScalingPolicyConfiguration: {
                                            adjustmentType: tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                            coolDown: 1,
                                            scalingAdjustment: 1
                                        }
                                    },
                                    description: 'Description',
                                    name: 'Name',
                                    trigger: {
                                        cloudWatchAlarmDefinition: {
                                            comparisonOperator: tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                            dimensions: [{
                                                    key: 'Key',
                                                    value: 'Value'
                                                }],
                                            evalutionPeriods: 1,
                                            metricName: 'Name',
                                            namespace: 'Namespace',
                                            period: cdk.Duration.seconds(300),
                                            statistic: tasks.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                            threshold: 1,
                                            unit: tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                        }
                                    }
                                }, {
                                    action: {
                                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                        simpleScalingPolicyConfiguration: {
                                            adjustmentType: tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                            coolDown: 1,
                                            scalingAdjustment: 1
                                        }
                                    },
                                    description: 'Description',
                                    name: 'Name',
                                    trigger: {
                                        cloudWatchAlarmDefinition: {
                                            comparisonOperator: tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                            dimensions: [{
                                                    key: 'Key',
                                                    value: 'Value'
                                                }],
                                            evalutionPeriods: 1,
                                            metricName: 'Name',
                                            namespace: 'Namespace',
                                            period: cdk.Duration.seconds(sfn.Data.numberAt('$.CloudWatchPeriod')),
                                            statistic: tasks.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                            threshold: 1,
                                            unit: tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                        }
                                    }
                                }]
                        },
                        bidPrice: '1',
                        configurations: [{
                                classification: 'Classification',
                                properties: {
                                    Key: 'Value'
                                }
                            }],
                        ebsConfiguration: {
                            ebsBlockDeviceConfigs: [{
                                    volumeSpecification: {
                                        iops: 1,
                                        sizeInGB: 1,
                                        volumeType: tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                    },
                                    volumesPerInstance: 1
                                }],
                            ebsOptimized: true
                        },
                        instanceCount: 1,
                        instanceRole: tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                        instanceType: 'm5.xlarge',
                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                        name: 'Name'
                    }]
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceGroups: [{
                        AutoScalingPolicy: {
                            Constraints: {
                                MaxCapacity: 2,
                                MinCapacity: 1
                            },
                            Rules: [{
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1
                                        }
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            ComparisonOperator: 'GREATER_THAN',
                                            Dimensions: [{
                                                    Key: 'Key',
                                                    Value: 'Value'
                                                }],
                                            EvaluationPeriods: 1,
                                            MetricName: 'Name',
                                            Namespace: 'Namespace',
                                            Period: 300,
                                            Statistic: 'AVERAGE',
                                            Threshold: 1,
                                            Unit: 'NONE'
                                        }
                                    }
                                }, {
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1
                                        }
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            'ComparisonOperator': 'GREATER_THAN',
                                            'Dimensions': [{
                                                    Key: 'Key',
                                                    Value: 'Value'
                                                }],
                                            'EvaluationPeriods': 1,
                                            'MetricName': 'Name',
                                            'Namespace': 'Namespace',
                                            'Period.$': '$.CloudWatchPeriod',
                                            'Statistic': 'AVERAGE',
                                            'Threshold': 1,
                                            'Unit': 'NONE'
                                        }
                                    }
                                }]
                        },
                        BidPrice: '1',
                        Configurations: [{
                                Classification: 'Classification',
                                Properties: {
                                    Key: 'Value'
                                }
                            }],
                        EbsConfiguration: {
                            EbsBlockDeviceConfigs: [{
                                    VolumeSpecification: {
                                        Iops: 1,
                                        SizeInGB: 1,
                                        VolumeType: 'standard'
                                    },
                                    VolumesPerInstance: 1
                                }],
                            EbsOptimized: true
                        },
                        InstanceCount: 1,
                        InstanceRole: 'MASTER',
                        InstanceType: 'm5.xlarge',
                        Market: 'ON_DEMAND',
                        Name: 'Name'
                    }]
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.EmrCreateCluster({
                instances: {},
                clusterRole,
                name: 'Cluster',
                serviceRole,
                autoScalingRole,
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call CreateCluster./i);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1yLWNyZWF0ZS1jbHVzdGVyLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJlbXItY3JlYXRlLWNsdXN0ZXIudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5Qix3Q0FBd0M7QUFDeEMsa0RBQWtEO0FBQ2xELHFDQUFxQztBQUNyQyxnQ0FBZ0M7QUFFaEMsSUFBSSxLQUFnQixDQUFDO0FBQ3JCLElBQUksV0FBcUIsQ0FBQztBQUMxQixJQUFJLFdBQXFCLENBQUM7QUFDMUIsSUFBSSxlQUF5QixDQUFDO0FBRTlCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7O0lBQ2QsUUFBUTtJQUNSLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUN4QixXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7UUFDL0MsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO0tBQ3pELENBQUMsQ0FBQztJQUNILFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtRQUMvQyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsZ0NBQWdDLENBQUM7S0FDdEUsQ0FBQyxDQUFDO0lBQ0gsZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7UUFDdkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGdDQUFnQyxDQUFDO0tBQ3RFLENBQUMsQ0FBQztJQUNILE1BQUEsZUFBZSxDQUFDLGdCQUFnQiwwQ0FBRSxhQUFhLENBQzdDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztRQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1FBQ3hCLFVBQVUsRUFBRTtZQUNWLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHVDQUF1QyxDQUFDO1NBQ2xFO1FBQ0QsT0FBTyxFQUFFO1lBQ1AsZ0JBQWdCO1NBQ2pCO0tBQ0YsQ0FBQyxFQUNGO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUUsRUFBRTtZQUNiLFdBQVc7WUFDWCxJQUFJLEVBQUUsU0FBUztZQUNmLFdBQVc7WUFDWCxlQUFlO1lBQ2Ysa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsMENBQTBDO2lCQUMzQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7YUFDbEM7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO0lBQ3ZELE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUUsRUFBRTtZQUNiLFdBQVc7WUFDWCxJQUFJLEVBQUUsU0FBUztZQUNmLFdBQVc7WUFDWCxlQUFlO1lBQ2Ysa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUk7U0FDdkQsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsK0NBQStDO2lCQUNoRDthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7YUFDbEM7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO0lBQ3RFLE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUUsRUFBRTtZQUNiLFdBQVc7WUFDWCxJQUFJLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsZUFBZSxDQUFDLENBQUMsS0FBSztZQUNyRCxXQUFXO1lBQ1gsZUFBZTtZQUNmLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlO1NBQ2xFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixRQUFRLEVBQUUsZUFBZTtZQUN6QixXQUFXLEVBQUU7Z0JBQ1gsMkJBQTJCLEVBQUUsSUFBSTthQUNsQztZQUNELG1CQUFtQixFQUFFLElBQUk7WUFDekIsYUFBYSxFQUFFO2dCQUNiLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsR0FBRyxFQUFFLHFCQUFxQjthQUMzQjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7SUFDcEMsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQzFFLFNBQVMsRUFBRSxFQUFFO1lBQ2IsV0FBVztZQUNYLElBQUksRUFBRSxTQUFTO1lBQ2YsV0FBVztZQUNYLGVBQWU7WUFDZixJQUFJLEVBQUUsQ0FBQztvQkFDTCxHQUFHLEVBQUUsS0FBSztvQkFDVixLQUFLLEVBQUUsT0FBTztpQkFDZixDQUFDO1lBQ0Ysa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsMENBQTBDO2lCQUMzQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7YUFDbEM7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtZQUNELElBQUksRUFBRSxDQUFDO29CQUNMLEdBQUcsRUFBRSxLQUFLO29CQUNWLEtBQUssRUFBRSxPQUFPO2lCQUNmLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtJQUM1QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUM7WUFDMUUsU0FBUyxFQUFFLEVBQUU7WUFDYixXQUFXO1lBQ1gsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXO1lBQ1gsZUFBZTtZQUNmLFlBQVksRUFBRTtnQkFDWixFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRTtnQkFDaEMsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUU7YUFDbEM7WUFDRCxrQkFBa0IsRUFBRSxHQUFHLENBQUMseUJBQXlCLENBQUMsZUFBZTtTQUNsRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRU4sT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCwwQ0FBMEM7aUJBQzNDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsSUFBSSxFQUFFLFNBQVM7WUFDZixTQUFTLEVBQUU7Z0JBQ1QsMkJBQTJCLEVBQUUsSUFBSTthQUNsQztZQUNELGlCQUFpQixFQUFFLElBQUk7WUFDdkIsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxFQUFFLHFCQUFxQjthQUMzQjtZQUNELGVBQWUsRUFBRTtnQkFDZixHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFO2dCQUNoQyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRTthQUNsQztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdUNBQXVDLEVBQUUsR0FBRyxFQUFFO0lBQ2pELE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUUsRUFBRTtZQUNiLFdBQVc7WUFDWCxJQUFJLEVBQUUsU0FBUztZQUNmLFdBQVc7WUFDWCxlQUFlO1lBQ2YsZ0JBQWdCLEVBQUUsQ0FBQztvQkFDakIsSUFBSSxFQUFFLFdBQVc7b0JBQ2pCLHFCQUFxQixFQUFFO3dCQUNyQixJQUFJLEVBQUUsV0FBVzt3QkFDakIsSUFBSSxFQUFFLENBQUUsS0FBSyxDQUFFO3FCQUNoQjtpQkFDRixDQUFDO1lBQ0Ysa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsMENBQTBDO2lCQUMzQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7YUFDbEM7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtZQUNELGdCQUFnQixFQUFFLENBQUM7b0JBQ2pCLElBQUksRUFBRSxXQUFXO29CQUNqQixxQkFBcUIsRUFBRTt3QkFDckIsSUFBSSxFQUFFLFdBQVc7d0JBQ2pCLElBQUksRUFBRSxDQUFFLEtBQUssQ0FBRTtxQkFDaEI7aUJBQ0YsQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO0lBQzlDLE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUUsRUFBRTtZQUNiLFdBQVc7WUFDWCxJQUFJLEVBQUUsU0FBUztZQUNmLFdBQVc7WUFDWCxlQUFlO1lBQ2YsY0FBYyxFQUFFLENBQUM7b0JBQ2YsY0FBYyxFQUFFLGdCQUFnQjtvQkFDaEMsVUFBVSxFQUFFO3dCQUNWLEdBQUcsRUFBRSxPQUFPO3FCQUNiO2lCQUNGLENBQUM7WUFDRixrQkFBa0IsRUFBRSxHQUFHLENBQUMseUJBQXlCLENBQUMsZUFBZTtTQUNsRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRU4sT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCwwQ0FBMEM7aUJBQzNDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsSUFBSSxFQUFFLFNBQVM7WUFDZixTQUFTLEVBQUU7Z0JBQ1QsMkJBQTJCLEVBQUUsSUFBSTthQUNsQztZQUNELGlCQUFpQixFQUFFLElBQUk7WUFDdkIsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxFQUFFLHFCQUFxQjthQUMzQjtZQUNELGVBQWUsRUFBRTtnQkFDZixHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1lBQ0QsY0FBYyxFQUFFLENBQUM7b0JBQ2YsY0FBYyxFQUFFLGdCQUFnQjtvQkFDaEMsVUFBVSxFQUFFO3dCQUNWLEdBQUcsRUFBRSxPQUFPO3FCQUNiO2lCQUNGLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtJQUNsRCxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUM7WUFDMUUsU0FBUyxFQUFFLEVBQUU7WUFDYixXQUFXO1lBQ1gsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXO1lBQ1gsZUFBZTtZQUNmLGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsT0FBTztnQkFDZCxvQkFBb0IsRUFBRSxXQUFXO2dCQUNqQyxnQkFBZ0IsRUFBRSxNQUFNO2dCQUN4QixnQ0FBZ0MsRUFBRSxXQUFXO2dCQUM3QyxnQkFBZ0IsRUFBRSxXQUFXO2FBQzlCO1lBQ0Qsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsMENBQTBDO2lCQUMzQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7YUFDbEM7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsT0FBTztnQkFDZCxvQkFBb0IsRUFBRSxXQUFXO2dCQUNqQyxnQkFBZ0IsRUFBRSxNQUFNO2dCQUN4QixnQ0FBZ0MsRUFBRSxXQUFXO2dCQUM3QyxnQkFBZ0IsRUFBRSxXQUFXO2FBQzlCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsT0FBTztJQUNQLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUM7UUFDbkQsU0FBUyxFQUFFLEVBQUU7UUFDYixJQUFJLEVBQUUsU0FBUztRQUNmLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJO0tBQ3ZELENBQUMsQ0FBQztJQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQztJQUVyRSxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELCtDQUErQztpQkFDaEQ7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixJQUFJLEVBQUUsU0FBUztZQUNmLFNBQVMsRUFBRTtnQkFDVCwyQkFBMkIsRUFBRSxJQUFJO2FBQ2xDO1lBQ0QsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxFQUFFLDBCQUEwQjthQUNoQztZQUNELFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEdBQUcsRUFBRSw2QkFBNkI7YUFDbkM7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsRUFBRTtRQUNqRCx3QkFBd0IsRUFBRTtZQUN4QixPQUFPLEVBQUUsWUFBWTtZQUNyQixTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsU0FBUyxFQUFFLEVBQUUsT0FBTyxFQUFFLGdDQUFnQyxFQUFFO29CQUN4RCxNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixNQUFNLEVBQUUsT0FBTztpQkFDaEI7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsc0VBQXNFO0lBQ3RFLHdCQUF3QjtJQUN4QixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLEVBQUU7UUFDakQsd0JBQXdCLEVBQUU7WUFDeEIsT0FBTyxFQUFFLFlBQVk7WUFDckIsU0FBUyxFQUFFO2dCQUNUO29CQUNFLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFDbEI7NEJBQ0UsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0UsTUFBTTtvQ0FDTjt3Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3FDQUN0QjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRCxNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixNQUFNLEVBQUUsT0FBTztpQkFDaEI7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixFQUFFO1FBQ2pELHdCQUF3QixFQUFFO1lBQ3hCLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsZ0NBQWdDLEVBQUU7b0JBQ3hELE1BQU0sRUFBRSxnQkFBZ0I7b0JBQ3hCLE1BQU0sRUFBRSxPQUFPO2lCQUNoQjtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsdUNBQXVDLEVBQUU7b0JBQy9ELE1BQU0sRUFBRSxnQkFBZ0I7b0JBQ3hCLE1BQU0sRUFBRSxPQUFPO2lCQUNoQjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2Q0FBNkMsRUFBRSxHQUFHLEVBQUU7SUFDdkQsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQzFFLFNBQVMsRUFBRTtnQkFDVCw4QkFBOEIsRUFBRSxDQUFDLGFBQWEsQ0FBQztnQkFDL0MsNkJBQTZCLEVBQUUsQ0FBQyxZQUFZLENBQUM7Z0JBQzdDLFVBQVUsRUFBRSxZQUFZO2dCQUN4QixXQUFXLEVBQUUsYUFBYTtnQkFDMUIsWUFBWSxFQUFFLENBQUMsYUFBYSxDQUFDO2dCQUM3Qiw2QkFBNkIsRUFBRSxhQUFhO2dCQUM1Qyw0QkFBNEIsRUFBRSxZQUFZO2dCQUMxQyxhQUFhLEVBQUUsZUFBZTtnQkFDOUIsYUFBYSxFQUFFLENBQUM7Z0JBQ2hCLGtCQUFrQixFQUFFLG9CQUFvQjtnQkFDeEMsU0FBUyxFQUFFO29CQUNULGdCQUFnQixFQUFFLGtCQUFrQjtvQkFDcEMsaUJBQWlCLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQztpQkFDeEM7Z0JBQ0QsMEJBQTBCLEVBQUUsb0JBQW9CO2dCQUNoRCxpQkFBaUIsRUFBRSxtQkFBbUI7Z0JBQ3RDLG9CQUFvQixFQUFFLElBQUk7YUFDM0I7WUFDRCxXQUFXO1lBQ1gsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXO1lBQ1gsZUFBZTtZQUNmLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlO1NBQ2xFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixJQUFJLEVBQUUsU0FBUztZQUNmLFNBQVMsRUFBRTtnQkFDVCw4QkFBOEIsRUFBRSxDQUFDLGFBQWEsQ0FBQztnQkFDL0MsNkJBQTZCLEVBQUUsQ0FBQyxZQUFZLENBQUM7Z0JBQzdDLFVBQVUsRUFBRSxZQUFZO2dCQUN4QixXQUFXLEVBQUUsYUFBYTtnQkFDMUIsWUFBWSxFQUFFLENBQUMsYUFBYSxDQUFDO2dCQUM3Qiw2QkFBNkIsRUFBRSxhQUFhO2dCQUM1Qyw0QkFBNEIsRUFBRSxZQUFZO2dCQUMxQyxhQUFhLEVBQUUsZUFBZTtnQkFDOUIsYUFBYSxFQUFFLENBQUM7Z0JBQ2hCLDJCQUEyQixFQUFFLElBQUk7Z0JBQ2pDLGtCQUFrQixFQUFFLG9CQUFvQjtnQkFDeEMsU0FBUyxFQUFFO29CQUNULGdCQUFnQixFQUFFLGtCQUFrQjtvQkFDcEMsaUJBQWlCLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQztpQkFDeEM7Z0JBQ0QsMEJBQTBCLEVBQUUsb0JBQW9CO2dCQUNoRCxpQkFBaUIsRUFBRSxtQkFBbUI7Z0JBQ3RDLG9CQUFvQixFQUFFLElBQUk7YUFDM0I7WUFDRCxpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO0lBQzdDLE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMxRSxTQUFTLEVBQUU7Z0JBQ1QsY0FBYyxFQUFFLENBQUM7d0JBQ2YsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLE1BQU07d0JBQ2pFLG1CQUFtQixFQUFFLENBQUM7Z0NBQ3BCLFFBQVEsRUFBRSxHQUFHO2dDQUNiLG1DQUFtQyxFQUFFLENBQUM7Z0NBQ3RDLGNBQWMsRUFBRSxDQUFDO3dDQUNmLGNBQWMsRUFBRSxnQkFBZ0I7d0NBQ2hDLFVBQVUsRUFBRTs0Q0FDVixHQUFHLEVBQUUsT0FBTzt5Q0FDYjtxQ0FDRixDQUFDO2dDQUNGLGdCQUFnQixFQUFFO29DQUNoQixxQkFBcUIsRUFBRSxDQUFDOzRDQUN0QixtQkFBbUIsRUFBRTtnREFDbkIsSUFBSSxFQUFFLENBQUM7Z0RBQ1AsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyx3QkFBd0IsQ0FBQyxRQUFROzZDQUNyRTs0Q0FDRCxrQkFBa0IsRUFBRSxDQUFDO3lDQUN0QixDQUFDO29DQUNGLFlBQVksRUFBRSxJQUFJO2lDQUNuQjtnQ0FDRCxZQUFZLEVBQUUsV0FBVztnQ0FDekIsZ0JBQWdCLEVBQUUsQ0FBQzs2QkFDcEIsQ0FBQzt3QkFDRixvQkFBb0IsRUFBRTs0QkFDcEIsaUJBQWlCLEVBQUU7Z0NBQ2pCLG9CQUFvQixFQUFFLENBQUM7Z0NBQ3ZCLGFBQWEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsaUJBQWlCLENBQUMsaUJBQWlCO2dDQUN6RSxzQkFBc0IsRUFBRSxDQUFDOzZCQUMxQjt5QkFDRjt3QkFDRCxJQUFJLEVBQUUsUUFBUTt3QkFDZCxzQkFBc0IsRUFBRSxDQUFDO3dCQUN6QixrQkFBa0IsRUFBRSxDQUFDO3FCQUN0QixDQUFDO2FBQ0g7WUFDRCxXQUFXO1lBQ1gsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXO1lBQ1gsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLGVBQWU7U0FDbEUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUVOLE9BQU87SUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QsMENBQTBDO2lCQUMzQzthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLElBQUksRUFBRSxTQUFTO1lBQ2YsU0FBUyxFQUFFO2dCQUNULDJCQUEyQixFQUFFLElBQUk7Z0JBQ2pDLGNBQWMsRUFBRSxDQUFDO3dCQUNmLGlCQUFpQixFQUFFLFFBQVE7d0JBQzNCLG1CQUFtQixFQUFFLENBQUM7Z0NBQ3BCLFFBQVEsRUFBRSxHQUFHO2dDQUNiLG1DQUFtQyxFQUFFLENBQUM7Z0NBQ3RDLGNBQWMsRUFBRSxDQUFDO3dDQUNmLGNBQWMsRUFBRSxnQkFBZ0I7d0NBQ2hDLFVBQVUsRUFBRTs0Q0FDVixHQUFHLEVBQUUsT0FBTzt5Q0FDYjtxQ0FDRixDQUFDO2dDQUNGLGdCQUFnQixFQUFFO29DQUNoQixxQkFBcUIsRUFBRSxDQUFDOzRDQUN0QixtQkFBbUIsRUFBRTtnREFDbkIsSUFBSSxFQUFFLENBQUM7Z0RBQ1AsUUFBUSxFQUFFLENBQUM7Z0RBQ1gsVUFBVSxFQUFFLFVBQVU7NkNBQ3ZCOzRDQUNELGtCQUFrQixFQUFFLENBQUM7eUNBQ3RCLENBQUM7b0NBQ0YsWUFBWSxFQUFFLElBQUk7aUNBQ25CO2dDQUNELFlBQVksRUFBRSxXQUFXO2dDQUN6QixnQkFBZ0IsRUFBRSxDQUFDOzZCQUNwQixDQUFDO3dCQUNGLG9CQUFvQixFQUFFOzRCQUNwQixpQkFBaUIsRUFBRTtnQ0FDakIsb0JBQW9CLEVBQUUsQ0FBQztnQ0FDdkIsYUFBYSxFQUFFLG1CQUFtQjtnQ0FDbEMsc0JBQXNCLEVBQUUsQ0FBQzs2QkFDMUI7eUJBQ0Y7d0JBQ0QsSUFBSSxFQUFFLFFBQVE7d0JBQ2Qsc0JBQXNCLEVBQUUsQ0FBQzt3QkFDekIsa0JBQWtCLEVBQUUsQ0FBQztxQkFDdEIsQ0FBQzthQUNIO1lBQ0QsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxFQUFFLHFCQUFxQjthQUMzQjtZQUNELFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUscUJBQXFCO2FBQzNCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7SUFDN0MsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQzFFLFNBQVMsRUFBRTtnQkFDVCxjQUFjLEVBQUUsQ0FBQzt3QkFDZixpQkFBaUIsRUFBRTs0QkFDakIsV0FBVyxFQUFFO2dDQUNYLFdBQVcsRUFBRSxDQUFDO2dDQUNkLFdBQVcsRUFBRSxDQUFDOzZCQUNmOzRCQUNELEtBQUssRUFBRSxDQUFDO29DQUNOLE1BQU0sRUFBRTt3Q0FDTixNQUFNLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxTQUFTO3dDQUN2RCxnQ0FBZ0MsRUFBRTs0Q0FDaEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0I7NENBQy9FLFFBQVEsRUFBRSxDQUFDOzRDQUNYLGlCQUFpQixFQUFFLENBQUM7eUNBQ3JCO3FDQUNGO29DQUNELFdBQVcsRUFBRSxhQUFhO29DQUMxQixJQUFJLEVBQUUsTUFBTTtvQ0FDWixPQUFPLEVBQUU7d0NBQ1AseUJBQXlCLEVBQUU7NENBQ3pCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxpQ0FBaUMsQ0FBQyxZQUFZOzRDQUN6RixVQUFVLEVBQUUsQ0FBQztvREFDWCxHQUFHLEVBQUUsS0FBSztvREFDVixLQUFLLEVBQUUsT0FBTztpREFDZixDQUFDOzRDQUNGLGdCQUFnQixFQUFFLENBQUM7NENBQ25CLFVBQVUsRUFBRSxNQUFNOzRDQUNsQixTQUFTLEVBQUUsV0FBVzs0Q0FDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQzs0Q0FDakMsU0FBUyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPOzRDQUNsRSxTQUFTLEVBQUUsQ0FBQzs0Q0FDWixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLElBQUk7eUNBQ3REO3FDQUNGO2lDQUNGLEVBQUU7b0NBQ0QsTUFBTSxFQUFFO3dDQUNOLE1BQU0sRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLFNBQVM7d0NBQ3ZELGdDQUFnQyxFQUFFOzRDQUNoQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLHFCQUFxQixDQUFDLGtCQUFrQjs0Q0FDL0UsUUFBUSxFQUFFLENBQUM7NENBQ1gsaUJBQWlCLEVBQUUsQ0FBQzt5Q0FDckI7cUNBQ0Y7b0NBQ0QsV0FBVyxFQUFFLGFBQWE7b0NBQzFCLElBQUksRUFBRSxNQUFNO29DQUNaLE9BQU8sRUFBRTt3Q0FDUCx5QkFBeUIsRUFBRTs0Q0FDekIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGlDQUFpQyxDQUFDLFlBQVk7NENBQ3pGLFVBQVUsRUFBRSxDQUFDO29EQUNYLEdBQUcsRUFBRSxLQUFLO29EQUNWLEtBQUssRUFBRSxPQUFPO2lEQUNmLENBQUM7NENBQ0YsZ0JBQWdCLEVBQUUsQ0FBQzs0Q0FDbkIsVUFBVSxFQUFFLE1BQU07NENBQ2xCLFNBQVMsRUFBRSxXQUFXOzRDQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsQ0FBQzs0Q0FDckUsU0FBUyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPOzRDQUNsRSxTQUFTLEVBQUUsQ0FBQzs0Q0FDWixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLElBQUk7eUNBQ3REO3FDQUNGO2lDQUNGLENBQUM7eUJBQ0g7d0JBQ0QsUUFBUSxFQUFFLEdBQUc7d0JBQ2IsY0FBYyxFQUFFLENBQUM7Z0NBQ2YsY0FBYyxFQUFFLGdCQUFnQjtnQ0FDaEMsVUFBVSxFQUFFO29DQUNWLEdBQUcsRUFBRSxPQUFPO2lDQUNiOzZCQUNGLENBQUM7d0JBQ0YsZ0JBQWdCLEVBQUU7NEJBQ2hCLHFCQUFxQixFQUFFLENBQUM7b0NBQ3RCLG1CQUFtQixFQUFFO3dDQUNuQixJQUFJLEVBQUUsQ0FBQzt3Q0FDUCxRQUFRLEVBQUUsQ0FBQzt3Q0FDWCxVQUFVLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLHdCQUF3QixDQUFDLFFBQVE7cUNBQ3JFO29DQUNELGtCQUFrQixFQUFFLENBQUM7aUNBQ3RCLENBQUM7NEJBQ0YsWUFBWSxFQUFFLElBQUk7eUJBQ25CO3dCQUNELGFBQWEsRUFBRSxDQUFDO3dCQUNoQixZQUFZLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLE1BQU07d0JBQzVELFlBQVksRUFBRSxXQUFXO3dCQUN6QixNQUFNLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxTQUFTO3dCQUN2RCxJQUFJLEVBQUUsTUFBTTtxQkFDYixDQUFDO2FBQ0g7WUFDRCxXQUFXO1lBQ1gsSUFBSSxFQUFFLFNBQVM7WUFDZixXQUFXO1lBQ1gsZUFBZTtZQUNmLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlO1NBQ2xFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixJQUFJLEVBQUUsU0FBUztZQUNmLFNBQVMsRUFBRTtnQkFDVCwyQkFBMkIsRUFBRSxJQUFJO2dCQUNqQyxjQUFjLEVBQUUsQ0FBQzt3QkFDZixpQkFBaUIsRUFBRTs0QkFDakIsV0FBVyxFQUFFO2dDQUNYLFdBQVcsRUFBRSxDQUFDO2dDQUNkLFdBQVcsRUFBRSxDQUFDOzZCQUNmOzRCQUNELEtBQUssRUFBRSxDQUFDO29DQUNOLE1BQU0sRUFBRTt3Q0FDTixNQUFNLEVBQUUsV0FBVzt3Q0FDbkIsZ0NBQWdDLEVBQUU7NENBQ2hDLGNBQWMsRUFBRSxvQkFBb0I7NENBQ3BDLFFBQVEsRUFBRSxDQUFDOzRDQUNYLGlCQUFpQixFQUFFLENBQUM7eUNBQ3JCO3FDQUNGO29DQUNELFdBQVcsRUFBRSxhQUFhO29DQUMxQixJQUFJLEVBQUUsTUFBTTtvQ0FDWixPQUFPLEVBQUU7d0NBQ1AseUJBQXlCLEVBQUU7NENBQ3pCLGtCQUFrQixFQUFFLGNBQWM7NENBQ2xDLFVBQVUsRUFBRSxDQUFDO29EQUNYLEdBQUcsRUFBRSxLQUFLO29EQUNWLEtBQUssRUFBRSxPQUFPO2lEQUNmLENBQUM7NENBQ0YsaUJBQWlCLEVBQUUsQ0FBQzs0Q0FDcEIsVUFBVSxFQUFFLE1BQU07NENBQ2xCLFNBQVMsRUFBRSxXQUFXOzRDQUN0QixNQUFNLEVBQUUsR0FBRzs0Q0FDWCxTQUFTLEVBQUUsU0FBUzs0Q0FDcEIsU0FBUyxFQUFFLENBQUM7NENBQ1osSUFBSSxFQUFFLE1BQU07eUNBQ2I7cUNBQ0Y7aUNBQ0YsRUFBRTtvQ0FDRCxNQUFNLEVBQUU7d0NBQ04sTUFBTSxFQUFFLFdBQVc7d0NBQ25CLGdDQUFnQyxFQUFFOzRDQUNoQyxjQUFjLEVBQUUsb0JBQW9COzRDQUNwQyxRQUFRLEVBQUUsQ0FBQzs0Q0FDWCxpQkFBaUIsRUFBRSxDQUFDO3lDQUNyQjtxQ0FDRjtvQ0FDRCxXQUFXLEVBQUUsYUFBYTtvQ0FDMUIsSUFBSSxFQUFFLE1BQU07b0NBQ1osT0FBTyxFQUFFO3dDQUNQLHlCQUF5QixFQUFFOzRDQUN6QixvQkFBb0IsRUFBRSxjQUFjOzRDQUNwQyxZQUFZLEVBQUUsQ0FBQztvREFDYixHQUFHLEVBQUUsS0FBSztvREFDVixLQUFLLEVBQUUsT0FBTztpREFDZixDQUFDOzRDQUNGLG1CQUFtQixFQUFFLENBQUM7NENBQ3RCLFlBQVksRUFBRSxNQUFNOzRDQUNwQixXQUFXLEVBQUUsV0FBVzs0Q0FDeEIsVUFBVSxFQUFFLG9CQUFvQjs0Q0FDaEMsV0FBVyxFQUFFLFNBQVM7NENBQ3RCLFdBQVcsRUFBRSxDQUFDOzRDQUNkLE1BQU0sRUFBRSxNQUFNO3lDQUNmO3FDQUNGO2lDQUNGLENBQUM7eUJBQ0g7d0JBQ0QsUUFBUSxFQUFFLEdBQUc7d0JBQ2IsY0FBYyxFQUFFLENBQUM7Z0NBQ2YsY0FBYyxFQUFFLGdCQUFnQjtnQ0FDaEMsVUFBVSxFQUFFO29DQUNWLEdBQUcsRUFBRSxPQUFPO2lDQUNiOzZCQUNGLENBQUM7d0JBQ0YsZ0JBQWdCLEVBQUU7NEJBQ2hCLHFCQUFxQixFQUFFLENBQUM7b0NBQ3RCLG1CQUFtQixFQUFFO3dDQUNuQixJQUFJLEVBQUUsQ0FBQzt3Q0FDUCxRQUFRLEVBQUUsQ0FBQzt3Q0FDWCxVQUFVLEVBQUUsVUFBVTtxQ0FDdkI7b0NBQ0Qsa0JBQWtCLEVBQUUsQ0FBQztpQ0FDdEIsQ0FBQzs0QkFDRixZQUFZLEVBQUUsSUFBSTt5QkFDbkI7d0JBQ0QsYUFBYSxFQUFFLENBQUM7d0JBQ2hCLFlBQVksRUFBRSxRQUFRO3dCQUN0QixZQUFZLEVBQUUsV0FBVzt3QkFDekIsTUFBTSxFQUFFLFdBQVc7d0JBQ25CLElBQUksRUFBRSxNQUFNO3FCQUNiLENBQUM7YUFDSDtZQUNELGlCQUFpQixFQUFFLElBQUk7WUFDdkIsV0FBVyxFQUFFO2dCQUNYLEdBQUcsRUFBRSxxQkFBcUI7YUFDM0I7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxFQUFFLHFCQUFxQjthQUMzQjtZQUNELGVBQWUsRUFBRTtnQkFDZixHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrRUFBK0UsRUFBRSxHQUFHLEVBQUU7SUFDekYsTUFBTSxDQUFDLEdBQUcsRUFBRTtRQUNWLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzFCLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDL0IsU0FBUyxFQUFFLEVBQUU7Z0JBQ2IsV0FBVztnQkFDWCxJQUFJLEVBQUUsU0FBUztnQkFDZixXQUFXO2dCQUNYLGVBQWU7Z0JBQ2Ysa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLG1CQUFtQjthQUN0RSxDQUFDO1NBQ0gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLG1HQUFtRyxDQUFDLENBQUM7QUFDbEgsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyB0YXNrcyBmcm9tICcuLi9saWInO1xuXG5sZXQgc3RhY2s6IGNkay5TdGFjaztcbmxldCBjbHVzdGVyUm9sZTogaWFtLlJvbGU7XG5sZXQgc2VydmljZVJvbGU6IGlhbS5Sb2xlO1xubGV0IGF1dG9TY2FsaW5nUm9sZTogaWFtLlJvbGU7XG5cbmJlZm9yZUVhY2goKCkgPT4ge1xuICAvLyBHSVZFTlxuICBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY2x1c3RlclJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdDbHVzdGVyUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKVxuICB9KTtcbiAgc2VydmljZVJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTZXJ2aWNlUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWxhc3RpY21hcHJlZHVjZS5hbWF6b25hd3MuY29tJylcbiAgfSk7XG4gIGF1dG9TY2FsaW5nUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ0F1dG9TY2FsaW5nUm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWxhc3RpY21hcHJlZHVjZS5hbWF6b25hd3MuY29tJylcbiAgfSk7XG4gIGF1dG9TY2FsaW5nUm9sZS5hc3N1bWVSb2xlUG9saWN5Py5hZGRTdGF0ZW1lbnRzKFxuICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIHByaW5jaXBhbHM6IFtcbiAgICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdhcHBsaWNhdGlvbi1hdXRvc2NhbGluZy5hbWF6b25hd3MuY29tJylcbiAgICAgIF0sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdzdHM6QXNzdW1lUm9sZSdcbiAgICAgIF1cbiAgICB9KVxuICApO1xufSk7XG5cbnRlc3QoJ0NyZWF0ZSBDbHVzdGVyIHdpdGggRklSRV9BTkRfRk9SR0VUIGludGVncmF0aW9uUGF0dGVybicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge30sXG4gICAgY2x1c3RlclJvbGUsXG4gICAgbmFtZTogJ0NsdXN0ZXInLFxuICAgIHNlcnZpY2VSb2xlLFxuICAgIGF1dG9TY2FsaW5nUm9sZSxcbiAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLkZJUkVfQU5EX0ZPUkdFVFxuICB9KSB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmVsYXN0aWNtYXByZWR1Y2U6Y3JlYXRlQ2x1c3RlcicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIE5hbWU6ICdDbHVzdGVyJyxcbiAgICAgIEluc3RhbmNlczoge1xuICAgICAgICBLZWVwSm9iRmxvd0FsaXZlV2hlbk5vU3RlcHM6IHRydWVcbiAgICAgIH0sXG4gICAgICBWaXNpYmxlVG9BbGxVc2VyczogdHJ1ZSxcbiAgICAgIEpvYkZsb3dSb2xlOiB7XG4gICAgICAgIFJlZjogJ0NsdXN0ZXJSb2xlRDlDQTc0NzEnLFxuICAgICAgfSxcbiAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgIFJlZjogJ1NlcnZpY2VSb2xlNDI4OEIxOTInXG4gICAgICB9LFxuICAgICAgQXV0b1NjYWxpbmdSb2xlOiB7XG4gICAgICAgIFJlZjogJ0F1dG9TY2FsaW5nUm9sZTAxNUFEQTBBJ1xuICAgICAgfVxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NyZWF0ZSBDbHVzdGVyIHdpdGggU1lOQyBpbnRlZ3JhdGlvblBhdHRlcm4nLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyKHtcbiAgICBpbnN0YW5jZXM6IHt9LFxuICAgIGNsdXN0ZXJSb2xlLFxuICAgIG5hbWU6ICdDbHVzdGVyJyxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DXG4gIH0pIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6ZWxhc3RpY21hcHJlZHVjZTpjcmVhdGVDbHVzdGVyLnN5bmMnLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBOYW1lOiAnQ2x1c3RlcicsXG4gICAgICBJbnN0YW5jZXM6IHtcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlXG4gICAgICB9LFxuICAgICAgVmlzaWJsZVRvQWxsVXNlcnM6IHRydWUsXG4gICAgICBKb2JGbG93Um9sZToge1xuICAgICAgICBSZWY6ICdDbHVzdGVyUm9sZUQ5Q0E3NDcxJyxcbiAgICAgIH0sXG4gICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICBSZWY6ICdTZXJ2aWNlUm9sZTQyODhCMTkyJ1xuICAgICAgfSxcbiAgICAgIEF1dG9TY2FsaW5nUm9sZToge1xuICAgICAgICBSZWY6ICdBdXRvU2NhbGluZ1JvbGUwMTVBREEwQSdcbiAgICAgIH1cbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDcmVhdGUgQ2x1c3RlciB3aXRoIGNsdXN0ZXJDb25maWd1cmF0aW9uIE5hbWUgZnJvbSBwYXlsb2FkJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlcih7XG4gICAgaW5zdGFuY2VzOiB7fSxcbiAgICBjbHVzdGVyUm9sZSxcbiAgICBuYW1lOiBzZm4uVGFza0lucHV0LmZyb21EYXRhQXQoJyQuQ2x1c3Rlck5hbWUnKS52YWx1ZSxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVRcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOmNyZWF0ZUNsdXN0ZXInLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAnTmFtZS4kJzogJyQuQ2x1c3Rlck5hbWUnLFxuICAgICAgJ0luc3RhbmNlcyc6IHtcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlXG4gICAgICB9LFxuICAgICAgJ1Zpc2libGVUb0FsbFVzZXJzJzogdHJ1ZSxcbiAgICAgICdKb2JGbG93Um9sZSc6IHtcbiAgICAgICAgUmVmOiAnQ2x1c3RlclJvbGVEOUNBNzQ3MScsXG4gICAgICB9LFxuICAgICAgJ1NlcnZpY2VSb2xlJzoge1xuICAgICAgICBSZWY6ICdTZXJ2aWNlUm9sZTQyODhCMTkyJ1xuICAgICAgfSxcbiAgICAgICdBdXRvU2NhbGluZ1JvbGUnOiB7XG4gICAgICAgIFJlZjogJ0F1dG9TY2FsaW5nUm9sZTAxNUFEQTBBJ1xuICAgICAgfVxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NyZWF0ZSBDbHVzdGVyIHdpdGggVGFncycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge30sXG4gICAgY2x1c3RlclJvbGUsXG4gICAgbmFtZTogJ0NsdXN0ZXInLFxuICAgIHNlcnZpY2VSb2xlLFxuICAgIGF1dG9TY2FsaW5nUm9sZSxcbiAgICB0YWdzOiBbe1xuICAgICAga2V5OiAnS2V5JyxcbiAgICAgIHZhbHVlOiAnVmFsdWUnXG4gICAgfV0sXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVRcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOmNyZWF0ZUNsdXN0ZXInLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBOYW1lOiAnQ2x1c3RlcicsXG4gICAgICBJbnN0YW5jZXM6IHtcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlXG4gICAgICB9LFxuICAgICAgVmlzaWJsZVRvQWxsVXNlcnM6IHRydWUsXG4gICAgICBKb2JGbG93Um9sZToge1xuICAgICAgICBSZWY6ICdDbHVzdGVyUm9sZUQ5Q0E3NDcxJyxcbiAgICAgIH0sXG4gICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICBSZWY6ICdTZXJ2aWNlUm9sZTQyODhCMTkyJ1xuICAgICAgfSxcbiAgICAgIEF1dG9TY2FsaW5nUm9sZToge1xuICAgICAgICBSZWY6ICdBdXRvU2NhbGluZ1JvbGUwMTVBREEwQSdcbiAgICAgIH0sXG4gICAgICBUYWdzOiBbe1xuICAgICAgICBLZXk6ICdLZXknLFxuICAgICAgICBWYWx1ZTogJ1ZhbHVlJ1xuICAgICAgfV1cbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDcmVhdGUgQ2x1c3RlciB3aXRoIEFwcGxpY2F0aW9ucycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge30sXG4gICAgY2x1c3RlclJvbGUsXG4gICAgbmFtZTogJ0NsdXN0ZXInLFxuICAgIHNlcnZpY2VSb2xlLFxuICAgIGF1dG9TY2FsaW5nUm9sZSxcbiAgICBhcHBsaWNhdGlvbnM6IFtcbiAgICAgIHsgbmFtZTogJ0hpdmUnLCB2ZXJzaW9uOiAnMC4wJyB9LFxuICAgICAgeyBuYW1lOiAnU3BhcmsnLCB2ZXJzaW9uOiAnMC4wJyB9XG4gICAgXSxcbiAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLkZJUkVfQU5EX0ZPUkdFVFxuICB9KSB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmVsYXN0aWNtYXByZWR1Y2U6Y3JlYXRlQ2x1c3RlcicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIE5hbWU6ICdDbHVzdGVyJyxcbiAgICAgIEluc3RhbmNlczoge1xuICAgICAgICBLZWVwSm9iRmxvd0FsaXZlV2hlbk5vU3RlcHM6IHRydWVcbiAgICAgIH0sXG4gICAgICBWaXNpYmxlVG9BbGxVc2VyczogdHJ1ZSxcbiAgICAgIEpvYkZsb3dSb2xlOiB7XG4gICAgICAgIFJlZjogJ0NsdXN0ZXJSb2xlRDlDQTc0NzEnLFxuICAgICAgfSxcbiAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgIFJlZjogJ1NlcnZpY2VSb2xlNDI4OEIxOTInXG4gICAgICB9LFxuICAgICAgQXV0b1NjYWxpbmdSb2xlOiB7XG4gICAgICAgIFJlZjogJ0F1dG9TY2FsaW5nUm9sZTAxNUFEQTBBJ1xuICAgICAgfSxcbiAgICAgIEFwcGxpY2F0aW9uczogW1xuICAgICAgICB7IE5hbWU6ICdIaXZlJywgVmVyc2lvbjogJzAuMCcgfSxcbiAgICAgICAgeyBOYW1lOiAnU3BhcmsnLCBWZXJzaW9uOiAnMC4wJyB9XG4gICAgICBdXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnQ3JlYXRlIENsdXN0ZXIgd2l0aCBCb290c3RyYXAgQWN0aW9ucycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge30sXG4gICAgY2x1c3RlclJvbGUsXG4gICAgbmFtZTogJ0NsdXN0ZXInLFxuICAgIHNlcnZpY2VSb2xlLFxuICAgIGF1dG9TY2FsaW5nUm9sZSxcbiAgICBib290c3RyYXBBY3Rpb25zOiBbe1xuICAgICAgbmFtZTogJ0Jvb3RzdHJhcCcsXG4gICAgICBzY3JpcHRCb290c3RyYXBBY3Rpb246IHtcbiAgICAgICAgcGF0aDogJ3MzOi8vbnVsbCcsXG4gICAgICAgIGFyZ3M6IFsgJ0FyZycgXVxuICAgICAgfVxuICAgIH1dLFxuICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uRklSRV9BTkRfRk9SR0VUXG4gIH0pIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6ZWxhc3RpY21hcHJlZHVjZTpjcmVhdGVDbHVzdGVyJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgTmFtZTogJ0NsdXN0ZXInLFxuICAgICAgSW5zdGFuY2VzOiB7XG4gICAgICAgIEtlZXBKb2JGbG93QWxpdmVXaGVuTm9TdGVwczogdHJ1ZVxuICAgICAgfSxcbiAgICAgIFZpc2libGVUb0FsbFVzZXJzOiB0cnVlLFxuICAgICAgSm9iRmxvd1JvbGU6IHtcbiAgICAgICAgUmVmOiAnQ2x1c3RlclJvbGVEOUNBNzQ3MScsXG4gICAgICB9LFxuICAgICAgU2VydmljZVJvbGU6IHtcbiAgICAgICAgUmVmOiAnU2VydmljZVJvbGU0Mjg4QjE5MidcbiAgICAgIH0sXG4gICAgICBBdXRvU2NhbGluZ1JvbGU6IHtcbiAgICAgICAgUmVmOiAnQXV0b1NjYWxpbmdSb2xlMDE1QURBMEEnXG4gICAgICB9LFxuICAgICAgQm9vdHN0cmFwQWN0aW9uczogW3tcbiAgICAgICAgTmFtZTogJ0Jvb3RzdHJhcCcsXG4gICAgICAgIFNjcmlwdEJvb3RzdHJhcEFjdGlvbjoge1xuICAgICAgICAgIFBhdGg6ICdzMzovL251bGwnLFxuICAgICAgICAgIEFyZ3M6IFsgJ0FyZycgXVxuICAgICAgICB9XG4gICAgICB9XVxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NyZWF0ZSBDbHVzdGVyIHdpdGggQ29uZmlndXJhdGlvbnMnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyKHtcbiAgICBpbnN0YW5jZXM6IHt9LFxuICAgIGNsdXN0ZXJSb2xlLFxuICAgIG5hbWU6ICdDbHVzdGVyJyxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgY29uZmlndXJhdGlvbnM6IFt7XG4gICAgICBjbGFzc2lmaWNhdGlvbjogJ2NsYXNzaWZpY2F0aW9uJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgS2V5OiAnVmFsdWUnXG4gICAgICB9XG4gICAgfV0sXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVRcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOmNyZWF0ZUNsdXN0ZXInLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBOYW1lOiAnQ2x1c3RlcicsXG4gICAgICBJbnN0YW5jZXM6IHtcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlXG4gICAgICB9LFxuICAgICAgVmlzaWJsZVRvQWxsVXNlcnM6IHRydWUsXG4gICAgICBKb2JGbG93Um9sZToge1xuICAgICAgICBSZWY6ICdDbHVzdGVyUm9sZUQ5Q0E3NDcxJyxcbiAgICAgIH0sXG4gICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICBSZWY6ICdTZXJ2aWNlUm9sZTQyODhCMTkyJ1xuICAgICAgfSxcbiAgICAgIEF1dG9TY2FsaW5nUm9sZToge1xuICAgICAgICBSZWY6ICdBdXRvU2NhbGluZ1JvbGUwMTVBREEwQSdcbiAgICAgIH0sXG4gICAgICBDb25maWd1cmF0aW9uczogW3tcbiAgICAgICAgQ2xhc3NpZmljYXRpb246ICdjbGFzc2lmaWNhdGlvbicsXG4gICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICBLZXk6ICdWYWx1ZSdcbiAgICAgICAgfVxuICAgICAgfV1cbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDcmVhdGUgQ2x1c3RlciB3aXRoIEtlcmJlcm9zQXR0cmlidXRlcycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge30sXG4gICAgY2x1c3RlclJvbGUsXG4gICAgbmFtZTogJ0NsdXN0ZXInLFxuICAgIHNlcnZpY2VSb2xlLFxuICAgIGF1dG9TY2FsaW5nUm9sZSxcbiAgICBrZXJiZXJvc0F0dHJpYnV0ZXM6IHtcbiAgICAgIHJlYWxtOiAncmVhbG0nLFxuICAgICAgYWREb21haW5Kb2luUGFzc3dvcmQ6ICdwYXNzd29yZDEnLFxuICAgICAgYWREb21haW5Kb2luVXNlcjogJ3VzZXInLFxuICAgICAgY3Jvc3NSZWFsbVRydXN0UHJpbmNpcGFsUGFzc3dvcmQ6ICdwYXNzd29yZDInLFxuICAgICAga2RjQWRtaW5QYXNzd29yZDogJ3Bhc3N3b3JkMydcbiAgICB9LFxuICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uRklSRV9BTkRfRk9SR0VUXG4gIH0pIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6ZWxhc3RpY21hcHJlZHVjZTpjcmVhdGVDbHVzdGVyJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgTmFtZTogJ0NsdXN0ZXInLFxuICAgICAgSW5zdGFuY2VzOiB7XG4gICAgICAgIEtlZXBKb2JGbG93QWxpdmVXaGVuTm9TdGVwczogdHJ1ZVxuICAgICAgfSxcbiAgICAgIFZpc2libGVUb0FsbFVzZXJzOiB0cnVlLFxuICAgICAgSm9iRmxvd1JvbGU6IHtcbiAgICAgICAgUmVmOiAnQ2x1c3RlclJvbGVEOUNBNzQ3MScsXG4gICAgICB9LFxuICAgICAgU2VydmljZVJvbGU6IHtcbiAgICAgICAgUmVmOiAnU2VydmljZVJvbGU0Mjg4QjE5MidcbiAgICAgIH0sXG4gICAgICBBdXRvU2NhbGluZ1JvbGU6IHtcbiAgICAgICAgUmVmOiAnQXV0b1NjYWxpbmdSb2xlMDE1QURBMEEnXG4gICAgICB9LFxuICAgICAgS2VyYmVyb3NBdHRyaWJ1dGVzOiB7XG4gICAgICAgIFJlYWxtOiAncmVhbG0nLFxuICAgICAgICBBRERvbWFpbkpvaW5QYXNzd29yZDogJ3Bhc3N3b3JkMScsXG4gICAgICAgIEFERG9tYWluSm9pblVzZXI6ICd1c2VyJyxcbiAgICAgICAgQ3Jvc3NSZWFsbVRydXN0UHJpbmNpcGFsUGFzc3dvcmQ6ICdwYXNzd29yZDInLFxuICAgICAgICBLZGNBZG1pblBhc3N3b3JkOiAncGFzc3dvcmQzJ1xuICAgICAgfVxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NyZWF0ZSBDbHVzdGVyIHdpdGhvdXQgUm9sZXMnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgY3JlYXRlQ2x1c3RlclRhc2sgPSBuZXcgdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlcih7XG4gICAgaW5zdGFuY2VzOiB7fSxcbiAgICBuYW1lOiAnQ2x1c3RlcicsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DXG4gIH0pO1xuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogY3JlYXRlQ2x1c3RlclRhc2t9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmVsYXN0aWNtYXByZWR1Y2U6Y3JlYXRlQ2x1c3Rlci5zeW5jJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgTmFtZTogJ0NsdXN0ZXInLFxuICAgICAgSW5zdGFuY2VzOiB7XG4gICAgICAgIEtlZXBKb2JGbG93QWxpdmVXaGVuTm9TdGVwczogdHJ1ZVxuICAgICAgfSxcbiAgICAgIFZpc2libGVUb0FsbFVzZXJzOiB0cnVlLFxuICAgICAgSm9iRmxvd1JvbGU6IHtcbiAgICAgICAgUmVmOiAnVGFza0luc3RhbmNlUm9sZUI3MjA3MkJGJ1xuICAgICAgfSxcbiAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgIFJlZjogJ1Rhc2tTZXJ2aWNlUm9sZUJGNTVGNjFFJ1xuICAgICAgfSxcbiAgICAgIEF1dG9TY2FsaW5nUm9sZToge1xuICAgICAgICBSZWY6ICdUYXNrQXV0b1NjYWxpbmdSb2xlRDA2Rjg0MjMnXG4gICAgICB9XG4gICAgfSxcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBQcmluY2lwYWw6IHsgU2VydmljZTogJ2VsYXN0aWNtYXByZWR1Y2UuYW1hem9uYXdzLmNvbScgfSxcbiAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnXG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfVxuICB9KTtcblxuICAvLyBUaGUgc3RhY2sgcmVuZGVycyB0aGUgZWMyLmFtYXpvbmF3cy5jb20gU2VydmljZSBwcmluY2lwYWwgaWQgd2l0aCBhXG4gIC8vIEpvaW4gdG8gdGhlIFVSTFN1ZmZpeFxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOlxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2VjMi4nLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlVSTFN1ZmZpeCdcbiAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIEFjdGlvbjogJ3N0czpBc3N1bWVSb2xlJyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdydcbiAgICAgICAgfVxuICAgICAgXSxcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdlbGFzdGljbWFwcmVkdWNlLmFtYXpvbmF3cy5jb20nIH0sXG4gICAgICAgICAgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93J1xuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdhcHBsaWNhdGlvbi1hdXRvc2NhbGluZy5hbWF6b25hd3MuY29tJyB9LFxuICAgICAgICAgIEFjdGlvbjogJ3N0czpBc3N1bWVSb2xlJyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdydcbiAgICAgICAgfVxuICAgICAgXSxcbiAgICB9XG4gIH0pO1xuXG59KTtcblxudGVzdCgnQ3JlYXRlIENsdXN0ZXIgd2l0aCBJbnN0YW5jZXMgY29uZmlndXJhdGlvbicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIoe1xuICAgIGluc3RhbmNlczoge1xuICAgICAgYWRkaXRpb25hbE1hc3RlclNlY3VyaXR5R3JvdXBzOiBbJ01hc3Rlckdyb3VwJ10sXG4gICAgICBhZGRpdGlvbmFsU2xhdmVTZWN1cml0eUdyb3VwczogWydTbGF2ZUdyb3VwJ10sXG4gICAgICBlYzJLZXlOYW1lOiAnRWMyS2V5TmFtZScsXG4gICAgICBlYzJTdWJuZXRJZDogJ0VjMlN1Ym5ldElkJyxcbiAgICAgIGVjMlN1Ym5ldElkczogWydFYzJTdWJuZXRJZCddLFxuICAgICAgZW1yTWFuYWdlZE1hc3RlclNlY3VyaXR5R3JvdXA6ICdNYXN0ZXJHcm91cCcsXG4gICAgICBlbXJNYW5hZ2VkU2xhdmVTZWN1cml0eUdyb3VwOiAnU2xhdmVHcm91cCcsXG4gICAgICBoYWRvb3BWZXJzaW9uOiAnSGFkb29wVmVyc2lvbicsXG4gICAgICBpbnN0YW5jZUNvdW50OiAxLFxuICAgICAgbWFzdGVySW5zdGFuY2VUeXBlOiAnTWFzdGVySW5zdGFuY2VUeXBlJyxcbiAgICAgIHBsYWNlbWVudDoge1xuICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnQXZhaWxhYmlsaXR5Wm9uZScsXG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmVzOiBbJ0F2YWlsYWJpbGl0eVpvbmUnXVxuICAgICAgfSxcbiAgICAgIHNlcnZpY2VBY2Nlc3NTZWN1cml0eUdyb3VwOiAnU2VydmljZUFjY2Vzc0dyb3VwJyxcbiAgICAgIHNsYXZlSW5zdGFuY2VUeXBlOiAnU2xhdmVJbnN0YW5jZVR5cGUnLFxuICAgICAgdGVybWluYXRpb25Qcm90ZWN0ZWQ6IHRydWVcbiAgICB9LFxuICAgIGNsdXN0ZXJSb2xlLFxuICAgIG5hbWU6ICdDbHVzdGVyJyxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVRcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOmNyZWF0ZUNsdXN0ZXInLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBOYW1lOiAnQ2x1c3RlcicsXG4gICAgICBJbnN0YW5jZXM6IHtcbiAgICAgICAgQWRkaXRpb25hbE1hc3RlclNlY3VyaXR5R3JvdXBzOiBbJ01hc3Rlckdyb3VwJ10sXG4gICAgICAgIEFkZGl0aW9uYWxTbGF2ZVNlY3VyaXR5R3JvdXBzOiBbJ1NsYXZlR3JvdXAnXSxcbiAgICAgICAgRWMyS2V5TmFtZTogJ0VjMktleU5hbWUnLFxuICAgICAgICBFYzJTdWJuZXRJZDogJ0VjMlN1Ym5ldElkJyxcbiAgICAgICAgRWMyU3VibmV0SWRzOiBbJ0VjMlN1Ym5ldElkJ10sXG4gICAgICAgIEVtck1hbmFnZWRNYXN0ZXJTZWN1cml0eUdyb3VwOiAnTWFzdGVyR3JvdXAnLFxuICAgICAgICBFbXJNYW5hZ2VkU2xhdmVTZWN1cml0eUdyb3VwOiAnU2xhdmVHcm91cCcsXG4gICAgICAgIEhhZG9vcFZlcnNpb246ICdIYWRvb3BWZXJzaW9uJyxcbiAgICAgICAgSW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlLFxuICAgICAgICBNYXN0ZXJJbnN0YW5jZVR5cGU6ICdNYXN0ZXJJbnN0YW5jZVR5cGUnLFxuICAgICAgICBQbGFjZW1lbnQ6IHtcbiAgICAgICAgICBBdmFpbGFiaWxpdHlab25lOiAnQXZhaWxhYmlsaXR5Wm9uZScsXG4gICAgICAgICAgQXZhaWxhYmlsaXR5Wm9uZXM6IFsnQXZhaWxhYmlsaXR5Wm9uZSddXG4gICAgICAgIH0sXG4gICAgICAgIFNlcnZpY2VBY2Nlc3NTZWN1cml0eUdyb3VwOiAnU2VydmljZUFjY2Vzc0dyb3VwJyxcbiAgICAgICAgU2xhdmVJbnN0YW5jZVR5cGU6ICdTbGF2ZUluc3RhbmNlVHlwZScsXG4gICAgICAgIFRlcm1pbmF0aW9uUHJvdGVjdGVkOiB0cnVlXG4gICAgICB9LFxuICAgICAgVmlzaWJsZVRvQWxsVXNlcnM6IHRydWUsXG4gICAgICBKb2JGbG93Um9sZToge1xuICAgICAgICBSZWY6ICdDbHVzdGVyUm9sZUQ5Q0E3NDcxJyxcbiAgICAgIH0sXG4gICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICBSZWY6ICdTZXJ2aWNlUm9sZTQyODhCMTkyJ1xuICAgICAgfSxcbiAgICAgIEF1dG9TY2FsaW5nUm9sZToge1xuICAgICAgICBSZWY6ICdBdXRvU2NhbGluZ1JvbGUwMTVBREEwQSdcbiAgICAgIH1cbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDcmVhdGUgQ2x1c3RlciB3aXRoIEluc3RhbmNlRmxlZXQnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyKHtcbiAgICBpbnN0YW5jZXM6IHtcbiAgICAgIGluc3RhbmNlRmxlZXRzOiBbe1xuICAgICAgICBpbnN0YW5jZUZsZWV0VHlwZTogdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlci5JbnN0YW5jZVJvbGVUeXBlLk1BU1RFUixcbiAgICAgICAgaW5zdGFuY2VUeXBlQ29uZmlnczogW3tcbiAgICAgICAgICBiaWRQcmljZTogJzEnLFxuICAgICAgICAgIGJpZFByaWNlQXNQZXJjZW50YWdlT2ZPbkRlbWFuZFByaWNlOiAxLFxuICAgICAgICAgIGNvbmZpZ3VyYXRpb25zOiBbe1xuICAgICAgICAgICAgY2xhc3NpZmljYXRpb246ICdDbGFzc2lmaWNhdGlvbicsXG4gICAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICAgIEtleTogJ1ZhbHVlJ1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1dLFxuICAgICAgICAgIGVic0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIGVic0Jsb2NrRGV2aWNlQ29uZmlnczogW3tcbiAgICAgICAgICAgICAgdm9sdW1lU3BlY2lmaWNhdGlvbjoge1xuICAgICAgICAgICAgICAgIGlvcHM6IDEsXG4gICAgICAgICAgICAgICAgc2l6ZUluR0I6IDEsXG4gICAgICAgICAgICAgICAgdm9sdW1lVHlwZTogdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlci5FYnNCbG9ja0RldmljZVZvbHVtZVR5cGUuU1RBTkRBUkQsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHZvbHVtZXNQZXJJbnN0YW5jZTogMVxuICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICBlYnNPcHRpbWl6ZWQ6IHRydWVcbiAgICAgICAgICB9LFxuICAgICAgICAgIGluc3RhbmNlVHlwZTogJ201LnhsYXJnZScsXG4gICAgICAgICAgd2VpZ2h0ZWRDYXBhY2l0eTogMVxuICAgICAgICB9XSxcbiAgICAgICAgbGF1bmNoU3BlY2lmaWNhdGlvbnM6IHtcbiAgICAgICAgICBzcG90U3BlY2lmaWNhdGlvbjoge1xuICAgICAgICAgICAgYmxvY2tEdXJhdGlvbk1pbnV0ZXM6IDEsXG4gICAgICAgICAgICB0aW1lb3V0QWN0aW9uOiB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyLlNwb3RUaW1lb3V0QWN0aW9uLlRFUk1JTkFURV9DTFVTVEVSLFxuICAgICAgICAgICAgdGltZW91dER1cmF0aW9uTWludXRlczogMVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgbmFtZTogJ01hc3RlcicsXG4gICAgICAgIHRhcmdldE9uRGVtYW5kQ2FwYWNpdHk6IDEsXG4gICAgICAgIHRhcmdldFNwb3RDYXBhY2l0eTogMVxuICAgICAgfV1cbiAgICB9LFxuICAgIGNsdXN0ZXJSb2xlLFxuICAgIG5hbWU6ICdDbHVzdGVyJyxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLkZJUkVfQU5EX0ZPUkdFVFxuICB9KSB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmVsYXN0aWNtYXByZWR1Y2U6Y3JlYXRlQ2x1c3RlcicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIE5hbWU6ICdDbHVzdGVyJyxcbiAgICAgIEluc3RhbmNlczoge1xuICAgICAgICBLZWVwSm9iRmxvd0FsaXZlV2hlbk5vU3RlcHM6IHRydWUsXG4gICAgICAgIEluc3RhbmNlRmxlZXRzOiBbe1xuICAgICAgICAgIEluc3RhbmNlRmxlZXRUeXBlOiAnTUFTVEVSJyxcbiAgICAgICAgICBJbnN0YW5jZVR5cGVDb25maWdzOiBbe1xuICAgICAgICAgICAgQmlkUHJpY2U6ICcxJyxcbiAgICAgICAgICAgIEJpZFByaWNlQXNQZXJjZW50YWdlT2ZPbkRlbWFuZFByaWNlOiAxLFxuICAgICAgICAgICAgQ29uZmlndXJhdGlvbnM6IFt7XG4gICAgICAgICAgICAgIENsYXNzaWZpY2F0aW9uOiAnQ2xhc3NpZmljYXRpb24nLFxuICAgICAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICAgICAgS2V5OiAnVmFsdWUnXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1dLFxuICAgICAgICAgICAgRWJzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICBFYnNCbG9ja0RldmljZUNvbmZpZ3M6IFt7XG4gICAgICAgICAgICAgICAgVm9sdW1lU3BlY2lmaWNhdGlvbjoge1xuICAgICAgICAgICAgICAgICAgSW9wczogMSxcbiAgICAgICAgICAgICAgICAgIFNpemVJbkdCOiAxLFxuICAgICAgICAgICAgICAgICAgVm9sdW1lVHlwZTogJ3N0YW5kYXJkJ1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgVm9sdW1lc1Blckluc3RhbmNlOiAxXG4gICAgICAgICAgICAgIH1dLFxuICAgICAgICAgICAgICBFYnNPcHRpbWl6ZWQ6IHRydWVcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBJbnN0YW5jZVR5cGU6ICdtNS54bGFyZ2UnLFxuICAgICAgICAgICAgV2VpZ2h0ZWRDYXBhY2l0eTogMVxuICAgICAgICAgIH1dLFxuICAgICAgICAgIExhdW5jaFNwZWNpZmljYXRpb25zOiB7XG4gICAgICAgICAgICBTcG90U3BlY2lmaWNhdGlvbjoge1xuICAgICAgICAgICAgICBCbG9ja0R1cmF0aW9uTWludXRlczogMSxcbiAgICAgICAgICAgICAgVGltZW91dEFjdGlvbjogJ1RFUk1JTkFURV9DTFVTVEVSJyxcbiAgICAgICAgICAgICAgVGltZW91dER1cmF0aW9uTWludXRlczogMVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgTmFtZTogJ01hc3RlcicsXG4gICAgICAgICAgVGFyZ2V0T25EZW1hbmRDYXBhY2l0eTogMSxcbiAgICAgICAgICBUYXJnZXRTcG90Q2FwYWNpdHk6IDFcbiAgICAgICAgfV1cbiAgICAgIH0sXG4gICAgICBWaXNpYmxlVG9BbGxVc2VyczogdHJ1ZSxcbiAgICAgIEpvYkZsb3dSb2xlOiB7XG4gICAgICAgIFJlZjogJ0NsdXN0ZXJSb2xlRDlDQTc0NzEnLFxuICAgICAgfSxcbiAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgIFJlZjogJ1NlcnZpY2VSb2xlNDI4OEIxOTInXG4gICAgICB9XG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnQ3JlYXRlIENsdXN0ZXIgd2l0aCBJbnN0YW5jZUdyb3VwJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlcih7XG4gICAgaW5zdGFuY2VzOiB7XG4gICAgICBpbnN0YW5jZUdyb3VwczogW3tcbiAgICAgICAgYXV0b1NjYWxpbmdQb2xpY3k6IHtcbiAgICAgICAgICBjb25zdHJhaW50czoge1xuICAgICAgICAgICAgbWF4Q2FwYWNpdHk6IDIsXG4gICAgICAgICAgICBtaW5DYXBhY2l0eTogMSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHJ1bGVzOiBbe1xuICAgICAgICAgICAgYWN0aW9uOiB7XG4gICAgICAgICAgICAgIG1hcmtldDogdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlci5JbnN0YW5jZU1hcmtldC5PTl9ERU1BTkQsXG4gICAgICAgICAgICAgIHNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgYWRqdXN0bWVudFR5cGU6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuU2NhbGluZ0FkanVzdG1lbnRUeXBlLkNIQU5HRV9JTl9DQVBBQ0lUWSxcbiAgICAgICAgICAgICAgICBjb29sRG93bjogMSxcbiAgICAgICAgICAgICAgICBzY2FsaW5nQWRqdXN0bWVudDogMVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgZGVzY3JpcHRpb246ICdEZXNjcmlwdGlvbicsXG4gICAgICAgICAgICBuYW1lOiAnTmFtZScsXG4gICAgICAgICAgICB0cmlnZ2VyOiB7XG4gICAgICAgICAgICAgIGNsb3VkV2F0Y2hBbGFybURlZmluaXRpb246IHtcbiAgICAgICAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuQ2xvdWRXYXRjaEFsYXJtQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTixcbiAgICAgICAgICAgICAgICBkaW1lbnNpb25zOiBbe1xuICAgICAgICAgICAgICAgICAga2V5OiAnS2V5JyxcbiAgICAgICAgICAgICAgICAgIHZhbHVlOiAnVmFsdWUnXG4gICAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICAgICAgZXZhbHV0aW9uUGVyaW9kczogMSxcbiAgICAgICAgICAgICAgICBtZXRyaWNOYW1lOiAnTmFtZScsXG4gICAgICAgICAgICAgICAgbmFtZXNwYWNlOiAnTmFtZXNwYWNlJyxcbiAgICAgICAgICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwMCksXG4gICAgICAgICAgICAgICAgc3RhdGlzdGljOiB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyLkNsb3VkV2F0Y2hBbGFybVN0YXRpc3RpYy5BVkVSQUdFLFxuICAgICAgICAgICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgICAgICAgICB1bml0OiB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyLkNsb3VkV2F0Y2hBbGFybVVuaXQuTk9ORVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSwge1xuICAgICAgICAgICAgYWN0aW9uOiB7XG4gICAgICAgICAgICAgIG1hcmtldDogdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlci5JbnN0YW5jZU1hcmtldC5PTl9ERU1BTkQsXG4gICAgICAgICAgICAgIHNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgYWRqdXN0bWVudFR5cGU6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuU2NhbGluZ0FkanVzdG1lbnRUeXBlLkNIQU5HRV9JTl9DQVBBQ0lUWSxcbiAgICAgICAgICAgICAgICBjb29sRG93bjogMSxcbiAgICAgICAgICAgICAgICBzY2FsaW5nQWRqdXN0bWVudDogMVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgZGVzY3JpcHRpb246ICdEZXNjcmlwdGlvbicsXG4gICAgICAgICAgICBuYW1lOiAnTmFtZScsXG4gICAgICAgICAgICB0cmlnZ2VyOiB7XG4gICAgICAgICAgICAgIGNsb3VkV2F0Y2hBbGFybURlZmluaXRpb246IHtcbiAgICAgICAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuQ2xvdWRXYXRjaEFsYXJtQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTixcbiAgICAgICAgICAgICAgICBkaW1lbnNpb25zOiBbe1xuICAgICAgICAgICAgICAgICAga2V5OiAnS2V5JyxcbiAgICAgICAgICAgICAgICAgIHZhbHVlOiAnVmFsdWUnXG4gICAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICAgICAgZXZhbHV0aW9uUGVyaW9kczogMSxcbiAgICAgICAgICAgICAgICBtZXRyaWNOYW1lOiAnTmFtZScsXG4gICAgICAgICAgICAgICAgbmFtZXNwYWNlOiAnTmFtZXNwYWNlJyxcbiAgICAgICAgICAgICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5zZWNvbmRzKHNmbi5EYXRhLm51bWJlckF0KCckLkNsb3VkV2F0Y2hQZXJpb2QnKSksXG4gICAgICAgICAgICAgICAgc3RhdGlzdGljOiB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyLkNsb3VkV2F0Y2hBbGFybVN0YXRpc3RpYy5BVkVSQUdFLFxuICAgICAgICAgICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgICAgICAgICB1bml0OiB0YXNrcy5FbXJDcmVhdGVDbHVzdGVyLkNsb3VkV2F0Y2hBbGFybVVuaXQuTk9ORVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfV1cbiAgICAgICAgfSxcbiAgICAgICAgYmlkUHJpY2U6ICcxJyxcbiAgICAgICAgY29uZmlndXJhdGlvbnM6IFt7XG4gICAgICAgICAgY2xhc3NpZmljYXRpb246ICdDbGFzc2lmaWNhdGlvbicsXG4gICAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgICAgS2V5OiAnVmFsdWUnXG4gICAgICAgICAgfVxuICAgICAgICB9XSxcbiAgICAgICAgZWJzQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIGVic0Jsb2NrRGV2aWNlQ29uZmlnczogW3tcbiAgICAgICAgICAgIHZvbHVtZVNwZWNpZmljYXRpb246IHtcbiAgICAgICAgICAgICAgaW9wczogMSxcbiAgICAgICAgICAgICAgc2l6ZUluR0I6IDEsXG4gICAgICAgICAgICAgIHZvbHVtZVR5cGU6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuRWJzQmxvY2tEZXZpY2VWb2x1bWVUeXBlLlNUQU5EQVJELFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHZvbHVtZXNQZXJJbnN0YW5jZTogMVxuICAgICAgICAgIH1dLFxuICAgICAgICAgIGVic09wdGltaXplZDogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBpbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICBpbnN0YW5jZVJvbGU6IHRhc2tzLkVtckNyZWF0ZUNsdXN0ZXIuSW5zdGFuY2VSb2xlVHlwZS5NQVNURVIsXG4gICAgICAgIGluc3RhbmNlVHlwZTogJ201LnhsYXJnZScsXG4gICAgICAgIG1hcmtldDogdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlci5JbnN0YW5jZU1hcmtldC5PTl9ERU1BTkQsXG4gICAgICAgIG5hbWU6ICdOYW1lJ1xuICAgICAgfV1cbiAgICB9LFxuICAgIGNsdXN0ZXJSb2xlLFxuICAgIG5hbWU6ICdDbHVzdGVyJyxcbiAgICBzZXJ2aWNlUm9sZSxcbiAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5GSVJFX0FORF9GT1JHRVRcbiAgfSkgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjplbGFzdGljbWFwcmVkdWNlOmNyZWF0ZUNsdXN0ZXInLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBOYW1lOiAnQ2x1c3RlcicsXG4gICAgICBJbnN0YW5jZXM6IHtcbiAgICAgICAgS2VlcEpvYkZsb3dBbGl2ZVdoZW5Ob1N0ZXBzOiB0cnVlLFxuICAgICAgICBJbnN0YW5jZUdyb3VwczogW3tcbiAgICAgICAgICBBdXRvU2NhbGluZ1BvbGljeToge1xuICAgICAgICAgICAgQ29uc3RyYWludHM6IHtcbiAgICAgICAgICAgICAgTWF4Q2FwYWNpdHk6IDIsXG4gICAgICAgICAgICAgIE1pbkNhcGFjaXR5OiAxXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUnVsZXM6IFt7XG4gICAgICAgICAgICAgIEFjdGlvbjoge1xuICAgICAgICAgICAgICAgIE1hcmtldDogJ09OX0RFTUFORCcsXG4gICAgICAgICAgICAgICAgU2ltcGxlU2NhbGluZ1BvbGljeUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgICAgIEFkanVzdG1lbnRUeXBlOiAnQ0hBTkdFX0lOX0NBUEFDSVRZJyxcbiAgICAgICAgICAgICAgICAgIENvb2xEb3duOiAxLFxuICAgICAgICAgICAgICAgICAgU2NhbGluZ0FkanVzdG1lbnQ6IDFcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIERlc2NyaXB0aW9uOiAnRGVzY3JpcHRpb24nLFxuICAgICAgICAgICAgICBOYW1lOiAnTmFtZScsXG4gICAgICAgICAgICAgIFRyaWdnZXI6IHtcbiAgICAgICAgICAgICAgICBDbG91ZFdhdGNoQWxhcm1EZWZpbml0aW9uOiB7XG4gICAgICAgICAgICAgICAgICBDb21wYXJpc29uT3BlcmF0b3I6ICdHUkVBVEVSX1RIQU4nLFxuICAgICAgICAgICAgICAgICAgRGltZW5zaW9uczogW3tcbiAgICAgICAgICAgICAgICAgICAgS2V5OiAnS2V5JyxcbiAgICAgICAgICAgICAgICAgICAgVmFsdWU6ICdWYWx1ZSdcbiAgICAgICAgICAgICAgICAgIH1dLFxuICAgICAgICAgICAgICAgICAgRXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgICAgICAgICAgICAgICBNZXRyaWNOYW1lOiAnTmFtZScsXG4gICAgICAgICAgICAgICAgICBOYW1lc3BhY2U6ICdOYW1lc3BhY2UnLFxuICAgICAgICAgICAgICAgICAgUGVyaW9kOiAzMDAsXG4gICAgICAgICAgICAgICAgICBTdGF0aXN0aWM6ICdBVkVSQUdFJyxcbiAgICAgICAgICAgICAgICAgIFRocmVzaG9sZDogMSxcbiAgICAgICAgICAgICAgICAgIFVuaXQ6ICdOT05FJ1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSwge1xuICAgICAgICAgICAgICBBY3Rpb246IHtcbiAgICAgICAgICAgICAgICBNYXJrZXQ6ICdPTl9ERU1BTkQnLFxuICAgICAgICAgICAgICAgIFNpbXBsZVNjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgICBBZGp1c3RtZW50VHlwZTogJ0NIQU5HRV9JTl9DQVBBQ0lUWScsXG4gICAgICAgICAgICAgICAgICBDb29sRG93bjogMSxcbiAgICAgICAgICAgICAgICAgIFNjYWxpbmdBZGp1c3RtZW50OiAxXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBEZXNjcmlwdGlvbjogJ0Rlc2NyaXB0aW9uJyxcbiAgICAgICAgICAgICAgTmFtZTogJ05hbWUnLFxuICAgICAgICAgICAgICBUcmlnZ2VyOiB7XG4gICAgICAgICAgICAgICAgQ2xvdWRXYXRjaEFsYXJtRGVmaW5pdGlvbjoge1xuICAgICAgICAgICAgICAgICAgJ0NvbXBhcmlzb25PcGVyYXRvcic6ICdHUkVBVEVSX1RIQU4nLFxuICAgICAgICAgICAgICAgICAgJ0RpbWVuc2lvbnMnOiBbe1xuICAgICAgICAgICAgICAgICAgICBLZXk6ICdLZXknLFxuICAgICAgICAgICAgICAgICAgICBWYWx1ZTogJ1ZhbHVlJ1xuICAgICAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICAgICAgICAnRXZhbHVhdGlvblBlcmlvZHMnOiAxLFxuICAgICAgICAgICAgICAgICAgJ01ldHJpY05hbWUnOiAnTmFtZScsXG4gICAgICAgICAgICAgICAgICAnTmFtZXNwYWNlJzogJ05hbWVzcGFjZScsXG4gICAgICAgICAgICAgICAgICAnUGVyaW9kLiQnOiAnJC5DbG91ZFdhdGNoUGVyaW9kJyxcbiAgICAgICAgICAgICAgICAgICdTdGF0aXN0aWMnOiAnQVZFUkFHRScsXG4gICAgICAgICAgICAgICAgICAnVGhyZXNob2xkJzogMSxcbiAgICAgICAgICAgICAgICAgICdVbml0JzogJ05PTkUnXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XVxuICAgICAgICAgIH0sXG4gICAgICAgICAgQmlkUHJpY2U6ICcxJyxcbiAgICAgICAgICBDb25maWd1cmF0aW9uczogW3tcbiAgICAgICAgICAgIENsYXNzaWZpY2F0aW9uOiAnQ2xhc3NpZmljYXRpb24nLFxuICAgICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgICBLZXk6ICdWYWx1ZSdcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XSxcbiAgICAgICAgICBFYnNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICBFYnNCbG9ja0RldmljZUNvbmZpZ3M6IFt7XG4gICAgICAgICAgICAgIFZvbHVtZVNwZWNpZmljYXRpb246IHtcbiAgICAgICAgICAgICAgICBJb3BzOiAxLFxuICAgICAgICAgICAgICAgIFNpemVJbkdCOiAxLFxuICAgICAgICAgICAgICAgIFZvbHVtZVR5cGU6ICdzdGFuZGFyZCdcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgVm9sdW1lc1Blckluc3RhbmNlOiAxXG4gICAgICAgICAgICB9XSxcbiAgICAgICAgICAgIEVic09wdGltaXplZDogdHJ1ZVxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgICBJbnN0YW5jZVJvbGU6ICdNQVNURVInLFxuICAgICAgICAgIEluc3RhbmNlVHlwZTogJ201LnhsYXJnZScsXG4gICAgICAgICAgTWFya2V0OiAnT05fREVNQU5EJyxcbiAgICAgICAgICBOYW1lOiAnTmFtZSdcbiAgICAgICAgfV1cbiAgICAgIH0sXG4gICAgICBWaXNpYmxlVG9BbGxVc2VyczogdHJ1ZSxcbiAgICAgIEpvYkZsb3dSb2xlOiB7XG4gICAgICAgIFJlZjogJ0NsdXN0ZXJSb2xlRDlDQTc0NzEnLFxuICAgICAgfSxcbiAgICAgIFNlcnZpY2VSb2xlOiB7XG4gICAgICAgIFJlZjogJ1NlcnZpY2VSb2xlNDI4OEIxOTInXG4gICAgICB9LFxuICAgICAgQXV0b1NjYWxpbmdSb2xlOiB7XG4gICAgICAgIFJlZjogJ0F1dG9TY2FsaW5nUm9sZTAxNUFEQTBBJ1xuICAgICAgfVxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ1Rhc2sgdGhyb3dzIGlmIFdBSVRfRk9SX1RBU0tfVE9LRU4gaXMgc3VwcGxpZWQgYXMgc2VydmljZSBpbnRlZ3JhdGlvbiBwYXR0ZXJuJywgKCkgPT4ge1xuICBleHBlY3QoKCkgPT4ge1xuICAgIG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiBuZXcgdGFza3MuRW1yQ3JlYXRlQ2x1c3Rlcih7XG4gICAgICAgIGluc3RhbmNlczoge30sXG4gICAgICAgIGNsdXN0ZXJSb2xlLFxuICAgICAgICBuYW1lOiAnQ2x1c3RlcicsXG4gICAgICAgIHNlcnZpY2VSb2xlLFxuICAgICAgICBhdXRvU2NhbGluZ1JvbGUsXG4gICAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTlxuICAgICAgfSlcbiAgICB9KTtcbiAgfSkudG9UaHJvdygvSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46IFdBSVRfRk9SX1RBU0tfVE9LRU4gaXMgbm90IHN1cHBvcnRlZCB0byBjYWxsIENyZWF0ZUNsdXN0ZXIuL2kpO1xufSk7Il19