"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Modify an InstanceGroup with static ClusterId, InstanceGroupName, and InstanceGroup', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                configurations: [{
                        classification: 'Classification',
                        properties: {
                            Key: 'Value'
                        }
                    }],
                eC2InstanceIdsToTerminate: ['InstanceToTerminate'],
                instanceCount: 1,
                shrinkPolicy: {
                    decommissionTimeout: cdk.Duration.seconds(1),
                    instanceResizePolicy: {
                        instanceTerminationTimeout: cdk.Duration.seconds(1),
                        instancesToProtect: ['InstanceToProtect'],
                        instancesToTerminate: ['InstanceToTerminate']
                    }
                }
            }
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                Configurations: [{
                        Classification: 'Classification',
                        Properties: {
                            Key: 'Value'
                        }
                    }],
                EC2InstanceIdsToTerminate: ['InstanceToTerminate'],
                InstanceCount: 1,
                ShrinkPolicy: {
                    DecommissionTimeout: 1,
                    InstanceResizePolicy: {
                        InstanceTerminationTimeout: 1,
                        InstancesToProtect: ['InstanceToProtect'],
                        InstancesToTerminate: ['InstanceToTerminate']
                    }
                }
            }
        }
    });
});
test('Modify an InstanceGroup with ClusterId from payload and static InstanceGroupName and InstanceGroupConfiguration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: sfn.Data.stringAt('$.ClusterId'),
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                instanceCount: 1
            }
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
            'InstanceGroupName': 'InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1
            }
        }
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupConfigurateion and InstanceGroupName from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: sfn.Data.stringAt('$.InstanceGroupName'),
            instanceGroup: {
                instanceCount: 1
            }
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId': 'ClusterId',
            'InstanceGroupName.$': '$.InstanceGroupName',
            'InstanceGroup': {
                InstanceCount: 1
            }
        },
    });
});
test('Modify an InstanceGroup with static ClusterId and InstanceGroupName and InstanceCount from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrModifyInstanceGroupByName({
            clusterId: 'ClusterId',
            instanceGroupName: 'InstanceGroupName',
            instanceGroup: {
                instanceCount: sfn.Data.numberAt('$.InstanceCount')
            }
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:modifyInstanceGroupByName',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            InstanceGroupName: 'InstanceGroupName',
            InstanceGroup: {
                'InstanceCount.$': '$.InstanceCount'
            }
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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