# AUTOGENERATED! DO NOT EDIT! File to edit: io-core.ipynb (unless otherwise specified).

__all__ = ['BUF_SIZE', 'DEFAULT_IMAGE_EXTENSIONS', 'FOLDER_FILTER', 'logger', 'scan_files', 'scan_path',
           'delete_folder', 'create_folder', 'get_file_sha', 'assign_filename_suffix', 'strip_path']


# Cell
import hashlib
import logging
import shutil
import os
from os import listdir, makedirs
from os.path import join, isfile, isdir, splitext


# Cell
BUF_SIZE = 65536  # lets read stuff in 64kb chunks!
DEFAULT_IMAGE_EXTENSIONS = ['.jpg']
FOLDER_FILTER = ['.DS_Store']


# Cell
logger = logging.getLogger(__name__)


# Cell
def scan_files(folder, file_extensions=None):
    """
    Scan the folder for files and filter files by file extension.
    If the optional `file_extension` is not set, **.jpg** as file extension is used by default.
    `folder`: the folder to scan for files
    `file_extensions`: the allowed file extensions
    return: the file path list
    """
    if file_extensions is None:
        file_extensions = DEFAULT_IMAGE_EXTENSIONS

    results = []
    if isdir(folder):
        files = listdir(folder)
        for file in files:
            file_path = join(folder, file)
            filename, file_extension = splitext(file_path)
            if isfile(file_path) and (file_extension in file_extensions):
                results.append(file_path)
            elif isdir(file_path):
                results.extend(scan_files(file_path, file_extensions))
    return results


# Cell
def scan_path(path):
    """
    Scan a folder and sub-folders until the file level.
    `path`: the current directory to parse for folders
    return: a list of folders prefixed with base directory
    """
    has_sub_folders = False
    results = []
    names = listdir(path)
    for name in names:
        sub_folder_path = join(path, name)
        if isdir(sub_folder_path) and name not in FOLDER_FILTER:
            has_sub_folders = True
            results = results + scan_path(sub_folder_path)
    if not has_sub_folders:
        results.append(path)
    return results


# Cell
def delete_folder(path):
    """
    Deletes a Folder, if exist.
    `path`: the folder path including the folder name
    return: the created folder path
    """
    if isdir(path):
        shutil.rmtree(path)
    return path


# Cell
def create_folder(path, clear=False):
    """
    Creates a Folder, if not exist.
    `path`: the folder path including the folder name
    `clear`: whether or not the created folder should be empty
    return: the created folder path
    """

    if clear:
        # remove previous data-set folder, if exist to clear the contents
        delete_folder(path)

    if not isdir(path):
        makedirs(path)

    return path


# Cell
def get_file_sha(file_path):
    """
    Calculates the SHA1 of a given file.
    `file_path`: the file path
    return: the calculated SHA1 as hex
    """
    result = None
    if isfile(file_path):
        sha1 = hashlib.sha1()
        with open(file_path, 'rb') as f:
            while True:
                data = f.read(BUF_SIZE)
                if not data:
                    break
                sha1.update(data)
            result = sha1.hexdigest()
    return result


# Cell
def assign_filename_suffix(file_path, suffix=None):
    """
    Assign a filename suffix to a given file_path. (e.g <path>/filename_<suffix>.txt)
    `file_path`: the file path to suffix
    `suffix`: the suffix
    return: the suffixed file path
    """
    if suffix is None:
        return file_path
    path, ext = splitext(file_path)
    return "{}_{}{}".format(path, suffix, ext)


# Cell
def strip_path(file_path, strip=os.sep):
    """
    Strip trailing separators from a path
    `file_path`: the file path to strip
    return: the striped file path
    """
    return file_path.rstrip(os.sep)
