# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/Docstring_helpers.ipynb (unless otherwise specified).

__all__ = ["run_examples_from_docstring"]

# Cell

from typing import *
import sys
import os

import re

import textwrap
from subprocess import run, CalledProcessError  # nosec: B404
from tempfile import TemporaryDirectory
from pathlib import Path

from rich import print
from rich.console import Group
from rich.panel import Panel

# Cell


def _extract_examples_from_docstring(o: Any) -> List[str]:
    try:
        import griffe
    except:
        raise Exception(
            "This function should only be used for testing where griffe package is installed."
        )

    if o.__doc__ is None:
        raise ValueError(f"{o.__name__}.__doc__ = {o.__doc__}")
    sections = griffe.docstrings.parse(
        griffe.dataclasses.Docstring(o.__doc__), griffe.docstrings.Parser.google
    )

    def _trim_if_needed(s: str) -> str:
        if s.startswith("```") and s.endswith("```"):
            s = "".join(s.splitlines(keepends=True)[1:-1])
        else:
            s = s + "\n"
        return s

    examples = [
        _trim_if_needed(section.value.description)  # type: ignore
        for section in sections
        if section.kind.value == "admonition" and section.value.annotation == "example"  # type: ignore
    ]

    return examples


# Cell


def _get_keywords(examples: List[str]) -> List[str]:
    keywords: List[str] = sum(
        [
            [x[9:-1] for x in re.findall("{fill in \w+}", example)]
            for example in examples
        ],
        start=[],
    )

    return keywords


# Cell


def _replace_keywords(examples: List[str], **kwargs) -> List[str]:
    keywords = _get_keywords(examples)

    if set(keywords) > set(kwargs.keys()):
        raise ValueError(f"{set(keywords)} > {set(kwargs.keys())}")

    for keyword in keywords:
        examples = [
            example.replace("{fill in " + keyword + "}", kwargs[keyword])
            for example in examples
        ]

    return examples


# Cell


def run_examples_from_docstring(o: Any, **kwargs):
    """Runs example from a docstring

    Parses docstring of an objects looking for examples. The examples are then saved into files and executed
    in a separate process.

    Note:
        Execution context is not the same as the one in the notebook because we want examples to work from
        user code. Make sure you compiled the library prior to executing the examples, otherwise you might
        be running them agains an old version of the library.

    Args:
        o: an object, typically a function or a class, for which docstring is being parsed for examples
        **kwargs: arguments use to replace "{fill in **param**}" in docstring with the actual values when running examples

    Raises:
        ValueError: if some params are missing from the **kwargs**
        RuntimeException: if example fails

    Example:
        ```python
        from .helpers import run_examples_from_docstring

        def f():
            '''
            Example:
                print("Hello {fill in name}!")
                print("Goodbye {fill in other_name}!")
            '''
            pass


        run_examples_from_docstring(f, name="John", other_name="Jane")
        ```
    """
    examples = _extract_examples_from_docstring(o)
    executable_examples = _replace_keywords(examples, **kwargs)
    for example, executable_example in zip(examples, executable_examples):
        with TemporaryDirectory() as d:
            cmd_path = (Path(d) / "example.py").absolute()
            with open(cmd_path, "w") as f:
                f.write(executable_example)
            process = run(  # nosec: B603
                [sys.executable, str(cmd_path)], capture_output=True, text=True
            )
            panel_group = Group(
                "Example:\n" + textwrap.indent(example, " " * 4),
                Panel(process.stdout, title="stdout"),
                Panel(process.stderr, title="stderr"),
            )
            print(Panel(panel_group))
            if process.returncode != 0:
                raise RuntimeError(process.stderr)
