import numpy as np


class ProtoElectronicDensity(object):

    def __init__(self, geometry):
        self._name = geometry.name
        self._sym_l = geometry.get_symbols()
        self._atm_l = geometry.get_positions()
        self._atomic_values = {
            "Titles": ['Atomic Number', 'Atomic weight', 'a1', 'w1', 'a2', 'w2', 'a3', 'w3', 'a4', 'w4', 'a5', 'w5',
                       'a6', 'w6', 'a7', 'w7',
                       'core electrons', 'wv1', 'wv2', 'wv3', 'wv4', 'wv5'],
            "H": [1, 1, 0.440920529854, 0.446848399957, 0.148523571410, 0.260661566642, 0.148522754226, 0.152052580541,
                  0.830533399065, 0.0886973386489, 1.55545886958, 0.0502242590213, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0],
            "He": [2, 4, 0.705128316152, 0.828896903794, 4.60226167419, 0.171103096206, 1, 0,
                   1, 0, 1, 0, 1, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0],
            "Li": [3, 7, 2.42959379596, 0.496667117048, 0.0931481410958, 0.428950159018, 14.6896849265, 0.0743827239327,
                   1, 0, 1, 0, 1, 0, 1, 0, 2, 0, 1 / 3, 0, 0, 0, 0, 0],
            "Be": [4, 9, 4.70365863654, 0.375887797196, 0.133243850840, 0.569822506390, 27.9324424462, 0.0542896964145,
                   1, 0, 1, 0, 1, 0, 1, 0, 2, 0, 1 / 2, 0, 0, 0, 0, 0],
            "B": [5, 11, 7.69672609374, 0.300184331682, 0.209275915739, 0.657081391054, 45.2385931510, 0.0427342772638,
                  1, 0, 1, 0, 1, 0, 1, 0, 2, 0, 0.6, 0, 0, 0, 0, 0],
            "C": [6, 12, 0.356462048751, 0.481662583478, 6.74589838364, 0.165621922856, 20.0053616858, 0.126630091695,
                  97.0508229097, 0.0169326970446, 0.122734548099, 0.209152704926, 1, 0, 1, 0, 2, 4 / 9, 0, 0, 0, 2 / 9,
                  0, 0],
            "N": [7, 14, 0.558690868187, 0.428044393139, 26.3091078048, 0.119405132277, 132.716385327, 0.0149293216485,
                  0.213410401209, 0.306224338493, 8.67117752141, 0.131396814443, 1, 0, 1, 0, 2, 0.396825, 0, 0, 0.31746,
                  0, 0, 0],
            "O": [8, 16, 0.764949764007, 0.462949674244, 36.3980482778, 0.0972634518752, 178.279439955, 0.0124323059053,
                  0.258345655165, 0.306747136544, 12.1009178518, 0.120607431432, 1, 0, 1, 0, 2, 0.416666, 0, 0, 1 / 3,
                  0, 0, 0],
            "F": [9, 19, 0.953231108644, 0.521113130773, 0.288290993131, 0.272227674853, 14.3826569656, 0.102468771030,
                  44.6764228163, 0.0927110692516, 223.415128313, 0.0114793540601, 1, 0, 1, 0, 2, 0, 7 / 9, 0, 0, 0, 0,
                  0],
            "Ne": [10, 20, 0.954374750854E+00, 0.828487153815E+00, 0.182218394930E+03, 0.229002856433E-01,
                   0.320674536956E+02, 0.148612560541E+00,
                   1, 0, 1, 0, 1, 0, 1, 0, 2, 0, 0.8, 0, 0, 0, 0, 0],
            "Na": [11, 23, 0.108854352851E+01, 0.825554838994E+00, 0.242558589286E+02, 0.112378067204E+00,
                   0.933932018858E+02, 0.577146198741E-01,
                   0.529038388855E+03, 0.435247392718E-02, 1, 0, 1, 0, 1, 0, 10, 0, 0, 0, 0, 0, 0, 0],
            "Mg": [12, 24, 0.218569903740E+01, 0.520863441795E+00, 0.301497473367E+00, 0.342512252791E+00,
                   0.490777626016E+02, 0.117633186674E+00,
                   0.267399143111E+03, 0.189911187397E-01, 1, 0, 1, 0, 1, 0, 10, 0, 0.166666, 0, 0, 0, 0, 0],
            "Al": [13, 27, 0.261388211922E+01, 0.523097525672E+00, 0.214149737434E+00, 0.351056420809E+00,
                   0.588955805607E+02, 0.109087878782E+00,
                   0.323698543235E+03, 0.167581747358E-01, 1, 0, 1, 0, 1, 0, 10, 0, 0.2307, 0, 0, 0, 0, 0],
            "Si": [14, 28, 0.316910421369E+01, .499707795463E+00, 0.204138192913E+00, 0.384254907417E+00,
                   0.694191161972E+02, 0.100765099351E+00,
                   0.381267247680E+03, 0.152721977687E-01, 1, 0, 1, 0, 1, 0, 10, 0, 0.2857, 0, 0, 0, 0, 0],
            "P": [15, 31, 0.361451144211E+01, 0.478831971088E+00, 0.203232522123E+00, 0.407091911330E+00,
                  0.642302360554E+02, 0.852748018285E-01,
                  0.241672185606E+03, 0.270948640067E-01, 0.140780726438E+04, 0.170645174514E-02, 1, 0, 1, 0, 10, 0,
                  1 / 3, 0, 0,
                  0, 0, 0],
            "S": [16, 32, 0.432801783704E+01, 0.450261917524E+00, 0.234919053755E+00, 0.443771174997E+00,
                  0.748183927869E+02, 0.797906413585E-01,
                  0.281418786843E+03, 0.246258047332E-01, 0.163076922864E+04, 0.155046138826E-02, 1, 0, 1, 0, 10, 0,
                  0.3375, 0, 0,
                  0.0375, 0, 0],
            "Cl": [17, 35, 0.512735107009E+01, 0.423533567847E+00, 0.274578692637E+00, 0.477496012097E+00,
                   0.856745844595E+02, 0.745409754409E-01, 0.319382224252E+03, 0.229544216482E-01, 1, 0, 1, 0, 1, 0, 10,
                   0, 0.4117,
                   0, 0, 0, 0, 0],
            "Ar": [18, 40, 0.630553641999E+01, 0.395500421392E+00, 0.334230771034E+00, 0.516912377231E+00,
                   0.121409292169E+03, 0.764819814552E-01,
                   0.663530875228E+03, 0.111052199211E-01, 1, 0, 1, 0, 1, 0, 10, 0, 4 / 9, 0, 0, 0, 0, 0],
            "K": [19, 39, 0.694111796576E+01, 0.378530044067E+00, 0.365552736285E+00, 0.533867111479E+00,
                  0.109272225958E+03, 0.660833543593E-01, 0.407346696568E+03, 0.202680540628E-01, 0.238410182916E+04,
                  0.125143603162E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Ca": [20, 40, 0.797169547630E+01, 0.359658109862E+00, 0.413114112614E+00, 0.557723102982E+00,
                   0.122588534955E+03, 0.621472337519E-01, 0.448342421376E+03, 0.192029093243E-01,
                   0.253816047853E+04, 0.126864408198E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Sc": [21, 45, 0.466098813111E+00, 0.580334271036E+00, 0.906753933931E+01, 0.341591384709E+00,
                   0.137924814616E+03, 0.591842967216E-01, 0.505591195738E+03, 0.177252545137E-01, 0.285677684998E+04,
                   0.116479301928E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Ti": [22, 48, 0.102030174401E+02, 0.325664004155E+00,
                   0.524119692331E+00, 0.600355878378E+00,
                   0.577967745935E+03, 0.160952910243E-01,
                   0.155127930139E+03, 0.568720663511E-01,
                   0.330393446417E+04, 0.101276009343E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "V": [23, 51, 0.587547598240E+00, 0.618279973747E+00,
                  0.113466698779E+02, 0.310770916577E+00,
                  0.166514895258E+03, 0.535397670350E-01,
                  0.603706148324E+03, 0.163180045782E-01,
                  0.339141857941E+04, 0.109133806241E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Cr": [24, 52, 0.657410201819E+00, 0.635212542832E+00,
                   0.125992022693E+02, 0.297076751089E+00,
                   0.183482934345E+03, 0.513494665838E-01,
                   0.667647053481E+03, 0.153471379380E-01,
                   0.376040644655E+04, 0.101410155738E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Mn": [25, 55, 0.731321183367E+00, 0.650256061721E+00,
                   0.138535780059E+02, 0.284649136425E+00,
                   0.197164905449E+03, 0.488499891915E-01,
                   0.707796493281E+03, 0.152125477571E-01,
                   0.396092243020E+04, 0.103226490660E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Fe": [26, 56, 0.151905423691E+02, 0.273660307790E+00,
                   0.806774579448E+00, 0.663994375191E+00,
                   0.797460211523E+03, 0.139959064655E-01,
                   0.217247761350E+03, 0.474618958503E-01,
                   0.457128545613E+04, 0.887514702503E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Co": [27, 59, 0.891910882476E+00, 0.677549460524E+00,
                   0.166294989201E+02, 0.262512271781E+00,
                   0.233921163867E+03, 0.453144213822E-01,
                   0.844940514841E+03, 0.137118031273E-01,
                   0.475087405721E+04, 0.912043186030E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Ni": [28, 59, 0.979482912947E+00, 0.689643509010E+00,
                   0.180956647897E+02, 0.252657778071E+00,
                   0.252881980624E+03, 0.437002946380E-01,
                   0.914887465474E+03, 0.131310044008E-01,
                   0.515468010086E+04, 0.867413880922E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Cu": [29, 64, 0.107209023218E+01, 0.701001714747E+00,
                   0.196289219872E+02, 0.243391832053E+00,
                   0.557928484261E+04, 0.825683041292E-03,
                   0.272777278563E+03, 0.422024493729E-01,
                   0.988563166715E+03, 0.125783207846E-01, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Zn": [30, 65, 0.116896633820E+01, 0.711390811658E+00,
                   0.211875063466E+02, 0.234878951870E+00,
                   0.292254770755E+03, 0.407506520837E-01,
                   0.105770323275E+04, 0.121783088582E-01,
                   0.596325228169E+04, 0.801275529490E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Ga": [31, 70, 0.127071328297E+01, 0.720501145467E+00,
                   0.228138862903E+02, 0.227540084557E+00,
                   0.312656278427E+03, 0.393903642201E-01,
                   0.113116306647E+04, 0.117959832057E-01,
                   0.639147101149E+04, 0.772422550138E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Ge": [32, 73, 0.137209377843E+01, 0.728978584017E+00,
                   0.245081692950E+02, 0.220729259029E+00,
                   0.333684785672E+03, 0.380925994677E-01,
                   0.120491094100E+04, 0.114485039259E-01,
                   0.680532805621E+04, 0.751053562037E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "As": [33, 75, 0.147332540718E+01, 0.736733293192E+00,
                   0.262478471292E+02, 0.214520302851E+00,
                   0.355134291546E+03, 0.368813778446E-01,
                   0.128002233514E+04, 0.111325591472E-01,
                   0.722234954865E+04, 0.732466965612E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Se": [34, 79, 0.157327466926E+01, 0.743804347197E+00,
                   0.280273779748E+02, 0.208887800103E+00,
                   0.376994044571E+03, 0.357480375904E-01,
                   0.135706043550E+04, 0.108462204650E-01,
                   0.766180781530E+04, 0.713594643572E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Br": [35, 80, 0.167194095812E+01, 0.750349123275E+00,
                   0.298572576521E+02, 0.203697409473E+00,
                   0.399453156703E+03, 0.346783255074E-01,
                   0.143533547208E+04, 0.105768861264E-01,
                   0.809523771040E+04, 0.698255620929E-03, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Kr": [36, 84, 0.277799657992E+01, 0.507053494416E+00,
                   0.234238179440E+00, 0.269316613444E+00,
                   0.355329010713E+02, 0.182351688078E+00,
                   0.541765502309E+03, 0.363567177024E-01,
                   0.292393327269E+04, 0.492148614584E-02, 1, 0, 1, 0, 18, 0, 0, 0, 0, 0, 0, 0],
            "Rb": [37, 86, 0.300515657300E+01, 0.494200116307E+00,
                   0.367410428429E+02, 0.177307415308E+00,
                   0.482612727030E+03, 0.329330972899E-01,
                   0.178116643401E+04, 0.858250476550E-02,
                   0.101265828568E+05, 0.534113016787E-03,
                   0.253226121174E+00, 0.286442753312E+00, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Sr": [38, 88, 0.324183215251E+01, 0.485062605671E+00,
                   0.386851331329E+02, 0.172496171463E+00,
                   0.480619491974E+03, 0.310951240660E-01,
                   0.167641318735E+04, 0.976008997776E-02,
                   0.932419241767E+04, 0.674588373583E-03,
                   0.265325600846E+00, 0.300911420448E+00, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Y": [39, 89, 0.352614893447E+01, 0.474223189855E+00,
                  0.412101321691E+02, 0.167327113465E+00,
                  0.508295795078E+03, 0.302451199854E-01,
                  0.177236603760E+04, 0.947658475371E-02,
                  0.988381767197E+04, 0.651557045109E-03,
                  0.280203985479E+00, 0.318076434896E+00, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Zr": [40, 91, 0.384381909852E+01, 0.463420113232E+00,
                   0.440985587415E+02, 0.161944808042E+00,
                   0.549992747467E+03, 0.297308765825E-01,
                   0.194435667035E+04, 0.868859971227E-02,
                   0.108202688202E+05, 0.587535081154E-03,
                   0.300233969681E+00, 0.335628067348E+00, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Nb": [41, 93, 0.414073506876E+01, 0.454419943549E+00,
                   0.315857327618E+00, 0.349991182588E+00,
                   0.466915336374E+02, 0.157656510847E+00,
                   0.123478251069E+05, 0.497866920883E-03,
                   0.587514850202E+03, 0.293533482117E-01,
                   0.213868967282E+04, 0.808114788304E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Mo": [42, 96, 0.448646806752E+01, 0.443460301630E+00,
                   0.341310114582E+00, 0.366657304295E+00,
                   0.623551995389E+03, 0.287037196625E-01,
                   0.228430692692E+04, 0.770415305665E-02,
                   0.132349990872E+05, 0.467691149047E-03,
                   0.495770936032E+02, 0.153006830206E+00, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Tc": [43, 99, 0.483270614068E+01, 0.433509081814E+00,
                   0.367506321048E+00, 0.381686120414E+00,
                   0.524074522044E+02, 0.148792289073E+00,
                   0.135973756473E+05, 0.474582997313E-03,
                   0.651053076880E+03, 0.278773308867E-01,
                   0.236153332139E+04, 0.766059481566E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Ru": [44, 101, 0.521240141600E+01, 0.422992661868E+00,
                   0.395841321465E+00, 0.397226356684E+00,
                   0.553161508570E+02, 0.144623336040E+00,
                   0.141451592474E+05, 0.469783120135E-03,
                   0.681993977054E+03, 0.271702659570E-01,
                   0.246399518743E+04, 0.751759633095E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Rh": [45, 103, 0.568491786936E+01, 0.411630819413E+00,
                   0.433662098119E+00, 0.415093792398E+00,
                   0.594941112089E+02, 0.139648400081E+00,
                   0.171579793601E+05, 0.345004651170E-03,
                   0.764278330890E+03, 0.271385597513E-01,
                   0.293134558925E+04, 0.614342370433E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Pd": [46, 106, 0.610639957306E+01, 0.401901741827E+00,
                   0.466645863035E+00, 0.429406313676E+00,
                   0.627193511652E+02, 0.135865732218E+00,
                   0.179678734478E+05, 0.336184295385E-03,
                   0.801905469572E+03, 0.265166562459E-01,
                   0.307511357546E+04, 0.597337173705E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Ag": [47, 108, 0.655758425582E+01, 0.392450147442E+00,
                   0.501988530808E+00, 0.443442645166E+00,
                   0.662197938048E+02, 0.132153497377E+00,
                   0.194061803321E+05, 0.308453205790E-03,
                   0.849686947490E+03, 0.260165999068E-01,
                   0.330141297982E+04, 0.562865690158E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Cd": [48, 112, 0.690845551858E+01, 0.385007600800E+00,
                   0.530007543703E+00, 0.453286390565E+00,
                   0.683084218481E+02, 0.129861670758E+00,
                   0.177086385346E+05, 0.392696626308E-03,
                   0.837206002816E+03, 0.249741193751E-01,
                   0.306845733912E+04, 0.647752187546E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "In": [49, 119, 0.750781310511E+01, 0.372729855486E+00,
                   0.580721361500E+00, 0.471241681099E+00,
                   0.732985640197E+02, 0.125412118426E+00,
                   0.201456192992E+05, 0.323841272863E-03,
                   0.916144447221E+03, 0.246636830545E-01,
                   0.347261968516E+04, 0.562882066132E-02, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Sn": [50, 119, 0.796568540561E+01, 0.364746802230E+00,
                   0.616817581645E+00, 0.482313943827E+00,
                   0.761102501554E+02, 0.122531268402E+00,
                   0.904445946373E+03, 0.234944613551E-01,
                   0.319357673542E+04, 0.647996798400E-02,
                   0.177560004677E+05, 0.433556201126E-03, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Sb": [51, 122, 0.850545618822E+01, 0.357389066020E+00,
                   0.654970567275E+00, 0.493985677829E+00,
                   0.805998485533E+02, 0.119212425962E+00,
                   0.983999714290E+03, 0.234358618009E-01,
                   0.363467693596E+04, 0.563169797318E-02,
                   0.206270032315E+05, 0.345270416060E-03, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Te": [52, 128, 0.899201587016E+01, 0.350870476560E+00,
                   0.688319291922E+00, 0.503606883802E+00,
                   0.840571534227E+02, 0.116567410645E+00,
                   0.100648588105E+04, 0.227615786524E-01,
                   0.363137563961E+04, 0.581963482291E-02,
                   0.203507326323E+05, 0.374015517550E-03, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "I": [53, 127, 0.952001650649E+01, 0.344614693286E+00,
                  0.722299115424E+00, 0.513296034215E+00,
                  0.881940625179E+02, 0.113815833668E+00,
                  0.106025797623E+04, 0.224156168634E-01,
                  0.387348255184E+04, 0.551256371806E-02,
                  0.218475287056E+05, 0.345258250321E-03, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Xe": [54, 132, 0.100184526282E+02, 0.339077078836E+00,
                   0.752782273579E+00, 0.521585841014E+00,
                   0.917594172416E+02, 0.111479465012E+00,
                   0.108291560348E+04, 0.217916640215E-01,
                   0.387000349766E+04, 0.569382108772E-02,
                   0.215938385297E+05, 0.372130029500E-03, 1, 0, 36, 0, 0, 0, 0, 0, 0, 0],
            "Cs": [55, 133, 0.107600686616E+02, 0.323256974444E+00,
                   0.108538339071E+01, 0.388537701954E+00,
                   0.952075874992E+02, 0.108975159898E+00,
                   0.109022112801E+04, 0.210782719805E-01,
                   0.378788147706E+04, 0.607859429957E-02,
                   0.116280226587E+00, 0.151660492685E+00,
                   0.210612848279E+05, 0.412804738448E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Ba": [56, 0.113874138423E+02, 0.316474134924E+00,
                   0.116973037595E+01, 0.384412166876E+00,
                   0.996283885914E+02, 0.106428447057E+00,
                   0.113985897262E+04, 0.207127969371E-01,
                   0.396923921699E+04, 0.587283909533E-02,
                   0.123786152795E+00, 0.165701288865E+00,
                   0.220378368040E+05, 0.398326246088E-03, 137, 54, 0, 0, 0, 0, 0, 0, 0],
            "La": [57, 139, 0.120828207720E+02, 0.309064401852E+00,
                   0.124872445397E+01, 0.388430012133E+00,
                   0.104940904147E+03, 0.103691599425E+00,
                   0.122815318768E+04, 0.206307825110E-01,
                   0.441439602294E+04, 0.521292058051E-02,
                   0.139229979475E+00, 0.172632671261E+00,
                   0.246409443666E+05, 0.337612239078E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Ce": [58, 140, 0.126074356414E+02, 0.303578555423E+00,
                   0.130009108144E+01, 0.395725826555E+00,
                   0.108341913269E+03, 0.101988842541E+00,
                   0.123919660512E+04, 0.200301611217E-01,
                   0.433803089176E+04, 0.551563376519E-02,
                   0.145526379437E+00, 0.172788755239E+00,
                   0.240493005528E+05, 0.372225353994E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Pr": [59, 141, 0.133211443381E+02, 0.297138278170E+00,
                   0.137533091467E+01, 0.402323382541E+00,
                   0.114111716886E+03, 0.995248622495E-01,
                   0.135027040830E+04, 0.200974586922E-01,
                   0.498705038404E+04, 0.469983877345E-02,
                   0.157272820844E+00, 0.175928672741E+00,
                   0.282517753513E+05, 0.287506833472E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Nd": [60, 144, 0.137752308176E+02, 0.292506190767E+00,
                   0.141330881722E+01, 0.411956446235E+00,
                   0.116604696868E+03, 0.983728807832E-01,
                   0.132141215591E+04, 0.192862148959E-01,
                   0.158881546076E+00, 0.172099484470E+00,
                   0.254797958642E+05, 0.367609391365E-03,
                   0.459811935950E+04, 0.541117328090E-02, 54, 0, 0, 0, 0, 0, 0, 0],
            "Pm": [61, 147, 0.145799609814E+02, 0.285640335955E+00,
                   0.150902378557E+01, 0.416029681409E+00,
                   0.122777031201E+03, 0.958659907042E-01,
                   0.143150145814E+04, 0.193090670958E-01,
                   0.520400682877E+04, 0.469117412248E-02,
                   0.176525867143E+00, 0.178167208940E+00,
                   0.292016815251E+05, 0.296541772922E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Sm": [62, 150, 0.144282400612E+02, 0.295273535183E+00,
                   0.108871063076E+01, 0.584554066394E+00,
                   0.124600880905E+03, 0.959452274089E-01,
                   0.140538802716E+04, 0.183068299312E-01,
                   0.415781198659E+04, 0.435405415769E-02,
                   0.763112852316E+04, 0.131750347924E-02,
                   0.322032005282E+05, 0.248783444323E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Eu": [63, 152, 0.158488929805E+02, 0.275193873490E+00,
                   0.164110678747E+01, 0.431491811727E+00,
                   0.131523592391E+03, 0.926217717010E-01,
                   0.151144298200E+04, 0.185679677554E-01,
                   0.541119833316E+04, 0.470563879645E-02,
                   0.192655639846E+00, 0.177113596265E+00,
                   0.301782741901E+05, 0.305340263378E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Gd": [64, 157, 0.162896769350E+02, 0.270908575628E+00,
                   0.169168776220E+01, 0.438488647613E+00,
                   0.133271620910E+03, 0.918418490610E-01,
                   0.143311587718E+04, 0.175040015495E-01,
                   0.474205588397E+04, 0.581808719649E-02,
                   0.197491460221E+00, 0.175009783062E+00,
                   0.258570223859E+05, 0.429055850124E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Tb": [65, 159, 0.170335821580E+02, 0.265797806533E+00,
                   0.177454151545E+01, 0.444050868158E+00,
                   0.139015589652E+03, 0.900600089703E-01,
                   0.153489399292E+04, 0.175913280766E-01,
                   0.209510786460E+00, 0.176935480771E+00,
                   0.290094654222E+05, 0.361229156000E-03,
                   0.524999010897E+04, 0.520327827771E-02, 54, 0, 0, 0, 0, 0, 0, 0],
            "Dy": [66, 163, 0.186198973917E+01, 0.448881826938E+00,
                   0.177600992249E+02, 0.260493400086E+00,
                   0.143781381023E+03, 0.884765259627E-01,
                   0.157405526285E+04, 0.172260274288E-01,
                   0.533008389347E+04, 0.522069689725E-02,
                   0.292384669858E+05, 0.370926192385E-03,
                   0.223598825409E+00, 0.179330595624E+00, 54, 0, 0, 0, 0, 0, 0, 0],
            "Ho": [67, 165, 0.183636521083E+02, 0.256331485479E+00,
                   0.192883543158E+01, 0.455604927172E+00,
                   0.147632945959E+03, 0.872772076152E-01,
                   0.157428413715E+04, 0.166433029687E-01,
                   0.517712336362E+04, 0.557600070800E-02,
                   0.230413831853E+00, 0.178150313363E+00,
                   0.281019349564E+05, 0.416762665312E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Er": [68, 167, 0.191667587039E+02, 0.251736872705E+00,
                   0.201674066729E+01, 0.461057622289E+00,
                   0.154097699093E+03, 0.856515596165E-01,
                   0.171271358082E+04, 0.169308977225E-01,
                   0.594646309261E+04, 0.474923766933E-02,
                   0.242816992893E+00, 0.179553091831E+00,
                   0.330518544241E+05, 0.320718126922E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Tm": [69, 169, 0.212369754429E+01, 0.463552177037E+00,
                   0.263695766186E+00, 0.184137082171E+00,
                   0.158948442721E+03, 0.841655743265E-01,
                   0.199662872176E+02, 0.246399799475E+00,
                   0.173745870652E+04, 0.165039105861E-01,
                   0.591305635631E+04, 0.489768303199E-02,
                   0.325390842763E+05, 0.343755428404E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Yb": [70, 173, 0.219658574537E+01, 0.470457682977E+00,
                   0.206034549746E+02, 0.242455477416E+00,
                   0.162863921808E+03, 0.830924703271E-01,
                   0.172698786161E+04, 0.158968597402E-01,
                   0.566744089491E+04, 0.531367995218E-02,
                   0.269543914308E+00, 0.182385419686E+00,
                   0.307112988656E+05, 0.398409694832E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Lu": [71, 175, 0.202059433031E+02, 0.254215506830E+00,
                   0.162465587704E+01, 0.641481753576E+00,
                   0.167774035672E+03, 0.833242387992E-01,
                   0.190438484124E+04, 0.163688407276E-01,
                   0.665866355734E+04, 0.427867400028E-02,
                   0.623943195771E+05, 0.587997591350E-04,
                   0.285701380130E+05, 0.272186308144E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Hf": [72, 179, 0.220820490222E+02, 0.234328095533E+00,
                   0.238155417791E+01, 0.477820650808E+00,
                   0.173147024625E+03, 0.809402431387E-01,
                   0.186637207143E+04, 0.157045886743E-01,
                   0.629203401916E+04, 0.487227319505E-02,
                   0.262732236345E+00, 0.185989979867E+00,
                   0.346816791841E+05, 0.344168707796E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Ta": [73, 181, 0.255854406332E+01, 0.472182991346E+00,
                   0.232847394275E+02, 0.228684060354E+00,
                   0.181642844146E+03, 0.789290509275E-01,
                   0.202742599219E+04, 0.158573602779E-01,
                   0.715940544035E+04, 0.416498598376E-02,
                   0.283129960975E+00, 0.199908628107E+00,
                   0.399827056380E+05, 0.272921855850E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "W": [74, 184, 0.238037716609E+02, 0.226025350345E+00,
                  0.264732698946E+01, 0.472288070804E+00,
                  0.184226656621E+03, 0.784198703457E-01,
                  0.198101387523E+04, 0.152835194257E-01,
                  0.668888539783E+04, 0.470100622683E-02,
                  0.277401882874E+00, 0.202951376702E+00,
                  0.368988258515E+05, 0.330806052575E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Re": [75, 186, 0.282304940571E+01, 0.466534003379E+00,
                   0.249323983166E+02, 0.221036079611E+00,
                   0.191677549677E+03, 0.767849656918E-01,
                   0.209725829061E+04, 0.152653385931E-01,
                   0.724339630549E+04, 0.430163095473E-02,
                   0.297860083817E+00, 0.215785643625E+00,
                   0.401623376751E+05, 0.292337384838E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Os": [76, 190, 0.297079860190E+01, 0.462354263364E+00,
                   0.258187627223E+02, 0.217191308882E+00,
                   0.196969109449E+03, 0.756917150773E-01,
                   0.213172588506E+04, 0.149657470563E-01,
                   0.727486876020E+04, 0.437520359242E-02,
                   0.310804801525E+00, 0.225117925776E+00,
                   0.401286872977E+05, 0.303835563977E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Ir": [77, 192, 0.317378806583E+01, 0.455483906393E+00,
                   0.337403905436E+00, 0.239147648318E+00,
                   0.205743526943E+03, 0.739774848854E-01,
                   0.271138830827E+02, 0.212278017660E+00,
                   0.229579604878E+04, 0.150813082541E-01,
                   0.821423013866E+04, 0.378999223249E-02,
                   0.461959591795E+05, 0.241632467447E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Pt": [78, 195, 0.330666183900E+01, 0.452252988381E+00,
                   0.278760195573E+02, 0.209285055288E+00,
                   0.210013036230E+03, 0.732166799264E-01,
                   0.229698110909E+04, 0.147096880057E-01,
                   0.798822513373E+04, 0.402495631898E-02,
                   0.347076154421E+00, 0.246240086187E+00,
                   0.443522887104E+05, 0.270544910245E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Au": [79, 197, 0.350288864768E+01, 0.445757121130E+00,
                   0.371825024502E+00, 0.258518552423E+00,
                   0.216928080493E+03, 0.719638192624E-01,
                   0.290106488072E+02, 0.205073783965E+00,
                   0.238781417213E+04, 0.145994292750E-01,
                   0.841509202112E+04, 0.383628027513E-02,
                   0.470628681888E+05, 0.251010577422E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Hg": [80, 201, 0.368323009122E+01, 0.440272184269E+00,
                   0.392879001556E+00, 0.268908804437E+00,
                   0.223072073510E+03, 0.709153888305E-01,
                   0.300444750281E+02, 0.201458389079E+00,
                   0.244992055174E+04, 0.144064015393E-01,
                   0.862622520163E+04, 0.379060852072E-02,
                   0.482450288799E+05, 0.248221555708E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Tl": [81, 204, 0.386284358229E+01, 0.434979162086E+00,
                   0.413201584374E+00, 0.278774510703E+00,
                   0.229341439259E+03, 0.699168595372E-01,
                   0.310612641356E+02, 0.198121129718E+00,
                   0.251284268429E+04, 0.142154277347E-01,
                   0.883776553806E+04, 0.374712472943E-02,
                   0.494056594337E+05, 0.245784458052E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Pb": [82, 207, 0.405059648947E+01, 0.431515442343E+00,
                   0.429810895104E+00, 0.287419116937E+00,
                   0.238855264731E+03, 0.685156017283E-01,
                   0.323723955936E+02, 0.194814377327E+00,
                   0.269190842344E+04, 0.142706878488E-01,
                   0.991044862285E+04, 0.326617712462E-02,
                   0.561942795479E+05, 0.198594228506E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Bi": [83, 209, 0.424623558161E+01, 0.426953844947E+00,
                   0.448541096314E+00, 0.296591275050E+00,
                   0.245085252709E+03, 0.675357313301E-01,
                   0.335004003542E+02, 0.191360383621E+00,
                   0.273483420155E+04, 0.140340832461E-01,
                   0.994515529059E+04, 0.331828179749E-02,
                   0.560830411943E+05, 0.206398769999E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Po": [84, 210, 0.439357005884E+01, 0.423979395277E+00,
                   0.458110728087E+00, 0.302372898209E+00,
                   0.248658003501E+03, 0.670825086631E-01,
                   0.342476220809E+02, 0.189025049790E+00,
                   0.270429904798E+04, 0.136744418296E-01,
                   0.948519585833E+04, 0.362726751784E-02,
                   0.530192063678E+05, 0.238438423299E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "At": [85, 210, 0.457321243663E+01, 0.420909369375E+00,
                   0.469862291315E+00, 0.309370884702E+00,
                   0.255260714253E+03, 0.661763696215E-01,
                   0.353502212690E+02, 0.186214003211E+00,
                   0.276905093797E+04, 0.135026287276E-01,
                   0.970248158868E+04, 0.359034100123E-02,
                   0.542085391881E+05, 0.236403160594E-03, 54, 0, 0, 0, 0, 0, 0, 0],
            "Rn": [86, 222, 0.472819693523E+01, 0.418890840331E+00,
                   0.475843250461E+00, 0.314514308956E+00,
                   0.261330692178E+03, 0.654403948340E-01,
                   0.363131472267E+02, 0.184001084801E+00,
                   0.281780500032E+04, 0.132957341045E-01,
                   0.979412980191E+04, 0.361570035352E-02,
                   0.544670675124E+05, 0.241936526950E-03, 54, 0, 0, 0, 0, 0, 0, 0]
        }

    def num_weights(self, element):
        atomic_values = self._atomic_values
        if atomic_values[element][15] == 0:
            total = 6
        elif atomic_values[element][13] == 0:
            total = 5
        elif atomic_values[element][11] == 0:
            total = 4
        elif atomic_values[element][9] == 0:
            total = 3
        elif atomic_values[element][7] == 0:
            total = 2
        elif atomic_values[element][5] == 0:
            total = 1
        else:
            total = 7
        return total

    def get_basis(self):
        name = self._name
        sym_l = self._sym_l
        basis = {
            'name': name,
            'primitive_type': 'gaussian',
        }

        atoms_data = []
        for atom in range(len(sym_l)):
            shells_data = []
            element = sym_l[atom]
            p_exp = []
            con_coef = []
            p_con = []

            for shell in range(self.num_weights(element)):
                expi = 2 * self._atomic_values[element][2 * (shell + 1)]
                coefi = self._atomic_values[element][2 * (shell + 1) + 1]
                nrmi = ((2*expi / np.pi) ** (3 / 4))

                p_exp.append(expi)
                con_coef.append(coefi * self._atomic_values[element][0] * nrmi)
                p_con.append(0.0)
            shells_data.append({
                'shell_type': 's',
                'p_exponents': p_exp,
                'con_coefficients': con_coef,
                'p_con_coefficients': p_con,
            })

            atoms_data.append({'shells': shells_data,
                               'symbol': element,
                               'atomic_number': self._atomic_values[element][0]})

        basis['atoms'] = atoms_data
        return basis

    def get_mo_coefficients(self):
        atomic_values=self._atomic_values
        sym_l=self._sym_l
        nat = len(sym_l)

        alpha_mo_coeff=[[]]
        for i in range(nat):
            num_weights1=self.num_weights(sym_l[i])
            selfsim = 0
            for k in range(num_weights1):
                for l in range(num_weights1):
                    a = 2 * atomic_values[sym_l[i]][2 * (k + 1)]
                    b = 2 * atomic_values[sym_l[i]][2 * (l + 1)]

                    normi = atomic_values[sym_l[i]][2 * (k + 1) + 1] * atomic_values[sym_l[i]][0] * (2*a / np.pi) ** (
                            3 / 2)
                    normj = atomic_values[sym_l[i]][2 * (l + 1) + 1] * atomic_values[sym_l[i]][0] * (2*b / np.pi) ** (
                            3 / 2)

                    coef = ((np.pi / (2*(a + b))) ** (3 / 2))/(2*np.sqrt(2))



                    selfsim = selfsim + normi * normj * coef

            alpha_mo_coeff[0].append(np.sqrt(selfsim))

        return alpha_mo_coeff

    def OLD_get_mo_coefficients(self):
        alpha_mo_coeff = [[]]
        for atom in range(len(self._sym_l)):
            alpha_mo_coeff[0].append(1.0)
        return alpha_mo_coeff


class ProtoElectronicStructure:
    """
    Protoelectronic structure class

    :param basis: The basis set
    :type basis: dict
    :param orbital_coefficients: Molecular orbital coefficients
    :type orbital_coefficients: list
    :param multiplicity: The multiplicity
    :type multiplicity: int
    :param alpha_energies: Alpha molecular orbital energies in Hartree
    :type alpha_energies: list
    :param beta_energies: Beta molecular orbital energies in Hartree
    :type beta_energies: list
    :param alpha_occupancy: Occupancy of alpha orbitals
    :type alpha_occupancy: list
    :param beta_occupancy: Occupancy of beta orbitals
    :type beta_occupancy: list

    """

    def __init__(self,
                 basis,
                 orbital_coefficients):

        self._basis = basis
        self._Ca = orbital_coefficients[0]
        if len(orbital_coefficients[1]) == 0:
            self._Cb = orbital_coefficients[0]
        else:
            self._Cb = orbital_coefficients[1]

    @property
    def basis(self):
        return self._basis

    @property
    def coefficients_a(self):
        return self._Ca

    @property
    def coefficients_b(self):
        return self._Cb
