'''
@anchor pydoc:grizzly.tasks Tasks
Tasks are functionality that is executed by `locust` at run time as they are specified in the feature file.

The most essential task is {@pylink grizzly.tasks.request}, which all {@pylink grizzly.users} is using to make
requests to the endpoint that is being load tested.

All other tasks are helper tasks for things that needs to happen after or before a {@pylink grizzly.tasks.request}, stuff like extracting information from
a previous response or fetching additional test data from a different endpoint ("{@link pydoc:grizzly.tasks.clients}").

## Custom

It is possible to implement custom tasks, the only requirement is that they inherit `grizzly.tasks.GrizzlyTask`. To get them to be executed by `grizzly`,
a step implementation is also needed.

Boilerplate example of a custom task:

```python
from typing import Any, cast

from grizzly.context import GrizzlyContext
from grizzly.tasks import GrizzlyTask, grizzlytask
from grizzly.scenarios import GrizzlyScenario
from grizzly.types.behave import Context, then


class TestTask(GrizzlyTask):
    def __call__(self) -> grizzlytask:
        @grizzlytask
        def task(parent: GrizzlyScenario) -> Any:
            print(f'{self.__class__.__name__}::task called')

        @task.on_start
        def on_start() -> None:
            print(f'{self.__class__.__name__}::on_start called')

        @task.on_stop
        def on_stop() -> None:
            print(f'{self.__class__.__name__}::on_stop called')

        return task


@then(u'run `TestTask`')
def step_run_testtask(context: Context) -> None:
    grizzly = cast(GrizzlyContext, context.grizzly)
    grizzly.scenario.tasks.add(TestTask())
```

There are examples of this in the {@link framework.example}.
'''
from abc import ABC, ABCMeta, abstractmethod
from typing import TYPE_CHECKING, Any, Callable, List, Type, Set, Optional
from os import environ
from pathlib import Path

from grizzly_extras.transformer import TransformerContentType

if TYPE_CHECKING:  # pragma: no cover
    from grizzly.types import GrizzlyResponse
    from grizzly.scenarios import GrizzlyScenario
    from grizzly.context import GrizzlyContextScenario

GrizzlyTaskType = Callable[['GrizzlyScenario'], Any]
GrizzlyTaskOnType = Optional[Callable[[], None]]


class grizzlytask:
    __name__ = 'grizzlytask'

    _on_start: Optional['OnGrizzlyTask'] = None
    _on_stop: Optional['OnGrizzlyTask'] = None

    class OnGrizzlyTask:
        _on_func: Optional[GrizzlyTaskOnType] = None

        def __init__(self, on_func: GrizzlyTaskOnType) -> None:
            self._on_func = on_func

        def __call__(self) -> None:
            if self._on_func is not None:
                self._on_func()

    def __init__(self, task: GrizzlyTaskType, doc: Optional[str] = None) -> None:
        self._task = task

        if doc is None and task is not None:
            self.__doc__ = doc

    def __call__(self, parent: 'GrizzlyScenario') -> Any:
        return self._task(parent)

    def on_start(self, on_start: GrizzlyTaskOnType = None) -> None:
        if self._on_start is None:
            self._on_start = self.OnGrizzlyTask(on_start)
        else:
            self._on_start()

    def on_stop(self, on_stop: GrizzlyTaskOnType = None) -> None:
        if self._on_stop is None:
            self._on_stop = self.OnGrizzlyTask(on_stop)
        else:
            self._on_stop()


class GrizzlyTask(ABC):
    __template_attributes__: List[str] = []

    _context_root: str

    scenario: 'GrizzlyContextScenario'
    step: str

    def __init__(self, scenario: Optional['GrizzlyContextScenario'] = None) -> None:
        self._context_root = environ.get('GRIZZLY_CONTEXT_ROOT', '.')
        if scenario is not None:
            self.scenario = scenario

    @abstractmethod
    def __call__(self) -> grizzlytask:
        raise NotImplementedError(f'{self.__class__.__name__} has not been implemented')

    def get_templates(self) -> List[str]:
        def is_template(value: str) -> bool:
            return '{{' in value and '}}' in value

        def _get_value_templates(value: Any) -> Set[str]:
            templates: Set[str] = set()

            if isinstance(value, str):
                try:
                    possible_file = Path(self._context_root) / 'requests' / value
                    if possible_file.is_file():
                        with open(possible_file, 'r', encoding='utf-8') as fd:
                            value = fd.read()
                except OSError:
                    pass

                if is_template(value):
                    templates.add(value)
            elif isinstance(value, list):
                for list_value in value:
                    if isinstance(list_value, GrizzlyTask):
                        templates.update(list_value.get_templates())
                    else:
                        templates.update(_get_value_templates(list_value))
            elif isinstance(value, dict):
                for dict_value in value.values():
                    if isinstance(dict_value, GrizzlyTask):
                        templates.update(dict_value.get_templates())
                    else:
                        templates.update(_get_value_templates(dict_value))
            elif isinstance(value, GrizzlyTask):
                templates.update(value.get_templates())

            return templates

        templates: Set[str] = set()
        for attribute in self.__template_attributes__:
            value = getattr(self, attribute, None)
            if value is None:
                continue

            templates.update(_get_value_templates(value))

        return list(templates)


class GrizzlyMetaRequestTask(GrizzlyTask, metaclass=ABCMeta):
    content_type: TransformerContentType
    name: Optional[str]
    endpoint: str

    def execute(self, parent: 'GrizzlyScenario') -> 'GrizzlyResponse':
        raise NotImplementedError(f'{self.__class__.name} has not implemented "execute"')


class GrizzlyTaskWrapper(GrizzlyTask, metaclass=ABCMeta):
    name: str

    @abstractmethod
    def add(self, task: GrizzlyTask) -> None:
        raise NotImplementedError(f'{self.__class__.__name__} has not implemented add')

    @abstractmethod
    def peek(self) -> List[GrizzlyTask]:
        raise NotImplementedError(f'{self.__class__.__name__} has not implemented peek')


class template:
    attributes: List[str]

    def __init__(self, attribute: str, *additional_attributes: str) -> None:
        self.attributes = [attribute]
        if len(additional_attributes) > 0:
            self.attributes += list(additional_attributes)

    def __call__(self, cls: Type[GrizzlyTask]) -> Type[GrizzlyTask]:
        cls.__template_attributes__ = self.attributes

        return cls


from .request import RequestTask, RequestTaskHandlers, RequestTaskResponse
from .wait import WaitTask
from .log_message import LogMessageTask
from .transformer import TransformerTask
from .until import UntilRequestTask
from .date import DateTask
from .async_group import AsyncRequestGroupTask
from .timer import TimerTask
from .task_wait import TaskWaitTask
from .conditional import ConditionalTask
from .loop import LoopTask
from .set_variable import SetVariableTask


__all__ = [
    'RequestTaskHandlers',
    'RequestTaskResponse',
    'RequestTask',
    'LogMessageTask',
    'WaitTask',
    'TransformerTask',
    'UntilRequestTask',
    'DateTask',
    'AsyncRequestGroupTask',
    'TimerTask',
    'TaskWaitTask',
    'ConditionalTask',
    'LoopTask',
    'SetVariableTask',
]
