"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotFleet = exports.EksSpotCluster = exports.InstanceInterruptionBehavior = exports.BlockDuration = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const launch_template_1 = require("./launch-template");
const user_data_1 = require("./user-data");
const DEFAULT_INSTANCE_TYPE = 't3.large';
/**
 * @experimental
 */
var BlockDuration;
(function (BlockDuration) {
    BlockDuration[BlockDuration["ONE_HOUR"] = 60] = "ONE_HOUR";
    BlockDuration[BlockDuration["TWO_HOURS"] = 120] = "TWO_HOURS";
    BlockDuration[BlockDuration["THREE_HOURS"] = 180] = "THREE_HOURS";
    BlockDuration[BlockDuration["FOUR_HOURS"] = 240] = "FOUR_HOURS";
    BlockDuration[BlockDuration["FIVE_HOURS"] = 300] = "FIVE_HOURS";
    BlockDuration[BlockDuration["SIX_HOURS"] = 360] = "SIX_HOURS";
})(BlockDuration = exports.BlockDuration || (exports.BlockDuration = {}));
/**
 * @experimental
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    InstanceInterruptionBehavior["STOP"] = "stop";
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior = exports.InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = {}));
/**
 * @experimental
 */
class EksSpotCluster extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // this.cluster = props.cluster;
        this.clusterVersion = props.clusterVersion;
        // use an existing vpc or create a new one
        this.vpc = this.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(this, 'Vpc', { isDefault: true }) :
            this.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(this, 'Vpc', { vpcId: this.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        const clusterAdmin = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole: clusterAdmin,
            defaultCapacity: 0,
            version: this.clusterVersion,
        });
    }
    /**
     * @experimental
     */
    addSpotFleet(id, props) {
        new SpotFleet(this, id, {
            cluster: this,
            ...props,
        });
    }
    /**
     * @experimental
     */
    addDays(date, days) {
        date.setDate(date.getDate() + days);
        return date;
    }
    /**
     * @experimental
     */
    addHours(date, hours) {
        date.setHours(date.getHours() + hours);
        return date;
    }
    /**
     * @experimental
     */
    addMinutes(date, minutes) {
        date.setMinutes(date.getMinutes() + minutes);
        return date;
    }
}
exports.EksSpotCluster = EksSpotCluster;
/**
 * @experimental
 */
class SpotFleet extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id, props);
        this.spotFleetId = id;
        this.clusterStack = props.cluster;
        this.launchTemplate = (_a = props.launchTemplate) !== null && _a !== void 0 ? _a : new launch_template_1.LaunchTemplate();
        this.targetCapacity = props.targetCapacity;
        this.defaultInstanceType = (_b = props.defaultInstanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType(DEFAULT_INSTANCE_TYPE);
        // isntance role
        this.instanceRole = props.instanceRole || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const sg = new ec2.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.clusterStack.cluster.vpc,
        });
        // self rules
        sg.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        sg.connections.allowFrom(this.clusterStack.cluster, ec2.Port.tcp(443));
        sg.connections.allowFrom(this.clusterStack.cluster, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        sg.connections.allowTo(this.clusterStack.cluster, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        sg.connections.allowToAnyIpv4(ec2.Port.allTcp());
        sg.connections.allowToAnyIpv4(ec2.Port.allUdp());
        sg.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const config = this.launchTemplate.bind(this);
        // const userData = renderAmazonLinuxUserData(cdk.Stack.of(this), this.cluster.clusterName, config.spotfleet);
        const userData = ec2.UserData.forLinux();
        userData.addCommands(...user_data_1.renderAmazonLinuxUserData(core_1.Stack.of(this), this.clusterStack.cluster.clusterName, config.spotfleet));
        this.defaultInstanceType = (_c = props.defaultInstanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType(DEFAULT_INSTANCE_TYPE);
        const imageId = (_d = props.customAmiId) !== null && _d !== void 0 ? _d : new eks.EksOptimizedImage({
            nodeType: nodeTypeForInstanceType(this.defaultInstanceType),
            kubernetesVersion: props.cluster.clusterVersion.version,
        }).getImage(this).imageId;
        const lt = new ec2.CfnLaunchTemplate(this, 'LaunchTemplate', {
            launchTemplateData: {
                imageId,
                instanceType: this.defaultInstanceType.toString(),
                tagSpecifications: [
                    {
                        resourceType: 'instance',
                        tags: [
                            {
                                key: 'Name',
                                value: `${core_1.Stack.of(this).stackName}/Cluster/${this.spotFleetId}`,
                            },
                            {
                                key: `kubernetes.io/cluster/${this.clusterStack.cluster.clusterName}`,
                                value: 'owned',
                            },
                        ],
                    },
                ],
                instanceMarketOptions: {
                    marketType: 'spot',
                    spotOptions: {
                        blockDurationMinutes: (_e = props.blockDuration) !== null && _e !== void 0 ? _e : BlockDuration.ONE_HOUR,
                        instanceInterruptionBehavior: (_f = props.instanceInterruptionBehavior) !== null && _f !== void 0 ? _f : InstanceInterruptionBehavior.TERMINATE,
                    },
                },
                // userData: cdk.Fn.base64(capacityAsg.userData.render()),
                userData: core_1.Fn.base64(userData.render()),
                securityGroupIds: sg.connections.securityGroups.map(m => m.securityGroupId),
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
            },
        });
        const spotFleetRole = new iam.Role(this, 'FleetRole', {
            assumedBy: new iam.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
        });
        const overrides = [];
        for (const s of this.clusterStack.cluster.vpc.privateSubnets) {
            overrides.push({ subnetId: s.subnetId });
        }
        new ec2.CfnSpotFleet(this, id, {
            spotFleetRequestConfigData: {
                launchTemplateConfigs: [
                    {
                        launchTemplateSpecification: {
                            launchTemplateId: lt.ref,
                            version: lt.attrLatestVersionNumber,
                        },
                        overrides,
                    },
                ],
                iamFleetRole: spotFleetRole.roleArn,
                targetCapacity: (_g = props.targetCapacity) !== null && _g !== void 0 ? _g : 1,
                validFrom: props.validFrom,
                validUntil: props.validUntil,
                terminateInstancesWithExpiration: props.terminateInstancesWithExpiration,
            },
        });
        this.clusterStack.cluster.awsAuth.addRoleMapping(this.instanceRole, {
            username: 'system:node:{{EC2PrivateDNSName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
            ],
        });
    }
}
exports.SpotFleet = SpotFleet;
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? eks.NodeType.GPU : eks.NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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