"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const _1 = require(".");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * (experimental) The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 *
 * @experimental
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * (experimental) Custom worker type.
     *
     * @param workerType custom worker type.
     * @experimental
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue-alpha.WorkerType", version: "2.0.0-alpha.7" };
/**
 * (experimental) Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 *
 * @experimental
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * (experimental) Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * (experimental) Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * (experimental) Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 * @experimental
 */
var JobState;
(function (JobState) {
    JobState["SUCCEEDED"] = "SUCCEEDED";
    JobState["FAILED"] = "FAILED";
    JobState["TIMEOUT"] = "TIMEOUT";
    JobState["STARTING"] = "STARTING";
    JobState["RUNNING"] = "RUNNING";
    JobState["STOPPING"] = "STOPPING";
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * (experimental) The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 * @experimental
 */
var MetricType;
(function (MetricType) {
    MetricType["GAUGE"] = "gauge";
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * (experimental) Create a CloudWatch Event Rule for this Glue Job when it's in a given state.
     *
     * @param id construct id.
     * @param options event options.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     * @experimental
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * (experimental) Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     * @experimental
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensions: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     *
     * @experimental
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     *
     * @experimental
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     *
     * @experimental
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        var _c;
        return (_c = this.node.tryFindChild(id)) !== null && _c !== void 0 ? _c : this.onStateChange(id, jobState);
    }
}
/**
 * (experimental) A Glue Job.
 *
 * @experimental
 */
class Job extends JobBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.jobName,
        });
        const executable = props.executable.bind();
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = ((_d = props.sparkUI) === null || _d === void 0 ? void 0 : _d.enabled) ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.location;
        const continuousLoggingArgs = ((_e = props.continuousLogging) === null || _e === void 0 ? void 0 : _e.enabled) ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new aws_glue_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: (_f = props.workerType) === null || _f === void 0 ? void 0 : _f.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: (_g = props.timeout) === null || _g === void 0 ? void 0 : _g.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: (_h = props.securityConfiguration) === null || _h === void 0 ? void 0 : _h.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * (experimental) Creates a Glue Job.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromJobAttributes(scope, id, attrs) {
        class Import extends JobBase {
            constructor() {
                var _c;
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = (_c = attrs.role) !== null && _c !== void 0 ? _c : new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--conf', '--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        var _c;
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && ((_c = config.extraJars) === null || _c === void 0 ? void 0 : _c.length) > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        var _c;
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        const bucket = (_c = props.bucket) !== null && _c !== void 0 ? _c : new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        var _c;
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': ((_c = props.quiet) !== null && _c !== void 0 ? _c : true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue-alpha.Job", version: "2.0.0-alpha.7" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensions: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiam9iLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiam9iLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEseURBQXlEO0FBQ3pELGlEQUFpRDtBQUNqRCwyQ0FBMkM7QUFFM0MseUNBQXlDO0FBQ3pDLG1DQUFtQztBQUVuQyx3QkFBc0U7QUFFdEUsbURBQThDOzs7Ozs7Ozs7QUFJOUMsTUFBYSxVQUFVO0lBa0JyQixZQUFvQixJQUFZO1FBQzlCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO0lBQ25CLENBQUM7Ozs7Ozs7SUFUTSxNQUFNLENBQUMsRUFBRSxDQUFDLFVBQWtCO1FBQ2pDLE9BQU8sSUFBSSxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDcEMsQ0FBQzs7QUFiSCxnQ0FxQkM7Ozs7Ozs7O0FBbkJ3QixtQkFBUSxHQUFHLElBQUksVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDOzs7Ozs7OztBQUd0QyxlQUFJLEdBQUcsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7Ozs7Ozs7O0FBRzlCLGVBQUksR0FBRyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQzs7Ozs7OztBQWdCdkQsSUFBWSxRQXFCWDtBQXJCRCxXQUFZLFFBQVE7SUFFbEIsbUNBQXVCLENBQUE7SUFHdkIsNkJBQWlCLENBQUE7SUFHakIsK0JBQW1CLENBQUE7SUFHbkIsaUNBQXFCLENBQUE7SUFHckIsK0JBQW1CLENBQUE7SUFHbkIsaUNBQXFCLENBQUE7SUFHckIsK0JBQW1CLENBQUE7QUFDckIsQ0FBQyxFQXJCVyxRQUFRLEdBQVIsZ0JBQVEsS0FBUixnQkFBUSxRQXFCbkI7Ozs7Ozs7QUFHRCxJQUFZLFVBTVg7QUFORCxXQUFZLFVBQVU7SUFFcEIsNkJBQWUsQ0FBQTtJQUdmLDZCQUFlLENBQUE7QUFDakIsQ0FBQyxFQU5XLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBTXJCO0FBc0NELE1BQWUsT0FBUSxTQUFRLEdBQUcsQ0FBQyxRQUFROzs7Ozs7Ozs7SUFPbEMsT0FBTyxDQUFDLEVBQVUsRUFBRSxVQUFpQyxFQUFFO1FBQzVELE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbkIsTUFBTSxFQUFFLENBQUMsVUFBVSxDQUFDO1lBQ3BCLFVBQVUsRUFBRSxDQUFDLHVCQUF1QixFQUFFLHFCQUFxQixDQUFDO1lBQzVELE1BQU0sRUFBRTtnQkFDTixPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO2FBQ3hCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOzs7Ozs7Ozs7SUFHTSxhQUFhLENBQUMsRUFBVSxFQUFFLFFBQWtCLEVBQUUsVUFBaUMsRUFBRTtRQUN0RixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRTtZQUM1QixXQUFXLEVBQUUsZ0NBQWdDLElBQUksQ0FBQyxPQUFPLFVBQVUsUUFBUSxRQUFRO1lBQ25GLEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbkIsTUFBTSxFQUFFO2dCQUNOLEtBQUssRUFBRSxDQUFDLFFBQVEsQ0FBQzthQUNsQjtTQUNGLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7Ozs7Ozs7SUFHTSxTQUFTLENBQUMsRUFBVSxFQUFFLFVBQWlDLEVBQUU7UUFDOUQsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxRQUFRLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzdELENBQUM7Ozs7Ozs7O0lBR00sU0FBUyxDQUFDLEVBQVUsRUFBRSxVQUFpQyxFQUFFO1FBQzlELE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsUUFBUSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMxRCxDQUFDOzs7Ozs7OztJQUdNLFNBQVMsQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTtRQUM5RCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDM0QsQ0FBQzs7Ozs7Ozs7OztJQUdNLE1BQU0sQ0FBQyxVQUFrQixFQUFFLElBQWdCLEVBQUUsS0FBZ0M7UUFDbEYsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDM0IsVUFBVTtZQUNWLFNBQVMsRUFBRSxNQUFNO1lBQ2pCLFVBQVUsRUFBRTtnQkFDVixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87Z0JBQ3JCLFFBQVEsRUFBRSxLQUFLO2dCQUNmLElBQUksRUFBRSxJQUFJO2FBQ1g7WUFDRCxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3BCLENBQUM7Ozs7Ozs7O0lBR00sYUFBYSxDQUFDLEtBQWdDO1FBQ25ELE9BQU8sVUFBVSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRSxRQUFRLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDN0YsQ0FBQzs7Ozs7Ozs7SUFHTSxhQUFhLENBQUMsS0FBZ0M7UUFDbkQsT0FBTyxVQUFVLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQixFQUFFLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMxRixDQUFDOzs7Ozs7OztJQUdNLGFBQWEsQ0FBQyxLQUFnQztRQUNuRCxPQUFPLFVBQVUsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzNGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSyxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsUUFBa0I7O1FBQ3ZELGFBQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFnQixtQ0FBSSxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUN2RixDQUFDO0NBQ0Y7Ozs7OztBQTRHRCxNQUFhLEdBQUksU0FBUSxPQUFPOzs7O0lBMkI5QixZQUFZLEtBQTJCLEVBQUUsRUFBVSxFQUFFLEtBQWU7O1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxPQUFPO1NBQzVCLENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFM0MsSUFBSSxDQUFDLElBQUksU0FBRyxLQUFLLENBQUMsSUFBSSxtQ0FBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUMxRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsb0JBQW9CLENBQUM7WUFDekQsZUFBZSxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQ2pHLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztRQUVoQyxNQUFNLE9BQU8sR0FBRyxPQUFBLEtBQUssQ0FBQyxPQUFPLDBDQUFFLE9BQU8sRUFBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUM3RyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLFFBQVEsQ0FBQztRQUNoRCxNQUFNLHFCQUFxQixHQUFHLE9BQUEsS0FBSyxDQUFDLGlCQUFpQiwwQ0FBRSxPQUFPLEVBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDdEksTUFBTSxvQkFBb0IsR0FBRyxLQUFLLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUU1RixNQUFNLGdCQUFnQixHQUFHO1lBQ3ZCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFVBQVUsQ0FBQztZQUN2QyxHQUFHLHFCQUFxQjtZQUN4QixHQUFHLG9CQUFvQjtZQUN2QixHQUFHLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxJQUFJO1lBQ2hCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztTQUNwRCxDQUFDO1FBRUYsTUFBTSxXQUFXLEdBQUcsSUFBSSxpQkFBTSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0MsSUFBSSxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ25CLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ3ZCLE9BQU8sRUFBRTtnQkFDUCxJQUFJLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJO2dCQUMxQixjQUFjLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDO2dCQUN2RCxhQUFhLEVBQUUsVUFBVSxDQUFDLGFBQWE7YUFDeEM7WUFDRCxXQUFXLEVBQUUsVUFBVSxDQUFDLFdBQVcsQ0FBQyxJQUFJO1lBQ3hDLFVBQVUsUUFBRSxLQUFLLENBQUMsVUFBVSwwQ0FBRSxJQUFJO1lBQ2xDLGVBQWUsRUFBRSxLQUFLLENBQUMsV0FBVztZQUNsQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsRUFBRSxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN2RyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEVBQUUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbkgsT0FBTyxRQUFFLEtBQUssQ0FBQyxPQUFPLDBDQUFFLFNBQVMsRUFBRTtZQUNuQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzlILHFCQUFxQixRQUFFLEtBQUssQ0FBQyxxQkFBcUIsMENBQUUseUJBQXlCO1lBQzdFLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixnQkFBZ0I7U0FDakIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQyxJQUFJLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLE9BQU8sR0FBRyxZQUFZLENBQUM7SUFDOUIsQ0FBQzs7Ozs7Ozs7O0lBN0VNLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUEyQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUMzRixNQUFNLE1BQU8sU0FBUSxPQUFPO1lBQTVCOzs7Z0JBQ2tCLFlBQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO2dCQUN4QixXQUFNLEdBQUcsTUFBTSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ3RDLG1CQUFjLFNBQUcsS0FBSyxDQUFDLElBQUksbUNBQUksSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUM5RixDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBdUVEOzs7O09BSUc7SUFDSyxtQkFBbUIsQ0FBQyxnQkFBNEM7UUFDdEUsSUFBSSxnQkFBZ0IsRUFBRTtZQUNwQixNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUM7WUFDNUUsTUFBTSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO2dCQUM1QyxJQUFJLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUU7b0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxHQUFHLDZDQUE2QyxDQUFDLENBQUM7aUJBQzFFO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQztJQUVPLG1CQUFtQixDQUFDLE1BQTJCOztRQUNyRCxNQUFNLElBQUksR0FBOEIsRUFBRSxDQUFDO1FBQzNDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxRQUFRLENBQUM7UUFDekMsSUFBSSxNQUFNLENBQUMsU0FBUyxFQUFFO1lBQ3BCLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDO1NBQ3BDO1FBQ0QsSUFBSSxNQUFNLENBQUMsU0FBUyxJQUFJLE9BQUEsTUFBTSxDQUFDLFNBQVMsMENBQUUsTUFBTSxJQUFHLENBQUMsRUFBRTtZQUNwRCxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQzNGO1FBQ0QsSUFBSSxNQUFNLENBQUMsZ0JBQWdCLElBQUksTUFBTSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDakUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsTUFBTSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDdEc7UUFDRCxJQUFJLE1BQU0sQ0FBQyxVQUFVLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3JELElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDN0Y7UUFDRCxJQUFJLE1BQU0sQ0FBQyxjQUFjLEVBQUU7WUFDekIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsTUFBTSxDQUFDO1NBQ3BDO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sWUFBWSxDQUFDLFVBQStCLEVBQUUsSUFBZSxFQUFFLEtBQW1COztRQUN4RixJQUFJLFVBQU8sQ0FBQyxZQUFZLEtBQUssVUFBVSxDQUFDLElBQUksRUFBRTtZQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLHlEQUF5RCxDQUFDLENBQUM7U0FDNUU7UUFFRCxNQUFNLE1BQU0sU0FBRyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ3BFLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUIsTUFBTSxJQUFJLEdBQUc7WUFDWCxtQkFBbUIsRUFBRSxNQUFNO1lBQzNCLHlCQUF5QixFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztTQUMvRCxDQUFDO1FBRUYsT0FBTztZQUNMLFFBQVEsRUFBRTtnQkFDUixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07Z0JBQ3BCLE1BQU07YUFDUDtZQUNELElBQUk7U0FDTCxDQUFDO0lBQ0osQ0FBQztJQUVPLHNCQUFzQixDQUFDLElBQWUsRUFBRSxLQUE2Qjs7UUFDM0UsTUFBTSxJQUFJLEdBQTRCO1lBQ3BDLG9DQUFvQyxFQUFFLE1BQU07WUFDNUMsZ0NBQWdDLEVBQUUsT0FBQyxLQUFLLENBQUMsS0FBSyxtQ0FBSSxJQUFJLENBQUMsQ0FBQyxRQUFRLEVBQUU7U0FDbkUsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsQixJQUFJLENBQUMsMkJBQTJCLENBQUMsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQztZQUNoRSxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNqQztRQUVELElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUN6QixJQUFJLENBQUMsa0NBQWtDLENBQUMsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO1NBQ2xFO1FBQ0QsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEVBQUU7WUFDM0IsSUFBSSxDQUFDLG9DQUFvQyxDQUFDLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1NBQ3RFO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sZUFBZSxDQUFDLElBQVU7UUFDaEMsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQztRQUN6RCxPQUFPLFFBQVEsVUFBVSxDQUFDLFVBQVUsSUFBSSxVQUFVLENBQUMsU0FBUyxFQUFFLENBQUM7SUFDakUsQ0FBQzs7QUFuS0gsa0JBb0tDOzs7QUFFRDs7Ozs7OztHQU9HO0FBQ0gsU0FBUyxVQUFVLENBQUMsSUFBa0IsRUFBRSxLQUFnQztJQUN0RSxPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztRQUMzQixTQUFTLEVBQUUsWUFBWTtRQUN2QixVQUFVLEVBQUUsZ0JBQWdCO1FBQzVCLFVBQVUsRUFBRSxFQUFFLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFO1FBQ3ZDLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUc7UUFDbkMsR0FBRyxLQUFLO0tBQ1QsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUNwQixDQUFDO0FBR0Q7Ozs7R0FJRztBQUNILFNBQVMsTUFBTSxDQUFDLEtBQTJCLEVBQUUsT0FBZTtJQUMxRCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUNuQyxPQUFPLEVBQUUsTUFBTTtRQUNmLFFBQVEsRUFBRSxLQUFLO1FBQ2YsWUFBWSxFQUFFLE9BQU87S0FDdEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGNvbnN0cnVjdHMgZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDb2RlLCBKb2JFeGVjdXRhYmxlLCBKb2JFeGVjdXRhYmxlQ29uZmlnLCBKb2JUeXBlIH0gZnJvbSAnLic7XG5pbXBvcnQgeyBJQ29ubmVjdGlvbiB9IGZyb20gJy4vY29ubmVjdGlvbic7XG5pbXBvcnQgeyBDZm5Kb2IgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZ2x1ZSc7XG5pbXBvcnQgeyBJU2VjdXJpdHlDb25maWd1cmF0aW9uIH0gZnJvbSAnLi9zZWN1cml0eS1jb25maWd1cmF0aW9uJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgV29ya2VyVHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU1RBTkRBUkQgPSBuZXcgV29ya2VyVHlwZSgnU3RhbmRhcmQnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgR18xWCA9IG5ldyBXb3JrZXJUeXBlKCdHLjFYJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBHXzJYID0gbmV3IFdvcmtlclR5cGUoJ0cuMlgnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBvZih3b3JrZXJUeXBlOiBzdHJpbmcpOiBXb3JrZXJUeXBlIHtcbiAgICByZXR1cm4gbmV3IFdvcmtlclR5cGUod29ya2VyVHlwZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZykge1xuICAgIHRoaXMubmFtZSA9IG5hbWU7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gSm9iU3RhdGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFNVQ0NFRURFRCA9ICdTVUNDRUVERUQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgRkFJTEVEID0gJ0ZBSUxFRCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBUSU1FT1VUID0gJ1RJTUVPVVQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFNUQVJUSU5HID0gJ1NUQVJUSU5HJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFJVTk5JTkcgPSAnUlVOTklORycsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgU1RPUFBJTkcgPSAnU1RPUFBJTkcnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgU1RPUFBFRCA9ICdTVE9QUEVEJyxcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gTWV0cmljVHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBHQVVHRSA9ICdnYXVnZScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQ09VTlQgPSAnY291bnQnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJSm9iIGV4dGVuZHMgY2RrLklSZXNvdXJjZSwgaWFtLklHcmFudGFibGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgam9iTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGpvYkFybjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBvblN0YXRlQ2hhbmdlKGlkOiBzdHJpbmcsIGpvYlN0YXRlOiBKb2JTdGF0ZSwgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBvblN1Y2Nlc3MoaWQ6IHN0cmluZywgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBvbkZhaWx1cmUoaWQ6IHN0cmluZywgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgb25UaW1lb3V0KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHR5cGU6IE1ldHJpY1R5cGUsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgbWV0cmljU3VjY2Vzcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG1ldHJpY0ZhaWx1cmUocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBtZXRyaWNUaW1lb3V0KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG59XG5cbmFic3RyYWN0IGNsYXNzIEpvYkJhc2UgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJSm9iIHtcblxuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgam9iQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBqb2JOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KTogZXZlbnRzLlJ1bGUge1xuICAgIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUodGhpcywgaWQsIG9wdGlvbnMpO1xuICAgIHJ1bGUuYWRkVGFyZ2V0KG9wdGlvbnMudGFyZ2V0KTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBzb3VyY2U6IFsnYXdzLmdsdWUnXSxcbiAgICAgIGRldGFpbFR5cGU6IFsnR2x1ZSBKb2IgU3RhdGUgQ2hhbmdlJywgJ0dsdWUgSm9iIFJ1biBTdGF0dXMnXSxcbiAgICAgIGRldGFpbDoge1xuICAgICAgICBqb2JOYW1lOiBbdGhpcy5qb2JOYW1lXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgcmV0dXJuIHJ1bGU7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgb25TdGF0ZUNoYW5nZShpZDogc3RyaW5nLCBqb2JTdGF0ZTogSm9iU3RhdGUsIG9wdGlvbnM6IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyA9IHt9KTogZXZlbnRzLlJ1bGUge1xuICAgIGNvbnN0IHJ1bGUgPSB0aGlzLm9uRXZlbnQoaWQsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBgUnVsZSB0cmlnZ2VyZWQgd2hlbiBHbHVlIGpvYiAke3RoaXMuam9iTmFtZX0gaXMgaW4gJHtqb2JTdGF0ZX0gc3RhdGVgLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBkZXRhaWw6IHtcbiAgICAgICAgc3RhdGU6IFtqb2JTdGF0ZV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJldHVybiBydWxlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG9uU3VjY2VzcyhpZDogc3RyaW5nLCBvcHRpb25zOiBldmVudHMuT25FdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICByZXR1cm4gdGhpcy5vblN0YXRlQ2hhbmdlKGlkLCBKb2JTdGF0ZS5TVUNDRUVERUQsIG9wdGlvbnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG9uRmFpbHVyZShpZDogc3RyaW5nLCBvcHRpb25zOiBldmVudHMuT25FdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICByZXR1cm4gdGhpcy5vblN0YXRlQ2hhbmdlKGlkLCBKb2JTdGF0ZS5GQUlMRUQsIG9wdGlvbnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBvblRpbWVvdXQoaWQ6IHN0cmluZywgb3B0aW9uczogZXZlbnRzLk9uRXZlbnRPcHRpb25zID0ge30pOiBldmVudHMuUnVsZSB7XG4gICAgcmV0dXJuIHRoaXMub25TdGF0ZUNoYW5nZShpZCwgSm9iU3RhdGUuVElNRU9VVCwgb3B0aW9ucyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCB0eXBlOiBNZXRyaWNUeXBlLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG1ldHJpY05hbWUsXG4gICAgICBuYW1lc3BhY2U6ICdHbHVlJyxcbiAgICAgIGRpbWVuc2lvbnM6IHtcbiAgICAgICAgSm9iTmFtZTogdGhpcy5qb2JOYW1lLFxuICAgICAgICBKb2JSdW5JZDogJ0FMTCcsXG4gICAgICAgIFR5cGU6IHR5cGUsXG4gICAgICB9LFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSkuYXR0YWNoVG8odGhpcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljU3VjY2Vzcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbWV0cmljUnVsZSh0aGlzLm1ldHJpY0pvYlN0YXRlUnVsZSgnU3VjY2Vzc01ldHJpY1J1bGUnLCBKb2JTdGF0ZS5TVUNDRUVERUQpLCBwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljRmFpbHVyZShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbWV0cmljUnVsZSh0aGlzLm1ldHJpY0pvYlN0YXRlUnVsZSgnRmFpbHVyZU1ldHJpY1J1bGUnLCBKb2JTdGF0ZS5GQUlMRUQpLCBwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljVGltZW91dChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbWV0cmljUnVsZSh0aGlzLm1ldHJpY0pvYlN0YXRlUnVsZSgnVGltZW91dE1ldHJpY1J1bGUnLCBKb2JTdGF0ZS5USU1FT1VUKSwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgcmV0cmlldmVzIGEgc2luZ2xldG9uIGV2ZW50IHJ1bGUgZm9yIHRoZSBpbnB1dCBqb2Igc3RhdGUgZm9yIHVzZSB3aXRoIHRoZSBtZXRyaWMgSm9iU3RhdGUgbWV0aG9kcy5cbiAgICpcbiAgICogQHBhcmFtIGlkIGNvbnN0cnVjdCBpZC5cbiAgICogQHBhcmFtIGpvYlN0YXRlIHRoZSBqb2Igc3RhdGUuXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIG1ldHJpY0pvYlN0YXRlUnVsZShpZDogc3RyaW5nLCBqb2JTdGF0ZTogSm9iU3RhdGUpOiBldmVudHMuUnVsZSB7XG4gICAgcmV0dXJuIHRoaXMubm9kZS50cnlGaW5kQ2hpbGQoaWQpIGFzIGV2ZW50cy5SdWxlID8/IHRoaXMub25TdGF0ZUNoYW5nZShpZCwgam9iU3RhdGUpO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNwYXJrVUlQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlZDogYm9vbGVhblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJ1Y2tldD86IHMzLklCdWNrZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTcGFya1VJTG9nZ2luZ0xvY2F0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDb250aW51b3VzTG9nZ2luZ1Byb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZWQ6IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ1N0cmVhbVByZWZpeD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHF1aWV0PzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29udmVyc2lvblBhdHRlcm4/OiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBKb2JBdHRyaWJ1dGVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgam9iTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEpvYlByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhlY3V0YWJsZTogSm9iRXhlY3V0YWJsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgam9iTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1heENhcGFjaXR5PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1heFJldHJpZXM/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1heENvbmN1cnJlbnRSdW5zPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbm90aWZ5RGVsYXlBZnRlcj86IGNkay5EdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0aW1lb3V0PzogY2RrLkR1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgd29ya2VyVHlwZT86IFdvcmtlclR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgd29ya2VyQ291bnQ/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25zPzogSUNvbm5lY3Rpb25bXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlDb25maWd1cmF0aW9uPzogSVNlY3VyaXR5Q29uZmlndXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlZmF1bHRBcmd1bWVudHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhZ3M/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZVByb2ZpbGluZ01ldHJpY3M/IDpib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3BhcmtVST86IFNwYXJrVUlQcm9wcyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29udGludW91c0xvZ2dpbmc/OiBDb250aW51b3VzTG9nZ2luZ1Byb3BzLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgSm9iIGV4dGVuZHMgSm9iQmFzZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21Kb2JBdHRyaWJ1dGVzKHNjb3BlOiBjb25zdHJ1Y3RzLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEpvYkF0dHJpYnV0ZXMpOiBJSm9iIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBKb2JCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBqb2JOYW1lID0gYXR0cnMuam9iTmFtZTtcbiAgICAgIHB1YmxpYyByZWFkb25seSBqb2JBcm4gPSBqb2JBcm4oc2NvcGUsIGF0dHJzLmpvYk5hbWUpO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsID0gYXR0cnMucm9sZSA/PyBuZXcgaWFtLlVua25vd25QcmluY2lwYWwoeyByZXNvdXJjZTogdGhpcyB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgam9iQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGpvYk5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHNwYXJrVUlMb2dnaW5nTG9jYXRpb24/OiBTcGFya1VJTG9nZ2luZ0xvY2F0aW9uO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjb25zdHJ1Y3RzLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEpvYlByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmpvYk5hbWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBleGVjdXRhYmxlID0gcHJvcHMuZXhlY3V0YWJsZS5iaW5kKCk7XG5cbiAgICB0aGlzLnJvbGUgPSBwcm9wcy5yb2xlID8/IG5ldyBpYW0uUm9sZSh0aGlzLCAnU2VydmljZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZ2x1ZS5hbWF6b25hd3MuY29tJyksXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFtpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NHbHVlU2VydmljZVJvbGUnKV0sXG4gICAgfSk7XG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMucm9sZTtcblxuICAgIGNvbnN0IHNwYXJrVUkgPSBwcm9wcy5zcGFya1VJPy5lbmFibGVkID8gdGhpcy5zZXR1cFNwYXJrVUkoZXhlY3V0YWJsZSwgdGhpcy5yb2xlLCBwcm9wcy5zcGFya1VJKSA6IHVuZGVmaW5lZDtcbiAgICB0aGlzLnNwYXJrVUlMb2dnaW5nTG9jYXRpb24gPSBzcGFya1VJPy5sb2NhdGlvbjtcbiAgICBjb25zdCBjb250aW51b3VzTG9nZ2luZ0FyZ3MgPSBwcm9wcy5jb250aW51b3VzTG9nZ2luZz8uZW5hYmxlZCA/IHRoaXMuc2V0dXBDb250aW51b3VzTG9nZ2luZyh0aGlzLnJvbGUsIHByb3BzLmNvbnRpbnVvdXNMb2dnaW5nKSA6IHt9O1xuICAgIGNvbnN0IHByb2ZpbGluZ01ldHJpY3NBcmdzID0gcHJvcHMuZW5hYmxlUHJvZmlsaW5nTWV0cmljcyA/IHsgJy0tZW5hYmxlLW1ldHJpY3MnOiAnJyB9IDoge307XG5cbiAgICBjb25zdCBkZWZhdWx0QXJndW1lbnRzID0ge1xuICAgICAgLi4udGhpcy5leGVjdXRhYmxlQXJndW1lbnRzKGV4ZWN1dGFibGUpLFxuICAgICAgLi4uY29udGludW91c0xvZ2dpbmdBcmdzLFxuICAgICAgLi4ucHJvZmlsaW5nTWV0cmljc0FyZ3MsXG4gICAgICAuLi5zcGFya1VJPy5hcmdzLFxuICAgICAgLi4udGhpcy5jaGVja05vUmVzZXJ2ZWRBcmdzKHByb3BzLmRlZmF1bHRBcmd1bWVudHMpLFxuICAgIH07XG5cbiAgICBjb25zdCBqb2JSZXNvdXJjZSA9IG5ldyBDZm5Kb2IodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgbmFtZTogcHJvcHMuam9iTmFtZSxcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIHJvbGU6IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgY29tbWFuZDoge1xuICAgICAgICBuYW1lOiBleGVjdXRhYmxlLnR5cGUubmFtZSxcbiAgICAgICAgc2NyaXB0TG9jYXRpb246IHRoaXMuY29kZVMzT2JqZWN0VXJsKGV4ZWN1dGFibGUuc2NyaXB0KSxcbiAgICAgICAgcHl0aG9uVmVyc2lvbjogZXhlY3V0YWJsZS5weXRob25WZXJzaW9uLFxuICAgICAgfSxcbiAgICAgIGdsdWVWZXJzaW9uOiBleGVjdXRhYmxlLmdsdWVWZXJzaW9uLm5hbWUsXG4gICAgICB3b3JrZXJUeXBlOiBwcm9wcy53b3JrZXJUeXBlPy5uYW1lLFxuICAgICAgbnVtYmVyT2ZXb3JrZXJzOiBwcm9wcy53b3JrZXJDb3VudCxcbiAgICAgIG1heENhcGFjaXR5OiBwcm9wcy5tYXhDYXBhY2l0eSxcbiAgICAgIG1heFJldHJpZXM6IHByb3BzLm1heFJldHJpZXMsXG4gICAgICBleGVjdXRpb25Qcm9wZXJ0eTogcHJvcHMubWF4Q29uY3VycmVudFJ1bnMgPyB7IG1heENvbmN1cnJlbnRSdW5zOiBwcm9wcy5tYXhDb25jdXJyZW50UnVucyB9IDogdW5kZWZpbmVkLFxuICAgICAgbm90aWZpY2F0aW9uUHJvcGVydHk6IHByb3BzLm5vdGlmeURlbGF5QWZ0ZXIgPyB7IG5vdGlmeURlbGF5QWZ0ZXI6IHByb3BzLm5vdGlmeURlbGF5QWZ0ZXIudG9NaW51dGVzKCkgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHRpbWVvdXQ6IHByb3BzLnRpbWVvdXQ/LnRvTWludXRlcygpLFxuICAgICAgY29ubmVjdGlvbnM6IHByb3BzLmNvbm5lY3Rpb25zID8geyBjb25uZWN0aW9uczogcHJvcHMuY29ubmVjdGlvbnMubWFwKChjb25uZWN0aW9uKSA9PiBjb25uZWN0aW9uLmNvbm5lY3Rpb25OYW1lKSB9IDogdW5kZWZpbmVkLFxuICAgICAgc2VjdXJpdHlDb25maWd1cmF0aW9uOiBwcm9wcy5zZWN1cml0eUNvbmZpZ3VyYXRpb24/LnNlY3VyaXR5Q29uZmlndXJhdGlvbk5hbWUsXG4gICAgICB0YWdzOiBwcm9wcy50YWdzLFxuICAgICAgZGVmYXVsdEFyZ3VtZW50cyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJlc291cmNlTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKGpvYlJlc291cmNlLnJlZik7XG4gICAgdGhpcy5qb2JBcm4gPSBqb2JBcm4odGhpcywgcmVzb3VyY2VOYW1lKTtcbiAgICB0aGlzLmpvYk5hbWUgPSByZXNvdXJjZU5hbWU7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgbm8gdXNhZ2Ugb2YgcmVzZXJ2ZWQgYXJndW1lbnRzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9nbHVlL2xhdGVzdC9kZy9hd3MtZ2x1ZS1wcm9ncmFtbWluZy1ldGwtZ2x1ZS1hcmd1bWVudHMuaHRtbFxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja05vUmVzZXJ2ZWRBcmdzKGRlZmF1bHRBcmd1bWVudHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9KSB7XG4gICAgaWYgKGRlZmF1bHRBcmd1bWVudHMpIHtcbiAgICAgIGNvbnN0IHJlc2VydmVkQXJncyA9IG5ldyBTZXQoWyctLWNvbmYnLCAnLS1kZWJ1ZycsICctLW1vZGUnLCAnLS1KT0JfTkFNRSddKTtcbiAgICAgIE9iamVjdC5rZXlzKGRlZmF1bHRBcmd1bWVudHMpLmZvckVhY2goKGFyZykgPT4ge1xuICAgICAgICBpZiAocmVzZXJ2ZWRBcmdzLmhhcyhhcmcpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgJHthcmd9IGFyZ3VtZW50IGlzIHJlc2VydmVkIGJ5IEdsdWUuIERvbid0IHNldCBpdGApO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIGRlZmF1bHRBcmd1bWVudHM7XG4gIH1cblxuICBwcml2YXRlIGV4ZWN1dGFibGVBcmd1bWVudHMoY29uZmlnOiBKb2JFeGVjdXRhYmxlQ29uZmlnKSB7XG4gICAgY29uc3QgYXJnczogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgIGFyZ3NbJy0tam9iLWxhbmd1YWdlJ10gPSBjb25maWcubGFuZ3VhZ2U7XG4gICAgaWYgKGNvbmZpZy5jbGFzc05hbWUpIHtcbiAgICAgIGFyZ3NbJy0tY2xhc3MnXSA9IGNvbmZpZy5jbGFzc05hbWU7XG4gICAgfVxuICAgIGlmIChjb25maWcuZXh0cmFKYXJzICYmIGNvbmZpZy5leHRyYUphcnM/Lmxlbmd0aCA+IDApIHtcbiAgICAgIGFyZ3NbJy0tZXh0cmEtamFycyddID0gY29uZmlnLmV4dHJhSmFycy5tYXAoY29kZSA9PiB0aGlzLmNvZGVTM09iamVjdFVybChjb2RlKSkuam9pbignLCcpO1xuICAgIH1cbiAgICBpZiAoY29uZmlnLmV4dHJhUHl0aG9uRmlsZXMgJiYgY29uZmlnLmV4dHJhUHl0aG9uRmlsZXMubGVuZ3RoID4gMCkge1xuICAgICAgYXJnc1snLS1leHRyYS1weS1maWxlcyddID0gY29uZmlnLmV4dHJhUHl0aG9uRmlsZXMubWFwKGNvZGUgPT4gdGhpcy5jb2RlUzNPYmplY3RVcmwoY29kZSkpLmpvaW4oJywnKTtcbiAgICB9XG4gICAgaWYgKGNvbmZpZy5leHRyYUZpbGVzICYmIGNvbmZpZy5leHRyYUZpbGVzLmxlbmd0aCA+IDApIHtcbiAgICAgIGFyZ3NbJy0tZXh0cmEtZmlsZXMnXSA9IGNvbmZpZy5leHRyYUZpbGVzLm1hcChjb2RlID0+IHRoaXMuY29kZVMzT2JqZWN0VXJsKGNvZGUpKS5qb2luKCcsJyk7XG4gICAgfVxuICAgIGlmIChjb25maWcuZXh0cmFKYXJzRmlyc3QpIHtcbiAgICAgIGFyZ3NbJy0tdXNlci1qYXJzLWZpcnN0J10gPSAndHJ1ZSc7XG4gICAgfVxuICAgIHJldHVybiBhcmdzO1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cFNwYXJrVUkoZXhlY3V0YWJsZTogSm9iRXhlY3V0YWJsZUNvbmZpZywgcm9sZTogaWFtLklSb2xlLCBwcm9wczogU3BhcmtVSVByb3BzKSB7XG4gICAgaWYgKEpvYlR5cGUuUFlUSE9OX1NIRUxMID09PSBleGVjdXRhYmxlLnR5cGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU3BhcmsgVUkgaXMgbm90IGF2YWlsYWJsZSBmb3IgSm9iVHlwZS5QWVRIT05fU0hFTEwgam9icycpO1xuICAgIH1cblxuICAgIGNvbnN0IGJ1Y2tldCA9IHByb3BzLmJ1Y2tldCA/PyBuZXcgczMuQnVja2V0KHRoaXMsICdTcGFya1VJQnVja2V0Jyk7XG4gICAgYnVja2V0LmdyYW50UmVhZFdyaXRlKHJvbGUpO1xuICAgIGNvbnN0IGFyZ3MgPSB7XG4gICAgICAnLS1lbmFibGUtc3BhcmstdWknOiAndHJ1ZScsXG4gICAgICAnLS1zcGFyay1ldmVudC1sb2dzLXBhdGgnOiBidWNrZXQuczNVcmxGb3JPYmplY3QocHJvcHMucHJlZml4KSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGxvY2F0aW9uOiB7XG4gICAgICAgIHByZWZpeDogcHJvcHMucHJlZml4LFxuICAgICAgICBidWNrZXQsXG4gICAgICB9LFxuICAgICAgYXJncyxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cENvbnRpbnVvdXNMb2dnaW5nKHJvbGU6IGlhbS5JUm9sZSwgcHJvcHM6IENvbnRpbnVvdXNMb2dnaW5nUHJvcHMpIHtcbiAgICBjb25zdCBhcmdzOiB7W2tleTogc3RyaW5nXTogc3RyaW5nfSA9IHtcbiAgICAgICctLWVuYWJsZS1jb250aW51b3VzLWNsb3Vkd2F0Y2gtbG9nJzogJ3RydWUnLFxuICAgICAgJy0tZW5hYmxlLWNvbnRpbnVvdXMtbG9nLWZpbHRlcic6IChwcm9wcy5xdWlldCA/PyB0cnVlKS50b1N0cmluZygpLFxuICAgIH07XG5cbiAgICBpZiAocHJvcHMubG9nR3JvdXApIHtcbiAgICAgIGFyZ3NbJy0tY29udGludW91cy1sb2ctbG9nR3JvdXAnXSA9IHByb3BzLmxvZ0dyb3VwLmxvZ0dyb3VwTmFtZTtcbiAgICAgIHByb3BzLmxvZ0dyb3VwLmdyYW50V3JpdGUocm9sZSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmxvZ1N0cmVhbVByZWZpeCkge1xuICAgICAgYXJnc1snLS1jb250aW51b3VzLWxvZy1sb2dTdHJlYW1QcmVmaXgnXSA9IHByb3BzLmxvZ1N0cmVhbVByZWZpeDtcbiAgICB9XG4gICAgaWYgKHByb3BzLmNvbnZlcnNpb25QYXR0ZXJuKSB7XG4gICAgICBhcmdzWyctLWNvbnRpbnVvdXMtbG9nLWNvbnZlcnNpb25QYXR0ZXJuJ10gPSBwcm9wcy5jb252ZXJzaW9uUGF0dGVybjtcbiAgICB9XG4gICAgcmV0dXJuIGFyZ3M7XG4gIH1cblxuICBwcml2YXRlIGNvZGVTM09iamVjdFVybChjb2RlOiBDb2RlKSB7XG4gICAgY29uc3QgczNMb2NhdGlvbiA9IGNvZGUuYmluZCh0aGlzLCB0aGlzLnJvbGUpLnMzTG9jYXRpb247XG4gICAgcmV0dXJuIGBzMzovLyR7czNMb2NhdGlvbi5idWNrZXROYW1lfS8ke3MzTG9jYXRpb24ub2JqZWN0S2V5fWA7XG4gIH1cbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBDbG91ZFdhdGNoIE1ldHJpYyB0aGF0J3MgYmFzZWQgb24gR2x1ZSBKb2IgZXZlbnRzXG4gKiB7QHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvZXZlbnRzL0V2ZW50VHlwZXMuaHRtbCNnbHVlLWV2ZW50LXR5cGVzfVxuICogVGhlIG1ldHJpYyBoYXMgbmFtZXNwYWNlID0gJ0FXUy9FdmVudHMnLCBtZXRyaWNOYW1lID0gJ1RyaWdnZXJlZFJ1bGVzJyBhbmQgUnVsZU5hbWUgPSBydWxlLnJ1bGVOYW1lIGRpbWVuc2lvbi5cbiAqXG4gKiBAcGFyYW0gcnVsZSBmb3IgdXNlIGluIHNldHRpbmcgUnVsZU5hbWUgZGltZW5zaW9uIHZhbHVlXG4gKiBAcGFyYW0gcHJvcHMgbWV0cmljIHByb3BlcnRpZXNcbiAqL1xuZnVuY3Rpb24gbWV0cmljUnVsZShydWxlOiBldmVudHMuSVJ1bGUsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICBuYW1lc3BhY2U6ICdBV1MvRXZlbnRzJyxcbiAgICBtZXRyaWNOYW1lOiAnVHJpZ2dlcmVkUnVsZXMnLFxuICAgIGRpbWVuc2lvbnM6IHsgUnVsZU5hbWU6IHJ1bGUucnVsZU5hbWUgfSxcbiAgICBzdGF0aXN0aWM6IGNsb3Vkd2F0Y2guU3RhdGlzdGljLlNVTSxcbiAgICAuLi5wcm9wcyxcbiAgfSkuYXR0YWNoVG8ocnVsZSk7XG59XG5cblxuLyoqXG4gKiBSZXR1cm5zIHRoZSBqb2IgYXJuXG4gKiBAcGFyYW0gc2NvcGVcbiAqIEBwYXJhbSBqb2JOYW1lXG4gKi9cbmZ1bmN0aW9uIGpvYkFybihzY29wZTogY29uc3RydWN0cy5Db25zdHJ1Y3QsIGpvYk5hbWU6IHN0cmluZykgOiBzdHJpbmcge1xuICByZXR1cm4gY2RrLlN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oe1xuICAgIHNlcnZpY2U6ICdnbHVlJyxcbiAgICByZXNvdXJjZTogJ2pvYicsXG4gICAgcmVzb3VyY2VOYW1lOiBqb2JOYW1lLFxuICB9KTtcbn1cbiJdfQ==