"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @stability stable
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, props);
        this.props = props;
        this.visibleToAllUsers = (_b = this.props.visibleToAllUsers) !== null && _b !== void 0 ? _b : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_c = this.props.serviceRole) !== null && _c !== void 0 ? _c : this.createServiceRole();
        this._clusterRole = (_d = this.props.clusterRole) !== null && _d !== void 0 ? _d : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
        if (this.props.releaseLabel !== undefined) {
            this.validateReleaseLabel(this.props.releaseLabel);
        }
        if (this.props.stepConcurrencyLevel !== undefined) {
            if (this.props.stepConcurrencyLevel < 1 || this.props.stepConcurrencyLevel > 256) {
                throw new Error(`Step concurrency level must be in range [1, 256], but got ${this.props.stepConcurrencyLevel}.`);
            }
            if (this.props.releaseLabel && this.props.stepConcurrencyLevel !== 1) {
                const [major, minor] = this.props.releaseLabel.substr(4).split('.');
                if (Number(major) < 5 || (Number(major) === 5 && Number(minor) < 28)) {
                    throw new Error(`Step concurrency is only supported in EMR release version 5.28.0 and above but got ${this.props.releaseLabel}.`);
                }
            }
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b, _c, _d;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_b = this._autoScalingRole) === null || _b === void 0 ? void 0 : _b.roleName),
                AutoTerminationPolicy: this.props.autoTerminationPolicy ? cluster_utils_1.AutoTerminationPolicyPropertyToJson(this.props.autoTerminationPolicy) : undefined,
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_c = this.props.ebsRootVolumeSize) === null || _c === void 0 ? void 0 : _c.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_d = this.props.scaleDownBehavior) === null || _d === void 0 ? void 0 : _d.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                StepConcurrencyLevel: cdk.numberToCloudFormation(this.props.stepConcurrencyLevel),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _b;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_b = role.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
    /**
     * Validates the release label string is in proper format.
     * Release labels are in the form `emr-x.x.x`. For example, `emr-5.33.0`.
     *
     * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-release-components.html
     */
    validateReleaseLabel(releaseLabel) {
        const prefix = releaseLabel.substr(0, 4);
        const versions = releaseLabel.substr(4).split('.');
        if (prefix !== 'emr-' || versions.length !== 3 || versions.some((e) => isNotANumber(e))) {
            throw new Error(`The release label must be in the format 'emr-x.x.x' but got ${releaseLabel}`);
        }
        return releaseLabel;
        function isNotANumber(value) {
            return value === '' || isNaN(Number(value));
        }
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrCreateCluster[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrCreateCluster", version: "1.132.0" };
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @stability stable
 */
(function (EmrCreateCluster) {
    /**
     * The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
     * @stability stable
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types.
     *
     * @stability stable
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        InstanceRoleType["MASTER"] = "MASTER";
        InstanceRoleType["CORE"] = "CORE";
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types.
     *
     * @stability stable
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions.
     *
     * @stability stable
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Spot Allocation Strategies
     *
     * Specifies the strategy to use in launching Spot Instance fleets. For example, "capacity-optimized" launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.
     *
     * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
     *
     */
    let SpotAllocationStrategy;
    (function (SpotAllocationStrategy) {
        SpotAllocationStrategy["CAPACITY_OPTIMIZED"] = "capacity-optimized";
    })(SpotAllocationStrategy = EmrCreateCluster.SpotAllocationStrategy || (EmrCreateCluster.SpotAllocationStrategy = {}));
    /**
     * CloudWatch Alarm Comparison Operators.
     *
     * @stability stable
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics.
     *
     * @stability stable
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units.
     *
     * @stability stable
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        CloudWatchAlarmUnit["NONE"] = "NONE";
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        CloudWatchAlarmUnit["BITS"] = "BITS";
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market.
     *
     * @stability stable
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type.
     *
     * @stability stable
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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