# AUTOGENERATED! DO NOT EDIT! File to edit: ../01_datasets.ipynb.

# %% ../01_datasets.ipynb 5
from __future__ import annotations  # for type hints, in LAION code samples
import numpy as np 
import torch
import torch.nn as nn
import torchaudio
from torchaudio import transforms as T
from torchvision import transforms as VT
import random
import os
import json
import tqdm
from multiprocessing import Pool, cpu_count
from functools import partial
from .core import load_audio, get_audio_filenames, is_silence, untuple
from .viz import playable_spectrogram
from fastcore.utils import *
import webdataset as wds
import subprocess
import re

# %% auto 0
__all__ = ['PadCrop', 'PhaseFlipper', 'FillTheNoise', 'RandPool', 'NormInputs', 'Mono', 'Stereo', 'RandomGain', 'get_s3_contents',
           'get_contiguous_range', 'fix_double_slashes', 'get_all_s3_urls', 'AudioDataset', 'IterableAudioDataset',
           'HybridAudioDataset', 'wds_preprocess', 'QuickWebDataLoader']

# %% ../01_datasets.ipynb 7
class PadCrop(nn.Module):
    def __init__(self, 
        n_samples,           # length of chunk to extract from longer signal
        randomize=True,      # draw cropped chunk from a random position in audio file
        redraw_silence=True, # a chunk containing silence will be replaced with a new one
        silence_thresh=-60,  # threshold in dB below which we declare to be silence
        max_redraws=2        # when redrawing silences, don't do it more than this many
        ):
        super().__init__()
        store_attr()     # sets self.___ vars automatically
    
    def draw_chunk(self, signal):
        "here's the part that actually draws a cropped/padded chunk of audio from signal"
        n, s = signal.shape
        start = 0 if (not self.randomize) else torch.randint(0, max(0, s - self.n_samples) + 1, []).item()
        end = start + self.n_samples
        chunk = signal.new_zeros([n, self.n_samples])
        chunk[:, :min(s, self.n_samples)] = signal[:, start:end]
        return chunk
    
    def __call__(self, signal):
        "when part of the pipline, this will grab a padded/cropped chunk from signal"
        chunk = self.draw_chunk(signal)
        num_redraws = 0
        while self.redraw_silence and is_silence(chunk, thresh=self.silence_thresh) and (num_redraws < self.max_redraws):
            #print(f"    PadCrop: Got silence.  Redrawing. Try {num_redraws+1} of {self.max_redraws}")
            chunk, num_redraws = self.draw_chunk(signal), num_redraws+1
        return chunk

# %% ../01_datasets.ipynb 8
class PhaseFlipper(nn.Module):
    "she was PHAAAAAAA-AAAASE FLIPPER, a random invert yeah"
    def __init__(self, 
        p=0.5  # probability that phase flip will be applied
        ):
        super().__init__()
        self.p = p
    def __call__(self, signal):
        return -signal if (random.random() < self.p) else signal

# %% ../01_datasets.ipynb 9
class FillTheNoise(nn.Module):
    "randomly adds a bit of noise, just to spice things up"
    def __init__(self, 
        p=0.33       # probability that noise will be added
        ):
        super().__init__()
        self.p = p
    def __call__(self, signal):
        return signal + 0.25*random.random()*(2*torch.rand_like(signal)-1) if (random.random() < self.p) else signal

# %% ../01_datasets.ipynb 10
class RandPool(nn.Module):
    def __init__(self, p=0.2):
        self.p, self.maxkern = p, 100
    def __call__(self, signal):
        if (random.random() < self.p):
            ksize = int(random.random()*self.maxkern)
            avger = nn.AvgPool1d(kernel_size=ksize, stride=1, padding=1)
            return avger(signal)
        else:
            return signal

# %% ../01_datasets.ipynb 11
class NormInputs(nn.Module):
    "Normalize inputs to [-1,1]. Useful for quiet inputs"
    def __init__(self, 
        do_norm=True    # controllable parameter for turning normalization on/off
        ):
        super().__init__()
        self.do_norm = do_norm
        self.eps = 1e-2
    def __call__(self, signal):
        return signal if (not self.do_norm) else signal/(torch.amax(signal,-1)[0] + self.eps)

# %% ../01_datasets.ipynb 12
class Mono(nn.Module):
    "convert audio to mono"
    def __call__(self, signal):
        return torch.mean(signal, dim=0) if len(signal.shape) > 1 else signal

# %% ../01_datasets.ipynb 13
class Stereo(nn.Module):
    "convert audio to stereo"
    def __call__(self, signal):
        signal_shape = signal.shape
        # Check if it's mono
        if len(signal.shape) == 1: # s -> 2, s
            signal = signal.unsqueeze(0).repeat(2, 1)
        elif len(signal_shape) == 2:
            if signal.shape[0] == 1: #1, s -> 2, s
                signal = signal.repeat(2, 1)
            elif signal.shape[0] > 2: #?, s -> 2,s
                signal = signal[:2, :]    
        return signal

# %% ../01_datasets.ipynb 14
class RandomGain(nn.Module):
    "apply a random gain to audio"
    def __init__(self, min_gain, max_gain):
        super().__init__()
        self.min_gain = min_gain
        self.max_gain = max_gain

    def __call__(self, signal):
        gain = random.uniform(self.min_gain, self.max_gain)
        signal = signal * gain
        return signal

# %% ../01_datasets.ipynb 18
def get_s3_contents(
    dataset_path,     # "name" of the dataset on s3
    s3_url_prefix='s3://s-laion-audio/webdataset_tar/',  # s3 bucket to check
    filter='',       # only grab certain filename / extensions
    recursive=True,  # check all subdirectories. RECOMMEND LEAVING THIS TRUE
    debug=False,     # print debugging info (don't rely on this info staying consistent)
    ):
    "Gets a list of names of files or subdirectories on an s3 path"
    if (dataset_path != '') and (not dataset_path.endswith('/')): 
        dataset_path = dataset_path + '/'
    dataset_path = dataset_path.replace('//','/') # aws is baffled by double slashes in dir names
    if not recursive:
        run_ls = subprocess.run(['aws','s3','ls',f'{s3_url_prefix}{dataset_path}'], capture_output=True)
    else:
        run_ls = subprocess.run(['aws','s3','ls',f'{s3_url_prefix}{dataset_path}','--recursive'], capture_output=True)
        run_ls = subprocess.run(["awk",'{$1=$2=$3=""; print $0}'], input=run_ls.stdout, capture_output=True)
        run_ls = subprocess.run(["sed",'s/^[ \t]*//'], input=run_ls.stdout, capture_output=True)
    contents = run_ls.stdout.decode('utf-8').split('\n') 
    contents = [x.strip() for x in contents if x]      # list of non-empty strings, without leading whitespace
    contents = [x.replace('PRE ','') if (x[-1]=='/') else x for x in contents]  # directories
    #contents = [''.join(x.split(' ')[4:]) if (x[-1]!='/') else x for x in contents]    # files
    if recursive:  # recursive flag weirdly adds redundant extr directory name taken from s3 url, so we should strip
        main_dir = s3_url_prefix.split('/')[-2]
        if debug: print("main_dir =",main_dir)
        contents = [x.replace(f'{main_dir}/','').replace(dataset_path,'').replace('//','/') for x in contents]
        if debug: print("contents = ",contents)
    return [x for x in contents if filter in x] # return filtered list

# %% ../01_datasets.ipynb 29
def get_contiguous_range(
    tar_names, # list of tar file names, although the .tar part is actually optional
    ):
    "given a string of tar file names, return a string of their range if the numbers are contiguous. Otherwise return empty string"
    if len(tar_names) == 0:  return ''
    elif len(tar_names) == 1: return tar_names[-1]
    just_nums = [x.replace('.tar','') for x in tar_names]
    just_nums.sort(key=int) # sorts numerically but meaningfully preserves leading zeros in strings
    nums_arr = np.asarray(just_nums,  dtype=int)
    is_contiguous =  np.abs( (nums_arr - np.roll(nums_arr,1)) [1:] ).max() == 1
    if is_contiguous:   # {000000..000999}
        return '{' + f'{just_nums[0]}..{just_nums[-1]}' +'}'
    else:
        print("get_contiguous_range: File numbers not continuous")  # have to do more work
        return '' # empty string will signify no dice; signal for more work to be done

# %% ../01_datasets.ipynb 47
def fix_double_slashes(s, debug=False):
    "aws is pretty unforgiving compared to 'normal' filesystems. so here's some 'cleanup'"
    cdsh_split = s.split('://')
    assert (len(cdsh_split) <= 2) and (len(cdsh_split) > 0), f'what kind of string are you using? s={s}'
    post = cdsh_split[-1]
    while '//' in post: 
        post = post.replace('//','/')
    if len(cdsh_split) > 1: 
        return cdsh_split[0] + '://' + post
    else:
        return post

# %% ../01_datasets.ipynb 51
def get_all_s3_urls(
    names=['FSD50K'],    # list of all valid [LAION AudioDataset] dataset names 
    subsets=[''],   # list of subsets you want from those datasets, e.g. ['train','valid']
    s3_url_prefix='s3://s-laion-audio/webdataset_tar/',   # prefix for those
    recursive=True,  # recursively list all tar files in all subdirs
    filter_str='tar', # only grab files with this substring
    debug=False,     # print debugging info -- note: info displayed likely to change at dev's whims
    ): 
    "get urls of shards (tar files) for multiple datasets in one s3 bucket"
    if s3_url_prefix[-1] != '/':  s3_url_prefix = s3_url_prefix + '/'
    urls = []
    names = [''] if names == [] else names # for loop below
    subsets = [''] if subsets == [] else subsets # for loop below
    for name in names:
        if debug: print(f"get_all_s3_urls: {s3_url_prefix}{name}:")
        for subset in subsets:
            contents_str = fix_double_slashes(f'{name}/{subset}/')
            if debug: print("contents_str =",contents_str)
            tar_list = get_s3_contents(contents_str, s3_url_prefix=s3_url_prefix, recursive=recursive, filter=filter_str, debug=debug)
            for tar in tar_list:
                tar = tar.replace(" ","\ ").replace("(","\(").replace(")","\)") # escape spaces and parentheses for shell
                s3_path  = f"{name}/{subset}/{tar} -"
                while '//' in s3_path:  # aws hates double-slashes
                    s3_path = s3_path.replace('//','/')
                request_str = f"pipe:aws s3 --cli-connect-timeout 0 cp {s3_url_prefix}{s3_path}" 
                if debug: print("request_str = ",request_str)
                urls.append(fix_double_slashes(request_str))
    #urls = [x.replace('tar//','tar/') for x in urls] # one last double-check
    return urls

# %% ../01_datasets.ipynb 55
class AudioDataset(torch.utils.data.Dataset):
    """
    Reads from a tree of directories and serves up cropped bits from any and all audio files
    found therein. For efficiency, best if you "chunk" these files via chunkadelic
    modified from https://github.com/drscotthawley/audio-diffusion/blob/main/dataset/dataset.py
    """
    def __init__(self, 
        paths,             # list of strings of directory (/tree) names to draw audio files from
        sample_rate=48000, # audio sample rate in Hz
        sample_size=65536, # how many audio samples in each "chunk"
        random_crop=True,  # take chunks from random positions within files
        load_frac=1.0,     # fraction of total dataset to load
        cache_training_data=False,  # True = pre-load whole dataset into memory (not fully supported)
        num_gpus=8,        # used only when `cache_training_data=True`, to avoid duplicates,
        redraw_silence=True, # a chunk containing silence will be replaced with a new one
        silence_thresh=-60,  # threshold in dB below which we declare to be silence
        max_redraws=2,        # when redrawing silences, don't do it more than this many
        augs='Stereo(), PhaseFlipper()', # list of augmentation transforms **after PadCrop**, as a string
        verbose=False,       # whether to print notices of reasampling or not
        ):
        super().__init__()
    
        print("augs =",augs)
        # base_augs are always applied
        base_augs = 'PadCrop(sample_size, randomize=random_crop, redraw_silence=redraw_silence, silence_thresh=silence_thresh, max_redraws=max_redraws)'
        self.augs = eval(f'torch.nn.Sequential( {base_augs}, {augs} )')  if augs is not None else None 
        self.silence_thresh = silence_thresh
        self.redraw_silence = redraw_silence
        self.max_redraws = max_redraws
        self.sr = sample_rate
        self.cache_training_data = cache_training_data
        self.verbose = verbose

        self.filenames = get_audio_filenames(paths)
        print(f"AudioDataset:{len(self.filenames)} files found.")
        self.n_files = int(len(self.filenames)*load_frac)
        self.filenames = self.filenames[0:self.n_files]
        if cache_training_data: self.preload_files()

        self.convert_tensor = VT.ToTensor()

    def load_file_ind(self, file_list,i): # used when caching training data
        return load_audio(file_list[i], sr=self.sr, verbose=self.verbose).cpu()

    def get_data_range(self): # for parallel runs, only grab part of the data -- OBVIATED BY CHUNKING.
        start, stop = 0, len(self.filenames)
        try:
            local_rank = int(os.environ["LOCAL_RANK"])
            world_size = int(os.environ["WORLD_SIZE"])
            interval = stop//world_size
            start, stop = local_rank*interval, (local_rank+1)*interval
            return start, stop
        except KeyError as e: # we're on GPU 0 and the others haven't been initialized yet
            start, stop = 0, len(self.filenames)//self.num_gpus
            return start, stop

    def preload_files(self):
        print(f"Caching {self.n_files} input audio files:")
        wrapper = partial(self.load_file_ind, self.filenames)
        start, stop = self.get_data_range()
        with Pool(processes=cpu_count()) as p:   # //8 to avoid FS bottleneck and/or too many processes (b/c * num_gpus)
            self.audio_files = list(tqdm.tqdm(p.imap(wrapper, range(start,stop)), total=stop-start))

    def __len__(self):
        return len(self.filenames)
    
    
    def get_next_chunk(self, 
        idx     # the index of the file within the list of files
        ):
        "The heart of this whole dataset routine"
        audio_filename = self.filenames[idx]
        try:
            if self.cache_training_data:
                audio = self.audio_files[idx] # .copy()
            else:
                audio = load_audio(audio_filename, sr=self.sr, verbose=self.verbose)
            if self.augs is not None:         #Run augmentations on this sample (including random crop)
                audio = self.augs(audio)
            audio = audio.clamp(-1, 1)
            return audio
        
        except Exception as e:
            print(f'AudioDataset.get_next_chunk: Error loading file {audio_filename}: {e}')
            return None
        
        
    def __getitem__(self, 
        idx     # the index of the file within the list of files
        ):
        audio = self.get_next_chunk(idx)
        # even with PadCrop set to reject silences, it could be that the whole file is silence; 
        num_redraws = 0 
        while (audio is None) or (self.redraw_silence and is_silence(audio, thresh=self.silence_thresh) \
            and (num_redraws < self.max_redraws)):
            #print(f"AudioDataset.__getitem__: Got None or silence (torch.max = {torch.max(audio)})  Redrawing. Attempt {num_redraws+1} of {self.max_redraws}")
            next_idx = random.randint(0,len(self.filenames)-1)     # pick some other file at random
            audio, num_redraws = self.get_next_chunk(next_idx), num_redraws+1
               
        return self[random.randrange(len(self))] if (audio is None) else audio

# %% ../01_datasets.ipynb 58
class IterableAudioDataset(torch.utils.data.IterableDataset):
    "Iterable version of AudioDataset, used with Chain (below)"
    def __init__(self, 
        paths,             # list of strings of directory (/tree) names to draw audio files from
        sample_rate=48000, # audio sample rate in Hz
        sample_size=65536, # how many audio samples in each "chunk"
        random_crop=True,  # take chunks from random positions within files
        load_frac=1.0,     # fraction of total dataset to load
        cache_training_data=False,  # True = pre-load whole dataset into memory (not fully supported)
        num_gpus=8,        # used only when `cache_training_data=True`, to avoid duplicates,
        redraw_silence=True, # a chunk containing silence will be replaced with a new one
        silence_thresh=-60,  # threshold in dB below which we declare to be silence
        max_redraws=2,        # when redrawing silences, don't do it more than this many
        augs='Stereo(), PhaseFlipper()', # list of augmentation transforms **after PadCrop**, as a string
        verbose=False,       # whether to print notices of reasampling or not
        ):
        super().__init__()
        self.this = AudioDataset(paths, sample_rate=sample_rate, sample_size=sample_size, random_crop=random_crop,
                                load_frac=load_frac, cache_training_data=cache_training_data, num_gpus=num_gpus,
                                redraw_silence=redraw_silence, silence_thresh=silence_thresh, max_redraws=max_redraws,
                                augs=augs, verbose=verbose)
        self.len = len(self.this)
        
    def __iter__(self):
        yield self.this.__getitem__(random.randint(0, self.len))

# %% ../01_datasets.ipynb 61
class HybridAudioDataset(torch.utils.data.IterableDataset):
    "Combines AudioDataset and WebDataset"
    def __init__(self, 
        local_paths:list,      # list of local paths names to draw audio files from (recursively)
        webdataset_names:list,    # list of LAION Audiodataset or Internet Archive dataset names
        sample_rate=48000, # audio sample rate in Hz
        sample_size=65536, # how many audio samples in each "chunk"
        random_crop=True,  # take chunks from random positions within files
        load_frac=1.0,     # fraction of total dataset to load
        cache_training_data=False,  # True = pre-load whole dataset into memory (not fully supported)
        num_gpus=8,        # used only when `cache_training_data=True`, to avoid duplicates,
        redraw_silence=True, # a chunk containing silence will be replaced with a new one
        silence_thresh=-60,  # threshold in dB below which we declare to be silence
        max_redraws=2,        # when redrawing silences, don't do it more than this many
        augs='Stereo(), PhaseFlipper()', # list of augmentation transforms **after PadCrop**, as a string
        verbose=False,      # whether to print notices of reasampling or not
        subsets=[],         # can specify, e.g. ['train','valid'] to exclude 'test'. default= grab everything!
        s3_url_prefixes=['s3://s-laion-audio/webdataset_tar/',
                         's3://iarchive-stability/',
                         's3://s-harmonai/datasets/'], # where to look on s3 for things
        recursive=True,     # grab all tar files ("shards") recursively
        debug=False,        # print debugging info
        ):
        "Combines local paths and WebDataset (s3:) datasets, streaming. Recommend leaving local_paths blank"
        super().__init__()
        
        if isinstance(webdataset_names, str): webdataset_names = [webdataset_names] # if it's just a string, make it a list
        base_augs = 'PadCrop(sample_size, randomize=random_crop, redraw_silence=redraw_silence, silence_thresh=silence_thresh, max_redraws=max_redraws)'
        self.augs = eval(f'torch.nn.Sequential( {base_augs}, {augs} )')  
        self.redraw_silence = redraw_silence
        self.max_redraws = max_redraws
        self.silence_thresh = silence_thresh

        self.local_ds, self.web_ds, self.len = None, None, 0
        # local paths
        if len(local_paths) > 0:
            self.local_ds = IterableAudioDataset(local_paths, sample_rate=sample_rate, sample_size=sample_size, random_crop=random_crop,
                                    load_frac=load_frac, cache_training_data=cache_training_data, num_gpus=num_gpus,
                                    redraw_silence=redraw_silence, silence_thresh=silence_thresh, max_redraws=max_redraws,
                                    augs=None, verbose=verbose)  # do augs later
        # web i.e. s3 paths
        urls= []
        for s3_url_prefix in s3_url_prefixes: # loop over various s3 bucket urls. this is maybe dumb/slow but generally will only execute once per run
            subset_list = [] if 'laion' not in s3_url_prefix else subsets
            urls0 = get_all_s3_urls(names=webdataset_names, subsets=subset_list, s3_url_prefix=s3_url_prefix, recursive=recursive, debug=debug)
            if debug: print("urls0 = \n",urls0)
            if len(urls0) > 0:  urls = urls + urls0
        self.web_ds = wds.WebDataset(urls,  nodesplitter=wds.split_by_node).decode(wds.torch_audio).to_tuple("flac") 
            
        self.ds_list = []
        if (self.local_ds != None): self.ds_list.append(self.local_ds)
        if (self.web_ds != None): self.ds_list.append(self.web_ds)


    def get_next_chunk(self):
        ds_choice = random.choice(self.ds_list)  # randomly pick from available ds list
        audio = untuple(next(iter(ds_choice)))   
        if self.augs is not None: #Run augmentations on this sample (including random crop)
            audio = self.augs(audio)
        audio = audio.clamp(-1, 1)
        return audio
            
        
    def __iter__(self):
        audio = self.get_next_chunk()
        num_redraws = 0 
        while (audio is None) or (self.redraw_silence and is_silence(audio, thresh=self.silence_thresh) \
            and (num_redraws < self.max_redraws)):
            audio, num_redraws = self.get_next_chunk(), num_redraws+1

        yield audio

# %% ../01_datasets.ipynb 68
def wds_preprocess(sample, sample_size=65536, sample_rate=48000, verbose=False):
    "utility routine for QuickWebDataLoader, below"
    audio_keys = ("flac", "wav", "mp3", "aiff")
    found_key, rewrite_key = '', ''
    for k,v in sample.items():  # print the all entries in dict
        for akey in audio_keys:
            if k.endswith(akey): 
                found_key, rewrite_key = k, akey  # to rename long/weird key with its simpler counterpart
                break
        if '' != found_key: break 
    if '' == found_key:  # got no audio!   
        print("  Error: No audio in this sample:")
        for k,v in sample.items():  # print the all entries in dict
            print(f"    {k:20s} {repr(v)[:50]}")
        print("       Skipping it.")
        return None  # try returning None to tell WebDataset to skip this one ?   
    
    audio, in_sr = sample[found_key]
    if in_sr != sample_rate:
        if verbose: print(f"Resampling {filename} from {in_sr} Hz to {sample_rate} Hz",flush=True)
        resample_tf = T.Resample(in_sr, sample_rate)
        audio = resample_tf(audio)        
    myop = torch.nn.Sequential(PadCrop(sample_size), Stereo(), PhaseFlipper())
    audio = myop(audio)
    if found_key != rewrite_key:   # rename long/weird key with its simpler counterpart
        del sample[found_key]
    sample[rewrite_key] = audio    
    return sample

# %% ../01_datasets.ipynb 69
def QuickWebDataLoader(
    names=['ekto/1'], # names of datasets. will search laion, harmonai & IA s3 buckets for these
    sample_size=65536, # how long each sample to grab via PadCrop
    sample_rate=48000, # standard sr in Hz
    num_workers=4,    # in the PyTorch DataLoader sense
    batch_size=4,     # typical batch size
    audio_file_ext='flac',  # yep this one only supports one extension at a time. try HybridAudioDataset for more
    shuffle_vals=[1000, 10000],  # values passed into shuffle as per WDS tutorials
    epoch_len=1000,    # how many passes/loads make for an epoch? wds part of this is not well documented IMHO
    debug=False,       # print info on internal workings
    verbose=False,     # not quite the same as debug. print things like notices of resampling
    callback=wds_preprocess, # function to call for additional user-based processing
    **kwargs,          # what else to pass to callback
    ):
    "Minimal/quick implementation: Sets up a WebDataLoader with some typical defaults"
    print("Note: 'Broken pipe' messages you might get aren't a big deal, but may indicate files that are too big.")
    if names is not list: names = [names]
    urls = get_all_s3_urls(names=names, recursive=True, debug=debug) 
    if debug: print("urls =\n",urls)
    dataset = wds.DataPipeline(
        wds.ResampledShards(urls),
        wds.tarfile_to_samples(),
        wds.shuffle(shuffle_vals[0]),
        wds.decode(wds.torch_audio),
        wds.map(partial(callback, sample_size=sample_size, sample_rate=sample_rate, verbose=verbose, **kwargs)),
        wds.shuffle(shuffle_vals[1]),
        wds.to_tuple(audio_file_ext),
        wds.batched(batch_size)
    ).with_epoch(epoch_len)
    return wds.WebLoader(dataset, num_workers=num_workers)
