"""
@author: Quantmoon Technologies
webpage: https://www.quantmoon.tech//
"""

from abc import ABCMeta, abstractmethod
from .event import FillEvent
from datetime import datetime

class ExecutionHandler(object):
    """
    The ExecutionHandler abstract class handles the interaction
    between a set of order objects generated by a Portfolio and
    the ultimate set of Fill objects that actually occur in the
    market. 

    The handlers can be used to subclass simulated brokerages
    or live brokerages, with identical interfaces. This allows
    strategies to be backtested in a very similar manner to the
    live trading engine.
    """

    __metaclass__ = ABCMeta

    @abstractmethod
    def execute_order(self, event):
        """
        Takes an Order event and executes it, producing
        a Fill event that gets placed onto the Events queue.

        Parameters:
        -----------
        - event: contains an Event object with order information.
        
        Output:
        -------
        - Implement/NotImplement Error Statement
        """
        raise NotImplementedError("Should implement execute_order()")

class SimulatedExecutionHandler(ExecutionHandler):
    """
    The simulated execution handler simply converts all order
    objects into their equivalent fill objects automatically
    without latency, slippage or fill-ratio issues.

    This allows a straightforward "first go" test of any strategy,
    before implementation with a more sophisticated execution
    handler.
    """
    
    def __init__(self, events):
        """
        Initialises the handler, setting the event queues
        up internally.

        Parameters:
        -----------
        - events: The Queue of Event objects.
        """
        self.events = events
        


    def execute_order(self, event):
        """
        Simply converts Order objects into Fill objects naively,
        i.e. without any latency, slippage or fill ratio problems.

        Parameters:
        -----------
        - event: contains an Event object with order information.
        """
        if event.type == 'ORDER':
            # The "FillEvent" use 'ARCA' as purely name reference.
       
            fill_event = FillEvent(datetime.utcnow(),
                                   event.symbol,
                                   'ARCA', 
                                   event.quantity, 
                                   event.direction, 
                                   None)
            
            self.events.put(fill_event)
        
            