/*
Copyright 2015 Filosoft OÜ

This file is part of Estnltk. It is available under the license of GPLv2 found
in the top-level directory of this distribution and
at http://www.gnu.org/licenses/old-licenses/gpl-2.0.en.html .
No part of this file, may be copied, modified, propagated, or distributed
except according to the terms contained in the license.

This software is distributed on an "AS IS" basis, without warranties or conditions
of any kind, either express or implied.
*/
#include "stdfsc.h"
#include "fstype.h"

#include "fsstring2.h"

long g_lFSCodePage=FSCP_SYSTEM;

///////////////////////////////////////////////////////////
// String conversion
///////////////////////////////////////////////////////////

static const CFSKeyValuePair<UINT16, char[9]> g_HTMLMap[]={
{0x0022, "quot"}, {0x0026, "amp"}, {0x003C, "lt"}, {0x003E, "gt"}, {0x00A0, "nbsp"}, {0x00A1, "iexcl"}, {0x00A2, "cent"}, {0x00A3, "pound"}, {0x00A4, "curren"}, {0x00A5, "yen"}, {0x00A6, "brvbar"}, {0x00A7, "sect"}, {0x00A8, "uml"}, {0x00A9, "copy"}, {0x00AA, "ordf"}, {0x00AB, "laquo"}, 
{0x00AC, "not"}, {0x00AD, "shy"}, {0x00AE, "reg"}, {0x00AF, "macr"}, {0x00B0, "deg"}, {0x00B1, "plusmn"}, {0x00B2, "sup2"}, {0x00B3, "sup3"}, {0x00B4, "acute"}, {0x00B5, "micro"}, {0x00B6, "para"}, {0x00B7, "middot"}, {0x00B8, "cedil"}, {0x00B9, "sup1"}, {0x00BA, "ordm"}, {0x00BB, "raquo"}, 
{0x00BC, "frac14"}, {0x00BD, "frac12"}, {0x00BE, "frac34"}, {0x00BF, "iquest"}, {0x00C0, "Agrave"}, {0x00C1, "Aacute"}, {0x00C2, "Acirc"}, {0x00C3, "Atilde"}, {0x00C4, "Auml"}, {0x00C5, "Aring"}, {0x00C6, "AElig"}, {0x00C7, "Ccedil"}, {0x00C8, "Egrave"}, {0x00C9, "Eacute"}, {0x00CA, "Ecirc"}, {0x00CB, "Euml"}, 
{0x00CC, "Igrave"}, {0x00CD, "Iacute"}, {0x00CE, "Icirc"}, {0x00CF, "Iuml"}, {0x00D0, "ETH"}, {0x00D1, "Ntilde"}, {0x00D2, "Ograve"}, {0x00D3, "Oacute"}, {0x00D4, "Ocirc"}, {0x00D5, "Otilde"}, {0x00D6, "Ouml"}, {0x00D7, "times"}, {0x00D8, "Oslash"}, {0x00D9, "Ugrave"}, {0x00DA, "Uacute"}, {0x00DB, "Ucirc"}, 
{0x00DC, "Uuml"}, {0x00DD, "Yacute"}, {0x00DE, "THORN"}, {0x00DF, "szlig"}, {0x00E0, "agrave"}, {0x00E1, "aacute"}, {0x00E2, "acirc"}, {0x00E3, "atilde"}, {0x00E4, "auml"}, {0x00E5, "aring"}, {0x00E6, "aelig"}, {0x00E7, "ccedil"}, {0x00E8, "egrave"}, {0x00E9, "eacute"}, {0x00EA, "ecirc"}, {0x00EB, "euml"}, 
{0x00EC, "igrave"}, {0x00ED, "iacute"}, {0x00EE, "icirc"}, {0x00EF, "iuml"}, {0x00F0, "eth"}, {0x00F1, "ntilde"}, {0x00F2, "ograve"}, {0x00F3, "oacute"}, {0x00F4, "ocirc"}, {0x00F5, "otilde"}, {0x00F6, "ouml"}, {0x00F7, "divide"}, {0x00F8, "oslash"}, {0x00F9, "ugrave"}, {0x00FA, "uacute"}, {0x00FB, "ucirc"}, 
{0x00FC, "uuml"}, {0x00FD, "yacute"}, {0x00FE, "thorn"}, {0x00FF, "yuml"}, {0x0152, "OElig"}, {0x0153, "oelig"}, {0x0160, "Scaron"}, {0x0161, "scaron"}, {0x0178, "Yuml"}, {0x017D, "Zcaron"}, {0x017E, "zcaron"}, {0x0192, "fnof"}, {0x02C6, "circ"}, {0x02DC, "tilde"}, {0x0391, "Alpha"}, {0x0392, "Beta"}, 
{0x0393, "Gamma"}, {0x0394, "Delta"}, {0x0395, "Epsilon"}, {0x0396, "Zeta"}, {0x0397, "Eta"}, {0x0398, "Theta"}, {0x0399, "Iota"}, {0x039A, "Kappa"}, {0x039B, "Lambda"}, {0x039C, "Mu"}, {0x039D, "Nu"}, {0x039E, "Xi"}, {0x039F, "Omicron"}, {0x03A0, "Pi"}, {0x03A1, "Rho"}, {0x03A3, "Sigma"}, 
{0x03A4, "Tau"}, {0x03A5, "Upsilon"}, {0x03A6, "Phi"}, {0x03A7, "Chi"}, {0x03A8, "Psi"}, {0x03A9, "Omega"}, {0x03B1, "alpha"}, {0x03B2, "beta"}, {0x03B3, "gamma"}, {0x03B4, "delta"}, {0x03B5, "epsilon"}, {0x03B6, "zeta"}, {0x03B7, "eta"}, {0x03B8, "theta"}, {0x03B9, "iota"}, {0x03BA, "kappa"}, 
{0x03BB, "lambda"}, {0x03BC, "mu"}, {0x03BD, "nu"}, {0x03BE, "xi"}, {0x03BF, "omicron"}, {0x03C0, "pi"}, {0x03C1, "rho"}, {0x03C2, "sigmaf"}, {0x03C3, "sigma"}, {0x03C4, "tau"}, {0x03C5, "upsilon"}, {0x03C6, "phi"}, {0x03C7, "chi"}, {0x03C8, "psi"}, {0x03C9, "omega"}, {0x03D1, "thetasym"}, 
{0x03D2, "upsih"}, {0x03D6, "piv"}, {0x2002, "ensp"}, {0x2003, "emsp"}, {0x2009, "thinsp"}, {0x200C, "zwnj"}, {0x200D, "zwj"}, {0x200E, "lrm"}, {0x200F, "rlm"}, {0x2013, "ndash"}, {0x2014, "mdash"}, {0x2018, "lsquo"}, {0x2019, "rsquo"}, {0x201A, "sbquo"}, {0x201C, "ldquo"}, {0x201D, "rdquo"}, 
{0x201E, "bdquo"}, {0x2020, "dagger"}, {0x2021, "Dagger"}, {0x2022, "bull"}, {0x2026, "hellip"}, {0x2030, "permil"}, {0x2032, "prime"}, {0x2033, "Prime"}, {0x2039, "lsaquo"}, {0x203A, "rsaquo"}, {0x203E, "oline"}, {0x2044, "frasl"}, {0x20AC, "euro"}, {0x2111, "image"}, {0x2118, "weierp"}, {0x211C, "real"}, 
{0x2122, "trade"}, {0x2135, "alefsym"}, {0x2190, "larr"}, {0x2191, "uarr"}, {0x2192, "rarr"}, {0x2193, "darr"}, {0x2194, "harr"}, {0x21B5, "crarr"}, {0x21D0, "lArr"}, {0x21D1, "uArr"}, {0x21D2, "rArr"}, {0x21D3, "dArr"}, {0x21D4, "hArr"}, {0x2200, "forall"}, {0x2202, "part"}, {0x2203, "exist"}, 
{0x2205, "empty"}, {0x2207, "nabla"}, {0x2208, "isin"}, {0x2209, "notin"}, {0x220B, "ni"}, {0x220F, "prod"}, {0x2211, "sum"}, {0x2212, "minus"}, {0x2217, "lowast"}, {0x221A, "radic"}, {0x221D, "prop"}, {0x221E, "infin"}, {0x2220, "ang"}, {0x2227, "and"}, {0x2228, "or"}, {0x2229, "cap"}, 
{0x222A, "cup"}, {0x222B, "int"}, {0x2234, "there4"}, {0x223C, "sim"}, {0x2245, "cong"}, {0x2248, "asymp"}, {0x2260, "ne"}, {0x2261, "equiv"}, {0x2264, "le"}, {0x2265, "ge"}, {0x2284, "nsub"}, {0x2286, "sube"}, {0x2287, "supe"}, {0x2295, "oplus"}, {0x2297, "otimes"}, {0x22A5, "perp"}, 
{0x22C5, "sdot"}, {0x2308, "lceil"}, {0x2309, "rceil"}, {0x230A, "lfloor"}, {0x230B, "rfloor"}, {0x2329, "lang"}, {0x232A, "rang"}, {0x25CA, "loz"}, {0x2660, "spades"}, {0x2663, "clubs"}, {0x2665, "hearts"}, {0x2666, "diams"}, 
};
static const UINT16 g_HTMLNameIndex[]={
42, 37, 38, 36, 110, 41, 39, 40, 111, 43, 131, 178, 113, 52, 45, 46, 
44, 114, 116, 47, 112, 49, 50, 48, 118, 51, 119, 120, 121, 53, 122, 100, 
55, 56, 54, 133, 124, 60, 57, 58, 130, 125, 183, 132, 126, 102, 127, 66, 
128, 117, 62, 63, 61, 129, 64, 123, 65, 104, 105, 115, 69, 70, 24, 74, 
68, 193, 134, 1, 221, 220, 73, 229, 71, 72, 176, 135, 10, 179, 223, 75, 
28, 6, 156, 108, 249, 228, 13, 199, 224, 8, 203, 177, 197, 20, 137, 251, 
91, 77, 78, 76, 208, 163, 162, 138, 231, 140, 84, 79, 188, 207, 107, 205, 
33, 32, 34, 187, 136, 233, 3, 204, 198, 250, 180, 81, 82, 5, 80, 189, 
219, 225, 142, 35, 210, 83, 143, 200, 144, 245, 15, 194, 241, 174, 232, 243, 
216, 247, 167, 184, 171, 2, 19, 170, 25, 27, 215, 145, 209, 4, 169, 230, 
212, 16, 211, 234, 85, 146, 87, 88, 101, 86, 186, 158, 148, 237, 222, 14, 
30, 92, 89, 238, 90, 26, 206, 181, 239, 155, 149, 161, 21, 7, 182, 213, 
218, 157, 0, 202, 217, 246, 31, 196, 242, 175, 191, 18, 244, 150, 168, 185, 
172, 173, 103, 240, 11, 17, 152, 151, 227, 248, 235, 214, 29, 22, 23, 236, 
67, 153, 226, 141, 159, 164, 98, 109, 59, 192, 201, 94, 195, 95, 93, 12, 
160, 154, 96, 190, 147, 97, 9, 99, 106, 139, 166, 165, 
};

static CFSAString __FSStrWtoHTML(const CFSWString &WString, bool *pbError)
{
	bool bError=false;
	CFSAString szResult;
	for (INTPTR ip=0; WString[ip]; ip++){
		LCHAR Char;
		INTPTR ipUsed=FSStrGetFSLCHAR(&Char, WString, ip);
		if (!ipUsed) {
			szResult+='?';
			bError=true;
			continue;
		}
		else {
			ip+=ipUsed-1;
		}

		INTPTR ipElem=FSArrayFind2(g_HTMLMap, Char);
		if (ipElem>=0) {
			szResult+="&";
			szResult+=g_HTMLMap[ipElem].Value;
			szResult+=";";
		} else {
			if (Char<0x80) {
				szResult+=(char)Char;
			}
			else {
				CFSAString szNumber;
				szNumber.Format("&#%d;", Char);
				szResult+=szNumber;
			}
		}
	}
	if (pbError) {
		*pbError=bError;
	}
	return szResult;
}

static CFSWString __FSStrHTMLtoW(const CFSAString &AString, bool *pbError)
{
	bool bError=false;
	CFSWString szResult;
	INTPTR ipMapSize=sizeof(g_HTMLNameIndex)/sizeof(UINT16);
	for (INTPTR ip=0; AString[ip]; ip++){
		if ((UCHAR)AString[ip]>=0x80) {
			szResult+='?'; bError=true;
			continue;
		}
		if (AString[ip]=='&') {
			INTPTR ipSemiPos=AString.Find(';', ip);
			if (ipSemiPos>=0 && ipSemiPos-ip<=10) {
				bool bFound=false;
				INTPTR ipStart=0;
				INTPTR ipEnd=ipMapSize-1;
				CFSAString szSymbol=AString.Mid(ip+1, ipSemiPos-ip-1);
				if (szSymbol[0]=='#') {
					LCHAR lChar=0;
					if (FSToUpper(szSymbol[1])=='X') {
						lChar=(LCHAR)strtol(szSymbol.Mid(2), 0, 16);
					} else{
						lChar=(LCHAR)strtol(szSymbol.Mid(1), 0, 10);
					}
					if (lChar){
						if (FSStrAppendFSLCHAR(szResult, lChar)==0) {
							szResult+='?';
							bError=true;
						}
						bFound=true;
					}
				}

				while (!bFound) {
					if (ipStart>=ipEnd-1) {
						if (ipStart==ipEnd-1 && szSymbol==g_HTMLMap[g_HTMLNameIndex[ipStart]].Value) {
							szResult+=g_HTMLMap[g_HTMLNameIndex[ipStart]].Key;
							bFound=true;
						}
						break;
					}

					INTPTR ipMiddle=(ipStart+ipEnd)/2;
					if (szSymbol<g_HTMLMap[g_HTMLNameIndex[ipMiddle]].Value) {
						ipEnd=ipMiddle;
					} else {
						ipStart=ipMiddle;
					}
				}
				if (!bFound){
					szResult+='?';
					bError=true;
				}
				ip=ipSemiPos;
			}
		}
		else {
			szResult+=(WCHAR)AString[ip];
		}
	}
	if (pbError) {
		*pbError=bError;
	}
	FSStrCombineW2(szResult);
	return szResult;
}

static CFSAString __FSStrWtoUTF8(const CFSWString &WString, bool *pbError)
{
	bool bError=false;
	CFSAString szResult;
	for (INTPTR ip=0; WString[ip]; ip++){
		LCHAR Char;
		INTPTR ipUsed=FSStrGetFSLCHAR(&Char, WString, ip);
		if (!ipUsed) {
			szResult+='?';
			bError=true;
			continue;
		}
		else {
			ip+=ipUsed-1;
		}

		if (Char<0x80) {
			szResult+=(char)Char;
		}
		else if (Char<0x800) {
			szResult+=(char)(0xC0|(Char>>6));
			szResult+=(char)(0x80|(Char&0x3F));
		}
		else if (Char<0x10000) {
			szResult+=(char)(0xE0|(Char>>12));
			szResult+=(char)(0x80|((Char>>6)&0x3F));
			szResult+=(char)(0x80|(Char&0x3F));
		}
		else if (Char<0x110000) {
			szResult+=(char)(0xF0|(Char>>18));
			szResult+=(char)(0x80|((Char>>12)&0x3F));
			szResult+=(char)(0x80|((Char>>6)&0x3F));
			szResult+=(char)(0x80|(Char&0x3F));
		}
		else {
			szResult+='?';
			bError=true;
		}
	}
	if (pbError) {
		*pbError=bError;
	}
	return szResult;
}

static CFSWString __FSStrUTF8toW(const CFSAString &AString, bool *pbError)
{
	bool bError=false;
	CFSWString szResult;
	for (INTPTR ip=0; AString[ip]; ip++) {
		UCHAR Char1=AString[ip];
		LCHAR lChar;
		if ((Char1&0x80)==0) {
			szResult+=(WCHAR)Char1;
			continue;
		}
		else if ((Char1&0xE0)==0xC0) { // 110????? + 10??????
			if ((AString[ip+1]&0xC0)!=0x80){
				lChar='?';
				bError=true;
			}
			else{
				BYTE Char2=AString[ip+1]&0x3f;
				lChar=(((LCHAR)Char1&0x1f)<<6) | Char2;
				ip++;
			}
		}
		else if ((Char1&0xF0)==0xE0){ // 1110???? + 10?????? + 10??????
			if ((AString[ip+1]&0xC0)!=0x80 || (AString[ip+2]&0xC0)!=0x80){
				lChar='?';
				bError=true;
			}
			else{
				BYTE Char2=AString[ip+1]&0x3f;
				BYTE Char3=AString[ip+2]&0x3f;
				lChar=(((LCHAR)Char1&0x0f)<<12) | (((LCHAR)Char2)<<6) | Char3;
				ip+=2;
			}
		}
		else if ((Char1&0xF8)==0xF0){ // 11110??? + 10?????? + 10?????? + 10??????
			if ((AString[ip+1]&0xC0)!=0x80 || (AString[ip+2]&0xC0)!=0x80 || (AString[ip+3]&0xC0)!=0x80){
				lChar='?';
				bError=true;
			}
			else{
				BYTE Char2=AString[ip+1]&0x3f;
				BYTE Char3=AString[ip+2]&0x3f;
				BYTE Char4=AString[ip+3]&0x3f;
				lChar=(((LCHAR)Char1&0x07)<<18) | (((LCHAR)Char2)<<12) | (((LCHAR)Char3)<<6) | Char4;
				ip+=3;
			}
		}
		else {
			lChar='?';
			bError=true;
		}

		if (FSStrAppendFSLCHAR(szResult, lChar)==0){
			szResult+='?';
			bError=true;
		}
	}
	if (pbError) {
		*pbError=bError;
	}
	return szResult;
}

static CFSAString __FSStrWtoA(const CFSWString &WString, long CodePage, bool *pbError)
{
	static const CFSKeyValuePair<UINT16, UCHAR> szAChar_1252[]={
	{0x81, 0x81}, {0x8d, 0x8d}, {0x8f, 0x8f}, {0x90, 0x90}, {0x9d, 0x9d}, {0xa0, 0xa0}, {0xa1, 0xa1}, {0xa2, 0xa2}, {0xa3, 0xa3}, {0xa4, 0xa4}, {0xa5, 0xa5}, {0xa6, 0xa6}, {0xa7, 0xa7}, {0xa8, 0xa8}, {0xa9, 0xa9}, {0xaa, 0xaa}, 
	{0xab, 0xab}, {0xac, 0xac}, {0xad, 0xad}, {0xae, 0xae}, {0xaf, 0xaf}, {0xb0, 0xb0}, {0xb1, 0xb1}, {0xb2, 0xb2}, {0xb3, 0xb3}, {0xb4, 0xb4}, {0xb5, 0xb5}, {0xb6, 0xb6}, {0xb7, 0xb7}, {0xb8, 0xb8}, {0xb9, 0xb9}, {0xba, 0xba}, 
	{0xbb, 0xbb}, {0xbc, 0xbc}, {0xbd, 0xbd}, {0xbe, 0xbe}, {0xbf, 0xbf}, {0xc0, 0xc0}, {0xc1, 0xc1}, {0xc2, 0xc2}, {0xc3, 0xc3}, {0xc4, 0xc4}, {0xc5, 0xc5}, {0xc6, 0xc6}, {0xc7, 0xc7}, {0xc8, 0xc8}, {0xc9, 0xc9}, {0xca, 0xca}, 
	{0xcb, 0xcb}, {0xcc, 0xcc}, {0xcd, 0xcd}, {0xce, 0xce}, {0xcf, 0xcf}, {0xd0, 0xd0}, {0xd1, 0xd1}, {0xd2, 0xd2}, {0xd3, 0xd3}, {0xd4, 0xd4}, {0xd5, 0xd5}, {0xd6, 0xd6}, {0xd7, 0xd7}, {0xd8, 0xd8}, {0xd9, 0xd9}, {0xda, 0xda}, 
	{0xdb, 0xdb}, {0xdc, 0xdc}, {0xdd, 0xdd}, {0xde, 0xde}, {0xdf, 0xdf}, {0xe0, 0xe0}, {0xe1, 0xe1}, {0xe2, 0xe2}, {0xe3, 0xe3}, {0xe4, 0xe4}, {0xe5, 0xe5}, {0xe6, 0xe6}, {0xe7, 0xe7}, {0xe8, 0xe8}, {0xe9, 0xe9}, {0xea, 0xea}, 
	{0xeb, 0xeb}, {0xec, 0xec}, {0xed, 0xed}, {0xee, 0xee}, {0xef, 0xef}, {0xf0, 0xf0}, {0xf1, 0xf1}, {0xf2, 0xf2}, {0xf3, 0xf3}, {0xf4, 0xf4}, {0xf5, 0xf5}, {0xf6, 0xf6}, {0xf7, 0xf7}, {0xf8, 0xf8}, {0xf9, 0xf9}, {0xfa, 0xfa}, 
	{0xfb, 0xfb}, {0xfc, 0xfc}, {0xfd, 0xfd}, {0xfe, 0xfe}, {0xff, 0xff}, {0x100, 0x41}, {0x101, 0x61}, {0x102, 0x41}, {0x103, 0x61}, {0x104, 0x41}, {0x105, 0x61}, {0x106, 0x43}, {0x107, 0x63}, {0x108, 0x43}, {0x109, 0x63}, {0x10a, 0x43}, 
	{0x10b, 0x63}, {0x10c, 0x43}, {0x10d, 0x63}, {0x10e, 0x44}, {0x10f, 0x64}, {0x110, 0xd0}, {0x111, 0x64}, {0x112, 0x45}, {0x113, 0x65}, {0x114, 0x45}, {0x115, 0x65}, {0x116, 0x45}, {0x117, 0x65}, {0x118, 0x45}, {0x119, 0x65}, {0x11a, 0x45}, 
	{0x11b, 0x65}, {0x11c, 0x47}, {0x11d, 0x67}, {0x11e, 0x47}, {0x11f, 0x67}, {0x120, 0x47}, {0x121, 0x67}, {0x122, 0x47}, {0x123, 0x67}, {0x124, 0x48}, {0x125, 0x68}, {0x126, 0x48}, {0x127, 0x68}, {0x128, 0x49}, {0x129, 0x69}, {0x12a, 0x49}, 
	{0x12b, 0x69}, {0x12c, 0x49}, {0x12d, 0x69}, {0x12e, 0x49}, {0x12f, 0x69}, {0x130, 0x49}, {0x131, 0x69}, {0x134, 0x4a}, {0x135, 0x6a}, {0x136, 0x4b}, {0x137, 0x6b}, {0x139, 0x4c}, {0x13a, 0x6c}, {0x13b, 0x4c}, {0x13c, 0x6c}, {0x13d, 0x4c}, 
	{0x13e, 0x6c}, {0x141, 0x4c}, {0x142, 0x6c}, {0x143, 0x4e}, {0x144, 0x6e}, {0x145, 0x4e}, {0x146, 0x6e}, {0x147, 0x4e}, {0x148, 0x6e}, {0x14c, 0x4f}, {0x14d, 0x6f}, {0x14e, 0x4f}, {0x14f, 0x6f}, {0x150, 0x4f}, {0x151, 0x6f}, {0x152, 0x8c}, 
	{0x153, 0x9c}, {0x154, 0x52}, {0x155, 0x72}, {0x156, 0x52}, {0x157, 0x72}, {0x158, 0x52}, {0x159, 0x72}, {0x15a, 0x53}, {0x15b, 0x73}, {0x15c, 0x53}, {0x15d, 0x73}, {0x15e, 0x53}, {0x15f, 0x73}, {0x160, 0x8a}, {0x161, 0x9a}, {0x162, 0x54}, 
	{0x163, 0x74}, {0x164, 0x54}, {0x165, 0x74}, {0x166, 0x54}, {0x167, 0x74}, {0x168, 0x55}, {0x169, 0x75}, {0x16a, 0x55}, {0x16b, 0x75}, {0x16c, 0x55}, {0x16d, 0x75}, {0x16e, 0x55}, {0x16f, 0x75}, {0x170, 0x55}, {0x171, 0x75}, {0x172, 0x55}, 
	{0x173, 0x75}, {0x174, 0x57}, {0x175, 0x77}, {0x176, 0x59}, {0x177, 0x79}, {0x178, 0x9f}, {0x179, 0x5a}, {0x17a, 0x7a}, {0x17b, 0x5a}, {0x17c, 0x7a}, {0x17d, 0x8e}, {0x17e, 0x9e}, {0x180, 0x62}, {0x189, 0xd0}, {0x191, 0x83}, {0x192, 0x83}, 
	{0x197, 0x49}, {0x19a, 0x6c}, {0x19f, 0x4f}, {0x1a0, 0x4f}, {0x1a1, 0x6f}, {0x1ab, 0x74}, {0x1ae, 0x54}, {0x1af, 0x55}, {0x1b0, 0x75}, {0x1b6, 0x7a}, {0x1c0, 0x7c}, {0x1c3, 0x21}, {0x1cd, 0x41}, {0x1ce, 0x61}, {0x1cf, 0x49}, {0x1d0, 0x69}, 
	{0x1d1, 0x4f}, {0x1d2, 0x6f}, {0x1d3, 0x55}, {0x1d4, 0x75}, {0x1d5, 0x55}, {0x1d6, 0x75}, {0x1d7, 0x55}, {0x1d8, 0x75}, {0x1d9, 0x55}, {0x1da, 0x75}, {0x1db, 0x55}, {0x1dc, 0x75}, {0x1de, 0x41}, {0x1df, 0x61}, {0x1e4, 0x47}, {0x1e5, 0x67}, 
	{0x1e6, 0x47}, {0x1e7, 0x67}, {0x1e8, 0x4b}, {0x1e9, 0x6b}, {0x1ea, 0x4f}, {0x1eb, 0x6f}, {0x1ec, 0x4f}, {0x1ed, 0x6f}, {0x1f0, 0x6a}, {0x261, 0x67}, {0x2b9, 0x27}, {0x2ba, 0x22}, {0x2bc, 0x27}, {0x2c4, 0x5e}, {0x2c6, 0x88}, {0x2c8, 0x27}, 
	{0x2c9, 0xaf}, {0x2ca, 0xb4}, {0x2cb, 0x60}, {0x2cd, 0x5f}, {0x2da, 0xb0}, {0x2dc, 0x98}, {0x300, 0x60}, {0x301, 0xb4}, {0x302, 0x5e}, {0x303, 0x7e}, {0x304, 0xaf}, {0x305, 0xaf}, {0x308, 0xa8}, {0x30a, 0xb0}, {0x30e, 0x22}, {0x327, 0xb8}, 
	{0x331, 0x5f}, {0x332, 0x5f}, {0x37e, 0x3b}, {0x393, 0x47}, {0x398, 0x54}, {0x3a3, 0x53}, {0x3a6, 0x46}, {0x3a9, 0x4f}, {0x3b1, 0x61}, {0x3b2, 0xdf}, {0x3b4, 0x64}, {0x3b5, 0x65}, {0x3bc, 0xb5}, {0x3c0, 0x70}, {0x3c3, 0x73}, {0x3c4, 0x74}, 
	{0x3c6, 0x66}, {0x4bb, 0x68}, {0x589, 0x3a}, {0x66a, 0x25}, {0x2000, 0x20}, {0x2001, 0x20}, {0x2002, 0x20}, {0x2003, 0x20}, {0x2004, 0x20}, {0x2005, 0x20}, {0x2006, 0x20}, {0x2010, 0x2d}, {0x2011, 0x2d}, {0x2013, 0x96}, {0x2014, 0x97}, {0x2017, 0x3d}, 
	{0x2018, 0x91}, {0x2019, 0x92}, {0x201a, 0x82}, {0x201c, 0x93}, {0x201d, 0x94}, {0x201e, 0x84}, {0x2020, 0x86}, {0x2021, 0x87}, {0x2022, 0x95}, {0x2024, 0xb7}, {0x2026, 0x85}, {0x2030, 0x89}, {0x2032, 0x27}, {0x2035, 0x60}, {0x2039, 0x8b}, {0x203a, 0x9b}, 
	{0x2044, 0x2f}, {0x2070, 0xb0}, {0x2074, 0x34}, {0x2075, 0x35}, {0x2076, 0x36}, {0x2077, 0x37}, {0x2078, 0x38}, {0x207f, 0x6e}, {0x2080, 0x30}, {0x2081, 0x31}, {0x2082, 0x32}, {0x2083, 0x33}, {0x2084, 0x34}, {0x2085, 0x35}, {0x2086, 0x36}, {0x2087, 0x37}, 
	{0x2088, 0x38}, {0x2089, 0x39}, {0x20a1, 0xa2}, {0x20a4, 0xa3}, {0x20a7, 0x50}, {0x20ac, 0x80}, {0x2102, 0x43}, {0x2107, 0x45}, {0x210a, 0x67}, {0x210b, 0x48}, {0x210c, 0x48}, {0x210d, 0x48}, {0x210e, 0x68}, {0x2110, 0x49}, {0x2111, 0x49}, {0x2112, 0x4c}, 
	{0x2113, 0x6c}, {0x2115, 0x4e}, {0x2118, 0x50}, {0x2119, 0x50}, {0x211a, 0x51}, {0x211b, 0x52}, {0x211c, 0x52}, {0x211d, 0x52}, {0x2122, 0x99}, {0x2124, 0x5a}, {0x2128, 0x5a}, {0x212a, 0x4b}, {0x212b, 0xc5}, {0x212c, 0x42}, {0x212d, 0x43}, {0x212e, 0x65}, 
	{0x212f, 0x65}, {0x2130, 0x45}, {0x2131, 0x46}, {0x2133, 0x4d}, {0x2134, 0x6f}, {0x2205, 0xd8}, {0x2212, 0x2d}, {0x2213, 0xb1}, {0x2215, 0x2f}, {0x2216, 0x5c}, {0x2217, 0x2a}, {0x2218, 0xb0}, {0x2219, 0xb7}, {0x221a, 0x76}, {0x221e, 0x38}, {0x2223, 0x7c}, 
	{0x2229, 0x6e}, {0x2236, 0x3a}, {0x223c, 0x7e}, {0x2248, 0x98}, {0x2261, 0x3d}, {0x2264, 0x3d}, {0x2265, 0x3d}, {0x226a, 0xab}, {0x226b, 0xbb}, {0x22c5, 0xb7}, {0x2302, 0xa6}, {0x2303, 0x5e}, {0x2310, 0xac}, {0x2320, 0x28}, {0x2321, 0x29}, {0x2329, 0x3c}, 
	{0x232a, 0x3e}, {0x2500, 0x2d}, {0x2502, 0xa6}, {0x250c, 0x2b}, {0x2510, 0x2b}, {0x2514, 0x2b}, {0x2518, 0x2b}, {0x251c, 0x2b}, {0x2524, 0xa6}, {0x252c, 0x2d}, {0x2534, 0x2d}, {0x253c, 0x2b}, {0x2550, 0x2d}, {0x2551, 0xa6}, {0x2552, 0x2b}, {0x2553, 0x2b}, 
	{0x2554, 0x2b}, {0x2555, 0x2b}, {0x2556, 0x2b}, {0x2557, 0x2b}, {0x2558, 0x2b}, {0x2559, 0x2b}, {0x255a, 0x2b}, {0x255b, 0x2b}, {0x255c, 0x2b}, {0x255d, 0x2b}, {0x255e, 0xa6}, {0x255f, 0xa6}, {0x2560, 0xa6}, {0x2561, 0xa6}, {0x2562, 0xa6}, {0x2563, 0xa6}, 
	{0x2564, 0x2d}, {0x2565, 0x2d}, {0x2566, 0x2d}, {0x2567, 0x2d}, {0x2568, 0x2d}, {0x2569, 0x2d}, {0x256a, 0x2b}, {0x256b, 0x2b}, {0x256c, 0x2b}, {0x2580, 0xaf}, {0x2584, 0x5f}, {0x2588, 0xa6}, {0x258c, 0xa6}, {0x2590, 0xa6}, {0x2591, 0xa6}, {0x2592, 0xa6}, 
	{0x2593, 0xa6}, {0x25a0, 0xa6}, {0x263c, 0xa4}, {0x2758, 0x7c}, {0x3000, 0x20}, {0x3008, 0x3c}, {0x3009, 0x3e}, {0x300a, 0xab}, {0x300b, 0xbb}, {0x301a, 0x5b}, {0x301b, 0x5d}, {0x30fb, 0xb7}, {0xff01, 0x21}, {0xff02, 0x22}, {0xff03, 0x23}, {0xff04, 0x24}, 
	{0xff05, 0x25}, {0xff06, 0x26}, {0xff07, 0x27}, {0xff08, 0x28}, {0xff09, 0x29}, {0xff0a, 0x2a}, {0xff0b, 0x2b}, {0xff0c, 0x2c}, {0xff0d, 0x2d}, {0xff0e, 0x2e}, {0xff0f, 0x2f}, {0xff10, 0x30}, {0xff11, 0x31}, {0xff12, 0x32}, {0xff13, 0x33}, {0xff14, 0x34}, 
	{0xff15, 0x35}, {0xff16, 0x36}, {0xff17, 0x37}, {0xff18, 0x38}, {0xff19, 0x39}, {0xff1a, 0x3a}, {0xff1b, 0x3b}, {0xff1c, 0x3c}, {0xff1d, 0x3d}, {0xff1e, 0x3e}, {0xff20, 0x40}, {0xff21, 0x41}, {0xff22, 0x42}, {0xff23, 0x43}, {0xff24, 0x44}, {0xff25, 0x45}, 
	{0xff26, 0x46}, {0xff27, 0x47}, {0xff28, 0x48}, {0xff29, 0x49}, {0xff2a, 0x4a}, {0xff2b, 0x4b}, {0xff2c, 0x4c}, {0xff2d, 0x4d}, {0xff2e, 0x4e}, {0xff2f, 0x4f}, {0xff30, 0x50}, {0xff31, 0x51}, {0xff32, 0x52}, {0xff33, 0x53}, {0xff34, 0x54}, {0xff35, 0x55}, 
	{0xff36, 0x56}, {0xff37, 0x57}, {0xff38, 0x58}, {0xff39, 0x59}, {0xff3a, 0x5a}, {0xff3b, 0x5b}, {0xff3c, 0x5c}, {0xff3d, 0x5d}, {0xff3e, 0x5e}, {0xff3f, 0x5f}, {0xff40, 0x60}, {0xff41, 0x61}, {0xff42, 0x62}, {0xff43, 0x63}, {0xff44, 0x64}, {0xff45, 0x65}, 
	{0xff46, 0x66}, {0xff47, 0x67}, {0xff48, 0x68}, {0xff49, 0x69}, {0xff4a, 0x6a}, {0xff4b, 0x6b}, {0xff4c, 0x6c}, {0xff4d, 0x6d}, {0xff4e, 0x6e}, {0xff4f, 0x6f}, {0xff50, 0x70}, {0xff51, 0x71}, {0xff52, 0x72}, {0xff53, 0x73}, {0xff54, 0x74}, {0xff55, 0x75}, 
	{0xff56, 0x76}, {0xff57, 0x77}, {0xff58, 0x78}, {0xff59, 0x79}, {0xff5a, 0x7a}, {0xff5b, 0x7b}, {0xff5c, 0x7c}, {0xff5d, 0x7d}, {0xff5e, 0x7e}, 
	};
	static const CFSKeyValuePair<UINT16, UCHAR> szAChar_1257[]={
	{0x81, 0x81}, {0x83, 0x83}, {0x88, 0x88}, {0x8a, 0x8a}, {0x8c, 0x8c}, {0x90, 0x90}, {0x98, 0x98}, {0x9a, 0x9a}, {0x9c, 0x9c}, {0x9f, 0x9f}, {0xa0, 0xa0}, {0xa2, 0xa2}, {0xa3, 0xa3}, {0xa4, 0xa4}, {0xa6, 0xa6}, {0xa7, 0xa7}, 
	{0xa8, 0x8d}, {0xa9, 0xa9}, {0xab, 0xab}, {0xac, 0xac}, {0xad, 0xad}, {0xae, 0xae}, {0xaf, 0x9d}, {0xb0, 0xb0}, {0xb1, 0xb1}, {0xb2, 0xb2}, {0xb3, 0xb3}, {0xb4, 0xb4}, {0xb5, 0xb5}, {0xb6, 0xb6}, {0xb7, 0xb7}, {0xb8, 0x8f}, 
	{0xb9, 0xb9}, {0xbb, 0xbb}, {0xbc, 0xbc}, {0xbd, 0xbd}, {0xbe, 0xbe}, {0xc4, 0xc4}, {0xc5, 0xc5}, {0xc6, 0xaf}, {0xc9, 0xc9}, {0xd3, 0xd3}, {0xd5, 0xd5}, {0xd6, 0xd6}, {0xd7, 0xd7}, {0xd8, 0xa8}, {0xdc, 0xdc}, {0xdf, 0xdf}, 
	{0xe4, 0xe4}, {0xe5, 0xe5}, {0xe6, 0xbf}, {0xe9, 0xe9}, {0xf3, 0xf3}, {0xf5, 0xf5}, {0xf6, 0xf6}, {0xf7, 0xf7}, {0xf8, 0xb8}, {0xfc, 0xfc}, {0x100, 0xc2}, {0x101, 0xe2}, {0x104, 0xc0}, {0x105, 0xe0}, {0x106, 0xc3}, {0x107, 0xe3}, 
	{0x10c, 0xc8}, {0x10d, 0xe8}, {0x112, 0xc7}, {0x113, 0xe7}, {0x116, 0xcb}, {0x117, 0xeb}, {0x118, 0xc6}, {0x119, 0xe6}, {0x122, 0xcc}, {0x123, 0xec}, {0x12a, 0xce}, {0x12b, 0xee}, {0x12e, 0xc1}, {0x12f, 0xe1}, {0x136, 0xcd}, {0x137, 0xed}, 
	{0x13b, 0xcf}, {0x13c, 0xef}, {0x141, 0xd9}, {0x142, 0xf9}, {0x143, 0xd1}, {0x144, 0xf1}, {0x145, 0xd2}, {0x146, 0xf2}, {0x14c, 0xd4}, {0x14d, 0xf4}, {0x156, 0xaa}, {0x157, 0xba}, {0x15a, 0xda}, {0x15b, 0xfa}, {0x160, 0xd0}, {0x161, 0xf0}, 
	{0x16a, 0xdb}, {0x16b, 0xfb}, {0x172, 0xd8}, {0x173, 0xf8}, {0x179, 0xca}, {0x17a, 0xea}, {0x17b, 0xdd}, {0x17c, 0xfd}, {0x17d, 0xde}, {0x17e, 0xfe}, {0x2c7, 0x8e}, {0x2d9, 0xff}, {0x2db, 0x9e}, {0x2013, 0x96}, {0x2014, 0x97}, {0x2018, 0x91}, 
	{0x2019, 0x92}, {0x201a, 0x82}, {0x201c, 0x93}, {0x201d, 0x94}, {0x201e, 0x84}, {0x2020, 0x86}, {0x2021, 0x87}, {0x2022, 0x95}, {0x2026, 0x85}, {0x2030, 0x89}, {0x2039, 0x8b}, {0x203a, 0x9b}, {0x20ac, 0x80}, {0x2122, 0x99}, {0xf8fc, 0xa1}, {0xf8fd, 0xa5}, 
	{0xff01, 0x21}, {0xff02, 0x22}, {0xff03, 0x23}, {0xff04, 0x24}, {0xff05, 0x25}, {0xff06, 0x26}, {0xff07, 0x27}, {0xff08, 0x28}, {0xff09, 0x29}, {0xff0a, 0x2a}, {0xff0b, 0x2b}, {0xff0c, 0x2c}, {0xff0d, 0x2d}, {0xff0e, 0x2e}, {0xff0f, 0x2f}, {0xff10, 0x30}, 
	{0xff11, 0x31}, {0xff12, 0x32}, {0xff13, 0x33}, {0xff14, 0x34}, {0xff15, 0x35}, {0xff16, 0x36}, {0xff17, 0x37}, {0xff18, 0x38}, {0xff19, 0x39}, {0xff1a, 0x3a}, {0xff1b, 0x3b}, {0xff1c, 0x3c}, {0xff1d, 0x3d}, {0xff1e, 0x3e}, {0xff20, 0x40}, {0xff21, 0x41}, 
	{0xff22, 0x42}, {0xff23, 0x43}, {0xff24, 0x44}, {0xff25, 0x45}, {0xff26, 0x46}, {0xff27, 0x47}, {0xff28, 0x48}, {0xff29, 0x49}, {0xff2a, 0x4a}, {0xff2b, 0x4b}, {0xff2c, 0x4c}, {0xff2d, 0x4d}, {0xff2e, 0x4e}, {0xff2f, 0x4f}, {0xff30, 0x50}, {0xff31, 0x51}, 
	{0xff32, 0x52}, {0xff33, 0x53}, {0xff34, 0x54}, {0xff35, 0x55}, {0xff36, 0x56}, {0xff37, 0x57}, {0xff38, 0x58}, {0xff39, 0x59}, {0xff3a, 0x5a}, {0xff3b, 0x5b}, {0xff3c, 0x5c}, {0xff3d, 0x5d}, {0xff3e, 0x5e}, {0xff3f, 0x5f}, {0xff40, 0x60}, {0xff41, 0x61}, 
	{0xff42, 0x62}, {0xff43, 0x63}, {0xff44, 0x64}, {0xff45, 0x65}, {0xff46, 0x66}, {0xff47, 0x67}, {0xff48, 0x68}, {0xff49, 0x69}, {0xff4a, 0x6a}, {0xff4b, 0x6b}, {0xff4c, 0x6c}, {0xff4d, 0x6d}, {0xff4e, 0x6e}, {0xff4f, 0x6f}, {0xff50, 0x70}, {0xff51, 0x71}, 
	{0xff52, 0x72}, {0xff53, 0x73}, {0xff54, 0x74}, {0xff55, 0x75}, {0xff56, 0x76}, {0xff57, 0x77}, {0xff58, 0x78}, {0xff59, 0x79}, {0xff5a, 0x7a}, {0xff5b, 0x7b}, {0xff5c, 0x7c}, {0xff5d, 0x7d}, {0xff5e, 0x7e}, 
	};

	CFSAString szResult;
	INTPTR ipMapSize;
	const CFSKeyValuePair<UINT16, UCHAR> *lpAChar;
	switch (CodePage){
		case 0:
		case 1252:
			lpAChar=szAChar_1252;
			ipMapSize=sizeof(szAChar_1252)/sizeof(CFSKeyValuePair<UINT16, UCHAR>);
		break;
		case 1257:
			lpAChar=szAChar_1257;
			ipMapSize=sizeof(szAChar_1257)/sizeof(CFSKeyValuePair<UINT16, UCHAR>);
		break;
		default:
			return "";
	};

	for (INTPTR ip=0; WString[ip]; ip++){
		if ((LCHAR)WString[ip]<0x80) {
			szResult+=(char)WString[ip];
		}
		else{
			INTPTR ipChar=FSArrayFind2(lpAChar, ipMapSize, WString[ip]);
			if (ipChar>=0) {
				szResult+=lpAChar[ipChar].Value;
			} else {
				szResult+='?';
				if (pbError) {
					*pbError=true;
				}
			}
		}
	}
	return szResult;
}

static CFSWString __FSStrAtoW(const CFSAString &AString, long CodePage, bool * /*pbError*/)
{
	static UINT16 const szWChar_1252[0x80]={
	0x20ac, 0x81, 0x201a, 0x192, 0x201e, 0x2026, 0x2020, 0x2021, 0x2c6, 0x2030, 0x160, 0x2039, 0x152, 0x8d, 0x17d, 0x8f, 
	0x90, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014, 0x2dc, 0x2122, 0x161, 0x203a, 0x153, 0x9d, 0x17e, 0x178, 
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
	};
	static UINT16 const szWChar_1257[0x80]={
	0x20ac, 0x81, 0x201a, 0x83, 0x201e, 0x2026, 0x2020, 0x2021, 0x88, 0x2030, 0x8a, 0x2039, 0x8c, 0xa8, 0x2c7, 0xb8, 
	0x90, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014, 0x98, 0x2122, 0x9a, 0x203a, 0x9c, 0xaf, 0x2db, 0x9f, 
	0xa0, 0xf8fc, 0xa2, 0xa3, 0xa4, 0xf8fd, 0xa6, 0xa7, 0xd8, 0xa9, 0x156, 0xab, 0xac, 0xad, 0xae, 0xc6, 
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xf8, 0xb9, 0x157, 0xbb, 0xbc, 0xbd, 0xbe, 0xe6, 
	0x104, 0x12e, 0x100, 0x106, 0xc4, 0xc5, 0x118, 0x112, 0x10c, 0xc9, 0x179, 0x116, 0x122, 0x136, 0x12a, 0x13b, 
	0x160, 0x143, 0x145, 0xd3, 0x14c, 0xd5, 0xd6, 0xd7, 0x172, 0x141, 0x15a, 0x16a, 0xdc, 0x17b, 0x17d, 0xdf, 
	0x105, 0x12f, 0x101, 0x107, 0xe4, 0xe5, 0x119, 0x113, 0x10d, 0xe9, 0x17a, 0x117, 0x123, 0x137, 0x12b, 0x13c, 
	0x161, 0x144, 0x146, 0xf3, 0x14d, 0xf5, 0xf6, 0xf7, 0x173, 0x142, 0x15b, 0x16b, 0xfc, 0x17c, 0x17e, 0x2d9, 
	};

	CFSWString szResult;
	const UINT16 *pszWChar=0;
	switch (CodePage){
		case 0:
		case 1252:
			pszWChar=szWChar_1252;
		break;
		case 1257:
			pszWChar=szWChar_1257;
		break;
		default:
			return CFSWString();
	};

	for (INTPTR ip=0; AString[ip]; ip++) {
		if ((UCHAR)AString[ip]<0x80) {
			szResult+=(WCHAR)AString[ip];
		}
		else {
			szResult+=pszWChar[(UCHAR)AString[ip]-0x80];
		}
	}
	return szResult;
}

CFSAString FSStrWtoA(const CFSWString &WString, long CodePage, bool *pbError)
{
	if (pbError) {
		*pbError=false;
	}
	CFSAString AString;
	if (WString.IsEmpty()) {
		return AString;
	}
	if (CodePage==FSCP_ACP) {
		CodePage=g_lFSCodePage;
	}
	if (CodePage==FSCP_SYSTEM){
#if defined (WIN32)
		BOOL bUsedDefault=0;
		int iSize=WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, WString, -1, 0, 0, "?", &bUsedDefault);
		WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, WString, -1, AString.GetBuffer(iSize), iSize, "?", &bUsedDefault);
		AString.ReleaseBuffer();
		if (pbError) {
			*pbError=(bUsedDefault!=0);
		}
		return AString;
#else
		CodePage=FSCP_UTF8;
#endif
	}
	if (CodePage==FSCP_HTML) {
		return __FSStrWtoHTML(WString, pbError);
	}
	if (CodePage==FSCP_UTF8) {
		return __FSStrWtoUTF8(WString, pbError);
	}
	return __FSStrWtoA(WString, CodePage, pbError);
}

CFSWString FSStrAtoW(const CFSAString &AString, long CodePage, bool *pbError)
{
	if (pbError) {
		*pbError=0;
	}
	CFSWString WString;
	if (AString.IsEmpty()) {
		return WString;
	}
	if (CodePage==FSCP_ACP) {
		CodePage=g_lFSCodePage;
	}
#if defined (WIN32)
	if (CodePage==FSCP_SYSTEM){
		int iSize=MultiByteToWideChar(CP_ACP, 0, AString, -1, 0, 0);
		MultiByteToWideChar(CP_ACP, 0, AString, -1, WString.GetBuffer(iSize), iSize);
		WString.ReleaseBuffer();
		return WString;
	}
#endif
	if (CodePage==FSCP_HTML) {
		return __FSStrHTMLtoW(AString, pbError);
	}
	if (CodePage==FSCP_UTF8) {
		return __FSStrUTF8toW(AString, pbError);
	}
	return __FSStrAtoW(AString, CodePage, pbError);
}

void FSStrCtoP(unsigned char *pszPStr, const char *pszCStr, unsigned char nMaxLength)
{
	INTPTR ipLength=(pszCStr ? FSStrLen(pszCStr) : 0);
	if (ipLength>nMaxLength) {
		ipLength=nMaxLength;
	}
	pszPStr[0]=(unsigned char)ipLength;
	memcpy(pszPStr+1, pszCStr, ipLength);
}

CFSAString FSStrPtoC(const unsigned char *pszPStr)
{
	CFSAString szCStr;
	memcpy(szCStr.GetBuffer(1+pszPStr[0]), pszPStr+1, pszPStr[0]);
	szCStr[pszPStr[0]]=0;
	szCStr.ReleaseBuffer();
	return szCStr;
}

///////////////////////////////////////////////////////////
// Baltic string manipulation
///////////////////////////////////////////////////////////

char FSToLower(char ch)
{
	static UCHAR const szToLowers_1257[128]={
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xba, 0xab, 0xac, 0xad, 0xae, 0xbf, 
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xd7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xdf, 
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
	};
	static UCHAR const szToLowers_1252[128]={
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x9a, 0x8b, 0x9c, 0x8d, 0x9e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0xff,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xd7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};
	if ((UCHAR)ch<128) {
		return (UCHAR)tolower(ch);
	}
	switch (g_lFSCodePage){
		case FSCP_BALTIC:
			return szToLowers_1257[(UCHAR)ch-128];
		default:
			return szToLowers_1252[(UCHAR)ch-128];
	}
}

char FSToUpper(char ch)
{
	static UCHAR const szToUppers_1257[128]={
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xaa, 0xbb, 0xbc, 0xbd, 0xbe, 0xaf, 
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xf7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xff, 
	};
	static UCHAR const szToUppers_1252[128]={
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x8a, 0x9b, 0x8c, 0x9d, 0x8e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xf7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0x9f,
	};
	if ((UCHAR)ch<128) {
		return (UCHAR)toupper(ch);
	}
	switch (g_lFSCodePage){
		case FSCP_BALTIC:
			return szToUppers_1257[(UCHAR)ch-128];
		default:
			return szToUppers_1252[(UCHAR)ch-128];
	}
}

///////////////////////////////////////////////////////////
// Unicode string manipulation
///////////////////////////////////////////////////////////

WCHAR FSToUpper(WCHAR Char){
	static const CFSKeyValuePair<UINT16, UINT16> s_CharMap[] = {
	{0x61, 0x41}, {0x62, 0x42}, {0x63, 0x43}, {0x64, 0x44}, {0x65, 0x45}, {0x66, 0x46}, {0x67, 0x47}, {0x68, 0x48}, {0x69, 0x49}, {0x6a, 0x4a}, 
	{0x6b, 0x4b}, {0x6c, 0x4c}, {0x6d, 0x4d}, {0x6e, 0x4e}, {0x6f, 0x4f}, {0x70, 0x50}, {0x71, 0x51}, {0x72, 0x52}, {0x73, 0x53}, {0x74, 0x54}, 
	{0x75, 0x55}, {0x76, 0x56}, {0x77, 0x57}, {0x78, 0x58}, {0x79, 0x59}, {0x7a, 0x5a}, {0xe0, 0xc0}, {0xe1, 0xc1}, {0xe2, 0xc2}, {0xe3, 0xc3}, 
	{0xe4, 0xc4}, {0xe5, 0xc5}, {0xe6, 0xc6}, {0xe7, 0xc7}, {0xe8, 0xc8}, {0xe9, 0xc9}, {0xea, 0xca}, {0xeb, 0xcb}, {0xec, 0xcc}, {0xed, 0xcd}, 
	{0xee, 0xce}, {0xef, 0xcf}, {0xf0, 0xd0}, {0xf1, 0xd1}, {0xf2, 0xd2}, {0xf3, 0xd3}, {0xf4, 0xd4}, {0xf5, 0xd5}, {0xf6, 0xd6}, {0xf8, 0xd8}, 
	{0xf9, 0xd9}, {0xfa, 0xda}, {0xfb, 0xdb}, {0xfc, 0xdc}, {0xfd, 0xdd}, {0xfe, 0xde}, {0xff, 0x178}, {0x101, 0x100}, {0x103, 0x102}, {0x105, 0x104}, 
	{0x107, 0x106}, {0x109, 0x108}, {0x10b, 0x10a}, {0x10d, 0x10c}, {0x10f, 0x10e}, {0x111, 0x110}, {0x113, 0x112}, {0x115, 0x114}, {0x117, 0x116}, {0x119, 0x118}, 
	{0x11b, 0x11a}, {0x11d, 0x11c}, {0x11f, 0x11e}, {0x121, 0x120}, {0x123, 0x122}, {0x125, 0x124}, {0x127, 0x126}, {0x129, 0x128}, {0x12b, 0x12a}, {0x12d, 0x12c}, 
	{0x12f, 0x12e}, {0x133, 0x132}, {0x135, 0x134}, {0x137, 0x136}, {0x13a, 0x139}, {0x13c, 0x13b}, {0x13e, 0x13d}, {0x140, 0x13f}, {0x142, 0x141}, {0x144, 0x143}, 
	{0x146, 0x145}, {0x148, 0x147}, {0x14b, 0x14a}, {0x14d, 0x14c}, {0x14f, 0x14e}, {0x151, 0x150}, {0x153, 0x152}, {0x155, 0x154}, {0x157, 0x156}, {0x159, 0x158}, 
	{0x15b, 0x15a}, {0x15d, 0x15c}, {0x15f, 0x15e}, {0x161, 0x160}, {0x163, 0x162}, {0x165, 0x164}, {0x167, 0x166}, {0x169, 0x168}, {0x16b, 0x16a}, {0x16d, 0x16c}, 
	{0x16f, 0x16e}, {0x171, 0x170}, {0x173, 0x172}, {0x175, 0x174}, {0x177, 0x176}, {0x17a, 0x179}, {0x17c, 0x17b}, {0x17e, 0x17d}, {0x183, 0x182}, {0x185, 0x184}, 
	{0x188, 0x187}, {0x18c, 0x18b}, {0x192, 0x191}, {0x199, 0x198}, {0x1a1, 0x1a0}, {0x1a3, 0x1a2}, {0x1a5, 0x1a4}, {0x1a8, 0x1a7}, {0x1ad, 0x1ac}, {0x1b0, 0x1af}, 
	{0x1b4, 0x1b3}, {0x1b6, 0x1b5}, {0x1b9, 0x1b8}, {0x1bd, 0x1bc}, {0x1c6, 0x1c4}, {0x1c9, 0x1c7}, {0x1cc, 0x1ca}, {0x1ce, 0x1cd}, {0x1d0, 0x1cf}, {0x1d2, 0x1d1}, 
	{0x1d4, 0x1d3}, {0x1d6, 0x1d5}, {0x1d8, 0x1d7}, {0x1da, 0x1d9}, {0x1dc, 0x1db}, {0x1dd, 0x18e}, {0x1df, 0x1de}, {0x1e1, 0x1e0}, {0x1e3, 0x1e2}, {0x1e5, 0x1e4}, 
	{0x1e7, 0x1e6}, {0x1e9, 0x1e8}, {0x1eb, 0x1ea}, {0x1ed, 0x1ec}, {0x1ef, 0x1ee}, {0x1f3, 0x1f1}, {0x1f5, 0x1f4}, {0x1fb, 0x1fa}, {0x1fd, 0x1fc}, {0x1ff, 0x1fe}, 
	{0x201, 0x200}, {0x203, 0x202}, {0x205, 0x204}, {0x207, 0x206}, {0x209, 0x208}, {0x20b, 0x20a}, {0x20d, 0x20c}, {0x20f, 0x20e}, {0x211, 0x210}, {0x213, 0x212}, 
	{0x215, 0x214}, {0x217, 0x216}, {0x253, 0x181}, {0x254, 0x186}, {0x256, 0x189}, {0x257, 0x18a}, {0x259, 0x18f}, {0x25b, 0x190}, {0x260, 0x193}, {0x263, 0x194}, 
	{0x268, 0x197}, {0x269, 0x196}, {0x26f, 0x19c}, {0x272, 0x19d}, {0x275, 0x19f}, {0x283, 0x1a9}, {0x288, 0x1ae}, {0x28a, 0x1b1}, {0x28b, 0x1b2}, {0x292, 0x1b7}, 
	{0x3ac, 0x386}, {0x3ad, 0x388}, {0x3ae, 0x389}, {0x3af, 0x38a}, {0x3b1, 0x391}, {0x3b2, 0x392}, {0x3b3, 0x393}, {0x3b4, 0x394}, {0x3b5, 0x395}, {0x3b6, 0x396}, 
	{0x3b7, 0x397}, {0x3b8, 0x398}, {0x3b9, 0x399}, {0x3ba, 0x39a}, {0x3bb, 0x39b}, {0x3bc, 0x39c}, {0x3bd, 0x39d}, {0x3be, 0x39e}, {0x3bf, 0x39f}, {0x3c0, 0x3a0}, 
	{0x3c1, 0x3a1}, {0x3c2, 0x3a3}, {0x3c3, 0x3a3}, {0x3c4, 0x3a4}, {0x3c5, 0x3a5}, {0x3c6, 0x3a6}, {0x3c7, 0x3a7}, {0x3c8, 0x3a8}, {0x3c9, 0x3a9}, {0x3ca, 0x3aa}, 
	{0x3cb, 0x3ab}, {0x3cc, 0x38c}, {0x3cd, 0x38e}, {0x3ce, 0x38f}, {0x3e3, 0x3e2}, {0x3e5, 0x3e4}, {0x3e7, 0x3e6}, {0x3e9, 0x3e8}, {0x3eb, 0x3ea}, {0x3ed, 0x3ec}, 
	{0x3ef, 0x3ee}, {0x430, 0x410}, {0x431, 0x411}, {0x432, 0x412}, {0x433, 0x413}, {0x434, 0x414}, {0x435, 0x415}, {0x436, 0x416}, {0x437, 0x417}, {0x438, 0x418}, 
	{0x439, 0x419}, {0x43a, 0x41a}, {0x43b, 0x41b}, {0x43c, 0x41c}, {0x43d, 0x41d}, {0x43e, 0x41e}, {0x43f, 0x41f}, {0x440, 0x420}, {0x441, 0x421}, {0x442, 0x422}, 
	{0x443, 0x423}, {0x444, 0x424}, {0x445, 0x425}, {0x446, 0x426}, {0x447, 0x427}, {0x448, 0x428}, {0x449, 0x429}, {0x44a, 0x42a}, {0x44b, 0x42b}, {0x44c, 0x42c}, 
	{0x44d, 0x42d}, {0x44e, 0x42e}, {0x44f, 0x42f}, {0x451, 0x401}, {0x452, 0x402}, {0x453, 0x403}, {0x454, 0x404}, {0x455, 0x405}, {0x456, 0x406}, {0x457, 0x407}, 
	{0x458, 0x408}, {0x459, 0x409}, {0x45a, 0x40a}, {0x45b, 0x40b}, {0x45c, 0x40c}, {0x45e, 0x40e}, {0x45f, 0x40f}, {0x461, 0x460}, {0x463, 0x462}, {0x465, 0x464}, 
	{0x467, 0x466}, {0x469, 0x468}, {0x46b, 0x46a}, {0x46d, 0x46c}, {0x46f, 0x46e}, {0x471, 0x470}, {0x473, 0x472}, {0x475, 0x474}, {0x477, 0x476}, {0x479, 0x478}, 
	{0x47b, 0x47a}, {0x47d, 0x47c}, {0x47f, 0x47e}, {0x481, 0x480}, {0x491, 0x490}, {0x493, 0x492}, {0x495, 0x494}, {0x497, 0x496}, {0x499, 0x498}, {0x49b, 0x49a}, 
	{0x49d, 0x49c}, {0x49f, 0x49e}, {0x4a1, 0x4a0}, {0x4a3, 0x4a2}, {0x4a5, 0x4a4}, {0x4a7, 0x4a6}, {0x4a9, 0x4a8}, {0x4ab, 0x4aa}, {0x4ad, 0x4ac}, {0x4af, 0x4ae}, 
	{0x4b1, 0x4b0}, {0x4b3, 0x4b2}, {0x4b5, 0x4b4}, {0x4b7, 0x4b6}, {0x4b9, 0x4b8}, {0x4bb, 0x4ba}, {0x4bd, 0x4bc}, {0x4bf, 0x4be}, {0x4c2, 0x4c1}, {0x4c4, 0x4c3}, 
	{0x4c8, 0x4c7}, {0x4cc, 0x4cb}, {0x4d1, 0x4d0}, {0x4d3, 0x4d2}, {0x4d5, 0x4d4}, {0x4d7, 0x4d6}, {0x4d9, 0x4d8}, {0x4db, 0x4da}, {0x4dd, 0x4dc}, {0x4df, 0x4de}, 
	{0x4e1, 0x4e0}, {0x4e3, 0x4e2}, {0x4e5, 0x4e4}, {0x4e7, 0x4e6}, {0x4e9, 0x4e8}, {0x4eb, 0x4ea}, {0x4ef, 0x4ee}, {0x4f1, 0x4f0}, {0x4f3, 0x4f2}, {0x4f5, 0x4f4}, 
	{0x4f9, 0x4f8}, {0x561, 0x531}, {0x562, 0x532}, {0x563, 0x533}, {0x564, 0x534}, {0x565, 0x535}, {0x566, 0x536}, {0x567, 0x537}, {0x568, 0x538}, {0x569, 0x539}, 
	{0x56a, 0x53a}, {0x56b, 0x53b}, {0x56c, 0x53c}, {0x56d, 0x53d}, {0x56e, 0x53e}, {0x56f, 0x53f}, {0x570, 0x540}, {0x571, 0x541}, {0x572, 0x542}, {0x573, 0x543}, 
	{0x574, 0x544}, {0x575, 0x545}, {0x576, 0x546}, {0x577, 0x547}, {0x578, 0x548}, {0x579, 0x549}, {0x57a, 0x54a}, {0x57b, 0x54b}, {0x57c, 0x54c}, {0x57d, 0x54d}, 
	{0x57e, 0x54e}, {0x57f, 0x54f}, {0x580, 0x550}, {0x581, 0x551}, {0x582, 0x552}, {0x583, 0x553}, {0x584, 0x554}, {0x585, 0x555}, {0x586, 0x556}, {0x1e01, 0x1e00}, 
	{0x1e03, 0x1e02}, {0x1e05, 0x1e04}, {0x1e07, 0x1e06}, {0x1e09, 0x1e08}, {0x1e0b, 0x1e0a}, {0x1e0d, 0x1e0c}, {0x1e0f, 0x1e0e}, {0x1e11, 0x1e10}, {0x1e13, 0x1e12}, {0x1e15, 0x1e14}, 
	{0x1e17, 0x1e16}, {0x1e19, 0x1e18}, {0x1e1b, 0x1e1a}, {0x1e1d, 0x1e1c}, {0x1e1f, 0x1e1e}, {0x1e21, 0x1e20}, {0x1e23, 0x1e22}, {0x1e25, 0x1e24}, {0x1e27, 0x1e26}, {0x1e29, 0x1e28}, 
	{0x1e2b, 0x1e2a}, {0x1e2d, 0x1e2c}, {0x1e2f, 0x1e2e}, {0x1e31, 0x1e30}, {0x1e33, 0x1e32}, {0x1e35, 0x1e34}, {0x1e37, 0x1e36}, {0x1e39, 0x1e38}, {0x1e3b, 0x1e3a}, {0x1e3d, 0x1e3c}, 
	{0x1e3f, 0x1e3e}, {0x1e41, 0x1e40}, {0x1e43, 0x1e42}, {0x1e45, 0x1e44}, {0x1e47, 0x1e46}, {0x1e49, 0x1e48}, {0x1e4b, 0x1e4a}, {0x1e4d, 0x1e4c}, {0x1e4f, 0x1e4e}, {0x1e51, 0x1e50}, 
	{0x1e53, 0x1e52}, {0x1e55, 0x1e54}, {0x1e57, 0x1e56}, {0x1e59, 0x1e58}, {0x1e5b, 0x1e5a}, {0x1e5d, 0x1e5c}, {0x1e5f, 0x1e5e}, {0x1e61, 0x1e60}, {0x1e63, 0x1e62}, {0x1e65, 0x1e64}, 
	{0x1e67, 0x1e66}, {0x1e69, 0x1e68}, {0x1e6b, 0x1e6a}, {0x1e6d, 0x1e6c}, {0x1e6f, 0x1e6e}, {0x1e71, 0x1e70}, {0x1e73, 0x1e72}, {0x1e75, 0x1e74}, {0x1e77, 0x1e76}, {0x1e79, 0x1e78}, 
	{0x1e7b, 0x1e7a}, {0x1e7d, 0x1e7c}, {0x1e7f, 0x1e7e}, {0x1e81, 0x1e80}, {0x1e83, 0x1e82}, {0x1e85, 0x1e84}, {0x1e87, 0x1e86}, {0x1e89, 0x1e88}, {0x1e8b, 0x1e8a}, {0x1e8d, 0x1e8c}, 
	{0x1e8f, 0x1e8e}, {0x1e91, 0x1e90}, {0x1e93, 0x1e92}, {0x1e95, 0x1e94}, {0x1ea1, 0x1ea0}, {0x1ea3, 0x1ea2}, {0x1ea5, 0x1ea4}, {0x1ea7, 0x1ea6}, {0x1ea9, 0x1ea8}, {0x1eab, 0x1eaa}, 
	{0x1ead, 0x1eac}, {0x1eaf, 0x1eae}, {0x1eb1, 0x1eb0}, {0x1eb3, 0x1eb2}, {0x1eb5, 0x1eb4}, {0x1eb7, 0x1eb6}, {0x1eb9, 0x1eb8}, {0x1ebb, 0x1eba}, {0x1ebd, 0x1ebc}, {0x1ebf, 0x1ebe}, 
	{0x1ec1, 0x1ec0}, {0x1ec3, 0x1ec2}, {0x1ec5, 0x1ec4}, {0x1ec7, 0x1ec6}, {0x1ec9, 0x1ec8}, {0x1ecb, 0x1eca}, {0x1ecd, 0x1ecc}, {0x1ecf, 0x1ece}, {0x1ed1, 0x1ed0}, {0x1ed3, 0x1ed2}, 
	{0x1ed5, 0x1ed4}, {0x1ed7, 0x1ed6}, {0x1ed9, 0x1ed8}, {0x1edb, 0x1eda}, {0x1edd, 0x1edc}, {0x1edf, 0x1ede}, {0x1ee1, 0x1ee0}, {0x1ee3, 0x1ee2}, {0x1ee5, 0x1ee4}, {0x1ee7, 0x1ee6}, 
	{0x1ee9, 0x1ee8}, {0x1eeb, 0x1eea}, {0x1eed, 0x1eec}, {0x1eef, 0x1eee}, {0x1ef1, 0x1ef0}, {0x1ef3, 0x1ef2}, {0x1ef5, 0x1ef4}, {0x1ef7, 0x1ef6}, {0x1ef9, 0x1ef8}, {0x1f00, 0x1f08}, 
	{0x1f01, 0x1f09}, {0x1f02, 0x1f0a}, {0x1f03, 0x1f0b}, {0x1f04, 0x1f0c}, {0x1f05, 0x1f0d}, {0x1f06, 0x1f0e}, {0x1f07, 0x1f0f}, {0x1f10, 0x1f18}, {0x1f11, 0x1f19}, {0x1f12, 0x1f1a}, 
	{0x1f13, 0x1f1b}, {0x1f14, 0x1f1c}, {0x1f15, 0x1f1d}, {0x1f20, 0x1f28}, {0x1f21, 0x1f29}, {0x1f22, 0x1f2a}, {0x1f23, 0x1f2b}, {0x1f24, 0x1f2c}, {0x1f25, 0x1f2d}, {0x1f26, 0x1f2e}, 
	{0x1f27, 0x1f2f}, {0x1f30, 0x1f38}, {0x1f31, 0x1f39}, {0x1f32, 0x1f3a}, {0x1f33, 0x1f3b}, {0x1f34, 0x1f3c}, {0x1f35, 0x1f3d}, {0x1f36, 0x1f3e}, {0x1f37, 0x1f3f}, {0x1f40, 0x1f48}, 
	{0x1f41, 0x1f49}, {0x1f42, 0x1f4a}, {0x1f43, 0x1f4b}, {0x1f44, 0x1f4c}, {0x1f45, 0x1f4d}, {0x1f51, 0x1f59}, {0x1f53, 0x1f5b}, {0x1f55, 0x1f5d}, {0x1f57, 0x1f5f}, {0x1f60, 0x1f68}, 
	{0x1f61, 0x1f69}, {0x1f62, 0x1f6a}, {0x1f63, 0x1f6b}, {0x1f64, 0x1f6c}, {0x1f65, 0x1f6d}, {0x1f66, 0x1f6e}, {0x1f67, 0x1f6f}, {0x1f70, 0x1fba}, {0x1f71, 0x1fbb}, {0x1f72, 0x1fc8}, 
	{0x1f73, 0x1fc9}, {0x1f74, 0x1fca}, {0x1f75, 0x1fcb}, {0x1f76, 0x1fda}, {0x1f77, 0x1fdb}, {0x1f78, 0x1ff8}, {0x1f79, 0x1ff9}, {0x1f7a, 0x1fea}, {0x1f7b, 0x1feb}, {0x1f7c, 0x1ffa}, 
	{0x1f7d, 0x1ffb}, {0x1fb0, 0x1fb8}, {0x1fb1, 0x1fb9}, {0x1fd0, 0x1fd8}, {0x1fd1, 0x1fd9}, {0x1fe0, 0x1fe8}, {0x1fe1, 0x1fe9}, {0x1fe5, 0x1fec}, {0x2170, 0x2160}, {0x2171, 0x2161}, 
	{0x2172, 0x2162}, {0x2173, 0x2163}, {0x2174, 0x2164}, {0x2175, 0x2165}, {0x2176, 0x2166}, {0x2177, 0x2167}, {0x2178, 0x2168}, {0x2179, 0x2169}, {0x217a, 0x216a}, {0x217b, 0x216b}, 
	{0x217c, 0x216c}, {0x217d, 0x216d}, {0x217e, 0x216e}, {0x217f, 0x216f}, {0x24d0, 0x24b6}, {0x24d1, 0x24b7}, {0x24d2, 0x24b8}, {0x24d3, 0x24b9}, {0x24d4, 0x24ba}, {0x24d5, 0x24bb}, 
	{0x24d6, 0x24bc}, {0x24d7, 0x24bd}, {0x24d8, 0x24be}, {0x24d9, 0x24bf}, {0x24da, 0x24c0}, {0x24db, 0x24c1}, {0x24dc, 0x24c2}, {0x24dd, 0x24c3}, {0x24de, 0x24c4}, {0x24df, 0x24c5}, 
	{0x24e0, 0x24c6}, {0x24e1, 0x24c7}, {0x24e2, 0x24c8}, {0x24e3, 0x24c9}, {0x24e4, 0x24ca}, {0x24e5, 0x24cb}, {0x24e6, 0x24cc}, {0x24e7, 0x24cd}, {0x24e8, 0x24ce}, {0x24e9, 0x24cf}, 
	{0xff41, 0xff21}, {0xff42, 0xff22}, {0xff43, 0xff23}, {0xff44, 0xff24}, {0xff45, 0xff25}, {0xff46, 0xff26}, {0xff47, 0xff27}, {0xff48, 0xff28}, {0xff49, 0xff29}, {0xff4a, 0xff2a}, 
	{0xff4b, 0xff2b}, {0xff4c, 0xff2c}, {0xff4d, 0xff2d}, {0xff4e, 0xff2e}, {0xff4f, 0xff2f}, {0xff50, 0xff30}, {0xff51, 0xff31}, {0xff52, 0xff32}, {0xff53, 0xff33}, {0xff54, 0xff34}, 
	{0xff55, 0xff35}, {0xff56, 0xff36}, {0xff57, 0xff37}, {0xff58, 0xff38}, {0xff59, 0xff39}, {0xff5a, 0xff3a}, };

	INTPTR ipChar=FSArrayFind2(s_CharMap, Char);
	return (ipChar>=0 ? s_CharMap[ipChar].Value : Char);
}

WCHAR FSToLower(WCHAR Char){
	static const CFSKeyValuePair<UINT16, UINT16> s_CharMap[] = {
	{0x41, 0x61}, {0x42, 0x62}, {0x43, 0x63}, {0x44, 0x64}, {0x45, 0x65}, {0x46, 0x66}, {0x47, 0x67}, {0x48, 0x68}, {0x49, 0x69}, {0x4a, 0x6a}, 
	{0x4b, 0x6b}, {0x4c, 0x6c}, {0x4d, 0x6d}, {0x4e, 0x6e}, {0x4f, 0x6f}, {0x50, 0x70}, {0x51, 0x71}, {0x52, 0x72}, {0x53, 0x73}, {0x54, 0x74}, 
	{0x55, 0x75}, {0x56, 0x76}, {0x57, 0x77}, {0x58, 0x78}, {0x59, 0x79}, {0x5a, 0x7a}, {0xc0, 0xe0}, {0xc1, 0xe1}, {0xc2, 0xe2}, {0xc3, 0xe3}, 
	{0xc4, 0xe4}, {0xc5, 0xe5}, {0xc6, 0xe6}, {0xc7, 0xe7}, {0xc8, 0xe8}, {0xc9, 0xe9}, {0xca, 0xea}, {0xcb, 0xeb}, {0xcc, 0xec}, {0xcd, 0xed}, 
	{0xce, 0xee}, {0xcf, 0xef}, {0xd0, 0xf0}, {0xd1, 0xf1}, {0xd2, 0xf2}, {0xd3, 0xf3}, {0xd4, 0xf4}, {0xd5, 0xf5}, {0xd6, 0xf6}, {0xd8, 0xf8}, 
	{0xd9, 0xf9}, {0xda, 0xfa}, {0xdb, 0xfb}, {0xdc, 0xfc}, {0xdd, 0xfd}, {0xde, 0xfe}, {0x100, 0x101}, {0x102, 0x103}, {0x104, 0x105}, {0x106, 0x107}, 
	{0x108, 0x109}, {0x10a, 0x10b}, {0x10c, 0x10d}, {0x10e, 0x10f}, {0x110, 0x111}, {0x112, 0x113}, {0x114, 0x115}, {0x116, 0x117}, {0x118, 0x119}, {0x11a, 0x11b}, 
	{0x11c, 0x11d}, {0x11e, 0x11f}, {0x120, 0x121}, {0x122, 0x123}, {0x124, 0x125}, {0x126, 0x127}, {0x128, 0x129}, {0x12a, 0x12b}, {0x12c, 0x12d}, {0x12e, 0x12f}, 
	{0x132, 0x133}, {0x134, 0x135}, {0x136, 0x137}, {0x139, 0x13a}, {0x13b, 0x13c}, {0x13d, 0x13e}, {0x13f, 0x140}, {0x141, 0x142}, {0x143, 0x144}, {0x145, 0x146}, 
	{0x147, 0x148}, {0x14a, 0x14b}, {0x14c, 0x14d}, {0x14e, 0x14f}, {0x150, 0x151}, {0x152, 0x153}, {0x154, 0x155}, {0x156, 0x157}, {0x158, 0x159}, {0x15a, 0x15b}, 
	{0x15c, 0x15d}, {0x15e, 0x15f}, {0x160, 0x161}, {0x162, 0x163}, {0x164, 0x165}, {0x166, 0x167}, {0x168, 0x169}, {0x16a, 0x16b}, {0x16c, 0x16d}, {0x16e, 0x16f}, 
	{0x170, 0x171}, {0x172, 0x173}, {0x174, 0x175}, {0x176, 0x177}, {0x178, 0xff}, {0x179, 0x17a}, {0x17b, 0x17c}, {0x17d, 0x17e}, {0x181, 0x253}, {0x182, 0x183}, 
	{0x184, 0x185}, {0x186, 0x254}, {0x187, 0x188}, {0x189, 0x256}, {0x18a, 0x257}, {0x18b, 0x18c}, {0x18e, 0x1dd}, {0x18f, 0x259}, {0x190, 0x25b}, {0x191, 0x192}, 
	{0x193, 0x260}, {0x194, 0x263}, {0x196, 0x269}, {0x197, 0x268}, {0x198, 0x199}, {0x19c, 0x26f}, {0x19d, 0x272}, {0x19f, 0x275}, {0x1a0, 0x1a1}, {0x1a2, 0x1a3}, 
	{0x1a4, 0x1a5}, {0x1a7, 0x1a8}, {0x1a9, 0x283}, {0x1ac, 0x1ad}, {0x1ae, 0x288}, {0x1af, 0x1b0}, {0x1b1, 0x28a}, {0x1b2, 0x28b}, {0x1b3, 0x1b4}, {0x1b5, 0x1b6}, 
	{0x1b7, 0x292}, {0x1b8, 0x1b9}, {0x1bc, 0x1bd}, {0x1c4, 0x1c6}, {0x1c7, 0x1c9}, {0x1ca, 0x1cc}, {0x1cd, 0x1ce}, {0x1cf, 0x1d0}, {0x1d1, 0x1d2}, {0x1d3, 0x1d4}, 
	{0x1d5, 0x1d6}, {0x1d7, 0x1d8}, {0x1d9, 0x1da}, {0x1db, 0x1dc}, {0x1de, 0x1df}, {0x1e0, 0x1e1}, {0x1e2, 0x1e3}, {0x1e4, 0x1e5}, {0x1e6, 0x1e7}, {0x1e8, 0x1e9}, 
	{0x1ea, 0x1eb}, {0x1ec, 0x1ed}, {0x1ee, 0x1ef}, {0x1f1, 0x1f3}, {0x1f4, 0x1f5}, {0x1fa, 0x1fb}, {0x1fc, 0x1fd}, {0x1fe, 0x1ff}, {0x200, 0x201}, {0x202, 0x203}, 
	{0x204, 0x205}, {0x206, 0x207}, {0x208, 0x209}, {0x20a, 0x20b}, {0x20c, 0x20d}, {0x20e, 0x20f}, {0x210, 0x211}, {0x212, 0x213}, {0x214, 0x215}, {0x216, 0x217}, 
	{0x386, 0x3ac}, {0x388, 0x3ad}, {0x389, 0x3ae}, {0x38a, 0x3af}, {0x38c, 0x3cc}, {0x38e, 0x3cd}, {0x38f, 0x3ce}, {0x391, 0x3b1}, {0x392, 0x3b2}, {0x393, 0x3b3}, 
	{0x394, 0x3b4}, {0x395, 0x3b5}, {0x396, 0x3b6}, {0x397, 0x3b7}, {0x398, 0x3b8}, {0x399, 0x3b9}, {0x39a, 0x3ba}, {0x39b, 0x3bb}, {0x39c, 0x3bc}, {0x39d, 0x3bd}, 
	{0x39e, 0x3be}, {0x39f, 0x3bf}, {0x3a0, 0x3c0}, {0x3a1, 0x3c1}, {0x3a3, 0x3c3}, {0x3a4, 0x3c4}, {0x3a5, 0x3c5}, {0x3a6, 0x3c6}, {0x3a7, 0x3c7}, {0x3a8, 0x3c8}, 
	{0x3a9, 0x3c9}, {0x3aa, 0x3ca}, {0x3ab, 0x3cb}, {0x3e2, 0x3e3}, {0x3e4, 0x3e5}, {0x3e6, 0x3e7}, {0x3e8, 0x3e9}, {0x3ea, 0x3eb}, {0x3ec, 0x3ed}, {0x3ee, 0x3ef}, 
	{0x401, 0x451}, {0x402, 0x452}, {0x403, 0x453}, {0x404, 0x454}, {0x405, 0x455}, {0x406, 0x456}, {0x407, 0x457}, {0x408, 0x458}, {0x409, 0x459}, {0x40a, 0x45a}, 
	{0x40b, 0x45b}, {0x40c, 0x45c}, {0x40e, 0x45e}, {0x40f, 0x45f}, {0x410, 0x430}, {0x411, 0x431}, {0x412, 0x432}, {0x413, 0x433}, {0x414, 0x434}, {0x415, 0x435}, 
	{0x416, 0x436}, {0x417, 0x437}, {0x418, 0x438}, {0x419, 0x439}, {0x41a, 0x43a}, {0x41b, 0x43b}, {0x41c, 0x43c}, {0x41d, 0x43d}, {0x41e, 0x43e}, {0x41f, 0x43f}, 
	{0x420, 0x440}, {0x421, 0x441}, {0x422, 0x442}, {0x423, 0x443}, {0x424, 0x444}, {0x425, 0x445}, {0x426, 0x446}, {0x427, 0x447}, {0x428, 0x448}, {0x429, 0x449}, 
	{0x42a, 0x44a}, {0x42b, 0x44b}, {0x42c, 0x44c}, {0x42d, 0x44d}, {0x42e, 0x44e}, {0x42f, 0x44f}, {0x460, 0x461}, {0x462, 0x463}, {0x464, 0x465}, {0x466, 0x467}, 
	{0x468, 0x469}, {0x46a, 0x46b}, {0x46c, 0x46d}, {0x46e, 0x46f}, {0x470, 0x471}, {0x472, 0x473}, {0x474, 0x475}, {0x476, 0x477}, {0x478, 0x479}, {0x47a, 0x47b}, 
	{0x47c, 0x47d}, {0x47e, 0x47f}, {0x480, 0x481}, {0x490, 0x491}, {0x492, 0x493}, {0x494, 0x495}, {0x496, 0x497}, {0x498, 0x499}, {0x49a, 0x49b}, {0x49c, 0x49d}, 
	{0x49e, 0x49f}, {0x4a0, 0x4a1}, {0x4a2, 0x4a3}, {0x4a4, 0x4a5}, {0x4a6, 0x4a7}, {0x4a8, 0x4a9}, {0x4aa, 0x4ab}, {0x4ac, 0x4ad}, {0x4ae, 0x4af}, {0x4b0, 0x4b1}, 
	{0x4b2, 0x4b3}, {0x4b4, 0x4b5}, {0x4b6, 0x4b7}, {0x4b8, 0x4b9}, {0x4ba, 0x4bb}, {0x4bc, 0x4bd}, {0x4be, 0x4bf}, {0x4c1, 0x4c2}, {0x4c3, 0x4c4}, {0x4c7, 0x4c8}, 
	{0x4cb, 0x4cc}, {0x4d0, 0x4d1}, {0x4d2, 0x4d3}, {0x4d4, 0x4d5}, {0x4d6, 0x4d7}, {0x4d8, 0x4d9}, {0x4da, 0x4db}, {0x4dc, 0x4dd}, {0x4de, 0x4df}, {0x4e0, 0x4e1}, 
	{0x4e2, 0x4e3}, {0x4e4, 0x4e5}, {0x4e6, 0x4e7}, {0x4e8, 0x4e9}, {0x4ea, 0x4eb}, {0x4ee, 0x4ef}, {0x4f0, 0x4f1}, {0x4f2, 0x4f3}, {0x4f4, 0x4f5}, {0x4f8, 0x4f9}, 
	{0x531, 0x561}, {0x532, 0x562}, {0x533, 0x563}, {0x534, 0x564}, {0x535, 0x565}, {0x536, 0x566}, {0x537, 0x567}, {0x538, 0x568}, {0x539, 0x569}, {0x53a, 0x56a}, 
	{0x53b, 0x56b}, {0x53c, 0x56c}, {0x53d, 0x56d}, {0x53e, 0x56e}, {0x53f, 0x56f}, {0x540, 0x570}, {0x541, 0x571}, {0x542, 0x572}, {0x543, 0x573}, {0x544, 0x574}, 
	{0x545, 0x575}, {0x546, 0x576}, {0x547, 0x577}, {0x548, 0x578}, {0x549, 0x579}, {0x54a, 0x57a}, {0x54b, 0x57b}, {0x54c, 0x57c}, {0x54d, 0x57d}, {0x54e, 0x57e}, 
	{0x54f, 0x57f}, {0x550, 0x580}, {0x551, 0x581}, {0x552, 0x582}, {0x553, 0x583}, {0x554, 0x584}, {0x555, 0x585}, {0x556, 0x586}, {0x10a0, 0x10d0}, {0x10a1, 0x10d1}, 
	{0x10a2, 0x10d2}, {0x10a3, 0x10d3}, {0x10a4, 0x10d4}, {0x10a5, 0x10d5}, {0x10a6, 0x10d6}, {0x10a7, 0x10d7}, {0x10a8, 0x10d8}, {0x10a9, 0x10d9}, {0x10aa, 0x10da}, {0x10ab, 0x10db}, 
	{0x10ac, 0x10dc}, {0x10ad, 0x10dd}, {0x10ae, 0x10de}, {0x10af, 0x10df}, {0x10b0, 0x10e0}, {0x10b1, 0x10e1}, {0x10b2, 0x10e2}, {0x10b3, 0x10e3}, {0x10b4, 0x10e4}, {0x10b5, 0x10e5}, 
	{0x10b6, 0x10e6}, {0x10b7, 0x10e7}, {0x10b8, 0x10e8}, {0x10b9, 0x10e9}, {0x10ba, 0x10ea}, {0x10bb, 0x10eb}, {0x10bc, 0x10ec}, {0x10bd, 0x10ed}, {0x10be, 0x10ee}, {0x10bf, 0x10ef}, 
	{0x10c0, 0x10f0}, {0x10c1, 0x10f1}, {0x10c2, 0x10f2}, {0x10c3, 0x10f3}, {0x10c4, 0x10f4}, {0x10c5, 0x10f5}, {0x1e00, 0x1e01}, {0x1e02, 0x1e03}, {0x1e04, 0x1e05}, {0x1e06, 0x1e07}, 
	{0x1e08, 0x1e09}, {0x1e0a, 0x1e0b}, {0x1e0c, 0x1e0d}, {0x1e0e, 0x1e0f}, {0x1e10, 0x1e11}, {0x1e12, 0x1e13}, {0x1e14, 0x1e15}, {0x1e16, 0x1e17}, {0x1e18, 0x1e19}, {0x1e1a, 0x1e1b}, 
	{0x1e1c, 0x1e1d}, {0x1e1e, 0x1e1f}, {0x1e20, 0x1e21}, {0x1e22, 0x1e23}, {0x1e24, 0x1e25}, {0x1e26, 0x1e27}, {0x1e28, 0x1e29}, {0x1e2a, 0x1e2b}, {0x1e2c, 0x1e2d}, {0x1e2e, 0x1e2f}, 
	{0x1e30, 0x1e31}, {0x1e32, 0x1e33}, {0x1e34, 0x1e35}, {0x1e36, 0x1e37}, {0x1e38, 0x1e39}, {0x1e3a, 0x1e3b}, {0x1e3c, 0x1e3d}, {0x1e3e, 0x1e3f}, {0x1e40, 0x1e41}, {0x1e42, 0x1e43}, 
	{0x1e44, 0x1e45}, {0x1e46, 0x1e47}, {0x1e48, 0x1e49}, {0x1e4a, 0x1e4b}, {0x1e4c, 0x1e4d}, {0x1e4e, 0x1e4f}, {0x1e50, 0x1e51}, {0x1e52, 0x1e53}, {0x1e54, 0x1e55}, {0x1e56, 0x1e57}, 
	{0x1e58, 0x1e59}, {0x1e5a, 0x1e5b}, {0x1e5c, 0x1e5d}, {0x1e5e, 0x1e5f}, {0x1e60, 0x1e61}, {0x1e62, 0x1e63}, {0x1e64, 0x1e65}, {0x1e66, 0x1e67}, {0x1e68, 0x1e69}, {0x1e6a, 0x1e6b}, 
	{0x1e6c, 0x1e6d}, {0x1e6e, 0x1e6f}, {0x1e70, 0x1e71}, {0x1e72, 0x1e73}, {0x1e74, 0x1e75}, {0x1e76, 0x1e77}, {0x1e78, 0x1e79}, {0x1e7a, 0x1e7b}, {0x1e7c, 0x1e7d}, {0x1e7e, 0x1e7f}, 
	{0x1e80, 0x1e81}, {0x1e82, 0x1e83}, {0x1e84, 0x1e85}, {0x1e86, 0x1e87}, {0x1e88, 0x1e89}, {0x1e8a, 0x1e8b}, {0x1e8c, 0x1e8d}, {0x1e8e, 0x1e8f}, {0x1e90, 0x1e91}, {0x1e92, 0x1e93}, 
	{0x1e94, 0x1e95}, {0x1ea0, 0x1ea1}, {0x1ea2, 0x1ea3}, {0x1ea4, 0x1ea5}, {0x1ea6, 0x1ea7}, {0x1ea8, 0x1ea9}, {0x1eaa, 0x1eab}, {0x1eac, 0x1ead}, {0x1eae, 0x1eaf}, {0x1eb0, 0x1eb1}, 
	{0x1eb2, 0x1eb3}, {0x1eb4, 0x1eb5}, {0x1eb6, 0x1eb7}, {0x1eb8, 0x1eb9}, {0x1eba, 0x1ebb}, {0x1ebc, 0x1ebd}, {0x1ebe, 0x1ebf}, {0x1ec0, 0x1ec1}, {0x1ec2, 0x1ec3}, {0x1ec4, 0x1ec5}, 
	{0x1ec6, 0x1ec7}, {0x1ec8, 0x1ec9}, {0x1eca, 0x1ecb}, {0x1ecc, 0x1ecd}, {0x1ece, 0x1ecf}, {0x1ed0, 0x1ed1}, {0x1ed2, 0x1ed3}, {0x1ed4, 0x1ed5}, {0x1ed6, 0x1ed7}, {0x1ed8, 0x1ed9}, 
	{0x1eda, 0x1edb}, {0x1edc, 0x1edd}, {0x1ede, 0x1edf}, {0x1ee0, 0x1ee1}, {0x1ee2, 0x1ee3}, {0x1ee4, 0x1ee5}, {0x1ee6, 0x1ee7}, {0x1ee8, 0x1ee9}, {0x1eea, 0x1eeb}, {0x1eec, 0x1eed}, 
	{0x1eee, 0x1eef}, {0x1ef0, 0x1ef1}, {0x1ef2, 0x1ef3}, {0x1ef4, 0x1ef5}, {0x1ef6, 0x1ef7}, {0x1ef8, 0x1ef9}, {0x1f08, 0x1f00}, {0x1f09, 0x1f01}, {0x1f0a, 0x1f02}, {0x1f0b, 0x1f03}, 
	{0x1f0c, 0x1f04}, {0x1f0d, 0x1f05}, {0x1f0e, 0x1f06}, {0x1f0f, 0x1f07}, {0x1f18, 0x1f10}, {0x1f19, 0x1f11}, {0x1f1a, 0x1f12}, {0x1f1b, 0x1f13}, {0x1f1c, 0x1f14}, {0x1f1d, 0x1f15}, 
	{0x1f28, 0x1f20}, {0x1f29, 0x1f21}, {0x1f2a, 0x1f22}, {0x1f2b, 0x1f23}, {0x1f2c, 0x1f24}, {0x1f2d, 0x1f25}, {0x1f2e, 0x1f26}, {0x1f2f, 0x1f27}, {0x1f38, 0x1f30}, {0x1f39, 0x1f31}, 
	{0x1f3a, 0x1f32}, {0x1f3b, 0x1f33}, {0x1f3c, 0x1f34}, {0x1f3d, 0x1f35}, {0x1f3e, 0x1f36}, {0x1f3f, 0x1f37}, {0x1f48, 0x1f40}, {0x1f49, 0x1f41}, {0x1f4a, 0x1f42}, {0x1f4b, 0x1f43}, 
	{0x1f4c, 0x1f44}, {0x1f4d, 0x1f45}, {0x1f59, 0x1f51}, {0x1f5b, 0x1f53}, {0x1f5d, 0x1f55}, {0x1f5f, 0x1f57}, {0x1f68, 0x1f60}, {0x1f69, 0x1f61}, {0x1f6a, 0x1f62}, {0x1f6b, 0x1f63}, 
	{0x1f6c, 0x1f64}, {0x1f6d, 0x1f65}, {0x1f6e, 0x1f66}, {0x1f6f, 0x1f67}, {0x1fb8, 0x1fb0}, {0x1fb9, 0x1fb1}, {0x1fba, 0x1f70}, {0x1fbb, 0x1f71}, {0x1fc8, 0x1f72}, {0x1fc9, 0x1f73}, 
	{0x1fca, 0x1f74}, {0x1fcb, 0x1f75}, {0x1fd8, 0x1fd0}, {0x1fd9, 0x1fd1}, {0x1fda, 0x1f76}, {0x1fdb, 0x1f77}, {0x1fe8, 0x1fe0}, {0x1fe9, 0x1fe1}, {0x1fea, 0x1f7a}, {0x1feb, 0x1f7b}, 
	{0x1fec, 0x1fe5}, {0x1ff8, 0x1f78}, {0x1ff9, 0x1f79}, {0x1ffa, 0x1f7c}, {0x1ffb, 0x1f7d}, {0x2160, 0x2170}, {0x2161, 0x2171}, {0x2162, 0x2172}, {0x2163, 0x2173}, {0x2164, 0x2174}, 
	{0x2165, 0x2175}, {0x2166, 0x2176}, {0x2167, 0x2177}, {0x2168, 0x2178}, {0x2169, 0x2179}, {0x216a, 0x217a}, {0x216b, 0x217b}, {0x216c, 0x217c}, {0x216d, 0x217d}, {0x216e, 0x217e}, 
	{0x216f, 0x217f}, {0x24b6, 0x24d0}, {0x24b7, 0x24d1}, {0x24b8, 0x24d2}, {0x24b9, 0x24d3}, {0x24ba, 0x24d4}, {0x24bb, 0x24d5}, {0x24bc, 0x24d6}, {0x24bd, 0x24d7}, {0x24be, 0x24d8}, 
	{0x24bf, 0x24d9}, {0x24c0, 0x24da}, {0x24c1, 0x24db}, {0x24c2, 0x24dc}, {0x24c3, 0x24dd}, {0x24c4, 0x24de}, {0x24c5, 0x24df}, {0x24c6, 0x24e0}, {0x24c7, 0x24e1}, {0x24c8, 0x24e2}, 
	{0x24c9, 0x24e3}, {0x24ca, 0x24e4}, {0x24cb, 0x24e5}, {0x24cc, 0x24e6}, {0x24cd, 0x24e7}, {0x24ce, 0x24e8}, {0x24cf, 0x24e9}, {0xff21, 0xff41}, {0xff22, 0xff42}, {0xff23, 0xff43}, 
	{0xff24, 0xff44}, {0xff25, 0xff45}, {0xff26, 0xff46}, {0xff27, 0xff47}, {0xff28, 0xff48}, {0xff29, 0xff49}, {0xff2a, 0xff4a}, {0xff2b, 0xff4b}, {0xff2c, 0xff4c}, {0xff2d, 0xff4d}, 
	{0xff2e, 0xff4e}, {0xff2f, 0xff4f}, {0xff30, 0xff50}, {0xff31, 0xff51}, {0xff32, 0xff52}, {0xff33, 0xff53}, {0xff34, 0xff54}, {0xff35, 0xff55}, {0xff36, 0xff56}, {0xff37, 0xff57}, 
	{0xff38, 0xff58}, {0xff39, 0xff59}, {0xff3a, 0xff5a}, };

	INTPTR ipChar=FSArrayFind2(s_CharMap, Char);
	return (ipChar>=0 ? s_CharMap[ipChar].Value : Char);
}

///////////////////////////////////////////////////////////////////////////////
// UCS4/UTF16 conversion
///////////////////////////////////////////////////////////////////////////////

INTPTR FSStrAppendFSLCHAR(CFSWString &szStr, LCHAR lChar)
{
#if defined FSUTF16
	if (lChar>=0x10000){
#else
	if (sizeof(WCHAR)==2 && lChar>=0x10000){
#endif
		WCHAR Char1, Char2;
		if (FSLtoW2(lChar, &Char1, &Char2)!=0) {
			return 0;
		}
		szStr+=Char1;
		szStr+=Char2;
		return 2;
	}
	szStr+=(WCHAR)lChar;
	return 1;
}

INTPTR FSStrGetFSLCHAR(LCHAR *plChar, const CFSWString &szStr, INTPTR ipPos)
{
#if defined FSUTF16
	if (FSIsW21(szStr[ipPos])){
#else
	if (sizeof(WCHAR)==2 && FSIsW21(szStr[ipPos])){
#endif
		if (FSW2toL(szStr[ipPos], szStr[ipPos+1], plChar)!=0) {
			return 0;
		}
		return 2;
	}		
	else {
		*plChar=szStr[ipPos];
	}
	return 1;
}

int FSStrCombineW2(CFSWString &szStr)
{
#if defined FSUTF16
	FSUNUSED(szStr);
#else
	if (sizeof(WCHAR)>2){
		for (INTPTR ip=0; szStr[ip]; ip++){
			LCHAR Char;
			if (FSIsW21(szStr[ip])){
				if (FSW2toL(szStr[ip], szStr[ip+1], &Char)!=0) {
					return -1;
				}
				szStr[ip]=(WCHAR)Char;
				szStr.Delete(ip+1);
			}
		}
	}
#endif
	return 0;
}

CFSAString FSStrToLowerUTF8(const CFSAString &szStr)
{
	return FSStrWtoA(FSStrAtoW(szStr, FSCP_UTF8).ToLower(), FSCP_UTF8);
}

CFSAString FSStrToUpperUTF8(const CFSAString &szStr)
{
	return FSStrWtoA(FSStrAtoW(szStr, FSCP_UTF8).ToUpper(), FSCP_UTF8);
}

///////////////////////////////////////////////////////////////////////////////
// Character types
///////////////////////////////////////////////////////////////////////////////

bool FSIsSpace(char Char) {
	return (Char>=0x0009 && Char<=0x000D) || // (control characters, containing Tab, CR and LF)
		Char==0x0020; // SPACE
}

bool FSIsSpace(wchar_t Char) {
	return (Char>=0x0009 && Char<=0x000D) || // (control characters, containing Tab, CR and LF)
		Char==0x0020 || // SPACE
		Char==0x0085 || // NEL (control character next line) !!! Windows skips that !!!
		Char==0x00A0 || // NBSP (NO-BREAK SPACE)
		Char==0x1680 || // OGHAM SPACE MARK
		Char==0x180E || // MONGOLIAN VOWEL SEPARATOR
		(Char>=0x2000 && Char<=0x200A) || // (different sorts of spaces)
		Char==0x2028 || // LS (LINE SEPARATOR)
		Char==0x2029 || // PS (PARAGRAPH SEPARATOR)
		Char==0x202F || // NNBSP (NARROW NO-BREAK SPACE)
		Char==0x205F || // MMSP (MEDIUM MATHEMATICAL SPACE)
		Char==0x3000; // IDEOGRAPHIC SPACE
}

bool FSIsLetter(char Char) {
	return (FSToLower(Char)!=FSToUpper(Char));
}

bool FSIsLetter(wchar_t Char) {
	static struct {
		UINT16 First;
		UINT16 Last;
	} s_Letters[]={
	{0x0041, 0x005a}, {0x0061, 0x007a}, {0x00c0, 0x00d6}, {0x00d8, 0x00f6}, {0x00f8, 0x02b8}, {0x02bb, 0x02c1}, {0x02d0, 0x02d1}, {0x02e0, 0x02e4}, {0x02ee, 0x02ee}, {0x0370, 0x0373}, 
	{0x0376, 0x0377}, {0x037a, 0x037d}, {0x0386, 0x0386}, {0x0388, 0x038a}, {0x038c, 0x038c}, {0x038e, 0x03a1}, {0x03a3, 0x03f5}, {0x03f7, 0x0481}, {0x048a, 0x0523}, {0x0531, 0x0556}, 
	{0x0559, 0x0559}, {0x0561, 0x0587}, {0x05d0, 0x05ea}, {0x05f0, 0x05f2}, {0x0621, 0x064a}, {0x066e, 0x066f}, {0x0671, 0x06d3}, {0x06d5, 0x06d5}, {0x06e5, 0x06e6}, {0x06ee, 0x06ef}, 
	{0x06fa, 0x06fc}, {0x06ff, 0x06ff}, {0x0710, 0x0710}, {0x0712, 0x072f}, {0x074d, 0x07a5}, {0x07b1, 0x07b1}, {0x07ca, 0x07ea}, {0x07f4, 0x07f5}, {0x07fa, 0x07fa}, {0x0904, 0x0939}, 
	{0x093d, 0x093d}, {0x0950, 0x0950}, {0x0958, 0x0961}, {0x0971, 0x0972}, {0x097b, 0x097f}, {0x0985, 0x098c}, {0x098f, 0x0990}, {0x0993, 0x09a8}, {0x09aa, 0x09b0}, {0x09b2, 0x09b2}, 
	{0x09b6, 0x09b9}, {0x09bd, 0x09bd}, {0x09ce, 0x09ce}, {0x09dc, 0x09dd}, {0x09df, 0x09e1}, {0x09f0, 0x09f1}, {0x0a05, 0x0a0a}, {0x0a0f, 0x0a10}, {0x0a13, 0x0a28}, {0x0a2a, 0x0a30}, 
	{0x0a32, 0x0a33}, {0x0a35, 0x0a36}, {0x0a38, 0x0a39}, {0x0a59, 0x0a5c}, {0x0a5e, 0x0a5e}, {0x0a72, 0x0a74}, {0x0a85, 0x0a8d}, {0x0a8f, 0x0a91}, {0x0a93, 0x0aa8}, {0x0aaa, 0x0ab0}, 
	{0x0ab2, 0x0ab3}, {0x0ab5, 0x0ab9}, {0x0abd, 0x0abd}, {0x0ad0, 0x0ad0}, {0x0ae0, 0x0ae1}, {0x0b05, 0x0b0c}, {0x0b0f, 0x0b10}, {0x0b13, 0x0b28}, {0x0b2a, 0x0b30}, {0x0b32, 0x0b33}, 
	{0x0b35, 0x0b39}, {0x0b3d, 0x0b3d}, {0x0b5c, 0x0b5d}, {0x0b5f, 0x0b61}, {0x0b71, 0x0b71}, {0x0b83, 0x0b83}, {0x0b85, 0x0b8a}, {0x0b8e, 0x0b90}, {0x0b92, 0x0b95}, {0x0b99, 0x0b9a}, 
	{0x0b9c, 0x0b9c}, {0x0b9e, 0x0b9f}, {0x0ba3, 0x0ba4}, {0x0ba8, 0x0baa}, {0x0bae, 0x0bb9}, {0x0bd0, 0x0bd0}, {0x0c05, 0x0c0c}, {0x0c0e, 0x0c10}, {0x0c12, 0x0c28}, {0x0c2a, 0x0c33}, 
	{0x0c35, 0x0c39}, {0x0c3d, 0x0c3d}, {0x0c58, 0x0c59}, {0x0c60, 0x0c61}, {0x0c85, 0x0c8c}, {0x0c8e, 0x0c90}, {0x0c92, 0x0ca8}, {0x0caa, 0x0cb3}, {0x0cb5, 0x0cb9}, {0x0cbd, 0x0cbd}, 
	{0x0cde, 0x0cde}, {0x0ce0, 0x0ce1}, {0x0d05, 0x0d0c}, {0x0d0e, 0x0d10}, {0x0d12, 0x0d28}, {0x0d2a, 0x0d39}, {0x0d3d, 0x0d3d}, {0x0d60, 0x0d61}, {0x0d7a, 0x0d7f}, {0x0d85, 0x0d96}, 
	{0x0d9a, 0x0db1}, {0x0db3, 0x0dbb}, {0x0dbd, 0x0dbd}, {0x0dc0, 0x0dc6}, {0x0e01, 0x0e3a}, {0x0e40, 0x0e4e}, {0x0e81, 0x0e82}, {0x0e84, 0x0e84}, {0x0e87, 0x0e88}, {0x0e8a, 0x0e8a}, 
	{0x0e8d, 0x0e8d}, {0x0e94, 0x0e97}, {0x0e99, 0x0e9f}, {0x0ea1, 0x0ea3}, {0x0ea5, 0x0ea5}, {0x0ea7, 0x0ea7}, {0x0eaa, 0x0eab}, {0x0ead, 0x0eb0}, {0x0eb2, 0x0eb3}, {0x0ebd, 0x0ebd}, 
	{0x0ec0, 0x0ec4}, {0x0ec6, 0x0ec6}, {0x0edc, 0x0edd}, {0x0f00, 0x0f00}, {0x0f40, 0x0f47}, {0x0f49, 0x0f6c}, {0x0f88, 0x0f8b}, {0x1000, 0x102a}, {0x103f, 0x103f}, {0x1050, 0x1055}, 
	{0x105a, 0x105d}, {0x1061, 0x1061}, {0x1065, 0x1066}, {0x106e, 0x1070}, {0x1075, 0x1081}, {0x108e, 0x108e}, {0x10a0, 0x10c5}, {0x10d0, 0x10fa}, {0x10fc, 0x10fc}, {0x1100, 0x1159}, 
	{0x115f, 0x11a2}, {0x11a8, 0x11f9}, {0x1200, 0x1248}, {0x124a, 0x124d}, {0x1250, 0x1256}, {0x1258, 0x1258}, {0x125a, 0x125d}, {0x1260, 0x1288}, {0x128a, 0x128d}, {0x1290, 0x12b0}, 
	{0x12b2, 0x12b5}, {0x12b8, 0x12be}, {0x12c0, 0x12c0}, {0x12c2, 0x12c5}, {0x12c8, 0x12d6}, {0x12d8, 0x1310}, {0x1312, 0x1315}, {0x1318, 0x135a}, {0x1380, 0x138f}, {0x13a0, 0x13f4}, 
	{0x1401, 0x166c}, {0x166f, 0x1676}, {0x1681, 0x169a}, {0x16a0, 0x16ea}, {0x16ee, 0x16f0}, {0x1700, 0x170c}, {0x170e, 0x1711}, {0x1720, 0x1731}, {0x1740, 0x1751}, {0x1760, 0x176c}, 
	{0x176e, 0x1770}, {0x1780, 0x17b3}, {0x17d7, 0x17d7}, {0x17dc, 0x17dc}, {0x1820, 0x1877}, {0x1880, 0x18a8}, {0x18aa, 0x18aa}, {0x1900, 0x191c}, {0x1950, 0x196d}, {0x1970, 0x1974}, 
	{0x1980, 0x19a9}, {0x19c1, 0x19c7}, {0x1a00, 0x1a16}, {0x1b05, 0x1b33}, {0x1b45, 0x1b4b}, {0x1b83, 0x1ba0}, {0x1bae, 0x1baf}, {0x1c00, 0x1c23}, {0x1c4d, 0x1c4f}, {0x1c5a, 0x1c7d}, 
	{0x1d00, 0x1dbf}, {0x1e00, 0x1f15}, {0x1f18, 0x1f1d}, {0x1f20, 0x1f45}, {0x1f48, 0x1f4d}, {0x1f50, 0x1f57}, {0x1f59, 0x1f59}, {0x1f5b, 0x1f5b}, {0x1f5d, 0x1f5d}, {0x1f5f, 0x1f7d}, 
	{0x1f80, 0x1fb4}, {0x1fb6, 0x1fbc}, {0x1fbe, 0x1fbe}, {0x1fc2, 0x1fc4}, {0x1fc6, 0x1fcc}, {0x1fd0, 0x1fd3}, {0x1fd6, 0x1fdb}, {0x1fe0, 0x1fec}, {0x1ff2, 0x1ff4}, {0x1ff6, 0x1ffc}, 
	{0x2071, 0x2071}, {0x207f, 0x207f}, {0x2090, 0x2094}, {0x2102, 0x2102}, {0x2107, 0x2107}, {0x210a, 0x2113}, {0x2115, 0x2115}, {0x2119, 0x211d}, {0x2124, 0x2124}, {0x2126, 0x2126}, 
	{0x2128, 0x2128}, {0x212a, 0x212d}, {0x212f, 0x2139}, {0x213c, 0x213f}, {0x2145, 0x2149}, {0x214e, 0x214e}, {0x2160, 0x2188}, {0x2c00, 0x2c2e}, {0x2c30, 0x2c5e}, {0x2c60, 0x2c6f}, 
	{0x2c71, 0x2c7d}, {0x2c80, 0x2ce4}, {0x2d00, 0x2d25}, {0x2d30, 0x2d65}, {0x2d6f, 0x2d6f}, {0x2d80, 0x2d96}, {0x2da0, 0x2da6}, {0x2da8, 0x2dae}, {0x2db0, 0x2db6}, {0x2db8, 0x2dbe}, 
	{0x2dc0, 0x2dc6}, {0x2dc8, 0x2dce}, {0x2dd0, 0x2dd6}, {0x2dd8, 0x2dde}, {0x3005, 0x3007}, {0x3021, 0x3029}, {0x3031, 0x3035}, {0x3038, 0x303c}, {0x3041, 0x3096}, {0x309d, 0x309f}, 
	{0x30a1, 0x30fa}, {0x30fc, 0x30ff}, {0x3105, 0x312d}, {0x3131, 0x318e}, {0x31a0, 0x31b7}, {0x31f0, 0x31ff}, {0x3400, 0x4db5}, {0x4e00, 0x9fc3}, {0xa000, 0xa48c}, {0xa500, 0xa60c}, 
	{0xa610, 0xa61f}, {0xa62a, 0xa62b}, {0xa640, 0xa65f}, {0xa662, 0xa66e}, {0xa680, 0xa697}, {0xa722, 0xa787}, {0xa78b, 0xa78c}, {0xa7fb, 0xa801}, {0xa803, 0xa805}, {0xa807, 0xa80a}, 
	{0xa80c, 0xa822}, {0xa840, 0xa873}, {0xa882, 0xa8b3}, {0xa90a, 0xa925}, {0xa930, 0xa946}, {0xaa00, 0xaa28}, {0xaa40, 0xaa42}, {0xaa44, 0xaa4b}, {0xac00, 0xd7a3}, {0xf900, 0xfa2d}, 
	{0xfa30, 0xfa6a}, {0xfa70, 0xfad9}, {0xfb00, 0xfb06}, {0xfb13, 0xfb17}, {0xfb1d, 0xfb1d}, {0xfb1f, 0xfb28}, {0xfb2a, 0xfb36}, {0xfb38, 0xfb3c}, {0xfb3e, 0xfb3e}, {0xfb40, 0xfb41}, 
	{0xfb43, 0xfb44}, {0xfb46, 0xfbb1}, {0xfbd3, 0xfd3d}, {0xfd50, 0xfd8f}, {0xfd92, 0xfdc7}, {0xfdf0, 0xfdfb}, {0xfe70, 0xfe74}, {0xfe76, 0xfefc}, {0xff21, 0xff3a}, {0xff41, 0xff5a}, 
	{0xff66, 0xffbe}, {0xffc2, 0xffc7}, {0xffca, 0xffcf}, {0xffd2, 0xffd7}, {0xffda, 0xffdc}
	};

	INTPTR ipStart=0;
	INTPTR ipEnd=sizeof(s_Letters)/sizeof(s_Letters[0]);
	if (Char<s_Letters[ipStart].First || Char>s_Letters[ipEnd-1].Last) return false;
	UINT16 Char16=(UINT16)Char;

	// Handle ASCII first
	if (Char16<0x80) {
		return ((Char16>=s_Letters[0].First && Char16<=s_Letters[0].Last) ||
				(Char16>=s_Letters[1].First && Char16<=s_Letters[1].Last));
	}
	ipStart=2;

	for (;;) {
		if (ipStart>=ipEnd-1) {
			return (ipStart==ipEnd-1 && Char16>=s_Letters[ipStart].First && Char16<=s_Letters[ipStart].Last);
		}

		INTPTR ipMiddle=(ipStart+ipEnd)/2;
		if (Char16<s_Letters[ipMiddle].First) {
			ipEnd=ipMiddle;
		} else {
			ipStart=ipMiddle;
		}
	}
}
