import * as s3 from '@aws-cdk/aws-s3';
import * as s3_assets from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
/**
 * Represents an OpenAPI definition asset.
 *
 * @stability stable
 */
export declare abstract class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket.
     *
     * @stability stable
     */
    static fromBucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3ApiDefinition;
    /**
     * Create an API definition from an inline object.
     *
     * The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @stability stable
     * @example
     *
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition: any): InlineApiDefinition;
    /**
     * Loads the API specification from a local disk asset.
     *
     * @stability stable
     */
    static fromAsset(file: string, options?: s3_assets.AssetOptions): AssetApiDefinition;
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @param scope The binding scope.
     * @stability stable
     */
    abstract bind(scope: Construct): ApiDefinitionConfig;
}
/**
 * S3 location of the API definition file.
 *
 * @stability stable
 */
export interface ApiDefinitionS3Location {
    /**
     * The S3 bucket.
     *
     * @stability stable
     */
    readonly bucket: string;
    /**
     * The S3 key.
     *
     * @stability stable
     */
    readonly key: string;
    /**
     * An optional version.
     *
     * @default - latest version
     * @stability stable
     */
    readonly version?: string;
}
/**
 * Post-Binding Configuration for a CDK construct.
 *
 * @stability stable
 */
export interface ApiDefinitionConfig {
    /**
     * The location of the specification in S3 (mutually exclusive with `inlineDefinition`).
     *
     * @default - API definition is not an S3 location
     * @stability stable
     */
    readonly s3Location?: ApiDefinitionS3Location;
    /**
     * Inline specification (mutually exclusive with `s3Location`).
     *
     * @default - API definition is not defined inline
     * @stability stable
     */
    readonly inlineDefinition?: any;
}
/**
 * OpenAPI specification from an S3 archive.
 *
 * @stability stable
 */
export declare class S3ApiDefinition extends ApiDefinition {
    private key;
    private objectVersion?;
    private bucketName;
    /**
     * @stability stable
     */
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope: Construct): ApiDefinitionConfig;
}
/**
 * OpenAPI specification from an inline JSON object.
 *
 * @stability stable
 */
export declare class InlineApiDefinition extends ApiDefinition {
    private definition;
    /**
     * @stability stable
     */
    constructor(definition: any);
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope: Construct): ApiDefinitionConfig;
}
/**
 * OpenAPI specification from a local file.
 *
 * @stability stable
 */
export declare class AssetApiDefinition extends ApiDefinition {
    private readonly path;
    private readonly options;
    private asset?;
    /**
     * @stability stable
     */
    constructor(path: string, options?: s3_assets.AssetOptions);
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope: Construct): ApiDefinitionConfig;
}
