import { Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IApiKey } from './api-key';
import { Method } from './method';
import { IRestApi } from './restapi';
import { Stage } from './stage';
/**
 * Container for defining throttling parameters to API stages or methods.
 *
 * @stability stable
 * @link https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
 */
export interface ThrottleSettings {
    /**
     * The API request steady-state rate limit (average requests per second over an extended period of time).
     *
     * @default none
     * @stability stable
     */
    readonly rateLimit?: number;
    /**
     * The maximum API request rate limit over a time ranging from one to a few seconds.
     *
     * @default none
     * @stability stable
     */
    readonly burstLimit?: number;
}
/**
 * Time period for which quota settings apply.
 *
 * @stability stable
 */
export declare enum Period {
    /**
     * @stability stable
     */
    DAY = "DAY",
    /**
     * @stability stable
     */
    WEEK = "WEEK",
    /**
     * @stability stable
     */
    MONTH = "MONTH"
}
/**
 * Specifies the maximum number of requests that clients can make to API Gateway APIs.
 *
 * @stability stable
 */
export interface QuotaSettings {
    /**
     * The maximum number of requests that users can make within the specified time period.
     *
     * @default none
     * @stability stable
     */
    readonly limit?: number;
    /**
     * For the initial time period, the number of requests to subtract from the specified limit.
     *
     * @default none
     * @stability stable
     */
    readonly offset?: number;
    /**
     * The time period for which the maximum limit of requests applies.
     *
     * @default none
     * @stability stable
     */
    readonly period?: Period;
}
/**
 * Represents per-method throttling for a resource.
 *
 * @stability stable
 */
export interface ThrottlingPerMethod {
    /**
     * [disable-awslint:ref-via-interface] The method for which you specify the throttling settings.
     *
     * @default none
     * @stability stable
     */
    readonly method: Method;
    /**
     * Specifies the overall request rate (average requests per second) and burst capacity.
     *
     * @default none
     * @stability stable
     */
    readonly throttle: ThrottleSettings;
}
/**
 * Represents the API stages that a usage plan applies to.
 *
 * @stability stable
 */
export interface UsagePlanPerApiStage {
    /**
     * @default none
     * @stability stable
     */
    readonly api?: IRestApi;
    /**
     * [disable-awslint:ref-via-interface].
     *
     * @default none
     * @stability stable
     */
    readonly stage?: Stage;
    /**
     * @default none
     * @stability stable
     */
    readonly throttle?: ThrottlingPerMethod[];
}
/**
 * @stability stable
 */
export interface UsagePlanProps {
    /**
     * API Stages to be associated with the usage plan.
     *
     * @default none
     * @stability stable
     */
    readonly apiStages?: UsagePlanPerApiStage[];
    /**
     * Represents usage plan purpose.
     *
     * @default none
     * @stability stable
     */
    readonly description?: string;
    /**
     * Number of requests clients can make in a given time period.
     *
     * @default none
     * @stability stable
     */
    readonly quota?: QuotaSettings;
    /**
     * Overall throttle settings for the API.
     *
     * @default none
     * @stability stable
     */
    readonly throttle?: ThrottleSettings;
    /**
     * Name for this usage plan.
     *
     * @default none
     * @stability stable
     */
    readonly name?: string;
    /**
     * (deprecated) ApiKey to be associated with the usage plan.
     *
     * @default none
     * @deprecated use `addApiKey()`
     */
    readonly apiKey?: IApiKey;
}
/**
 * Options to the UsagePlan.addApiKey() method.
 *
 * @stability stable
 */
export interface AddApiKeyOptions {
    /**
     * Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource.
     *
     * @default - autogenerated by the CDK
     * @stability stable
     */
    readonly overrideLogicalId?: string;
}
/**
 * @stability stable
 */
export declare class UsagePlan extends Resource {
    /**
     * @stability stable
     * @attribute true
     */
    readonly usagePlanId: string;
    private readonly apiStages;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: UsagePlanProps);
    /**
     * Adds an ApiKey.
     *
     * @param apiKey the api key to associate with this usage plan.
     * @param options options that control the behaviour of this method.
     * @stability stable
     */
    addApiKey(apiKey: IApiKey, options?: AddApiKeyOptions): void;
    /**
     * Adds an apiStage.
     *
     * @stability stable
     */
    addApiStage(apiStage: UsagePlanPerApiStage): void;
    /**
     *
     * @param props
     */
    private renderApiStages;
    private createStage;
    private renderQuota;
    private renderThrottle;
    private renderThrottlePerMethod;
}
