from typing import Optional, List, Any

from aws_cdk.aws_apigatewayv2 import CfnRoute, CfnRouteResponse
from aws_cdk.core import Stack

from b_aws_websocket_api.ws_api import WsApi


class WsRoute(CfnRoute):
    """
    Creates a route for a web socket API.
    """

    def __init__(
            self,
            scope: Stack,
            id: str,
            ws_api: WsApi,
            route_key: str,
            api_key_required: Optional[bool] = None,
            authorization_scopes: Optional[List[str]] = None,
            authorization_type: Optional[str] = None,
            authorizer_id: Optional[str] = None,
            model_selection_expression: Optional[str] = None,
            operation_name: Optional[str] = None,
            request_models: Any = None,
            request_parameters: Any = None,
            route_response_selection_expression: Optional[str] = None,
            target: Optional[str] = None,
            default_route_response: Optional[bool] = None,
            *args,
            **kwargs
    ) -> None:
        """
        Constructor.

        :param scope: CloudFormation-equivalent stack.
        :param id: Resource id.
        :param ws_api: Web socket API for which to create the route.
        :param route_key: The route key for the route.
        :param api_key_required: Specifies whether an API key is required for the route.
        :param authorization_scopes: The authorization scopes supported by this route.
        :param authorization_type: The authorization type for the route. For WebSocket APIs, valid values are NONE
        for open access, AWS_IAM for using AWS IAM permissions, and CUSTOM for using a Lambda authorizer.
        :param authorizer_id: The identifier of the Authorizer resource to be associated with this route.
        The authorizer identifier is generated by API Gateway when you created the authorizer.
        :param model_selection_expression: The model selection expression for the route.
        :param operation_name: The operation name for the route.
        :param request_models: The request models for the route.
        :param request_parameters: The request parameters for the route.
        :param route_response_selection_expression: The route response selection expression for the route.
        :param target: The target for the route.
        :param default_route_response: Specify whether to create a route response resource.
        """
        super().__init__(
            scope=scope,
            id=id,
            api_id=ws_api.ref,
            route_key=route_key,
            api_key_required=api_key_required,
            authorization_scopes=authorization_scopes,
            authorization_type=authorization_type,
            authorizer_id=authorizer_id,
            model_selection_expression=model_selection_expression,
            operation_name=operation_name,
            request_models=request_models,
            request_parameters=request_parameters,
            route_response_selection_expression=route_response_selection_expression,
            target=target,
            *args,
            **kwargs
        )

        if default_route_response in [True, None]:
            CfnRouteResponse(
                scope=scope,
                id=f'{id}Response',
                api_id=ws_api.ref,
                route_id=self.ref,
                route_response_key='$default',
            )
