# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['peopledatalabs', 'peopledatalabs.endpoints', 'peopledatalabs.models']

package_data = \
{'': ['*']}

install_requires = \
['email-validator>=1.1,<2.0',
 'pydantic>=1.8,<1.10',
 'python-dotenv>=0,<1',
 'requests>=2,<3']

setup_kwargs = {
    'name': 'peopledatalabs',
    'version': '1.1.1',
    'description': 'Official Python client for the People Data Labs API',
    'long_description': '<p align="center">\n<img src="https://i.imgur.com/S7DkZtr.png" width="250" alt="People Data Labs Logo">\n</p>\n<h1 align="center">People Data Labs Python Client</h1>\n<p align="center">Official Python client for the People Data Labs API.</p>\n\n<p align="center">\n  <a href="https://github.com/peopledatalabs/peopledatalabs-python">\n    <img src="https://img.shields.io/badge/repo%20status-Active-limegreen" alt="Repo Status">\n  </a>&nbsp;\n  <a href="https://pypi.org/project/peopledatalabs/">\n    <img src="https://img.shields.io/pypi/v/peopledatalabs.svg?logo=pypi&logoColor=fff&label=PyPI+package&color=limegreen" alt="People Data Labs on PyPI" />\n  </a>&nbsp;\n  <a href="https://pypi.org/project/peopledatalabs/">\n    <img src="https://img.shields.io/pypi/pyversions/peopledatalabs.svg" alt="People Data Labs on PyPI" />\n  </a>&nbsp;\n  <a href="https://github.com/peopledatalabs/peopledatalabs-python/actions/workflows/python-poetry.yml">\n    <img src="https://github.com/peopledatalabs/peopledatalabs-python/actions/workflows/python-poetry.yml/badge.svg" alt="Tests Status" />\n  </a>\n</p>\n\n## Table of Contents\n\n- [🔧 Installation](#installation)\n- [🚀 Usage](#usage)\n- [🏝 Sandbox Usage](#sandbox)\n- [🌐 Endpoints](#endpoints)\n- [📘 Documentation](#documentation)\n\n## 🔧 Installation <a name="installation"></a>\n\n1. Install from PyPi using [pip](https://pip.pypa.io/en/latest/), a package manager for Python.\n\n    ```bash\n    pip install peopledatalabs\n    ```\n\n2. Sign up for a [free PDL API key](https://www.peopledatalabs.com/signup).\n3. Set your API key in the `PDL_API_KEY` environment variable.\n\n## 🚀 Usage <a name="usage"></a>\n\nFirst, create the PDLPY client:\n\n```python\nfrom peopledatalabs import PDLPY\n\n\n# specifying an API key\nclient = PDLPY(\n    api_key="YOUR API KEY",\n)\n\n# or leave blank if you have PDL_API_KEY set in your environment or .env file\nclient = PDLPY()\n```\n\n**Note:** You can provide your API key directly in code, or alternatively set a `PDL_API_KEY` variable in your environment or `.env` file.\n\nThen, send requests to any PDL API Endpoint.\n\n### Getting Person Data\n\n#### By Enrichment\n\n```python\nresult = client.person.enrichment(\n    phone="4155688415",\n    pretty=True,\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code};"\n        f"\\nReason: {result.reason};"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']};"\n    )\n```\n\n#### By Bulk Enrichment\n\n```python\nresult = client.person.bulk(\n    required="emails AND profiles",\n    requests=[\n        {\n            "metadata": {\n                "user_id": "123"\n            },\n            "params": {\n                "profile": ["linkedin.com/in/seanthorne"],\n                "location": ["SF Bay Area"],\n                "name": ["Sean F. Thorne"],\n            }\n        },\n        {\n            "metadata": {\n                "user_id": "345"\n            },\n            "params": {\n                "profile": ["https://www.linkedin.com/in/haydenconrad/"],\n                "first_name": "Hayden",\n                "last_name": "Conrad",\n            }\n        }\n    ]\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By Search (Elasticsearch)\n\n```python\nes_query = {\n    "query": {\n        "bool": {\n            "must": [\n                {"term": {"location_country": "mexico"}},\n                {"term": {"job_title_role": "health"}},\n            ]\n        }\n    }\n}\ndata = {\n    "query": es_query,\n    "size": 10,\n    "pretty": True,\n    "dataset": "phone, mobile_phone",\n}\nresult = client.person.search(**data)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By Search (SQL)\n\n```python\nsql_query = (\n    "SELECT * FROM person"\n    " WHERE location_country=\'mexico\'"\n    " AND job_title_role=\'health\'"\n    " AND phone_numbers IS NOT NULL;"\n)\ndata = {\n    "sql": sql_query,\n    "size": 10,\n    "pretty": True,\n    "dataset": "phone, mobile_phone",\n}\nresult = client.person.search(**data)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By `PDL_ID` (Retrieve API)\n\n```python\nresult = client.person.retrieve(\n    person_id="qEnOZ5Oh0poWnQ1luFBfVw_0000",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By Fuzzy Enrichment (Identify API)\n\n```python\nresult = client.person.enrichment(\n    name="sean thorne",\n    pretty=True,\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n### Getting Company Data\n\n#### By Enrichment\n\n```python\nresult = client.company.enrichment(\n    website="peopledatalabs.com",\n    pretty=True,\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By Search (Elasticsearch)\n\n```python\nes_query = {\n    "query": {\n        "bool": {\n            "must": [\n                {"term": {"tags": "big data"}},\n                {"term": {"industry": "financial services"}},\n                {"term": {"location.country": "united states"}},\n            ]\n        }\n    }\n}\ndata = {\n    "query": es_query,\n    "size": 10,\n    "pretty": True,\n}\nresult = client.company.search(**data)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### By Search (SQL)\n\n```python\nsql_query = (\n    "SELECT * FROM company"\n    " WHERE tags=\'big data\'"\n    " AND industry=\'financial services\'"\n    " AND location.country=\'united states\';"\n)\ndata = {\n    "sql": sql_query,\n    "size": 10,\n    "pretty": True,\n}\nresult = client.company.search(**data)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n### Using supporting APIs\n\n#### Get Autocomplete Suggestions\n\n```python\nresult = client.autocomplete(\n    field="title",\n    text="full",\n    size=10,\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### Clean Raw Company Strings\n\n```python\nresult = client.company.cleaner(\n    name="peOple DaTa LabS",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### Clean Raw Location Strings\n\n```python\nresult = client.location.cleaner(\n    location="455 Market Street, San Francisco, California 94105, US",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### Clean Raw School Strings\n\n```python\nresult = client.school.cleaner(\n    name="university of oregon",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### Get Job Title Enrichment\n\n```python\nresult = client.job_title(\n    job_title="data scientist",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n#### Get Skill Enrichment\n\n```python\nresult = client.skill(\n    skill="c++",\n)\nif result.ok:\n    print(result.text)\nelse:\n    print(\n        f"Status: {result.status_code}"\n        f"\\nReason: {result.reason}"\n        f"\\nMessage: {result.json()[\'error\'][\'message\']}"\n    )\n```\n\n## 🏝 Sandbox Usage <a name="sandbox"></a>\n#### To enable sandbox usage, use the sandbox flag on PDLPY\n\n```python\nPDLPY(sandbox=True)\n```\n\n## 🌐 Endpoints <a name="endpoints"></a>\n\n**Person Endpoints**\n\n| API Endpoint                                                                           | PDLPY Function                      |\n| -------------------------------------------------------------------------------------- | ----------------------------------- |\n| [Person Enrichment API](https://docs.peopledatalabs.com/docs/enrichment-api)           | `PDLPY.person.enrichment(**params)` |\n| [Person Bulk Enrichment API](https://docs.peopledatalabs.com/docs/bulk-enrichment-api) | `PDLPY.person.bulk(**params)`       |\n| [Person Search API](https://docs.peopledatalabs.com/docs/search-api)                   | `PDLPY.person.search(**params)`     |\n| [Person Retrieve API](https://docs.peopledatalabs.com/docs/person-retrieve-api)        | `PDLPY.person.retrieve(**params)`   |\n| [Person Identify API](https://docs.peopledatalabs.com/docs/identify-api)               | `PDLPY.person.identify(**params)`   |\n\n**Company Endpoints**\n\n| API Endpoint                                                                          | PDLPY Function                       |\n| ------------------------------------------------------------------------------------- | ------------------------------------ |\n| [Company Enrichment API](https://docs.peopledatalabs.com/docs/company-enrichment-api) | `PDLPY.company.enrichment(**params)` |\n| [Company Search API](https://docs.peopledatalabs.com/docs/company-search-api)         | `PDLPY.company.search(**params)`     |\n\n**Supporting Endpoints**\n\n| API Endpoint                                                                            | PDLJS Function                     |\n| --------------------------------------------------------------------------------------- | ---------------------------------- |\n| [Autocomplete API](https://docs.peopledatalabs.com/docs/autocomplete-api)               | `PDLPY.autocomplete(**params)`     |\n| [Company Cleaner API](https://docs.peopledatalabs.com/docs/cleaner-apis#companyclean)   | `PDLPY.company.cleaner(**params)`  |\n| [Location Cleaner API](https://docs.peopledatalabs.com/docs/cleaner-apis#locationclean) | `PDLPY.location.cleaner(**params)` |\n| [School Cleaner API](https://docs.peopledatalabs.com/docs/cleaner-apis#schoolclean)     | `PDLPY.school.cleaner(**params)`   |\n| [Job Title Enrichment API](https://docs.peopledatalabs.com/docs/job-title-enrichment-api) | `PDLPY.job_title(**params)` |\n| [Skill Enrichment API](https://docs.peopledatalabs.com/docs/skill-enrichment-api) | `PDLPY.skill(**params)` |\n\n## 📘 Documentation <a name="documentation"></a>\n\nAll of our API endpoints are documented at: https://docs.peopledatalabs.com/\n\nThese docs describe the supported input parameters, output responses and also provide additional technical context.\n\nAs illustrated in the [Endpoints](#endpoints) section above, each of our API endpoints is mapped to a specific method in the PDLPY class.  For each of these class methods, **all function inputs are mapped as input parameters to the respective API endpoint**, meaning that you can use the API documentation linked above to determine the input parameters for each endpoint.\n\nAs an example:\n\nThe following is **valid** because `name` is a [supported input parameter to the Person Identify API](https://docs.peopledatalabs.com/docs/identify-api-reference#input-parameters):\n\n```python\nPDLPY().person.identify({"name": "sean thorne"})\n```\n\nConversely, this would be **invalid** because `fake_parameter` is not an input parameter to the Person Identify API:\n\n```python\nPDLPY().person.identify({"fake_parameter": "anything"})\n```\n',
    'author': 'People Data Labs',
    'author_email': 'hello@peopledatalabs.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://www.peopledatalabs.com',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
