from typing import Dict, List

# https://commons.wikimedia.org/wiki/Commons:File_types
IMAGE_EXTENSIONS: List[str] = [
    ".jpg",
    ".png",
    ".svg",
    ".gif",
    ".jpeg",
    ".tif",
    ".bmp",
    ".webp",
    ".xcf",
]
VIDEO_EXTENSIONS: List[str] = [".ogv", ".webm", ".mpg", ".mpeg"]
AUDIO_EXTENSIONS: List[str] = [
    ".ogg",
    ".mp3",
    ".mid",
    ".webm",
    ".flac",
    ".wav",
    ".oga",
]
MEDIA_EXTENSIONS: List[str] = list(
    set(IMAGE_EXTENSIONS + VIDEO_EXTENSIONS + AUDIO_EXTENSIONS)
)

# Last updated: December 16 2022
DEF_OPTION_TAGS: Dict[str, List[str]] = {
    "keywords": [
        "middle",
        "right",
        "super",
        "border",
        "text-bottom",
        "frame",
        "enframed",
        "left",
        "center",
        "thumb",
        "thumbnail",
        "frameless",
        "upright",
        "sup",
        "text-top",
        "none",
        "framed",
        "bottom",
        "top",
        "baseline",
        "centre",
        "sub",
    ],
    "params": [
        "lang",
        "class",
        "link",
        "alt",
        "thumbnail",
        "lossy",
        "upright",
        "page",
        "thumb",
    ],
    "startswith": ["page ", "upright "],
    "endswith": ["px"],
}

IMG_OPTION_ALIASES: Dict[str, Dict[str, List[str]]] = {
    "ab": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ace": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "af": {
        "keywords": [
            "omraam",
            "teks-onder",
            "bo",
            "middel",
            "teks-bo",
            "raam",
            "regs",
            "raamloos",
            "senter",
            "onder",
            "geen",
            "duimnael",
            "links",
        ],
        "params": ["skakel"],
    },
    "als": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "alt": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ami": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "an": {
        "keywords": [
            "centro",
            "zurda",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "dcha",
            "sinenmarcar",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "dreita",
            "enmarcada",
            "cucha",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "ar": {
        "keywords": [
            "يمين",
            "تصغير",
            "إطار",
            "بدون",
            "أعلى",
            "نص_أسفل",
            "سب",
            "حدود",
            "لاإطار",
            "بإطار",
            "خط_أساسي",
            "معدول",
            "أسفل",
            "سوبر",
            "وسط",
            "بلا",
            "مركز",
            "فرعي",
            "نص_أعلى",
            "يسار",
        ],
        "params": [
            "رتبة",
            "وصلة",
            "تصغير",
            "بديل",
            "صفحة",
            "معدول",
            "رابط",
            "مصغر",
            "لغة",
        ],
        "startswith": ["صفحة_", "معدول_"],
        "endswith": ["عن", "بك"],
    },
    "arc": {
        "keywords": [
            "ܙܥܘܪܬܐ",
            "ܬܚܘܡܐ",
            "ܡܨܥܐ",
            "ܝܡܝܢܐ",
            "ܦܪܥܝܐ",
            "ܣܡܠܐ",
            "ܣܪܛܐ_ܫܪܫܝܐ",
            "ܠܐ_ܡܕܡ",
        ],
        "params": ["ܙܥܘܪܬܐ", "ܦܐܬܐ"],
        "startswith": ["ܦܐܬܐ "],
    },
    "ary": {
        "keywords": [
            "يمين",
            "تصغير",
            "إطار",
            "بدون",
            "أعلى",
            "نص_أسفل",
            "سب",
            "حدود",
            "لاإطار",
            "بإطار",
            "خط_أساسي",
            "معدول",
            "أسفل",
            "سوبر",
            "وسط",
            "بلا",
            "مركز",
            "فرعي",
            "نص_أعلى",
            "يسار",
        ],
        "params": [
            "رتبة",
            "وصلة",
            "تصغير",
            "بديل",
            "صفحة",
            "معدول",
            "رابط",
            "مصغر",
            "لغة",
        ],
        "startswith": ["صفحة_", "معدول_"],
        "endswith": ["عن", "بك"],
    },
    "arz": {
        "keywords": [
            "يمين",
            "تصغير",
            "إطار",
            "بدون",
            "أعلى",
            "مصغر",
            "نص_أسفل",
            "سب",
            "حدود",
            "لاإطار",
            "بإطار",
            "خط_أساسي",
            "معدول",
            "حد",
            "خط_اساسى",
            "أسفل",
            "سوبر",
            "وسط",
            "من_غير_اطار",
            "بلا",
            "فرعى",
            "مركز",
            "فرعي",
            "نص_أعلى",
            "يسار",
        ],
        "params": [
            "رتبة",
            "وصلة",
            "تصغير",
            "صفح",
            "رابط",
            "مصغر",
            "بديل",
            "صفحة",
            "معدول",
            "لغة",
            "لغه",
            "رتبه",
        ],
        "startswith": ["صفحة_", "صفحه_", "معدول_"],
        "endswith": ["عن", "بك"],
    },
    "ast": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "atj": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "av": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "avk": {
        "keywords": [
            "neant",
            "сверху",
            "сверхусправа",
            "mini",
            "bas-txt",
            "текст-сверху",
            "encadre",
            "no",
            "мини",
            "центр",
            "ninguno",
            "haut",
            "néant",
            "izquierda",
            "ind",
            "миниатюра",
            "основание",
            "exposant",
            "под",
            "enmarcada",
            "centré",
            "безрамки",
            "derecha",
            "centrar",
            "encadré",
            "haut-txt",
            "sinmarco",
            "miniaturadeimagen",
            "indice",
            "снизу",
            "haut-texte",
            "ligne-de-base",
            "над",
            "exp",
            "borde",
            "centrado",
            "bas-texte",
            "medio",
            "droite",
            "non_encadré",
            "centrada",
            "izq",
            "enmarcado",
            "слева",
            "vignette",
            "non_encadre",
            "ninguna",
            "abajo",
            "dcha",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "base",
            "bas",
            "centro",
            "без",
            "nada",
            "gauche",
            "sans_cadre",
            "arriba",
            "redresse",
            "обрамить",
            "milieu",
            "bordure",
            "der",
            "справа",
            "посередине",
            "sinenmarcar",
            "текст-снизу",
            "izda",
            "граница",
            "cadre",
        ],
        "params": [
            "página",
            "сверхусправа",
            "ссылка",
            "redresse",
            "мини",
            "classe",
            "альт",
            "vignette",
            "vínculo",
            "idioma",
            "miniaturadeimagen",
            "lien",
            "миниатюра",
            "страница",
            "pagina",
            "vinculo",
            "enlace",
            "langue",
            "miniatura",
        ],
        "startswith": ["redresse_", "сверхусправа ", "página_", "pagina_", "страница "],
        "endswith": ["пкс"],
    },
    "awa": {
        "keywords": [
            "किनारा",
            "फ्रेमहीन",
            "केन्द्र",
            "बाएं",
            "बाएँ",
            "दाएँ",
            "तल",
            "दायें",
            "बॉर्डर",
            "फ़्रेमहीन",
            "मध्य",
            "आधार_रेखा",
            "फ्रेम",
            "कोई_नहीं",
            "केंद्र",
            "दाएं",
            "मूर्ध",
            "खड़ी",
            "शीर्ष",
            "पाठ-शीर्ष",
            "फ़्रेम",
            "केंद्रित",
            "केन्द्रित",
            "अंगूठाकार",
            "पद",
            "अंगूठा",
            "पाठ-तल",
            "बायें",
        ],
        "params": [
            "अंगूठाकार",
            "अंगूठा",
            "भाषा",
            "कड़ी",
            "खड़ी",
            "पाठ",
            "वर्ग",
            "पृष्ठ",
        ],
        "startswith": ["खड़ी_", "पृष्ठ_"],
        "endswith": ["पिक्सेल"],
    },
    "ay": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "azb": {
        "keywords": [
            "بی_قاب",
            "بندانگشتی",
            "همکف",
            "بالا",
            "متن-پایین",
            "قاب",
            "حاشیه",
            "چپ",
            "قابیق",
            "انگشتی",
            "سول",
            "زبر",
            "ایستاده",
            "متن-بالا",
            "میانه",
            "انگشتدان",
            "وسط",
            "هئچ",
            "بی\u200cقاب",
            "بیقاب",
            "ساغ",
            "زیر",
            "راست",
            "پایین",
            "هیچ",
        ],
        "params": [
            "کلاس",
            "پیوند",
            "بندانگشتی",
            "انگشتی",
            "صفحه",
            "زبان",
            "جایگزین",
            "انگشتدان",
            "ایستاده",
        ],
        "startswith": ["صفحه_", "ایستاده_"],
        "endswith": ["پیکسل"],
    },
    "ba": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ban": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "bar": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "bat-smg": {
        "keywords": ["kairėje", "dešinėje", "mini", "miniatiūra"],
        "params": ["mini", "miniatiūra"],
    },
    "bcl": {
        "keywords": [
            "wala",
            "tangâ",
            "too",
            "kwadro",
            "daing kwadro",
            "mayò",
            "sentro",
            "nakakawadro",
        ],
        "params": ["pahina"],
        "startswith": ["pahina "],
    },
    "be": {
        "keywords": [
            "міні",
            "мініяцюра",
            "справа",
            "няма",
            "безрамкі",
            "цэнтр",
            "злева",
        ],
        "params": ["міні", "мініяцюра"],
        "endswith": ["пкс"],
    },
    "be-x-old": {
        "keywords": [
            "рамка",
            "зьлева",
            "цэнтар",
            "значак",
            "пасярэдзіне",
            "міні",
            "мініяцюра",
            "справа",
            "няма",
            "зьверху",
            "безрамкі",
            "зьнізу",
            "цэнтр",
            "злева",
        ],
        "params": ["міні", "старонка", "мініяцюра", "значак", "спасылка"],
        "startswith": ["старонка "],
        "endswith": ["пкс"],
    },
    "bg": {
        "keywords": [
            "рамка",
            "д",
            "контур",
            "врамка",
            "вляво",
            "вдясно",
            "център",
            "безрамка",
            "ляво",
            "мини",
            "л",
            "центр",
            "ц",
            "дясно",
            "ръб",
            "н",
        ],
        "params": ["мини"],
        "endswith": ["пкс", "п"],
    },
    "bjn": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "bm": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "bn": {
        "keywords": [
            "থাম্ব",
            "নিম্নে",
            "লেখা-নীচ",
            "উপর",
            "পাঠ্য-উপরে",
            "মধ্য",
            "সীমান্ত",
            "পাঠ্য-উপর",
            "ডান",
            "কোনটিনা",
            "শীর্ষ",
            "পাঠ্য-নীচে",
            "ডানদিকে_উপরে",
            "তলদেশ",
            "নিচে",
            "ফ্রেম",
            "শীর্ষে",
            "লেখা-উপর",
            "বামে",
            "ফ্রেমবিহীন",
            "ফ্রেমসহ",
            "ডানে",
            "মধ্যে",
            "নীচে",
            "ফ্রেমহীন",
            "সীমানা",
            "নীচ",
            "কেন্দ্র",
            "থাম্বনেইল",
            "কিছুইনা",
            "উপরে",
            "কিছুই_না",
            "লেখা-উপরে",
            "ডানে_উপরে",
            "কেন্দ্রে",
            "বাম",
            "নিচ",
            "কোনটি_না",
        ],
        "params": [
            "থাম্ব",
            "সংযোগ",
            "পাতা",
            "ক্লাস",
            "পৃষ্ঠা",
            "ডানদিকে_উপরে",
            "ভাষা",
            "থাম্বনেইল",
            "ডানেউপরে",
            "লিঙ্ক",
        ],
        "startswith": ["পাতা ", "পৃষ্ঠা ", "ডানদিকে_উপরে ", "ডানেউপরে "],
        "endswith": ["পিক্সেল", "পিক্স"],
    },
    "bpy": {
        "keywords": [
            "থাম্ব",
            "নিম্নে",
            "লেখা-নীচ",
            "উপর",
            "পাঠ্য-উপরে",
            "মধ্য",
            "সীমান্ত",
            "পাঠ্য-উপর",
            "ডান",
            "কোনটিনা",
            "শীর্ষ",
            "পাঠ্য-নীচে",
            "ডানদিকে_উপরে",
            "তলদেশ",
            "নিচে",
            "ফ্রেম",
            "শীর্ষে",
            "লেখা-উপর",
            "বামে",
            "ফ্রেমবিহীন",
            "ফ্রেমসহ",
            "ডানে",
            "মধ্যে",
            "নীচে",
            "ফ্রেমহীন",
            "সীমানা",
            "নীচ",
            "কেন্দ্র",
            "থাম্বনেইল",
            "কিছুইনা",
            "উপরে",
            "কিছুই_না",
            "লেখা-উপরে",
            "ডানে_উপরে",
            "কেন্দ্রে",
            "বাম",
            "নিচ",
            "কোনটি_না",
        ],
        "params": [
            "থাম্ব",
            "সংযোগ",
            "পাতা",
            "ক্লাস",
            "পৃষ্ঠা",
            "ডানদিকে_উপরে",
            "ভাষা",
            "থাম্বনেইল",
            "ডানেউপরে",
            "লিঙ্ক",
        ],
        "startswith": ["পাতা ", "পৃষ্ঠা ", "ডানদিকে_উপরে ", "ডানেউপরে "],
        "endswith": ["পিক্সেল", "পিক্স"],
    },
    "br": {
        "keywords": [
            "kreiz",
            "kreizenn",
            "traoñ",
            "kleiz",
            "krec'h",
            "is",
            "dehou",
            "netra",
        ],
        "params": ["pajenn", "liamm"],
        "startswith": ["pajenn "],
    },
    "bs": {
        "keywords": [
            "lijevo",
            "vrh",
            "sredina",
            "mini",
            "odjeljak",
            "ivica",
            "centar",
            "granica",
            "okvir",
            "pocetna_linija",
            "na_gore",
            "početna_linija",
            "d",
            "dugme",
            "c",
            "desno",
            "ram",
            "bez_okvira",
            "vrh_teksta",
            "bez",
            "tekst-dugme",
            "l",
            "n",
        ],
        "params": ["mini", "na_gore", "stranica"],
        "startswith": ["na_gore_", "stranica "],
        "endswith": ["p", "piksel"],
    },
    "bug": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "bxr": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ca": {
        "keywords": [
            "neant",
            "baix-text",
            "quadre",
            "esquèrra",
            "sense marc",
            "bas",
            "exp",
            "senèstra",
            "centrat",
            "linha_de_basa",
            "bas-tèxte",
            "marc",
            "bordadura",
            "mitan",
            "bas-txt",
            "dalt-text",
            "redreçat",
            "baix",
            "vinheta",
            "nonrés",
            "gaucha",
            "naut-txt",
            "mitjà",
            "naut",
            "línia de base",
            "dreta",
            "ind",
            "indici",
            "redreça",
            "sens_quadre",
            "enquagrat",
            "cap",
            "drecha",
            "dalt",
            "vora",
            "naut-tèxte",
            "esquerra",
            "miniatura",
        ],
        "params": [
            "llengua",
            "ligam",
            "idioma",
            "pàgina",
            "redreçat",
            "vinheta",
            "miniatura",
            "enllaç",
        ],
        "startswith": ["pàgina ", "redreça ", "redreça", "redreçat "],
    },
    "cbk-zam": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "cdo": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "ce": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "доза",
            "бух",
            "лакхахь",
            "гурабе",
            "бухара-йоза",
            "без",
            "юккъе",
            "гурабоцаш",
            "харцхьа",
            "тӀе",
            "текст-сверху",
            "жима",
            "бакъхьалакхо",
            "юккъ",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "лакххьара-йоза",
            "йоцуш",
            "справа",
            "буха",
            "посередине",
            "бакъхьа",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "бухар",
            "граница",
            "жимо",
            "под",
            "над",
        ],
        "params": [
            "агlо",
            "сверхусправа",
            "ссылка",
            "жима",
            "бакъхьалакхо",
            "мини",
            "альт",
            "хьажорг",
            "миниатюра",
            "страница",
            "жимо",
        ],
        "startswith": [
            "сверхусправа_",
            "сверхусправа ",
            "агlо_",
            "страница_",
            "бакъхьалакхо_",
            "upright_",
            "page_",
            "страница ",
        ],
        "endswith": ["пкс"],
    },
    "ckb": {
        "keywords": [
            "سنوور",
            "ڕاست",
            "چەپ",
            "چوارچێوە",
            "وێنۆک",
            "ناوەڕاست",
            "بێچوارچێوە",
        ],
        "endswith": ["پیکسڵ"],
    },
    "co": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "cs": {
        "keywords": [
            "stred",
            "náhled",
            "žádné",
            "střed",
            "žiadny",
            "vpravo",
            "náhľad",
            "bezrámu",
            "náhľadobrázka",
            "vľavo",
            "rám",
            "okraj",
            "vlevo",
        ],
        "params": ["náhled", "odkaz", "strana", "jazyk", "třída"],
        "startswith": ["strana_"],
        "endswith": ["bod", "pixelů"],
    },
    "csb": {
        "keywords": [
            "centruj",
            "mały",
            "bezramki",
            "bez_ramki",
            "środek",
            "lewo",
            "góra",
            "ramka",
            "prawo",
            "brak",
            "tło",
            "dół",
        ],
        "params": ["mały", "strona"],
    },
    "cv": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "cy": {
        "keywords": [
            "canol",
            "gwaelod",
            "uwch",
            "bawd",
            "brig",
            "is",
            "ewin_bawd",
            "dim",
            "chwith",
            "mân-lun",
            "unionsyth",
            "godre",
            "de",
        ],
        "params": ["mân-lun", "unionsyth", "tudalen", "bawd"],
        "startswith": ["unionsyth_", "tudalen_"],
    },
    "de": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "diq": {
        "keywords": [
            "miyan",
            "resmoqıckek",
            "Sinoréerdi",
            "merkez",
            "qıckek",
            "çerçewe",
            "çıniyo",
            "corén",
            "erd",
            "gedug-metin",
            "çep",
            "gedug",
            "sinor",
            "çerçeweyın",
            "anvar",
            "disleg",
            "werte",
            "cor",
            "erd-metin",
            "raşt",
            "béçerçewe",
            "çerçewekerden",
        ],
        "params": [
            "pera",
            "zuwan",
            "gre",
            "resmoqıckek",
            "qıckek",
            "klik",
            "disleg",
            "sınıf",
        ],
        "startswith": ["disleg_", "pera_"],
        "endswith": ["pik", "piksel"],
    },
    "dsb": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "el": {
        "keywords": [
            "άνω",
            "αριστερά",
            "καθόλου",
            "μέσο",
            "μικρογραφία",
            "κάτω",
            "εκθέτης",
            "μινιατούρα",
            "πλαίσιο",
            "χωρίς-πλαίσιο",
            "δείκτης",
            "με-πλαίσιο",
            "κάτω-από-το-κείμενο",
            "δεξιά",
            "κατακόρυφα",
            "πάνω-από-το-κείμενο",
            "κέντρο",
            "γραμμήβάσης",
        ],
        "params": [
            "κατακόρυφα",
            "σελίδα",
            "σύνδεσμος",
            "μινιατούρα",
            "εναλλ.",
            "μικρογραφία",
        ],
        "startswith": ["σελίδα_", "κατακόρυφα_"],
        "endswith": ["εσ"],
    },
    "eml": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "eo": {
        "keywords": [
            "kadro",
            "meza",
            "dekstra",
            "alte",
            "nenio",
            "tekst-alte",
            "centra",
            "supre",
            "malalte",
            "kadrita",
            "meze",
            "dekstre",
            "maldekstra",
            "altdekstre",
            "kadrite",
            "senkadra",
            "sube",
            "eta",
            "malsupre",
            "kadrigita",
            "suba-teksto",
            "kadrigite",
            "enkadrita",
            "malsube",
            "kadra",
            "enkadrite",
            "maldekstre",
            "neniu",
        ],
        "params": [
            "eta",
            "ligilo",
            "klaso",
            "altdekstre",
            "alternative",
            "pagxo",
            "paĝo",
        ],
        "startswith": ["pagxo_", "paĝo ", "altdekstre_"],
        "endswith": ["ra"],
    },
    "es": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "et": {
        "keywords": [
            "keskel",
            "ääris",
            "pisipilt",
            "vasakul",
            "tühi",
            "pisi",
            "raam",
            "püsti",
            "raamita",
            "paremal",
        ],
        "params": ["lehekülg", "pisi", "püsti", "pisipilt", "keel"],
        "startswith": ["lehekülg_"],
    },
    "eu": {"keywords": ["eskuinera", "erdian", "ezkerrera"]},
    "ext": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "fa": {
        "keywords": [
            "بی_قاب",
            "بندانگشتی",
            "همکف",
            "بالا",
            "متن-پایین",
            "قاب",
            "چپ",
            "حاشیه",
            "انگشتی",
            "زبر",
            "ایستاده",
            "متن-بالا",
            "میانه",
            "انگشتدان",
            "وسط",
            "بی\u200cقاب",
            "بیقاب",
            "زیر",
            "راست",
            "پایین",
            "هیچ",
        ],
        "params": [
            "کلاس",
            "پیوند",
            "بندانگشتی",
            "انگشتی",
            "صفحه",
            "زبان",
            "جایگزین",
            "انگشتدان",
            "ایستاده",
        ],
        "startswith": ["صفحه_", "ایستاده_"],
        "endswith": ["پیکسل"],
    },
    "fi": {
        "keywords": [
            "pienois",
            "alaindeksi",
            "alas",
            "reunus",
            "ylös",
            "vasen",
            "alhaalla",
            "oikea",
            "pysty",
            "keskellä",
            "kehystetty",
            "keski",
            "kehys",
            "yläindeksi",
            "pienoiskuva",
            "perustaso",
            "ylhäällä",
            "kehyksetön",
            "yläoikea",
            "keskitetty",
            "tyhjä",
        ],
        "params": ["pienois", "sivu", "yläoikea", "pysty", "linkki", "pienoiskuva"],
        "startswith": ["yläoikea_", "pysty_", "sivu_"],
    },
    "fiu-vro": {
        "keywords": [
            "keskel",
            "ääris",
            "pisipilt",
            "vasakul",
            "tühi",
            "pisi",
            "raam",
            "püsti",
            "raamita",
            "paremal",
        ],
        "params": ["lehekülg", "pisi", "püsti", "pisipilt", "keel"],
        "startswith": ["lehekülg_"],
    },
    "fr": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "frp": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "èxp",
            "bordura",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "drêt",
            "haut-texte",
            "figura",
            "sans_cadre",
            "bas-txt",
            "d'avâl",
            "tèxto-d'avâl",
            "centrâ",
            "encadré",
            "encadre",
            "pas_encâdrâ",
            "drêta",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "redresse",
            "segno",
            "ligne_de_base",
            "ind",
            "sen_câdro",
            "indice",
            "d'amont",
            "tèxto-d'amont",
            "câdro",
            "entre-mié",
            "vouedo",
            "èxposent",
            "encâdrâ",
            "exposant",
            "ligne-de-base",
            "legne_de_bâsa",
            "gôche",
            "base",
            "cadre",
            "centré",
        ],
        "params": [
            "vignette",
            "lim",
            "drêt",
            "figura",
            "pâge",
            "redresse",
            "lien",
            "classe",
            "langue",
        ],
        "startswith": ["redresse_", "drêt ", "pâge ", "redresse "],
    },
    "frr": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "fur": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "ga": {
        "keywords": [
            "mion",
            "frámaithe",
            "clé",
            "fráma",
            "lár",
            "faic",
            "mionsamhail",
            "deas",
        ]
    },
    "gag": {
        "keywords": [
            "sınır",
            "çerçevesiz",
            "tavan",
            "orta",
            "tabançizgisi",
            "merkez",
            "metin-tavan",
            "dikey",
            "sol",
            "üst",
            "metin-taban",
            "sağ",
            "çerçeveli",
            "tepe",
            "altçizgi",
            "üs",
            "çerçeve",
            "alt",
            "taban",
            "metin-tepe",
            "yok",
            "küçükresim",
            "küçük",
        ],
        "params": ["bağlantı", "dikey", "sayfa", "küçük", "küçükresim", "sınıf"],
        "startswith": ["dikey ", "sayfa "],
        "endswith": ["pik", "piksel"],
    },
    "gan": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "gcr": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "gl": {
        "keywords": [
            "direita",
            "conbordo",
            "miniatura",
            "senmarco",
            "centro",
            "texto-abaixo",
            "comborda",
            "miniaturadaimagem",
            "esquerda",
            "arribaádereita",
            "borda",
            "medio",
            "semborda",
            "liñadebase",
            "arriba",
            "miniaturadaimaxe",
            "dereita",
            "linhadebase",
            "bordo",
            "semmoldura",
            "nenhum",
            "superiordireito",
            "acima",
            "commoldura",
            "senbordo",
            "abaixo",
            "conmarco",
            "marco",
            "meio",
            "ningún",
            "texto-arriba",
        ],
        "params": [
            "página",
            "miniaturadaimagem",
            "arribaádereita",
            "ligação",
            "miniaturadaimaxe",
            "ligazón",
            "clase",
            "superiordireito",
            "páxina",
            "miniatura",
        ],
        "startswith": [
            "superiordireito_",
            "arribaádereita_",
            "páxina_",
            "página_",
            "página ",
            "superiordireito ",
        ],
    },
    "glk": {
        "keywords": [
            "بی_قاب",
            "بندانگشتی",
            "همکف",
            "بالا",
            "متن-پایین",
            "قاب",
            "چپ",
            "حاشیه",
            "انگشتی",
            "زبر",
            "ایستاده",
            "متن-بالا",
            "میانه",
            "انگشتدان",
            "وسط",
            "بی\u200cقاب",
            "بیقاب",
            "زیر",
            "راست",
            "پایین",
            "هیچ",
        ],
        "params": [
            "کلاس",
            "پیوند",
            "بندانگشتی",
            "انگشتی",
            "صفحه",
            "زبان",
            "جایگزین",
            "انگشتدان",
            "ایستاده",
        ],
        "startswith": ["صفحه_", "ایستاده_"],
        "endswith": ["پیکسل"],
    },
    "gn": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "gor": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "hak": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "haw": {
        "keywords": ["waena", "ākau", "hema", "ʻākau", "akau", "aohe", "ʻaʻohe"],
        "params": ["loulou"],
    },
    "he": {
        "keywords": [
            "ממוסגר",
            "גבולות",
            "למטה",
            "ללא מסגרת",
            "גבול",
            "ללא",
            "מסגרת",
            "ממוזער",
            "באמצע",
            "עילי",
            "בתחתית הטקסט",
            "לא ממוסגר",
            "ימין למעלה",
            "ימין",
            "בראש הטקסט",
            "מרכז",
            "שורת הבסיס",
            "שמאל",
            "למעלה",
            "תחתי",
        ],
        "params": ["ממוזער", "טקסט", "דף", "ימין למעלה", "קישור"],
        "startswith": ["דף ", "ימין למעלה "],
        "endswith": [" פיקסלים"],
    },
    "hi": {
        "keywords": [
            "किनारा",
            "फ्रेमहीन",
            "केन्द्र",
            "बाएं",
            "बाएँ",
            "दाएँ",
            "तल",
            "दायें",
            "बॉर्डर",
            "फ़्रेमहीन",
            "मध्य",
            "आधार_रेखा",
            "फ्रेम",
            "कोई_नहीं",
            "केंद्र",
            "दाएं",
            "मूर्ध",
            "खड़ी",
            "शीर्ष",
            "पाठ-शीर्ष",
            "फ़्रेम",
            "केंद्रित",
            "केन्द्रित",
            "अंगूठाकार",
            "पद",
            "अंगूठा",
            "पाठ-तल",
            "बायें",
        ],
        "params": [
            "अंगूठाकार",
            "अंगूठा",
            "भाषा",
            "कड़ी",
            "खड़ी",
            "पाठ",
            "वर्ग",
            "पृष्ठ",
        ],
        "startswith": ["खड़ी_", "पृष्ठ_"],
        "endswith": ["पिक्सेल"],
    },
    "hr": {
        "keywords": [
            "lijevo",
            "vrh",
            "ništa",
            "natpis",
            "dno",
            "mini",
            "osnovnacrta",
            "bezokvira",
            "okvir",
            "pola",
            "tekst-vrh",
            "potpis",
            "minijatura",
            "desno",
            "središte",
            "eks",
            "ind",
            "tekst-dno",
            "obrub",
        ],
        "params": ["jezik", "uspravno", "minijatura", "poveznica", "stranica"],
        "startswith": ["uspravno ", "stranica "],
    },
    "hsb": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "ht": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "hu": {
        "keywords": [
            "keretes",
            "lent",
            "bal",
            "középre",
            "szöveg-lenn",
            "lenn",
            "vközépre",
            "alsóindex",
            "szöveg-fenn",
            "miniatűr",
            "balra",
            "jobbra",
            "semmi",
            "keretben",
            "keretezett",
            "közép",
            "vközépen",
            "kerettel",
            "alapvonal",
            "keretnélküli",
            "jobb",
            "ai",
            "fennjobbra",
            "bélyegkép",
            "fi",
            "fent",
            "felsőindex",
            "fenn",
            "szöveg-fent",
            "keret",
            "bélyeg",
            "szöveg-lent",
        ],
        "params": ["bélyeg", "fennjobbra", "miniatűr", "bélyegkép", "oldal"],
        "startswith": ["oldal ", "fennjobbra "],
    },
    "hy": {
        "keywords": ["ձախից", "շրջափակել", "մինի", "աջից", "առանց", "կենտրոն"],
        "params": ["մինի", "էջը"],
        "startswith": ["էջ "],
        "endswith": ["փքս"],
    },
    "hyw": {
        "keywords": ["ձախից", "շրջափակել", "մինի", "աջից", "առանց", "կենտրոն"],
        "params": ["մինի", "էջը"],
        "startswith": ["էջ "],
        "endswith": ["փքս"],
    },
    "id": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "ig": {"keywords": ["élú", "áká_èkpè", "okpúrù-ede", "áká_ịkẹngạ", "ẹtítì"]},
    "inh": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "io": {
        "keywords": [
            "kadro",
            "meza",
            "dekstra",
            "alte",
            "nenio",
            "tekst-alte",
            "centra",
            "supre",
            "malalte",
            "kadrita",
            "meze",
            "dekstre",
            "maldekstra",
            "altdekstre",
            "kadrite",
            "senkadra",
            "sube",
            "eta",
            "malsupre",
            "kadrigita",
            "suba-teksto",
            "kadrigite",
            "enkadrita",
            "malsube",
            "kadra",
            "enkadrite",
            "maldekstre",
            "neniu",
        ],
        "params": [
            "eta",
            "ligilo",
            "klaso",
            "altdekstre",
            "alternative",
            "pagxo",
            "paĝo",
        ],
        "startswith": ["pagxo_", "paĝo ", "altdekstre_"],
        "endswith": ["ra"],
    },
    "is": {
        "keywords": [
            "neðst",
            "texti-neðst",
            "miðja",
            "yfir",
            "hægri",
            "undir",
            "þumall",
            "engin",
            "vinstri",
            "efst",
        ],
        "endswith": ["dp"],
    },
    "it": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "ja": {
        "keywords": [
            "フレーム",
            "フレームなし",
            "ボーダー",
            "下線",
            "文上端",
            "サムネイル",
            "下付き",
            "中心",
            "中央",
            "なし",
            "境界",
            "文下端",
            "ベースライン",
            "右",
            "右上",
            "上付き",
            "左",
            "下端",
            "無し",
            "上端",
        ],
        "params": ["ページ", "リンク", "代替文", "サムネイル", "代替画像"],
        "startswith": ["ページ "],
        "endswith": ["ピクセル"],
    },
    "jv": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "ka": {
        "keywords": [
            "მარჯვნივ",
            "ცენტრი",
            "არა",
            "მარცხნივ",
            "საზღვარი",
            "მინი",
            "ქვედა",
            "ცეროდენა",
            "მინიატიურა",
            "მინიასლი",
            "ზედა",
            "შუა",
            "ცენტრში",
        ],
        "params": ["გვერდი", "მინიატიურა", "ალტ", "ბმული", "მინი"],
        "startswith": ["გვერდი_"],
        "endswith": ["პქ"],
    },
    "kaa": {
        "keywords": [
            "ортаға",
            "oñ",
            "sürmeli",
            "üstine",
            "orta",
            "оңға",
            "ortağa",
            "аст",
            "орта",
            "жоқ",
            "sürmesiz",
            "üstiligi",
            "mätin-üstinde",
            "сүрмесіз",
            "sol",
            "аралығына",
            "joq",
            "оң",
            "üst",
            "tikti",
            "үстіне",
            "мәтін-астында",
            "nobaý",
            "тікті",
            "mätin-astında",
            "үстілігі",
            "ast",
            "сол",
            "астылығы",
            "aralığına",
            "astına",
            "oñğa",
            "тірекжол",
            "нобай",
            "астына",
            "мәтін-үстінде",
            "жиекті",
            "tirekjol",
            "солға",
            "solğa",
            "сүрмелі",
            "eşqandaý",
            "ешқандай",
            "үст",
            "jïekti",
            "astılığı",
        ],
        "params": ["бет", "tiktik", "nobaý", "нобай", "bet", "тіктік"],
        "startswith": ["bet ", "тіктік ", "бет ", "tiktik "],
        "endswith": [" nükte", " нүкте"],
    },
    "kab": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "kbp": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "kk": {
        "keywords": [
            "ортаға",
            "оңға",
            "аст",
            "орта",
            "жоқ",
            "сүрмесіз",
            "аралығына",
            "оң",
            "мәтін-астында",
            "тікті",
            "үстілігі",
            "сол",
            "астылығы",
            "тірекжол",
            "нобай",
            "астына",
            "мәтін-үстінде",
            "жиекті",
            "солға",
            "ешқандай",
            "сүрмелі",
            "үст",
            "үстіне",
        ],
        "params": ["бет", "нобай", "тіктік"],
        "startswith": ["тіктік ", "бет "],
        "endswith": [" нүкте"],
    },
    "km": {
        "keywords": [
            "ខាងធ្វេង",
            "ធ្វេង",
            "ស៊ុម",
            "ផ្នែកខាងលើ",
            "កណ្តាល",
            "ឃ្លានៅផ្នែកខាងលើ",
            "ស្តាំ",
            "ផ្នែកបាត",
            "ឃ្លានៅផ្នែកបាត",
            "ផ្នែកលើ",
            "ទទេ",
            "ឃ្លាផ្នែកបាត",
            "ខាងស្តាំ",
            "រូបតូច",
            "ផ្នែកកណ្តាល",
            "ឃ្លាផ្នែកខាងលើ",
            "គ្មានស៊ុម",
            "បាត",
            "រូបភាពតូច",
            "គ្មាន",
        ],
        "params": ["តំណភ្ជាប់", "រូបតូច", "តំនភ្ជាប់", "រូបភាពតូច", "ទំព័រ"],
        "startswith": ["ទំព័រ"],
        "endswith": ["ភស", "ភីកសែល"],
    },
    "ko": {
        "keywords": [
            "아래첨자",
            "가운데",
            "텍스트위",
            "축소판",
            "테두리",
            "썸네일",
            "프레임",
            "섬네일",
            "프레임없음",
            "글자위",
            "위첨자",
            "아래",
            "중간",
            "왼쪽",
            "텍스트아래",
            "오른쪽",
            "위",
            "위오른쪽",
            "없음",
            "밑줄",
            "글자아래",
        ],
        "params": ["위오른쪽", "축소판", "문서", "섬네일", "썸네일", "언어", "클래스", "링크", "대체글"],
        "endswith": ["픽셀"],
    },
    "koi": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "krc": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ksh": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "lengks",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "lenks",
            "miniatur",
            "grundlinie",
            "rähß",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
            "räts",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "ku": {"keywords": ["rast", "çep"], "params": ["girêdan"]},
    "kv": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "kw": {
        "keywords": [
            "gwartha",
            "dyhow",
            "nagonan",
            "goles",
            "kresel",
            "fremys",
            "hebfram",
            "skeusennik",
            "tekst-gwartha",
            "tekst-goles",
            "kledh",
            "kres",
        ],
        "params": ["skeusennik", "kevren", "folen"],
        "startswith": ["folen_"],
    },
    "lad": {
        "keywords": [
            "cierda",
            "centro",
            "borde",
            "enmarcada",
            "nada",
            "centrado",
            "mini",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "dinguna",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "dinguno",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "lb": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "ouni",
            "Miniatur",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "uewen",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "lénks",
            "gerummt",
            "miniatur",
            "riets",
            "grundlinie",
            "ënnen",
            "zentréiert",
            "text-oben",
            "mëtt",
            "net_gerummt",
            "hochkant",
            "links",
            "hochgestellt",
            "bord",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "Säit",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_", "Säit_"],
    },
    "lbe": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "lez": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "li": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "midden",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "geen",
            "omkaderd",
            "gecentreerd",
            "links",
            "beneden",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "pagina",
            "taal",
            "klasse",
            "rechtop",
            "verwijzing",
            "koppeling",
        ],
        "startswith": ["pagina_", "rechtop"],
    },
    "lij": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "lld": {
        "keywords": [
            "sinistra",
            "centro",
            "sidretg",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "sidretg", "miniatura"],
        "startswith": ["pagina_", "sidretg_", "verticale_"],
    },
    "lmo": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "drita",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "nissön",
            "manzína",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "ln": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "lt": {
        "keywords": ["kairėje", "dešinėje", "mini", "miniatiūra"],
        "params": ["mini", "miniatiūra"],
    },
    "mad": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "mai": {
        "keywords": [
            "किनारा",
            "फ्रेमहीन",
            "केन्द्र",
            "बाएं",
            "बाएँ",
            "दाएँ",
            "तल",
            "दायें",
            "बॉर्डर",
            "फ़्रेमहीन",
            "मध्य",
            "आधार_रेखा",
            "फ्रेम",
            "कोई_नहीं",
            "केंद्र",
            "दाएं",
            "मूर्ध",
            "खड़ी",
            "शीर्ष",
            "पाठ-शीर्ष",
            "फ़्रेम",
            "केंद्रित",
            "केन्द्रित",
            "अंगूठाकार",
            "पद",
            "अंगूठा",
            "पाठ-तल",
            "बायें",
        ],
        "params": [
            "अंगूठाकार",
            "अंगूठा",
            "भाषा",
            "कड़ी",
            "खड़ी",
            "पाठ",
            "वर्ग",
            "पृष्ठ",
        ],
        "startswith": ["खड़ी_", "पृष्ठ_"],
        "endswith": ["पिक्सेल"],
    },
    "map-bms": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "mdf": {
        "keywords": [
            "вить_кедь",
            "безрамки",
            "сверху",
            "сверхусправа",
            "кундовтомо",
            "без",
            "кенжешка",
            "вейкеяк_арась",
            "куншкасо",
            "текст-сверху",
            "керш_кедь",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "текст-верькс",
            "справа",
            "верькс",
            "посередине",
            "алкс",
            "текст-алкс",
            "миниатюра",
            "текст-снизу",
            "куншка",
            "снизу",
            "основание",
            "граница",
            "под",
            "кундсо",
            "над",
        ],
        "params": [
            "миниатюра",
            "страница",
            "сверхусправа",
            "лопа",
            "ссылка",
            "мини",
            "альт",
        ],
        "startswith": ["сверхусправа ", "лопа_", "страница_", "страница "],
        "endswith": ["пкс"],
    },
    "mg": {
        "keywords": [
            "ankavanana",
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "ankavia",
            "encadré",
            "encadre",
            "haut-txt",
            "anivo",
            "milieu",
            "haut",
            "vignette",
            "ambany",
            "néant",
            "non_encadre",
            "bordure",
            "tsymisy",
            "ind",
            "cadre",
            "anivony",
            "indice",
            "ambony",
            "haut-texte",
            "exposant",
            "sisiny",
            "ligne-de-base",
            "base",
            "ampivoany",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_", "pejy "],
    },
    "mhr": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "йымалне",
            "без",
            "шола",
            "текст-сверху",
            "ӱмбалне",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "покшелне",
            "справа",
            "кӱшычын",
            "посередине",
            "чек",
            "миниатюра",
            "текст-снизу",
            "пурла",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
            "ӱлычын",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "min": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "mk": {
        "keywords": [
            "рамка",
            "д",
            "горениндекс",
            "лево",
            "текст-најдолу",
            "текст-врв",
            "безрамка",
            "најгоре",
            "дно",
            "мини",
            "л",
            "врв",
            "ц",
            "ворамка",
            "граничник",
            "десно",
            "дол",
            "мини-слика",
            "средина",
            "текст-дно",
            "текст-најгоре",
            "долениндекс",
            "исправено",
            "нема",
            "основналинија",
            "најдолу",
            "центар",
            "граница",
            "гор",
            "н",
        ],
        "params": [
            "страница",
            "исправено",
            "јаз",
            "алт",
            "мини-слика",
            "класа",
            "мини",
            "врска",
        ],
        "startswith": ["страница_", "исправено_"],
        "endswith": ["пкс", "п"],
    },
    "ml": {
        "keywords": [
            "വലത്\u200c",
            "ശൂന്യം",
            "ചട്ടത്തിൽ",
            "നടുവിൽ",
            "എഴുത്ത്-താഴെ",
            "ഇടത്ത്\u200c",
            "അതിർവര",
            "കീഴെയെഴുത്ത്",
            "ലഘുചിത്രം",
            "താഴെയുള്ളവര",
            "എഴുത്ത്-മേലെ",
            "നേരേകുത്തനെ",
            "ചട്ടരഹിതം",
            "വലത്ത്\u200c",
            "താഴെ",
            "മദ്ധ്യം",
            "ചട്ടം",
            "മേലേയെഴുത്ത്",
            "ഇടത്\u200c",
            "മേലെ",
            "നടുക്ക്\u200c",
            "ലഘു",
        ],
        "params": [
            "പകരം",
            "ശ്രേണി",
            "ഭാഷ",
            "കണ്ണി",
            "ലഘുചിത്രം",
            "നേരേകുത്തനെ",
            "താൾ",
            "ലഘു",
        ],
        "startswith": ["താൾ_", "നേരേകുത്തനെ_"],
        "endswith": ["ബിന്ദു"],
    },
    "mr": {
        "keywords": [
            "बूड",
            "मध्यवर्ती",
            "कोणतेचनाही",
            "अधो",
            "मजकुरतळ",
            "विनाचौकट",
            "चौकट",
            "नन्ना",
            "विनाफ़्रेम",
            "मध्य",
            "शीर्ष-मजकूर",
            "तळ",
            "आधाररेषा",
            "उर्ध्व",
            "तळरेषा",
            "इवलेसे",
            "उभा",
            "मजकूर-शीर्ष",
            "फ़्रेम",
            "उजवे",
            "डावे",
            "अत्यूच्च",
            "सीमा",
        ],
        "params": ["पान", "अल्ट", "इवलेसे", "उभा", "दुवा"],
        "startswith": ["पान_", "उभा_"],
        "endswith": ["कणी", "अंश", "पक्ष"],
    },
    "mrj": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "йымалне",
            "без",
            "шола",
            "текст-сверху",
            "ӱмбалне",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "покшелне",
            "справа",
            "кӱшычын",
            "посередине",
            "чек",
            "миниатюра",
            "текст-снизу",
            "пурла",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
            "ӱлычын",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ms": {"keywords": ["bingkai", "tiada", "tengah", "kiri", "kanan"]},
    "mt": {
        "keywords": [
            "burdura",
            "nofs",
            "wieqaf",
            "fuq",
            "bordura",
            "ċentrali",
            "tilar",
            "bid",
            "bla_tilar",
            "daqsminuri",
            "ċentru",
            "taħt",
            "lemin",
            "tilat",
            "test-fuq",
            "bażi_tal-linja",
            "b'tilar",
            "tajjeb",
            "xellug",
            "minuri",
            "xejn",
            "test-taħt",
        ],
        "params": ["ħolqa", "minuri", "daqsminuri", "wieqaf", "paġna"],
        "startswith": ["paġna ", "wieqaf "],
    },
    "mwl": {
        "keywords": [
            "direita",
            "centro",
            "squierda",
            "comborda",
            "miniaturadaimagem",
            "esquerda",
            "borda",
            "semborda",
            "nanhun",
            "linhadebase",
            "semmoldura",
            "nenhum",
            "superiordireito",
            "acima",
            "commoldura",
            "abaixo",
            "meio",
            "miniatura",
            "dreita",
        ],
        "params": [
            "página",
            "ligação",
            "miniatura",
            "miniaturadaimagem",
            "superiordireito",
        ],
        "startswith": ["superiordireito_", "página_", "página ", "superiordireito "],
    },
    "myv": {
        "keywords": [
            "вить_кедь",
            "безрамки",
            "сверху",
            "сверхусправа",
            "кундовтомо",
            "без",
            "кенжешка",
            "вейкеяк_арась",
            "куншкасо",
            "текст-сверху",
            "керш_кедь",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "текст-верькс",
            "справа",
            "верькс",
            "посередине",
            "алкс",
            "текст-алкс",
            "миниатюра",
            "текст-снизу",
            "куншка",
            "снизу",
            "основание",
            "граница",
            "под",
            "кундсо",
            "над",
        ],
        "params": [
            "миниатюра",
            "страница",
            "сверхусправа",
            "лопа",
            "ссылка",
            "мини",
            "альт",
        ],
        "startswith": ["сверхусправа ", "лопа_", "страница_", "страница "],
        "endswith": ["пкс"],
    },
    "mzn": {
        "keywords": [
            "بی_قاب",
            "بندانگشتی",
            "همکف",
            "بالا",
            "متن-پایین",
            "قاب",
            "چپ",
            "حاشیه",
            "انگشتی",
            "زبر",
            "ایستاده",
            "متن-بالا",
            "میانه",
            "انگشتدان",
            "وسط",
            "بی\u200cقاب",
            "بیقاب",
            "زیر",
            "راست",
            "پایین",
            "هیچ",
        ],
        "params": [
            "کلاس",
            "پیوند",
            "بندانگشتی",
            "انگشتی",
            "صفحه",
            "زبان",
            "جایگزین",
            "انگشتدان",
            "ایستاده",
        ],
        "startswith": ["صفحه_", "ایستاده_"],
        "endswith": ["پیکسل"],
    },
    "nah": {
        "keywords": [
            "centro",
            "borde",
            "mini",
            "nada",
            "centrado",
            "medio",
            "derecha",
            "centrada",
            "centrar",
            "izq",
            "enmarcado",
            "arriba",
            "no",
            "sinmarco",
            "ninguno",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "abajo",
            "sinenmarcar",
            "dcha",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "enmarcada",
        ],
        "params": [
            "página",
            "pagina",
            "vínculo",
            "vinculo",
            "idioma",
            "enlace",
            "miniaturadeimagen",
            "miniatura",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "nap": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "nds": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "keen",
            "merrn",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "rahmt",
            "text-oben",
            "hochkant",
            "links",
            "duum",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "nds-nl": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "tekste-benejen",
            "benejen",
            "mini",
            "kaoderloos",
            "tekste-boven",
            "midden",
            "raand",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "grondliende",
            "gien",
            "esentreerd",
            "geen",
            "omkaderd",
            "gecentreerd",
            "umraand",
            "links",
            "beneden",
            "duum",
            "doem",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "zied",
            "verwiezing",
            "taal",
            "klasse",
            "doemnaegel",
            "rechtop",
            "verwijzing",
            "koppeling",
            "pagina",
            "duumnegel",
        ],
        "startswith": ["zied_", "pagina_", "pagina ", "rechtop"],
    },
    "nia": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "nl": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "midden",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "geen",
            "omkaderd",
            "gecentreerd",
            "links",
            "beneden",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "pagina",
            "taal",
            "klasse",
            "rechtop",
            "verwijzing",
            "koppeling",
        ],
        "startswith": ["pagina_", "rechtop"],
    },
    "nn": {
        "keywords": [
            "rammeløs",
            "sentrer",
            "tekst-bunn",
            "mini",
            "midtstilt",
            "ramme",
            "midt",
            "miniatyr",
            "rammelaus",
            "ramma",
            "grunnlinje",
            "topp",
            "høgre",
            "venstre",
            "portrett",
            "høyre",
            "senter",
            "sentrum",
            "bunn",
            "ingenramme",
            "ingen",
        ],
        "params": ["mini", "lenke", "miniatyr", "side", "lenkje", "portrett", "språk"],
        "startswith": ["portrett_", "side_", "side "],
        "endswith": ["pk"],
    },
    "no": {
        "keywords": [
            "rammeløs",
            "sentrer",
            "tekst-bunn",
            "mini",
            "midtstilt",
            "ramme",
            "midt",
            "miniatyr",
            "rammelaus",
            "grunnlinje",
            "ramma",
            "topp",
            "høgre",
            "venstre",
            "portrett",
            "høyre",
            "senter",
            "sentrum",
            "bunn",
            "ingenramme",
            "ingen",
        ],
        "params": ["lenke", "miniatyr", "side", "lenkje", "språk", "portrett", "mini"],
        "startswith": ["portrett_", "side_", "side "],
        "endswith": ["pk"],
    },
    "nrm": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "oc": {
        "keywords": [
            "baix-text",
            "neant",
            "sense marc",
            "senèstra",
            "bas-tèxte",
            "bas-txt",
            "encadre",
            "haut",
            "néant",
            "redreça",
            "ind",
            "vora",
            "exposant",
            "centré",
            "linha_de_basa",
            "bordadura",
            "encadré",
            "haut-txt",
            "nonrés",
            "gaucha",
            "naut-txt",
            "sens_quadre",
            "cap",
            "indice",
            "haut-texte",
            "ligne-de-base",
            "quadre",
            "exp",
            "centrat",
            "marc",
            "bas-texte",
            "droite",
            "non_encadré",
            "mitan",
            "redreçat",
            "vinheta",
            "vignette",
            "non_encadre",
            "naut",
            "dreta",
            "indici",
            "enquagrat",
            "drecha",
            "dalt",
            "miniatura",
            "base",
            "esquèrra",
            "bas",
            "gauche",
            "sans_cadre",
            "dalt-text",
            "redresse",
            "baix",
            "milieu",
            "bordure",
            "mitjà",
            "línia de base",
            "naut-tèxte",
            "esquerra",
            "cadre",
        ],
        "params": [
            "llengua",
            "ligam",
            "redresse",
            "redreçat",
            "pàgina",
            "vinheta",
            "classe",
            "vignette",
            "idioma",
            "lien",
            "enllaç",
            "langue",
            "miniatura",
        ],
        "startswith": ["redresse_", "redreça ", "pàgina ", "redreça", "redreçat "],
    },
    "olo": {
        "keywords": [
            "pienois",
            "alaindeksi",
            "alas",
            "reunus",
            "ylös",
            "vasen",
            "alhaalla",
            "oikea",
            "pysty",
            "keskellä",
            "kehystetty",
            "keski",
            "kehys",
            "yläindeksi",
            "pienoiskuva",
            "perustaso",
            "ylhäällä",
            "kehyksetön",
            "yläoikea",
            "keskitetty",
            "tyhjä",
        ],
        "params": ["pienois", "sivu", "yläoikea", "pysty", "linkki", "pienoiskuva"],
        "startswith": ["yläoikea_", "pysty_", "sivu_"],
    },
    "or": {
        "keywords": [
            "ବେସଲାଇନ",
            "ବାଆଁ",
            "ଫ୍ରେମନଥିବା",
            "କେନ୍ଦ୍ର",
            "କିଛି_ନୁହେଁ",
            "ଉପର",
            "ଲେଖା-ତଳ",
            "ବର୍ଡର",
            "ଫ୍ରେମକରା",
            "ଲେଖା-ଉପର",
            "ଡାହାଣ",
            "ମଝି",
            "ତଳ",
        ],
        "params": ["ନଖଦେଖଣା", "ଦେଖଣା", "ଲିଙ୍କ"],
        "endswith": ["_ପିକସେଲ"],
    },
    "os": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "астæу",
            "къаддæргонд",
            "галиу",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "под",
            "справа",
            "рахиз",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "æнæ",
            "граница",
            "къаддæр",
            "над",
        ],
        "params": [
            "миниатюра",
            "страница",
            "сверхусправа",
            "ссылка",
            "мини",
            "альт",
            "къаддæр",
            "къаддæргонд",
        ],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "pcd": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "pdc": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "pfl": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "pl": {
        "keywords": [
            "centruj",
            "mały",
            "bezramki",
            "bez_ramki",
            "środek",
            "lewo",
            "góra",
            "ramka",
            "prawo",
            "brak",
            "tło",
            "dół",
        ],
        "params": ["mały", "strona"],
    },
    "pms": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "pnt": {
        "keywords": [
            "άνω",
            "αριστερά",
            "καθόλου",
            "μέσο",
            "μικρογραφία",
            "κάτω",
            "εκθέτης",
            "μινιατούρα",
            "πλαίσιο",
            "χωρίς-πλαίσιο",
            "δείκτης",
            "με-πλαίσιο",
            "κάτω-από-το-κείμενο",
            "δεξιά",
            "κατακόρυφα",
            "πάνω-από-το-κείμενο",
            "κέντρο",
            "γραμμήβάσης",
        ],
        "params": [
            "κατακόρυφα",
            "σελίδα",
            "σύνδεσμος",
            "μινιατούρα",
            "εναλλ.",
            "μικρογραφία",
        ],
        "startswith": ["σελίδα_", "κατακόρυφα_"],
        "endswith": ["εσ"],
    },
    "ps": {"keywords": ["ښي", "هېڅ", "مېنځ،_center", "کيڼ", "بټنوک"]},
    "pt": {
        "keywords": [
            "direita",
            "centro",
            "comborda",
            "miniaturadaimagem",
            "esquerda",
            "borda",
            "semborda",
            "linhadebase",
            "semmoldura",
            "nenhum",
            "superiordireito",
            "acima",
            "commoldura",
            "abaixo",
            "meio",
            "miniatura",
        ],
        "params": [
            "página",
            "ligação",
            "miniatura",
            "miniaturadaimagem",
            "superiordireito",
        ],
        "startswith": ["superiordireito_", "página_", "página ", "superiordireito "],
    },
    "pwn": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "qu": {
        "keywords": [
            "ichuq",
            "ukhupi",
            "centro",
            "borde",
            "lluqi",
            "mini",
            "nada",
            "centrado",
            "hanan",
            "inchuyuq",
            "medio",
            "derecha",
            "centrada",
            "saywa",
            "centrar",
            "izq",
            "manaima",
            "enmarcado",
            "uran",
            "arriba",
            "chawpi",
            "no",
            "sayaq",
            "inchu",
            "sinmarco",
            "tiksisiqi",
            "qillqasiki",
            "sin_embarcar",
            "ninguno",
            "rikchacha",
            "izquierda",
            "der",
            "ninguna",
            "miniaturadeimagen",
            "mana",
            "abajo",
            "sinenmarcar",
            "dcha",
            "qillqahawa",
            "hawa",
            "paña",
            "alliq",
            "inchunnaq",
            "izda",
            "marco",
            "sin_enmarcar",
            "miniatura",
            "sikipi",
            "enmarcada",
        ],
        "params": [
            "página",
            "sayaq",
            "rikchacha",
            "vínculo",
            "idioma",
            "miniaturadeimagen",
            "tinki",
            "pagina",
            "panqa",
            "vinculo",
            "enlace",
            "miniatura",
            "wak",
        ],
        "startswith": ["pagina_", "página_"],
    },
    "rm": {
        "keywords": ["sidretg", "miniatura"],
        "params": ["sidretg", "miniatura"],
        "startswith": ["sidretg_"],
    },
    "rmy": {
        "keywords": [
            "linia_de_bază",
            "faracadru",
            "sus",
            "chenar",
            "nu",
            "text-jos",
            "mini",
            "jos",
            "cadru",
            "dreapta",
            "dreaptasus",
            "mijloc",
            "exponent",
            "indice",
            "text-sus",
            "stanga",
            "miniatura",
            "centru",
        ],
        "params": ["pagina", "dreaptasus", "legătură", "miniatura", "mini"],
        "startswith": ["pagina ", "dreaptasus "],
    },
    "ro": {
        "keywords": [
            "linia_de_bază",
            "faracadru",
            "sus",
            "chenar",
            "nu",
            "text-jos",
            "mini",
            "jos",
            "cadru",
            "dreapta",
            "dreaptasus",
            "mijloc",
            "exponent",
            "indice",
            "text-sus",
            "stanga",
            "miniatura",
            "centru",
        ],
        "params": ["pagina", "dreaptasus", "legătură", "miniatura", "mini"],
        "startswith": ["pagina ", "dreaptasus "],
    },
    "roa-rup": {
        "keywords": [
            "linia_de_bază",
            "faracadru",
            "sus",
            "chenar",
            "nu",
            "text-jos",
            "mini",
            "jos",
            "cadru",
            "dreapta",
            "dreaptasus",
            "mijloc",
            "exponent",
            "indice",
            "text-sus",
            "stanga",
            "miniatura",
            "centru",
        ],
        "params": ["pagina", "dreaptasus", "legătură", "miniatura", "mini"],
        "startswith": ["pagina ", "dreaptasus "],
    },
    "roa-tara": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "ru": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "rue": {
        "keywords": [
            "рамка",
            "обрамити",
            "безрамки",
            "під",
            "сверху",
            "праворуч",
            "сверхусправа",
            "без",
            "межа",
            "текст-сверху",
            "границя",
            "основа",
            "мини",
            "обрамить",
            "центр",
            "текст-зверху",
            "слева",
            "міні",
            "зверхуправоруч",
            "справа",
            "посередині",
            "посередине",
            "знизу",
            "миниатюра",
            "текст-снизу",
            "ліворуч",
            "снизу",
            "основание",
            "зверху",
            "текст-знизу",
            "граница",
            "мініатюра",
            "под",
            "над",
        ],
        "params": [
            "сверхусправа",
            "ссылка",
            "посилання",
            "мини",
            "альт",
            "міні",
            "зверхуправоруч",
            "миниатюра",
            "страница",
            "сторінка",
            "мініатюра",
        ],
        "startswith": ["сверхусправа ", "сторінка_", "зверхуправоруч_", "страница "],
        "endswith": ["пкс"],
    },
    "sa": {
        "keywords": [
            "पाठ्य-शीर्षम्",
            "फ्रेमहीन",
            "अधः",
            "दक्षिणत",
            "दक्षिणतः",
            "अङ्गुष्ठ",
            "पाठ्य-अधः",
            "सङ्कुचितचित्र",
            "लघुत्तम",
            "उन्नत",
            "फ्रेम",
            "पाठ्य-अग्र",
            "केन्द्रम्",
            "शीर्ष",
            "आबन्धः",
            "सीमा",
            "नैव",
            "निराबन्ध",
            "आधाररेखा",
            "अग्र",
            "विषये",
            "बॉर्डर",
            "अधस",
            "खड़ी",
            "शीर्षम्",
            "अंगूठाकार",
            "पाठ-तल",
            "अङ्गुष्ठचित्रम्",
            "किनारा",
            "केन्द्र",
            "बाएं",
            "तल",
            "आबन्ध",
            "फ़्रेमहीन",
            "मध्य",
            "केंद्र",
            "पाठ्य-अधस",
            "पाठ-शीर्ष",
            "निराबन्धः",
            "लघुचित्रम्",
            "अंगूठा",
            "पद",
            "मध्ये",
            "बायें",
            "सङ्कुचितचित्रम्",
            "चित्रगमिका",
            "शीर्षदक्षिणतः",
            "वामतः",
            "बाएँ",
            "दाएँ",
            "अति",
            "दायें",
            "किमपि_न",
            "आधार_रेखा",
            "कोई_नहीं",
            "दाएं",
            "मूर्ध",
            "फ़्रेम",
            "तीव्र",
            "केंद्रित",
            "केन्द्रित",
            "चित्रसूचकम्",
            "शीर्षसङ्ख्या",
        ],
        "params": [
            "चित्रपाठ्यम्",
            "शीर्षदक्षिणतः",
            "कड़ी",
            "वर्ग",
            "चित्रपरिसन्धिः",
            "अङ्गुष्ठ",
            "चित्रपृष्ठम्",
            "पृष्ठ",
            "सङ्कुचितचित्र",
            "उन्नत",
            "खड़ी",
            "लघुचित्रम्",
            "लघुत्तमचित्र",
            "विकल्प",
            "अंगूठाकार",
            "अंगूठा",
            "भाषा",
            "सम्बद्धं",
            "पाठ",
            "सङ्कुचितचित्रम्",
            "अङ्गुष्ठचित्रम्",
        ],
        "startswith": [
            "उन्नत ",
            "खड़ी_",
            "शीर्षदक्षिणतः_",
            "पृष्ठ_",
            "पृष्ठ ",
            "चित्रपृष्ठम्_",
        ],
        "endswith": ["पिट", "पिक्सेल", "अणवः"],
    },
    "sah": {
        "keywords": [
            "үөһээуҥа",
            "орто",
            "безрамки",
            "сверху",
            "сверхусправа",
            "кирбии",
            "без",
            "раамката_суох",
            "олох-дьураа",
            "раамка",
            "хаҥас",
            "текст-сверху",
            "алын",
            "мини",
            "ойуучаан",
            "обрамить",
            "центр",
            "слева",
            "ортотунан",
            "уҥа",
            "справа",
            "аллараа",
            "посередине",
            "үрүт",
            "миниатюра",
            "текст-снизу",
            "үрдүнэн",
            "тиэкис-үрдүгэр",
            "снизу",
            "основание",
            "суох",
            "тиэкис-аллара",
            "граница",
            "под",
            "над",
        ],
        "params": [
            "үөһээуҥа",
            "сирэй",
            "сверхусправа",
            "ссылка",
            "сигэ",
            "мини",
            "ойуучаан",
            "альт",
            "миниатюра",
            "страница",
        ],
        "startswith": ["сверхусправа ", "сирэй ", "үөһээуҥа ", "страница "],
        "endswith": ["пкс"],
    },
    "scn": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "sd": {
        "keywords": ["ساڄو", "وچ", "مرڪز", "سِرُ", "تَرُ", "کاٻو", "ڪجهنه"],
        "endswith": [" عڪسلون"],
    },
    "se": {
        "keywords": [
            "pienois",
            "rammeløs",
            "alaindeksi",
            "alas",
            "gasku",
            "olgeš",
            "sentrer",
            "tekst-bunn",
            "mini",
            "gurut",
            "reunus",
            "midtstilt",
            "ramme",
            "ylös",
            "midt",
            "miniatyr",
            "vasen",
            "alhaalla",
            "oikea",
            "pysty",
            "grunnlinje",
            "keskellä",
            "kehystetty",
            "keski",
            "topp",
            "kehys",
            "yläindeksi",
            "venstre",
            "pienoiskuva",
            "perustaso",
            "ylhäällä",
            "portrett",
            "høyre",
            "senter",
            "kehyksetön",
            "bunn",
            "yläoikea",
            "keskitetty",
            "tyhjä",
            "ingenramme",
            "ingen",
        ],
        "params": [
            "pienois",
            "mini",
            "miniatyr",
            "liŋka",
            "pysty",
            "side",
            "lenke",
            "pienoiskuva",
            "portrett",
            "sivu",
            "yläoikea",
            "linkki",
        ],
        "startswith": ["yläoikea_", "pysty_", "portrett_", "side ", "sivu_"],
    },
    "sg": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "sh": {
        "keywords": [
            "lijevo",
            "vrh",
            "levo",
            "pod",
            "sredina",
            "ništa",
            "natpis",
            "dno",
            "mini",
            "osnovnacrta",
            "odjeljak",
            "ivica",
            "centar",
            "bezokvira",
            "granica",
            "okvir",
            "pola",
            "tekst-vrh",
            "vrhteksta",
            "minijatura",
            "umanjeno",
            "potpis",
            "pocetna_linija",
            "na_gore",
            "početna_linija",
            "bez_rama",
            "d",
            "uspravno",
            "dugme",
            "sredinateksta",
            "c",
            "desno",
            "ram",
            "bezrama",
            "bez_okvira",
            "središte",
            "tekst_vrh",
            "ind",
            "eks",
            "osnova",
            "vrh_teksta",
            "sredina_teksta",
            "bez",
            "tekst-dugme",
            "tekst-dno",
            "l",
            "obrub",
            "n",
        ],
        "params": [
            "poveznica",
            "mini",
            "jezik",
            "minijatura",
            "umanjeno",
            "na_gore",
            "uspravno",
            "veza",
            "stranica",
            "strana",
        ],
        "startswith": [
            "stranica_",
            "uspravno ",
            "stranica ",
            "strana_",
            "na_gore_",
            "uspravno_",
        ],
        "endswith": ["piskel", "p", "piksel"],
    },
    "shi": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "si": {
        "keywords": ["දකුණ", "දාරය", "මධ්\u200dයය", "උප", "නොමැත", "වම", "මැද"],
        "endswith": ["පික්"],
    },
    "sk": {
        "keywords": [
            "stred",
            "náhled",
            "žádné",
            "střed",
            "žiadny",
            "vpravo",
            "náhľad",
            "bezrámu",
            "náhľadobrázka",
            "vľavo",
            "rám",
            "okraj",
            "vlevo",
        ],
        "params": ["náhled", "odkaz", "strana", "jazyk", "třída"],
        "startswith": ["strana_"],
        "endswith": ["bod", "pixelů"],
    },
    "sl": {
        "keywords": [
            "vrh",
            "nadpisano",
            "levo",
            "pod",
            "sredina",
            "dno",
            "nad",
            "dno-besedila",
            "brezokvirja",
            "vrh-besedila",
            "okvir",
            "okvirjeno",
            "desno",
            "obroba",
            "zgorajdesno",
            "sličica",
            "sredinsko",
            "brez",
            "podpisano",
        ],
        "params": ["zgorajdesno", "sličica", "stran"],
        "startswith": ["zgorajdesno ", "m_stran "],
        "endswith": ["_pik"],
    },
    "smn": {
        "keywords": [
            "pienois",
            "alaindeksi",
            "alas",
            "reunus",
            "ylös",
            "vasen",
            "alhaalla",
            "oikea",
            "pysty",
            "keskellä",
            "kehystetty",
            "keski",
            "kehys",
            "yläindeksi",
            "pienoiskuva",
            "perustaso",
            "ylhäällä",
            "kehyksetön",
            "yläoikea",
            "keskitetty",
            "tyhjä",
        ],
        "params": ["pienois", "sivu", "yläoikea", "pysty", "linkki", "pienoiskuva"],
        "startswith": ["yläoikea_", "pysty_", "sivu_"],
    },
    "sq": {
        "keywords": [
            "pa_kornizë",
            "pamje",
            "qendrore",
            "djathtas",
            "s'ka",
            "lartdjathtas",
            "linjabazë",
            "fund",
            "parapamje",
            "nën",
            "vertikale",
            "majtas",
            "kornizë",
            "i_kornizuar",
            "mes",
            "qendër",
            "asnjë",
            "tekst-majë",
            "tekst-fund",
            "tekst-top",
            "kufi",
        ],
        "params": [
            "vertikale",
            "faqja",
            "parapamje",
            "lidhje",
            "lidhja",
            "pamje",
            "lartdjathtas",
        ],
        "startswith": ["lartdjathtas ", "faqja ", "vertikale "],
    },
    "sr": {
        "keywords": [
            "врх_текста",
            "д",
            "усправно",
            "оивичено",
            "без",
            "срединатекста",
            "ивица",
            "врхтекста",
            "оквир",
            "лево",
            "супер",
            "дно",
            "основа",
            "без_рама",
            "мини",
            "л",
            "безоквира",
            "ц",
            "врх",
            "десно",
            "c",
            "безрама",
            "средина",
            "умањено",
            "рам",
            "центар",
            "средина_текста",
            "под",
            "без_оквира",
            "н",
        ],
        "params": ["веза", "страница", "усправно", "алт", "страна", "мини", "умањено"],
        "startswith": ["страница_", "страна_", "усправно_"],
        "endswith": ["p", "п", "пискел"],
    },
    "srn": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "midden",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "geen",
            "omkaderd",
            "gecentreerd",
            "links",
            "beneden",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "pagina",
            "taal",
            "klasse",
            "rechtop",
            "verwijzing",
            "koppeling",
        ],
        "startswith": ["pagina_", "rechtop"],
    },
    "stq": {
        "keywords": [
            "ohne",
            "rahmenlos",
            "rand",
            "oben",
            "rechts",
            "zentriert",
            "mini",
            "text-unten",
            "mitte",
            "tief",
            "unten",
            "gerahmt",
            "tiefgestellt",
            "hoch",
            "miniatur",
            "grundlinie",
            "text-oben",
            "hochkant",
            "links",
            "hochgestellt",
        ],
        "params": [
            "verweis",
            "seite",
            "klasse",
            "sprache",
            "hochkant",
            "miniatur",
            "mini",
            "alternativtext",
        ],
        "startswith": ["seite_", "seite ", "hochkant ", "hochkant_"],
    },
    "su": {
        "keywords": [
            "jmpl",
            "bingkai",
            "tepi",
            "tegak",
            "atek",
            "mini",
            "bawah",
            "lurus",
            "gada",
            "ka",
            "garis_dasar",
            "batek",
            "nirbing",
            "bawah-teks",
            "pus",
            "tanpa",
            "atas",
            "tengah",
            "atas-teks",
            "nir",
            "miniatur",
            "ki",
            "tanpabingkai",
            "bing",
            "pusat",
            "jempol",
            "upa",
            "batas",
            "kiri",
            "kanan",
        ],
        "params": [
            "jmpl",
            "hal",
            "halaman",
            "tegak",
            "al",
            "mini",
            "pranala",
            "lurus",
            "alternatif",
            "miniatur",
            "pra",
            "jempol",
            "bhs",
        ],
        "startswith": ["hal_", "lurus_", "halaman_", "tegak_"],
    },
    "sv": {
        "keywords": [
            "höger",
            "upp",
            "mini",
            "miniatyr",
            "ned",
            "vänster",
            "text-botten",
            "kantlinje",
            "text-topp",
            "topp",
            "ram",
            "centrerad",
            "inramad",
            "botten",
            "stående",
            "ramlös",
            "baslinje",
            "mitten",
            "ingen",
        ],
        "params": ["länk", "miniatyr", "sida", "stående", "mini"],
        "startswith": ["sida ", "stående "],
    },
    "szl": {
        "keywords": [
            "centruj",
            "mały",
            "bezramki",
            "bez_ramki",
            "środek",
            "lewo",
            "góra",
            "ramka",
            "prawo",
            "brak",
            "tło",
            "dół",
        ],
        "params": ["mały", "strona"],
    },
    "szy": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "ta": {
        "keywords": [
            "கீழ்",
            "மையம்",
            "வலது",
            "மத்தியில்",
            "மேல்",
            "இடது",
            "ஒன்றுமில்லை",
        ]
    },
    "tay": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "te": {"keywords": ["కుడి", "ఎడమ"]},
    "tet": {
        "keywords": [
            "direita",
            "centro",
            "comborda",
            "miniaturadaimagem",
            "esquerda",
            "borda",
            "semborda",
            "linhadebase",
            "semmoldura",
            "nenhum",
            "superiordireito",
            "acima",
            "commoldura",
            "abaixo",
            "meio",
            "miniatura",
        ],
        "params": [
            "página",
            "ligação",
            "miniatura",
            "miniaturadaimagem",
            "superiordireito",
        ],
        "startswith": ["superiordireito_", "página_", "página ", "superiordireito "],
    },
    "tg": {
        "keywords": [
            "чаҳорчӯба",
            "рамка",
            "аз_боло",
            "чап",
            "бе_чаҳорчӯба",
            "асос",
            "марказ",
            "болорост",
            "бе",
            "мини",
            "боло",
            "дарпоён",
            "бе_рамка",
            "миниатюра",
            "сарҳад",
            "дарбайн",
            "матнпоён",
            "поён",
            "матнболо",
            "рост",
        ],
        "params": ["пайванд", "миниатюра", "алт", "мини", "саҳифа", "боло_рост"],
        "startswith": ["болорост ", "саҳифа "],
        "endswith": ["пкс"],
    },
    "tr": {
        "keywords": [
            "sınır",
            "çerçevesiz",
            "tavan",
            "orta",
            "tabançizgisi",
            "merkez",
            "metin-tavan",
            "dikey",
            "sol",
            "üst",
            "metin-taban",
            "sağ",
            "çerçeveli",
            "tepe",
            "altçizgi",
            "üs",
            "çerçeve",
            "alt",
            "taban",
            "metin-tepe",
            "yok",
            "küçükresim",
            "küçük",
        ],
        "params": ["bağlantı", "dikey", "sayfa", "küçük", "küçükresim", "sınıf"],
        "startswith": ["dikey ", "sayfa "],
        "endswith": ["pik", "piksel"],
    },
    "trv": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "tt": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "юк",
            "үзәк",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "сулда",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "уңда",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "ty": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "tyv": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "оң",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "төп",
            "солагай",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "udm": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "uk": {
        "keywords": [
            "рамка",
            "обрамити",
            "безрамки",
            "під",
            "сверху",
            "праворуч",
            "сверхусправа",
            "без",
            "межа",
            "текст-сверху",
            "границя",
            "основа",
            "мини",
            "обрамить",
            "центр",
            "текст-зверху",
            "слева",
            "міні",
            "зверхуправоруч",
            "справа",
            "посередині",
            "посередине",
            "знизу",
            "миниатюра",
            "текст-снизу",
            "ліворуч",
            "снизу",
            "основание",
            "зверху",
            "текст-знизу",
            "граница",
            "мініатюра",
            "под",
            "над",
        ],
        "params": [
            "сверхусправа",
            "ссылка",
            "посилання",
            "мини",
            "альт",
            "міні",
            "зверхуправоруч",
            "миниатюра",
            "страница",
            "сторінка",
            "мініатюра",
        ],
        "startswith": ["сверхусправа ", "сторінка_", "зверхуправоруч_", "страница "],
        "endswith": ["пкс"],
    },
    "ur": {
        "keywords": [
            "بالا",
            "بدون",
            "زیریں",
            "چوکھٹا",
            "بائیں",
            "حدود",
            "بدون_چوکھٹا",
            "متن-زیریں",
            "زبر",
            "خط_اساسی",
            "تصغیر",
            "بغیر",
            "متن-بالا",
            "درمیان",
            "دائیں",
            "وسط",
            "ایستادہ",
            "زیر",
        ],
        "params": ["متبادل", "لنک", "تصغیر", "ایستادہ", "ربط", "زبان", "درجہ", "صفحہ"],
        "startswith": ["ایستادہ_", "صفحہ_"],
        "endswith": ["پکسل", "پک"],
    },
    "vec": {
        "keywords": [
            "sinistra",
            "centro",
            "nessuno",
            "testo-sotto",
            "testo-sopra",
            "min",
            "verticale",
            "sotto",
            "destra",
            "originale",
            "bordo",
            "riquadrato",
            "incorniciato",
            "metà",
            "pedice",
            "sopra",
            "senza_cornice",
            "miniatura",
        ],
        "params": ["min", "verticale", "pagina", "miniatura"],
        "startswith": ["pagina_", "verticale_"],
    },
    "vep": {
        "keywords": [
            "keskel",
            "ääris",
            "pisipilt",
            "vasakul",
            "hura",
            "eile",
            "tühi",
            "pisi",
            "raam",
            "kesk",
            "röun",
            "oiged",
            "püsti",
            "üläh",
            "ala",
            "raamita",
            "paremal",
        ],
        "params": ["lehekülg", "pisi", "püsti", "pisipilt", "keel"],
        "startswith": ["lehekülg_"],
        "endswith": ["piks"],
    },
    "vi": {
        "keywords": [
            "trên",
            "giữa",
            "đứng",
            "viền",
            "trái",
            "chỉ-số-trên",
            "chỉ-số-dưới",
            "không",
            "khung",
            "phải",
            "chân-chữ",
            "nửa-chiều-cao",
            "dưới",
            "không_khung",
            "trên-chữ",
            "dưới-chữ",
            "nhỏ",
        ],
        "params": [
            "thay_thế",
            "thế",
            "tiếng",
            "đứng",
            "trang",
            "liên_kết",
            "lớp",
            "nhỏ",
            "ngôn_ngữ",
        ],
        "startswith": ["đứng_", "trang_"],
    },
    "vls": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "midden",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "geen",
            "omkaderd",
            "gecentreerd",
            "links",
            "beneden",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "pagina",
            "taal",
            "klasse",
            "rechtop",
            "verwijzing",
            "koppeling",
        ],
        "startswith": ["pagina_", "rechtop"],
    },
    "wa": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "wo": {
        "keywords": [
            "neant",
            "bas",
            "exp",
            "gauche",
            "bas-texte",
            "droite",
            "non_encadré",
            "sans_cadre",
            "bas-txt",
            "redresse",
            "encadré",
            "encadre",
            "haut-txt",
            "milieu",
            "haut",
            "vignette",
            "non_encadre",
            "néant",
            "bordure",
            "ind",
            "indice",
            "haut-texte",
            "exposant",
            "ligne-de-base",
            "base",
            "cadre",
            "centré",
        ],
        "params": ["vignette", "redresse", "lien", "langue", "classe"],
        "startswith": ["redresse_"],
    },
    "wuu": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "顶部",
            "垂直置頂",
            "置中",
            "文字置頂",
            "基线",
            "边框",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "链接",
            "連結",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "xal": {
        "keywords": [
            "безрамки",
            "сверху",
            "сверхусправа",
            "без",
            "текст-сверху",
            "мини",
            "обрамить",
            "центр",
            "слева",
            "справа",
            "посередине",
            "миниатюра",
            "текст-снизу",
            "снизу",
            "основание",
            "граница",
            "под",
            "над",
        ],
        "params": ["миниатюра", "страница", "сверхусправа", "ссылка", "мини", "альт"],
        "startswith": ["сверхусправа ", "страница "],
        "endswith": ["пкс"],
    },
    "xmf": {
        "keywords": [
            "მარჯვნივ",
            "ცენტრი",
            "არა",
            "მარცხნივ",
            "საზღვარი",
            "მინი",
            "ქვედა",
            "ცეროდენა",
            "მინიატიურა",
            "მინიასლი",
            "ზედა",
            "შუა",
            "ცენტრში",
        ],
        "params": ["გვერდი", "მინიატიურა", "ალტ", "ბმული", "მინი"],
        "startswith": ["გვერდი_"],
        "endswith": ["პქ"],
    },
    "yi": {
        "keywords": [
            "אויבן",
            "ממוסגר",
            "אונטער",
            "גבולות",
            "למטה",
            "ללא מסגרת",
            "צענטער",
            "ללא",
            "גבול",
            "אונטן",
            "אינמיטן",
            "מסגרת",
            "קליין",
            "לינקס",
            "אן",
            "ממוזער",
            "באמצע",
            "רעכטס",
            "עילי",
            "בתחתית הטקסט",
            "לא ממוסגר",
            "איבער",
            "ימין למעלה",
            "ימין",
            "בראש הטקסט",
            "מרכז",
            "שורת הבסיס",
            "שמאל",
            "למעלה",
            "תחתי",
        ],
        "params": [
            "טעקסט",
            "ממוזער",
            "טקסט",
            "קליין",
            "דף",
            "לינק",
            "ימין למעלה",
            "קישור",
        ],
        "startswith": ["דף ", "ימין למעלה "],
        "endswith": [" פיקסלים", "פיקס"],
    },
    "za": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "顶部",
            "垂直置頂",
            "置中",
            "文字置頂",
            "基线",
            "边框",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "链接",
            "連結",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "zea": {
        "keywords": [
            "rand",
            "miniatuur",
            "tekst-beneden",
            "rechts",
            "midden",
            "boven",
            "tekst-boven",
            "kaderloos",
            "rechtop",
            "geen",
            "omkaderd",
            "gecentreerd",
            "links",
            "beneden",
            "grondlijn",
        ],
        "params": [
            "miniatuur",
            "pagina",
            "taal",
            "klasse",
            "rechtop",
            "verwijzing",
            "koppeling",
        ],
        "startswith": ["pagina_", "rechtop"],
    },
    "zh": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "顶部",
            "垂直置頂",
            "置中",
            "文字置頂",
            "基线",
            "边框",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "链接",
            "連結",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "zh-classical": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
    "zh-min-nan": {
        "keywords": [
            "替代文字",
            "底部",
            "文字顶部",
            "居中",
            "中间",
            "缩略图",
            "有框",
            "垂直置頂",
            "顶部",
            "置中",
            "文字置頂",
            "边框",
            "基线",
            "超",
            "子",
            "邊框",
            "无",
            "文字置底",
            "垂直置中",
            "無框",
            "右",
            "右上",
            "無",
            "左",
            "縮圖",
            "垂直置底",
            "文字底部",
            "下標",
            "无框",
            "上標",
        ],
        "params": [
            "缩略图",
            "連結",
            "链接",
            "右上",
            "替代",
            "語言",
            "替代文本",
            "语言",
            "頁",
            "类",
            "縮圖",
            "页数",
            "類別",
        ],
        "startswith": ["右上"],
        "endswith": ["像素", "页", "頁"],
    },
}
