import { DockerImageAssetLocation, DockerImageAssetSource, FileAssetLocation, FileAssetSource } from '../assets';
import { ISynthesisSession } from '../construct-compat';
import { Stack } from '../stack';
import { StackSynthesizer } from './stack-synthesizer';
export declare const BOOTSTRAP_QUALIFIER_CONTEXT = "@aws-cdk/core:bootstrapQualifier";
/**
 * Configuration properties for DefaultStackSynthesizer.
 *
 * @stability stable
 */
export interface DefaultStackSynthesizerProps {
    /**
     * Name of the S3 bucket to hold file assets.
     *
     * You must supply this if you have given a non-standard name to the staging bucket.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME
     * @stability stable
     */
    readonly fileAssetsBucketName?: string;
    /**
     * Name of the ECR repository to hold Docker Image assets.
     *
     * You must supply this if you have given a non-standard name to the ECR repository.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME
     * @stability stable
     */
    readonly imageAssetsRepositoryName?: string;
    /**
     * The role to use to publish file assets to the S3 bucket in this environment.
     *
     * You must supply this if you have given a non-standard name to the publishing role.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN
     * @stability stable
     */
    readonly fileAssetPublishingRoleArn?: string;
    /**
     * External ID to use when assuming role for file asset publishing.
     *
     * @default - No external ID
     * @stability stable
     */
    readonly fileAssetPublishingExternalId?: string;
    /**
     * The role to use to publish image assets to the ECR repository in this environment.
     *
     * You must supply this if you have given a non-standard name to the publishing role.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN
     * @stability stable
     */
    readonly imageAssetPublishingRoleArn?: string;
    /**
     * The role to use to look up values from the target AWS account during synthesis.
     *
     * @default - None
     * @stability stable
     */
    readonly lookupRoleArn?: string;
    /**
     * External ID to use when assuming role for image asset publishing.
     *
     * @default - No external ID
     * @stability stable
     */
    readonly imageAssetPublishingExternalId?: string;
    /**
     * External ID to use when assuming role for cloudformation deployments.
     *
     * @default - No external ID
     * @stability stable
     */
    readonly deployRoleExternalId?: string;
    /**
     * The role to assume to initiate a deployment in this environment.
     *
     * You must supply this if you have given a non-standard name to the publishing role.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN
     * @stability stable
     */
    readonly deployRoleArn?: string;
    /**
     * The role CloudFormation will assume when deploying the Stack.
     *
     * You must supply this if you have given a non-standard name to the execution role.
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN
     * @stability stable
     */
    readonly cloudFormationExecutionRole?: string;
    /**
     * (deprecated) Name of the CloudFormation Export with the asset key name.
     *
     * You must supply this if you have given a non-standard name to the KMS key export
     *
     * The placeholders `${Qualifier}`, `${AWS::AccountId}` and `${AWS::Region}` will
     * be replaced with the values of qualifier and the stack's account and region,
     * respectively.
     *
     * @default DefaultStackSynthesizer.DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME
     * @deprecated This property is not used anymore
     */
    readonly fileAssetKeyArnExportName?: string;
    /**
     * Qualifier to disambiguate multiple environments in the same account.
     *
     * You can use this and leave the other naming properties empty if you have deployed
     * the bootstrap environment with standard names but only differnet qualifiers.
     *
     * @default - Value of context key '
     * @stability stable
     * @aws-cdk /core:bootstrapQualifier' if set, otherwise `DefaultStackSynthesizer.DEFAULT_QUALIFIER`
     */
    readonly qualifier?: string;
    /**
     * Whether to add a Rule to the stack template verifying the bootstrap stack version.
     *
     * This generally should be left set to `true`, unless you explicitly
     * want to be able to deploy to an unbootstrapped environment.
     *
     * @default true
     * @stability stable
     */
    readonly generateBootstrapVersionRule?: boolean;
    /**
     * bucketPrefix to use while storing S3 Assets.
     *
     * @default - DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX
     * @stability stable
     */
    readonly bucketPrefix?: string;
    /**
     * Bootstrap stack version SSM parameter.
     *
     * The placeholder `${Qualifier}` will be replaced with the value of qualifier.
     *
     * @default DefaultStackSynthesizer.DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER
     * @stability stable
     */
    readonly bootstrapStackVersionSsmParameter?: string;
}
/**
 * Uses conventionally named roles and reify asset storage locations.
 *
 * This synthesizer is the only StackSynthesizer that generates
 * an asset manifest, and is required to deploy CDK applications using the
 * `@aws-cdk/app-delivery` CI/CD library.
 *
 * Requires the environment to have been bootstrapped with Bootstrap Stack V2.
 *
 * @stability stable
 */
export declare class DefaultStackSynthesizer extends StackSynthesizer {
    private readonly props;
    /**
     * Default ARN qualifier.
     *
     * @stability stable
     */
    static readonly DEFAULT_QUALIFIER = "hnb659fds";
    /**
     * Default CloudFormation role ARN.
     *
     * @stability stable
     */
    static readonly DEFAULT_CLOUDFORMATION_ROLE_ARN = "arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-cfn-exec-role-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default deploy role ARN.
     *
     * @stability stable
     */
    static readonly DEFAULT_DEPLOY_ROLE_ARN = "arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-deploy-role-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default asset publishing role ARN for file (S3) assets.
     *
     * @stability stable
     */
    static readonly DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN = "arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-file-publishing-role-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default asset publishing role ARN for image (ECR) assets.
     *
     * @stability stable
     */
    static readonly DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN = "arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-image-publishing-role-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default lookup role ARN for missing values.
     *
     * @stability stable
     */
    static readonly DEFAULT_LOOKUP_ROLE_ARN = "arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-lookup-role-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default image assets repository name.
     *
     * @stability stable
     */
    static readonly DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME = "cdk-${Qualifier}-container-assets-${AWS::AccountId}-${AWS::Region}";
    /**
     * Default file assets bucket name.
     *
     * @stability stable
     */
    static readonly DEFAULT_FILE_ASSETS_BUCKET_NAME = "cdk-${Qualifier}-assets-${AWS::AccountId}-${AWS::Region}";
    /**
     * Name of the CloudFormation Export with the asset key name.
     *
     * @stability stable
     */
    static readonly DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME = "CdkBootstrap-${Qualifier}-FileAssetKeyArn";
    /**
     * Default file asset prefix.
     *
     * @stability stable
     */
    static readonly DEFAULT_FILE_ASSET_PREFIX = "";
    /**
     * Default bootstrap stack version SSM parameter.
     *
     * @stability stable
     */
    static readonly DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER = "/cdk-bootstrap/${Qualifier}/version";
    private _stack?;
    private bucketName?;
    private repositoryName?;
    private _deployRoleArn?;
    private _cloudFormationExecutionRoleArn?;
    private fileAssetPublishingRoleArn?;
    private imageAssetPublishingRoleArn?;
    private lookupRoleArn?;
    private qualifier?;
    private bucketPrefix?;
    private bootstrapStackVersionSsmParameter?;
    private readonly files;
    private readonly dockerImages;
    /**
     * @stability stable
     */
    constructor(props?: DefaultStackSynthesizerProps);
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack: Stack): void;
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset: FileAssetSource): FileAssetLocation;
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset: DockerImageAssetSource): DockerImageAssetLocation;
    /**
     * Have the stack write out its template.
     *
     * @stability stable
     */
    protected synthesizeStackTemplate(stack: Stack, session: ISynthesisSession): void;
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session: ISynthesisSession): void;
    /**
     * Returns the ARN of the deploy Role.
     *
     * @stability stable
     */
    get deployRoleArn(): string;
    /**
     * Returns the ARN of the CFN execution Role.
     *
     * @stability stable
     */
    get cloudFormationExecutionRoleArn(): string;
    /**
     * @stability stable
     */
    protected get stack(): Stack | undefined;
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    private addStackTemplateToAssetManifest;
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    private writeAssetManifest;
    private get manifestEnvName();
}
