"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeployments = exports.replaceEnvPlaceholders = void 0;
const cxapi = require("@aws-cdk/cx-api");
const cdk_assets_1 = require("cdk-assets");
const logging_1 = require("../logging");
const asset_publishing_1 = require("../util/asset-publishing");
const aws_auth_1 = require("./aws-auth");
const deploy_stack_1 = require("./deploy-stack");
const toolkit_info_1 = require("./toolkit-info");
const cloudformation_1 = require("./util/cloudformation");
/**
 * Replace the {ACCOUNT} and {REGION} placeholders in all strings found in a complex object.
 */
async function replaceEnvPlaceholders(object, env, sdkProvider) {
    return cxapi.EnvironmentPlaceholders.replaceAsync(object, {
        accountId: () => Promise.resolve(env.account),
        region: () => Promise.resolve(env.region),
        partition: async () => {
            var _a;
            // There's no good way to get the partition!
            // We should have had it already, except we don't.
            //
            // Best we can do is ask the "base credentials" for this environment for their partition. Cross-partition
            // AssumeRole'ing will never work anyway, so this answer won't be wrong (it will just be slow!)
            return (_a = (await sdkProvider.baseCredentialsPartition(env, aws_auth_1.Mode.ForReading))) !== null && _a !== void 0 ? _a : 'aws';
        },
    });
}
exports.replaceEnvPlaceholders = replaceEnvPlaceholders;
/**
 * Helper class for CloudFormation deployments
 *
 * Looks us the right SDK and Bootstrap stack to deploy a given
 * stack artifact.
 */
class CloudFormationDeployments {
    constructor(props) {
        this.sdkProvider = props.sdkProvider;
    }
    async readCurrentTemplate(stackArtifact) {
        logging_1.debug(`Reading existing template for stack ${stackArtifact.displayName}.`);
        const { stackSdk } = await this.prepareSdkFor(stackArtifact, undefined, aws_auth_1.Mode.ForReading);
        const cfn = stackSdk.cloudFormation();
        const stack = await cloudformation_1.CloudFormationStack.lookup(cfn, stackArtifact.stackName);
        return stack.template();
    }
    async deployStack(options) {
        const { stackSdk, resolvedEnvironment, cloudFormationRoleArn } = await this.prepareSdkFor(options.stack, options.roleArn);
        const toolkitInfo = await toolkit_info_1.ToolkitInfo.lookup(resolvedEnvironment, stackSdk, options.toolkitStackName);
        // Publish any assets before doing the actual deploy
        await this.publishStackAssets(options.stack, toolkitInfo);
        // Do a verification of the bootstrap stack version
        await this.validateBootstrapStackVersion(options.stack.stackName, options.stack.requiresBootstrapStackVersion, options.stack.bootstrapStackVersionSsmParameter, toolkitInfo);
        return deploy_stack_1.deployStack({
            stack: options.stack,
            resolvedEnvironment,
            deployName: options.deployName,
            notificationArns: options.notificationArns,
            quiet: options.quiet,
            sdk: stackSdk,
            sdkProvider: this.sdkProvider,
            roleArn: cloudFormationRoleArn,
            reuseAssets: options.reuseAssets,
            toolkitInfo,
            tags: options.tags,
            execute: options.execute,
            changeSetName: options.changeSetName,
            force: options.force,
            parameters: options.parameters,
            usePreviousParameters: options.usePreviousParameters,
            progress: options.progress,
            ci: options.ci,
        });
    }
    async destroyStack(options) {
        const { stackSdk, cloudFormationRoleArn: roleArn } = await this.prepareSdkFor(options.stack, options.roleArn);
        return deploy_stack_1.destroyStack({
            sdk: stackSdk,
            roleArn,
            stack: options.stack,
            deployName: options.deployName,
            quiet: options.quiet,
        });
    }
    async stackExists(options) {
        var _a;
        const { stackSdk } = await this.prepareSdkFor(options.stack, undefined, aws_auth_1.Mode.ForReading);
        const stack = await cloudformation_1.CloudFormationStack.lookup(stackSdk.cloudFormation(), (_a = options.deployName) !== null && _a !== void 0 ? _a : options.stack.stackName);
        return stack.exists;
    }
    /**
     * Get the environment necessary for touching the given stack
     *
     * Returns the following:
     *
     * - The resolved environment for the stack (no more 'unknown-account/unknown-region')
     * - SDK loaded with the right credentials for calling `CreateChangeSet`.
     * - The Execution Role that should be passed to CloudFormation.
     */
    async prepareSdkFor(stack, roleArn, mode = aws_auth_1.Mode.ForWriting) {
        if (!stack.environment) {
            throw new Error(`The stack ${stack.displayName} does not have an environment`);
        }
        const resolvedEnvironment = await this.sdkProvider.resolveEnvironment(stack.environment);
        // Substitute any placeholders with information about the current environment
        const arns = await replaceEnvPlaceholders({
            assumeRoleArn: stack.assumeRoleArn,
            // Use the override if given, otherwise use the field from the stack
            cloudFormationRoleArn: roleArn !== null && roleArn !== void 0 ? roleArn : stack.cloudFormationExecutionRoleArn,
        }, resolvedEnvironment, this.sdkProvider);
        const stackSdk = await this.sdkProvider.forEnvironment(resolvedEnvironment, mode, {
            assumeRoleArn: arns.assumeRoleArn,
            assumeRoleExternalId: stack.assumeRoleExternalId,
        });
        return {
            stackSdk,
            resolvedEnvironment,
            cloudFormationRoleArn: arns.cloudFormationRoleArn,
        };
    }
    /**
     * Publish all asset manifests that are referenced by the given stack
     */
    async publishStackAssets(stack, toolkitInfo) {
        const stackEnv = await this.sdkProvider.resolveEnvironment(stack.environment);
        const assetArtifacts = stack.dependencies.filter(isAssetManifestArtifact);
        for (const assetArtifact of assetArtifacts) {
            await this.validateBootstrapStackVersion(stack.stackName, assetArtifact.requiresBootstrapStackVersion, assetArtifact.bootstrapStackVersionSsmParameter, toolkitInfo);
            const manifest = cdk_assets_1.AssetManifest.fromFile(assetArtifact.file);
            await asset_publishing_1.publishAssets(manifest, this.sdkProvider, stackEnv);
        }
    }
    /**
     * Validate that the bootstrap stack has the right version for this stack
     */
    async validateBootstrapStackVersion(stackName, requiresBootstrapStackVersion, bootstrapStackVersionSsmParameter, toolkitInfo) {
        if (requiresBootstrapStackVersion === undefined) {
            return;
        }
        try {
            await toolkitInfo.validateVersion(requiresBootstrapStackVersion, bootstrapStackVersionSsmParameter);
        }
        catch (e) {
            throw new Error(`${stackName}: ${e.message}`);
        }
    }
}
exports.CloudFormationDeployments = CloudFormationDeployments;
function isAssetManifestArtifact(art) {
    return art instanceof cxapi.AssetManifestArtifact;
}
//# sourceMappingURL=data:application/json;base64,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