"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudExecutable = void 0;
const fs_1 = require("fs");
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const contextproviders = require("../../context-providers");
const logging_1 = require("../../logging");
const cloud_assembly_1 = require("./cloud-assembly");
/**
 * Represent the Cloud Executable and the synthesis we can do on it
 */
class CloudExecutable {
    constructor(props) {
        this.props = props;
    }
    /**
     * Return whether there is an app command from the configuration
     */
    get hasApp() {
        return !!this.props.configuration.settings.get(['app']);
    }
    /**
     * Synthesize a set of stacks
     */
    async synthesize() {
        if (!this._cloudAssembly) {
            this._cloudAssembly = await this.doSynthesize();
        }
        return this._cloudAssembly;
    }
    async doSynthesize() {
        const trackVersions = this.props.configuration.settings.get(['versionReporting']);
        // We may need to run the cloud executable multiple times in order to satisfy all missing context
        // (When the executable runs, it will tell us about context it wants to use
        // but it missing. We'll then look up the context and run the executable again, and
        // again, until it doesn't complain anymore or we've stopped making progress).
        let previouslyMissingKeys;
        while (true) {
            const assembly = await this.props.synthesizer(this.props.sdkProvider, this.props.configuration);
            if (assembly.manifest.missing && assembly.manifest.missing.length > 0) {
                const missingKeys = missingContextKeys(assembly.manifest.missing);
                if (!this.canLookup) {
                    throw new Error('Context lookups have been disabled. '
                        + 'Make sure all necessary context is already in \'cdk.context.json\' by running \'cdk synth\' on a machine with sufficient AWS credentials and committing the result. '
                        + `Missing context keys: '${Array.from(missingKeys).join(', ')}'`);
                }
                let tryLookup = true;
                if (previouslyMissingKeys && setsEqual(missingKeys, previouslyMissingKeys)) {
                    logging_1.debug('Not making progress trying to resolve environmental context. Giving up.');
                    tryLookup = false;
                }
                previouslyMissingKeys = missingKeys;
                if (tryLookup) {
                    logging_1.debug('Some context information is missing. Fetching...');
                    await contextproviders.provideContextValues(assembly.manifest.missing, this.props.configuration.context, this.props.sdkProvider);
                    // Cache the new context to disk
                    await this.props.configuration.saveContext();
                    // Execute again
                    continue;
                }
            }
            if (trackVersions) {
                // @deprecated(v2): remove this 'if' block and all code referenced by it.
                // This should honestly not be done here. The framework
                // should (and will, shortly) synthesize this information directly into
                // the template. However, in order to support old framework versions
                // that don't synthesize this info yet, we can only remove this code
                // once we break backwards compatibility.
                await this.addMetadataResource(assembly);
            }
            return new cloud_assembly_1.CloudAssembly(assembly);
        }
    }
    /**
     * Modify the templates in the assembly in-place to add metadata resource declarations
     */
    async addMetadataResource(rootAssembly) {
        if (!rootAssembly.runtime) {
            return;
        }
        const modules = formatModules(rootAssembly.runtime);
        await processAssembly(rootAssembly);
        async function processAssembly(assembly) {
            for (const stack of assembly.stacks) {
                await processStack(stack);
            }
            for (const nested of assembly.nestedAssemblies) {
                await processAssembly(nested.nestedAssembly);
            }
        }
        async function processStack(stack) {
            const resourcePresent = stack.environment.region === cxapi.UNKNOWN_REGION
                || region_info_1.RegionInfo.get(stack.environment.region).cdkMetadataResourceAvailable;
            if (!resourcePresent) {
                return;
            }
            if (!stack.template.Resources) {
                stack.template.Resources = {};
            }
            if (stack.template.Resources.CDKMetadata) {
                // Already added by framework, this is expected.
                return;
            }
            stack.template.Resources.CDKMetadata = {
                Type: 'AWS::CDK::Metadata',
                Properties: {
                    Modules: modules,
                },
            };
            if (stack.environment.region === cxapi.UNKNOWN_REGION) {
                stack.template.Conditions = stack.template.Conditions || {};
                const condName = 'CDKMetadataAvailable';
                if (!stack.template.Conditions[condName]) {
                    stack.template.Conditions[condName] = _makeCdkMetadataAvailableCondition();
                    stack.template.Resources.CDKMetadata.Condition = condName;
                }
                else {
                    logging_1.warning(`The stack ${stack.id} already includes a ${condName} condition`);
                }
            }
            // The template has changed in-memory, but the file on disk remains unchanged so far.
            // The CLI *might* later on deploy the in-memory version (if it's <50kB) or use the
            // on-disk version (if it's >50kB).
            //
            // Be sure to flush the changes we just made back to disk. The on-disk format is always
            // JSON.
            await fs_1.promises.writeFile(stack.templateFullPath, JSON.stringify(stack.template, undefined, 2), { encoding: 'utf-8' });
        }
    }
    get canLookup() {
        var _a;
        return !!((_a = this.props.configuration.settings.get(['lookups'])) !== null && _a !== void 0 ? _a : true);
    }
}
exports.CloudExecutable = CloudExecutable;
/**
 * Return all keys of missing context items
 */
function missingContextKeys(missing) {
    return new Set((missing || []).map(m => m.key));
}
function setsEqual(a, b) {
    if (a.size !== b.size) {
        return false;
    }
    for (const x of a) {
        if (!b.has(x)) {
            return false;
        }
    }
    return true;
}
function _makeCdkMetadataAvailableCondition() {
    return _fnOr(region_info_1.RegionInfo.regions
        .filter(ri => ri.cdkMetadataResourceAvailable)
        .map(ri => ({ 'Fn::Equals': [{ Ref: 'AWS::Region' }, ri.name] })));
}
/**
 * This takes a bunch of operands and crafts an `Fn::Or` for those. Funny thing is `Fn::Or` requires
 * at least 2 operands and at most 10 operands, so we have to... do this.
 */
function _fnOr(operands) {
    if (operands.length === 0) {
        throw new Error('Cannot build `Fn::Or` with zero operands!');
    }
    if (operands.length === 1) {
        return operands[0];
    }
    if (operands.length <= 10) {
        return { 'Fn::Or': operands };
    }
    return _fnOr(_inGroupsOf(operands, 10).map(group => _fnOr(group)));
}
function _inGroupsOf(array, maxGroup) {
    const result = new Array();
    for (let i = 0; i < array.length; i += maxGroup) {
        result.push(array.slice(i, i + maxGroup));
    }
    return result;
}
function formatModules(runtime) {
    const modules = new Array();
    // inject toolkit version to list of modules
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const toolkitVersion = require('../../../package.json').version;
    modules.push(`aws-cdk=${toolkitVersion}`);
    for (const key of Object.keys(runtime.libraries).sort()) {
        modules.push(`${key}=${runtime.libraries[key]}`);
    }
    return modules.join(',');
}
//# sourceMappingURL=data:application/json;base64,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