"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedZoneContextProviderPlugin = void 0;
const cxapi = require("@aws-cdk/cx-api");
const api_1 = require("../api");
const logging_1 = require("../logging");
class HostedZoneContextProviderPlugin {
    constructor(aws) {
        this.aws = aws;
    }
    async getValue(args) {
        const account = args.account;
        const region = args.region;
        if (!this.isHostedZoneQuery(args)) {
            throw new Error(`HostedZoneProvider requires domainName property to be set in ${args}`);
        }
        const domainName = args.domainName;
        logging_1.debug(`Reading hosted zone ${account}:${region}:${domainName}`);
        const options = { assumeRoleArn: args.lookupRoleArn };
        const r53 = (await this.aws.forEnvironment(cxapi.EnvironmentUtils.make(account, region), api_1.Mode.ForReading, options)).route53();
        const response = await r53.listHostedZonesByName({ DNSName: domainName }).promise();
        if (!response.HostedZones) {
            throw new Error(`Hosted Zone not found in account ${account}, region ${region}: ${domainName}`);
        }
        const candidateZones = await this.filterZones(r53, response.HostedZones, args);
        if (candidateZones.length !== 1) {
            const filteProps = `dns:${domainName}, privateZone:${args.privateZone}, vpcId:${args.vpcId}`;
            throw new Error(`Found zones: ${JSON.stringify(candidateZones)} for ${filteProps}, but wanted exactly 1 zone`);
        }
        return {
            Id: candidateZones[0].Id,
            Name: candidateZones[0].Name,
        };
    }
    async filterZones(r53, zones, props) {
        let candidates = [];
        const domainName = props.domainName.endsWith('.') ? props.domainName : `${props.domainName}.`;
        logging_1.debug(`Found the following zones ${JSON.stringify(zones)}`);
        candidates = zones.filter(zone => zone.Name === domainName);
        logging_1.debug(`Found the following matched name zones ${JSON.stringify(candidates)}`);
        if (props.privateZone) {
            candidates = candidates.filter(zone => zone.Config && zone.Config.PrivateZone);
        }
        else {
            candidates = candidates.filter(zone => !zone.Config || !zone.Config.PrivateZone);
        }
        if (props.vpcId) {
            const vpcZones = [];
            for (const zone of candidates) {
                const data = await r53.getHostedZone({ Id: zone.Id }).promise();
                if (!data.VPCs) {
                    logging_1.debug(`Expected VPC for private zone but no VPC found ${zone.Id}`);
                    continue;
                }
                if (data.VPCs.map(vpc => vpc.VPCId).includes(props.vpcId)) {
                    vpcZones.push(zone);
                }
            }
            return vpcZones;
        }
        return candidates;
    }
    isHostedZoneQuery(props) {
        return props.domainName !== undefined;
    }
}
exports.HostedZoneContextProviderPlugin = HostedZoneContextProviderPlugin;
//# sourceMappingURL=data:application/json;base64,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